/*
 * Copyright 2010, The Android Open Source Project
 * Copyright (C) 2012 Samsung Electronics. All rights reserved.
 * Copyright (C) 2014 Sony Interactive Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#if ENABLE(DEVICE_ORIENTATION)
#include "DeviceOrientationClientManx.h"

#include "DeviceOrientationController.h"
#include <wtf/MainThread.h>

namespace WebCore {

DeviceOrientationClientManx::DeviceOrientationClientManx()
{
    m_provider = motion::OrientationProvider::create(this);
}

DeviceOrientationClientManx::~DeviceOrientationClientManx()
{
    if (m_provider)
        motion::OrientationProvider::destroy(m_provider);

    cancelCallOnMainThread(handleDidChangeDeviceOrientationOnMainThread, this);
}

void DeviceOrientationClientManx::deviceOrientationControllerDestroyed()
{
    delete this;
}

void DeviceOrientationClientManx::setController(DeviceOrientationController* controller)
{
    m_controller = controller;
}

void DeviceOrientationClientManx::startUpdating()
{
    if (m_provider)
        m_provider->startUpdating();
}

void DeviceOrientationClientManx::stopUpdating()
{
    if (m_provider)
        m_provider->stopUpdating();

    cancelCallOnMainThread(handleDidChangeDeviceOrientationOnMainThread, this);
}

WebCore::DeviceOrientationData* DeviceOrientationClientManx::lastOrientation() const
{
    MutexLocker locker(m_mutex);

    return m_lastOrientation.get();
}

void DeviceOrientationClientManx::onUpdate(motion::OrientationData data)
{
    MutexLocker locker(m_mutex);

    m_lastOrientation = WebCore::DeviceOrientationData::create(
        data.canProvideAlpha, data.alpha,
        data.canProvideBeta, data.beta,
        data.canProvideGamma, data.gamma,
        data.canProvideAbsolute, data.absolute
        );

    callOnMainThread(handleDidChangeDeviceOrientationOnMainThread, this);
}

void DeviceOrientationClientManx::handleDidChangeDeviceOrientationOnMainThread(void* arg)
{
    DeviceOrientationClientManx* that = reinterpret_cast<DeviceOrientationClientManx*>(arg);

    that->m_mutex.lock();
    WTF::RefPtr<WebCore::DeviceOrientationData> data = that->m_lastOrientation;
    that->m_mutex.unlock();
    if (data)
        that->m_controller->didChangeDeviceOrientation(data.get());
}

} // namespace WebCore
#endif
