/*
 * Copyright 2010 Apple Inc. All rights reserved.
 * Copyright (C) 2012 Samsung Electronics. All rights reserved.
 * Copyright (C) 2014 Sony Interactive Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#if ENABLE(DEVICE_ORIENTATION)
#include "DeviceMotionClientManx.h"

#include "DeviceMotionController.h"
#include <wtf/MainThread.h>

namespace WebCore {

DeviceMotionClientManx::DeviceMotionClientManx()
{
    m_provider = motion::MotionProvider::create(this);
}

DeviceMotionClientManx::~DeviceMotionClientManx()
{
    if (m_provider)
        motion::MotionProvider::destroy(m_provider);

    cancelCallOnMainThread(handleDidChangeDeviceMotionOnMainThread, this);
}

void DeviceMotionClientManx::deviceMotionControllerDestroyed()
{
    delete this;
}

void DeviceMotionClientManx::setController(DeviceMotionController* controller)
{
    m_controller = controller;
}

void DeviceMotionClientManx::startUpdating()
{
    if (m_provider)
        m_provider->startUpdating();
}

void DeviceMotionClientManx::stopUpdating()
{
    if (m_provider)
        m_provider->stopUpdating();

    cancelCallOnMainThread(handleDidChangeDeviceMotionOnMainThread, this);
}

DeviceMotionData* DeviceMotionClientManx::lastMotion() const
{
    MutexLocker locker(m_mutex);

    return m_lastMotion.get();
}

void DeviceMotionClientManx::onUpdate(motion::MotionData data)
{
    MutexLocker locker(m_mutex);

    m_lastMotion = WebCore::DeviceMotionData::create(
        WebCore::DeviceMotionData::Acceleration::create(
            data.acceleration.canProvideX, data.acceleration.x,
            data.acceleration.canProvideY, data.acceleration.y,
            data.acceleration.canProvideZ, data.acceleration.z
            ),
        WebCore::DeviceMotionData::Acceleration::create(
            data.accelerationIncludingGravity.canProvideX, data.accelerationIncludingGravity.x,
            data.accelerationIncludingGravity.canProvideY, data.accelerationIncludingGravity.y,
            data.accelerationIncludingGravity.canProvideZ, data.accelerationIncludingGravity.z
            ),
        WebCore::DeviceMotionData::RotationRate::create(
            data.rotationRate.canProvideAlpha, data.rotationRate.alpha,
            data.rotationRate.canProvideBeta, data.rotationRate.beta,
            data.rotationRate.canProvideGamma, data.rotationRate.gamma
            ),
        data.canProvideInterval,
        data.interval
        );

    callOnMainThread(handleDidChangeDeviceMotionOnMainThread, this);
}

void DeviceMotionClientManx::handleDidChangeDeviceMotionOnMainThread(void* arg)
{
    DeviceMotionClientManx* that = reinterpret_cast<DeviceMotionClientManx*>(arg);

    that->m_mutex.lock();
    WTF::RefPtr<WebCore::DeviceMotionData> data = that->m_lastMotion;
    that->m_mutex.unlock();
    if (data)
        that->m_controller->didChangeDeviceMotion(data.get());
}

} // namespece WebCore

#endif
