/*
 * Copyright (C) 2011 Igalia, S.L.
 * Copyright (C) 2014 Sony Interactive Entertainment Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef AcceleratedCompositingContext_h
#define AcceleratedCompositingContext_h

#include "GraphicsLayer.h"
#include "GraphicsLayerClient.h"
#include "IntRect.h"
#include "IntSize.h"
#include "Timer.h"
#include "WebViewPrivate.h"
#include <wtf/PassOwnPtr.h>

#if USE(ACCELERATED_COMPOSITING) && USE(ACAGI)

#include "AcagiLayer.h"
#include "SigreCommandBuffer.h"

namespace WebKit {

class AcceleratedCompositingContext : public WebCore::GraphicsLayerClient {
    WTF_MAKE_FAST_ALLOCATED;
    WTF_MAKE_NONCOPYABLE(AcceleratedCompositingContext);
public:
    static PassOwnPtr<AcceleratedCompositingContext> create(WebViewPrivate* webViewPrivate)
    {
        return adoptPtr(new AcceleratedCompositingContext(webViewPrivate));
    }

    virtual ~AcceleratedCompositingContext();
    void setRootCompositingLayer(WebCore::GraphicsLayer*);
    void setNonCompositedContentsNeedDisplay(const WebCore::IntRect&);
    void scheduleLayerFlush();
    void resizeRootLayer(const WebCore::IntSize&);
    bool enabled();

    // GraphicsLayerClient
    virtual void notifyAnimationStarted(const WebCore::GraphicsLayer*, double time);
    virtual void notifyFlushRequired(const WebCore::GraphicsLayer*);
    virtual void paintContents(const WebCore::GraphicsLayer*, WebCore::GraphicsContext&, WebCore::GraphicsLayerPaintingPhase, const WebCore::IntRect& rectToPaint);

    void initialize();

    enum CompositePurpose { ForResize, NotForResize };
    void compositeLayersToContext(CompositePurpose = NotForResize);

    void flushAndRenderLayers();
    bool flushPendingLayerChanges();
    void setActualVisibleRect(const WebCore::IntRect&);

    void patchAndSubmitDrawCommands(const WebCore::IntRect& clipRect, WebCore::SigreCommandBuffer::TargetSurface*);

private:
    WebViewPrivate* m_webViewPrivate;
    WebCore::Timer<AcceleratedCompositingContext> m_layerFlushTimer;

    OwnPtr<WebCore::GraphicsLayer> m_rootLayer;
    OwnPtr<WebCore::GraphicsLayer> m_nonCompositedContentLayer;
    OwnPtr<WebCore::AcagiCompositor> m_compositor;
    OwnPtr<WebCore::SigreCommandBuffer> m_commandBuffer;

    WebCore::IntRect m_actualVisibleRect;
    float m_contentsScale;

    void layerFlushTimerFired();
    void layerFlushTimerFired(WebCore::Timer<AcceleratedCompositingContext>*);
    void stopAnyPendingLayerFlush();
    void clearEverywhere();

    AcceleratedCompositingContext(WebViewPrivate*);
};

} // namespace WebKit

#endif // USE(ACCELERATED_COMPOSITING) && USE(ACAGI)
#endif // AcceleratedCompositingContext_h
