/*
 * Copyright (C) 2012 Sony Interactive Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#ifndef TiledBufferPainter_h
#define TiledBufferPainter_h

#include "Color.h"
#include "TiledBuffer.h"
#include <JavaScriptCore/JavaScript.h>
#include <tilebackend/tilebackend.h>

namespace WebCore {

class FrameView;
class TiledBufferPainter {
    typedef HashMap<TiledBuffer::Coordinate, RefPtr<TiledBuffer> > TileMap;
    typedef tilebackend::TileSize TileSize;

public:
    static TiledBufferPainter* create(const IntSize& screenSize, tilebackend::PoolInterface*, tilebackend::PainterInterface*);

    ~TiledBufferPainter();

    float contentsScale() const { return m_contentsScale; }
    void setContentsScale(float);

    bool updateTiles(FrameView*);
    void paint(tilebackend::PlatformSurface*, const IntRect& viewRect);

    void invalidate(const IntRect& updateRect, bool immediate);

    void clear();

    tilebackend::PoolInterface* pool() { return m_pool; }
    tilebackend::PainterInterface* painter() { return m_painter; }

    int tileCount() const { return m_tiles.size(); }

protected:
    TiledBufferPainter();

    void paintContents(FrameView*, cairo_surface_t*, const IntRect& targetRect);
    void updateTileContents(TiledBuffer*, cairo_surface_t*, const IntRect& targetRect, TiledBuffer* original = 0);
    void commitTiles(TileMap &tiles, float contentsScale);

    void updateDirtyRects();
    void getDirtyTileCoords(const IntRect& coverRect, Vector<TiledBuffer::Coordinate>& dirtyTileCoords);
    bool releaseMostDistantTile(const IntRect& viewRect);

    IntSize tileSize() const
    {
        return IntSize(m_tileSize.width, m_tileSize.height);
    }

    WTF::Mutex m_mutex;

    tilebackend::PoolInterface* m_pool;
    tilebackend::PainterInterface* m_painter;

    IntSize m_screenSize;
    float m_contentsScale;
    float m_committedScale;

    TileSize m_tileSize;

    TileMap m_tiles;

    Color m_backgroundColor;

    Vector<IntRect> m_dirtyRects;
};

} // namespace WebCore

#endif // TiledBufferPainter_h
