/*
 * Copyright (C) 2014 Sony Interactive Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef SigreCommandBuffer_h
#define SigreCommandBuffer_h

#if USE(ACCELERATED_COMPOSITING) && USE(ACAGI)

#include "Color.h"
#include "IntRect.h"
#include "IntSize.h"
#include "SigreTexture.h"
#include "TransformationMatrix.h"
#include <wtf/PassOwnPtr.h>
#include <wtf/Vector.h>

namespace WebCore {

class SigreCommand;

class SigreCommandBuffer {
    WTF_MAKE_FAST_ALLOCATED;
    WTF_MAKE_NONCOPYABLE(SigreCommandBuffer);
public:
    typedef tilebackend::Target TargetSurface;
    typedef tilebackend::PoolInterface Allocator;
    typedef tilebackend::PainterInterface PlatformGraphics;

    static PassOwnPtr<SigreCommandBuffer> create(Allocator*, PlatformGraphics*);
    ~SigreCommandBuffer();

    // Texture creation
    PassRefPtr<SigreTexture> createTexture(IntSize textureSize);

    // Drawing  API
    void beginPainting();
    void endPainting();
    
    void drawBorder(const Color&, float borderWidth, const FloatRect&, const TransformationMatrix&);
    void drawNumber(int number, const Color&, const FloatPoint&, const TransformationMatrix&);

    void drawTexture(PassRefPtr<SigreTexture>, const FloatRect& target, const TransformationMatrix& modelViewMatrix = TransformationMatrix(), float opacity = 1.0f);
    void drawSolidColor(const FloatRect&, const TransformationMatrix&, const Color&);
    void bindSurface(PassRefPtr<SigreTexture> surface);

    void scissor(const FloatRect& scissorBox, bool enable);

    // Clear all commands in both front and back buffer.
    void clearCommands();

    void patchAndSubmitDrawCommands(const IntRect& clipRect, TargetSurface*);

private:
    explicit SigreCommandBuffer(Allocator*, PlatformGraphics*);

#if DEBUG_TILEDBACKEND_V2
    static bool memoryChunkInfoCallback(float pos, float size, bool inuse, void* usrdata);
#endif


    Allocator* m_allocator;
    PlatformGraphics* m_gfx;

    // Command List
    typedef Vector<OwnPtr<SigreCommand> > CommandList;
    OwnPtr<CommandList> m_front;
    OwnPtr<CommandList> m_back;
    Mutex m_frontLocker;
};

}

#endif // USE(ACCELERATED_COMPOSITING) && USE(ACAGI)

#endif
