/*
 Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies)
 Copyright (C) 2014 Sony Interactive Entertainment Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public License
 along with this library; see the file COPYING.LIB.  If not, write to
 the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 Boston, MA 02110-1301, USA.
 */

#ifndef AcagiLayer_h
#define AcagiLayer_h

#if USE(ACCELERATED_COMPOSITING) && USE(ACAGI)

#include "AcagiBackingStore.h"
#include "AcagiCompositor.h"
#include "FilterOperations.h"
#include "FloatRect.h"
#include "GraphicsLayerAnimation.h"
#include "GraphicsLayerTransform.h"

namespace WebCore {

class Region;
class AcagiPaintOptions;
class PlatformLayerAcagi;

class AcagiLayer : public GraphicsLayerAnimation::Client {
    WTF_MAKE_NONCOPYABLE(AcagiLayer);
    WTF_MAKE_FAST_ALLOCATED;
public:

    class ScrollingClient {
    public:
        virtual ~ScrollingClient() { }
        virtual void commitScrollOffset(uint32_t layerID, const IntSize& offset) = 0;
    };

    AcagiLayer()
        : m_parent(0)
        , m_effectTarget(0)
        , m_contentsLayer(0)
        , m_currentOpacity(1)
        , m_centerZ(0)
        , m_compositor(0)
        , m_fixedToViewport(false)
        , m_id(0)
        , m_scrollClient(0)
        , m_isScrollable(false)
        , m_patternTransformDirty(false)
    { }

    virtual ~AcagiLayer();

    void setID(uint32_t id) { m_id = id; }
    uint32_t id() { return m_id; }

    const Vector<AcagiLayer*>& children() const { return m_children; }
    AcagiLayer* findScrollableContentsLayerAt(const FloatPoint& pos);

    void setScrollClient(ScrollingClient* scrollClient) { m_scrollClient = scrollClient; }
    void scrollBy(const WebCore::FloatSize&);

    void didCommitScrollOffset(const IntSize&);
    void setIsScrollable(bool isScrollable) { m_isScrollable = isScrollable; }
    bool isScrollable() const { return m_isScrollable; }

    AcagiCompositor* compositor() const;
    void setCompositor(AcagiCompositor* compositor) { m_compositor = compositor; }

    void setChildren(const Vector<AcagiLayer*>&);
    void setMaskLayer(AcagiLayer*);
    void setReplicaLayer(AcagiLayer*);
    void setPosition(const FloatPoint&);
    void setSize(const FloatSize&);
    void setAnchorPoint(const FloatPoint3D&);
    void setPreserves3D(bool);
    void setTransform(const TransformationMatrix&);
    void setChildrenTransform(const TransformationMatrix&);
    void setContentsRect(const IntRect&);
    void setMasksToBounds(bool);
    void setDrawsContent(bool);
    bool drawsContent() const { return m_state.drawsContent; }
    bool contentsAreVisible() const { return m_state.contentsVisible; }
    FloatSize size() const { return m_state.size; }
    float opacity() const { return m_state.opacity; }
    TransformationMatrix transform() const { return m_state.transform; }
    void setContentsVisible(bool);
    void setContentsOpaque(bool);
    void setBackfaceVisibility(bool);
    void setOpacity(float);
    void setSolidColor(const Color&);
    void setContentsTileSize(const IntSize&);
    void setContentsTilePhase(const IntPoint&);

    bool hasFilters() const
    {
        return false;
    }

    void setDebugVisuals(bool showDebugBorders, const Color& debugBorderColor, float debugBorderWidth, bool showRepaintCounter);
    bool isShowingRepaintCounter() const { return m_state.showRepaintCounter; }
    void setRepaintCount(int);
    void setContentsLayer(PlatformLayerAcagi*);
    void setAnimations(const GraphicsLayerAnimations&);
    void setFixedToViewport(bool);
    bool fixedToViewport() const { return m_fixedToViewport; }
    void setBackingStore(PassRefPtr<AcagiBackingStore>);

    void syncAnimations();
    bool descendantsOrSelfHaveRunningAnimations() const;

    void paint();

    void setScrollPositionDeltaIfNeeded(const FloatSize&);

    void applyAnimationsRecursively();
    void addChild(AcagiLayer*);

    void computeTransformsRecursive();
    TransformationMatrix combinedTransform() { return m_currentTransform.combined(); }

private:
    const AcagiLayer* rootLayer() const;

    static int compareGraphicsLayersZValue(const void* a, const void* b);
    static void sortByZOrder(Vector<AcagiLayer* >& array);

    PassRefPtr<AcagiBitmapTexture> texture() { return m_backingStore ? m_backingStore->texture() : 0; }
    FloatPoint adjustedPosition() const { return m_state.pos + m_scrollPositionDelta - m_userScrollOffset; }
    bool isAncestorFixedToViewport() const;
    TransformationMatrix replicaTransform();
    void removeFromParent();
    void removeAllChildren();

    enum ResolveSelfOverlapMode {
        ResolveSelfOverlapAlways = 0,
        ResolveSelfOverlapIfNeeded
    };
    void computeOverlapRegions(Region& overlapRegion, Region& nonOverlapRegion, ResolveSelfOverlapMode);

    void paintRecursive(const AcagiPaintOptions&);
    void paintUsingOverlapRegions(const AcagiPaintOptions&);
    PassRefPtr<AcagiBitmapTexture> paintIntoSurface(const AcagiPaintOptions&, const IntSize&);
    void paintWithIntermediateSurface(const AcagiPaintOptions&, const IntRect&);
    void paintSelf(const AcagiPaintOptions&);
    void paintSelfAndChildren(const AcagiPaintOptions&);
    void paintSelfAndChildrenWithReplica(const AcagiPaintOptions&);
    void applyMask(const AcagiPaintOptions&);
    void computePatternTransformIfNeeded();

    // GraphicsLayerAnimation::Client
    virtual void setAnimatedTransform(const TransformationMatrix&) OVERRIDE;
    virtual void setAnimatedOpacity(float) OVERRIDE;

    bool isVisible() const;
    enum ContentsLayerCount {
        NoLayersWithContent,
        SingleLayerWithContents,
        MultipleLayersWithContents
    };

    bool shouldBlend() const;

    inline FloatRect layerRect() const
    {
        return FloatRect(FloatPoint::zero(), m_state.size);
    }

    Vector<AcagiLayer*> m_children;
    AcagiLayer* m_parent;
    AcagiLayer* m_effectTarget;
    RefPtr<AcagiBackingStore> m_backingStore;
    PlatformLayerAcagi* m_contentsLayer;
    GraphicsLayerTransform m_currentTransform;
    float m_currentOpacity;
    float m_centerZ;

    template<class HitTestCondition> AcagiLayer* hitTest(const FloatPoint&, HitTestCondition);
    static bool scrollableLayerHitTestCondition(AcagiLayer*, const FloatPoint&);

    FloatSize mapScrollOffset(const FloatSize&);
    void commitScrollOffset(const FloatSize&);

    struct State {
        FloatPoint pos;
        FloatPoint3D anchorPoint;
        FloatSize size;
        TransformationMatrix transform;
        TransformationMatrix childrenTransform;
        float opacity;
        FloatRect contentsRect;
        IntSize contentsTileSize;
        IntPoint contentsTilePhase;
        AcagiLayer* maskLayer;
        AcagiLayer* replicaLayer;
        Color solidColor;
        Color debugBorderColor;
        float debugBorderWidth;
        int repaintCount;

        bool preserves3D : 1;
        bool masksToBounds : 1;
        bool drawsContent : 1;
        bool contentsVisible : 1;
        bool contentsOpaque : 1;
        bool backfaceVisibility : 1;
        bool visible : 1;
        bool showDebugBorders : 1;
        bool showRepaintCounter : 1;

        State()
            : opacity(1)
            , maskLayer(0)
            , replicaLayer(0)
            , debugBorderWidth(0)
            , repaintCount(0)
            , preserves3D(false)
            , masksToBounds(false)
            , drawsContent(false)
            , contentsVisible(true)
            , contentsOpaque(false)
            , backfaceVisibility(true)
            , visible(true)
            , showDebugBorders(false)
            , showRepaintCounter(false)
        {
        }
    };

    State m_state;
    AcagiCompositor* m_compositor;
    GraphicsLayerAnimations m_animations;
    FloatSize m_scrollPositionDelta;
    bool m_fixedToViewport;
    uint32_t m_id;
    ScrollingClient* m_scrollClient;
    bool m_isScrollable;
    FloatSize m_userScrollOffset;
    FloatSize m_accumulatedScrollOffsetFractionalPart;
    TransformationMatrix m_patternTransform;
    bool m_patternTransformDirty;
};

}
#endif

#endif // AcagiLayer_h
