/*
 Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies)
 Copyright (C) 2014 Sony Interactive Entertainment Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public License
 along with this library; see the file COPYING.LIB.  If not, write to
 the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 Boston, MA 02110-1301, USA.
 */

#ifndef AcagiBitmapTexture_h
#define AcagiBitmapTexture_h

#if USE(ACCELERATED_COMPOSITING) && USE(ACAGI)

#include "AcagiPlatformTexture.h"
#include "GraphicsContext.h"
#include "IntRect.h"
#include "IntSize.h"
#include "TransformationMatrix.h"
#include <wtf/Noncopyable.h>

namespace WebCore {

class GraphicsLayer;
class AcagiCompositor;

// A 2D texture
class AcagiBitmapTexture : public RefCounted<AcagiBitmapTexture> {
    WTF_MAKE_NONCOPYABLE(AcagiBitmapTexture)
    typedef AcagiPlatformTexture PlatformTexture;
public:
    enum Flag {
        SupportsAlpha = 0x01
    };

    enum UpdateContentsFlag {
        UpdateCanModifyOriginalImageData,
        UpdateCannotModifyOriginalImageData
    };

    typedef unsigned Flags;

    AcagiBitmapTexture(AcagiCompositor*);
    ~AcagiBitmapTexture();

    IntSize size() const;
    void updateContents(Image*, const IntRect&, const IntPoint& offset, UpdateContentsFlag);
    void updateContents(AcagiCompositor*, GraphicsLayer*, const IntRect& targetRect, const IntPoint& targetOffset, const FloatRect& sourceRect, UpdateContentsFlag);
    void updateContents(const void*, const IntRect& target, const IntPoint& offset, int bytesPerLine, UpdateContentsFlag);
    bool isValid() const;
    inline Flags flags() const { return m_flags; }

    int bpp() const;
    bool canReuseWith(const IntSize& contentsSize, Flags = 0);
    void reset(const IntSize& size, Flags flags = 0)
    {
        m_flags = flags;
        m_contentSize = size;
        didReset();
    }
    void didReset();
    void duplicate(const AcagiBitmapTexture&);



    inline IntSize contentSize() const { return m_contentSize; }
    inline int numberOfBytes() const { return size().width() * size().height() * bpp() >> 3; }
    inline bool isOpaque() const { return !(m_flags & SupportsAlpha); }

    PassRefPtr<PlatformTexture> platformTexture() const { return m_texture; }

private:
    IntSize textureSize() const { return m_texture ? m_texture->textureSize() : IntSize(); }
    void resetTexture(const IntSize& newSize);

private:
    Flags m_flags;
    IntSize m_contentSize;
    AcagiCompositor* m_compositor;
    RefPtr<PlatformTexture> m_texture;
    bool m_shouldClear;
};

}

#endif

#endif
