/*
 * Copyright (C) 2003, 2006 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2006 Samuel Weinig <sam.weinig@gmail.com>
 * Copyright (C) 2012 Sony Interactive Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef ResourceRequest_h
#define ResourceRequest_h

#include "ResourceRequestBase.h"

typedef const struct _CFURLRequest* CFURLRequestRef;

namespace WebCore {

class Frame;
class ResourceRequest : public ResourceRequestBase {
public:
    ResourceRequest(const String& url)
        : ResourceRequestBase(KURL(ParsedURLString, url), UseProtocolCachePolicy)
    {
        m_isMainResource = false;
        m_redirected = false;
    }

    ResourceRequest(const KURL& url)
        : ResourceRequestBase(url, UseProtocolCachePolicy)
    {
        m_isMainResource = false;
        m_redirected = false;
    }

    ResourceRequest(const KURL& url, const String& referrer, ResourceRequestCachePolicy policy = UseProtocolCachePolicy)
        : ResourceRequestBase(url, policy)
    {
        m_isMainResource = false;
        m_redirected = false;
        setHTTPReferrer(referrer);
    }

    ResourceRequest()
        : ResourceRequestBase(KURL(), UseProtocolCachePolicy)
    {
        m_isMainResource = false;
        m_redirected = false;
    }

    ResourceRequest(CFURLRequestRef)
        : ResourceRequestBase()
    {
        m_isMainResource = false;
        m_redirected = false;
    }

    // Needed for compatibility.
    CFURLRequestRef cfURLRequest() const { return 0; }

    void setIsMainResource(bool isMain) const { m_isMainResource = isMain; }
    bool isMainResource() const { return m_isMainResource; }

    void setRedirected(bool isRedirected) const { m_redirected = isRedirected; }
    bool getRedirected() const { return m_redirected; }

    void clearHTTPHeaderMap() { m_httpHeaderFields.clear(); }

    // The following two stubs are for compatibility with CFNetwork, and are not used.
    static bool httpPipeliningEnabled() { return false; }
    static void setHTTPPipeliningEnabled(bool) { }

    void updateFromDelegatePreservingOldHTTPBody(const ResourceRequest& delegateProvidedRequest) { *this = delegateProvidedRequest; }

private:
    friend class ResourceRequestBase;

    void doUpdatePlatformRequest() { }
    void doUpdateResourceRequest() { }
    void doUpdatePlatformHTTPBody() { }
    void doUpdateResourceHTTPBody() { }

    PassOwnPtr<CrossThreadResourceRequestData> doPlatformCopyData(PassOwnPtr<CrossThreadResourceRequestData> data) const { return data; }
    void doPlatformAdopt(PassOwnPtr<CrossThreadResourceRequestData>) { }

    mutable bool m_isMainResource;
    mutable bool m_redirected;

    static bool s_httpPipeliningEnabled;
};

struct CrossThreadResourceRequestData : public CrossThreadResourceRequestDataBase {
};

} // namespace WebCore

#endif // ResourceRequest_h
