/*
 * Copyright (C) 2008 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2009 Google Inc.
 * Copyright (C) 2012 Sony Interactive Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "RenderThemeManx.h"

#include "BitmapImage.h"
#include "CSSValueKeywords.h"
#include "GraphicsContext.h"
#include "Image.h"
#include "NotImplemented.h"
#include "PaintInfo.h"
#include "SharedBuffer.h"
#include "UserAgentStyleSheets.h"
#include <wtf/text/StringConcatenate.h>

#if ENABLE(VIDEO)
#include "HTMLMediaElement.h"
#include "RenderMediaControls.h"
#endif

namespace WebCore {

static const unsigned aquaFocusRingColor = 0xFF7DADD9;

RenderThemeManx::~RenderThemeManx()
{
}

PassRefPtr<RenderTheme> RenderTheme::themeForPage(Page* page)
{
    static RenderTheme* rt = RenderThemeManx::create().leakRef();
    return rt;
}

PassRefPtr<RenderTheme> RenderThemeManx::create()
{
    return adoptRef(new RenderThemeManx());
}

#if ENABLE(VIDEO)
String RenderThemeManx::extraMediaControlsStyleSheet()
{
#if OS(ORBIS)
    return String(mediaControlsManxOrbisUserAgentStyleSheet, sizeof(mediaControlsManxOrbisUserAgentStyleSheet));
#endif
}
#endif
#if ENABLE(FULLSCREEN_API)
String RenderThemeManx::extraFullScreenStyleSheet()
{
#if OS(ORBIS)
    return String(fullscreenManxOrbisUserAgentStyleSheet, sizeof(fullscreenManxOrbisUserAgentStyleSheet));
#endif
}
#endif

PassRefPtr<Image> RenderThemeManx::loadThemeImage(const char* name)
{
    static const char *path = getenv("WEBKIT_THEME_PATH");

    RefPtr<BitmapImage> img = BitmapImage::create();

    if (path) {
        RefPtr<SharedBuffer> buffer = SharedBuffer::createWithContentsOfFile(makeString(path, "/", name, ".png"));
        if (buffer)
            img->setData(buffer.release(), true);
    }

    return img.release();
}

void RenderThemeManx::paintThemeImage(GraphicsContext* ctx, Image* image, const IntRect& rect,
    ColorSpace styleColorSpace)
{
    ctx->drawImage(image, styleColorSpace, rect);
}

void RenderThemeManx::paintThemeImage3x3(GraphicsContext* ctx, Image* image, const IntRect& rect,
    int left, int right, int top, int bottom,
    ColorSpace styleColorSpace)
{
    IntRect dstRect = rect;
    IntRect srcRect = image->rect();

    // top-left
    ctx->drawImage(image, styleColorSpace,
        IntRect(dstRect.x(), dstRect.y(), left, top),
        IntRect(srcRect.x(), srcRect.y(), left, top));
    // top-middle
    ctx->drawImage(image, styleColorSpace,
        IntRect(dstRect.x() + left, dstRect.y(), dstRect.width() - left - right, top),
        IntRect(srcRect.x() + left, srcRect.y(), srcRect.width() - left - right, top));
    // top-right
    ctx->drawImage(image, styleColorSpace,
        IntRect(dstRect.x() + dstRect.width() - right, dstRect.y(), right, top),
        IntRect(srcRect.x() + srcRect.width() - right, srcRect.y(), right, top));
    // middle-left
    ctx->drawImage(image, styleColorSpace,
        IntRect(dstRect.x(), dstRect.y() + top, left, dstRect.height() - top - bottom),
        IntRect(srcRect.x(), srcRect.y() + top, left, srcRect.height() - top - bottom));
    // middle-middle
    ctx->drawImage(image, styleColorSpace,
        IntRect(dstRect.x() + left, dstRect.y() + top, dstRect.width() - left - right, dstRect.height() - top - bottom),
        IntRect(srcRect.x() + left, srcRect.y() + top, srcRect.width() - left - right, srcRect.height() - top - bottom));
    // middle-right
    ctx->drawImage(image, styleColorSpace,
        IntRect(dstRect.x() + dstRect.width() - right, dstRect.y() + top, right, dstRect.height() - top - bottom),
        IntRect(srcRect.x() + srcRect.width() - right, srcRect.y() + top, right, srcRect.height() - top - bottom));
    // bottom-left
    ctx->drawImage(image, styleColorSpace,
        IntRect(dstRect.x(), dstRect.y() + dstRect.height() - bottom, left, bottom),
        IntRect(srcRect.x(), srcRect.y() + srcRect.height() - bottom, left, bottom));
    // bottom-middle
    ctx->drawImage(image, styleColorSpace,
        IntRect(dstRect.x() + left, dstRect.y() + dstRect.height() - bottom, dstRect.width() - left - right, bottom),
        IntRect(srcRect.x() + left, srcRect.y() + srcRect.height() - bottom, srcRect.width() - left - right, bottom));
    // bottom-right
    ctx->drawImage(image, styleColorSpace,
        IntRect(dstRect.x() + dstRect.width() - right, dstRect.y() + dstRect.height() - bottom, right, bottom),
        IntRect(srcRect.x() + srcRect.width() - right, srcRect.y() + srcRect.height() - bottom, right, bottom));
}

bool RenderThemeManx::supportsHover(const RenderStyle*) const
{
    return true;
}

bool RenderThemeManx::supportsFocusRing(const RenderStyle*) const
{
    return false;
}

Color RenderThemeManx::platformFocusRingColor() const
{
    return aquaFocusRingColor;
}

void RenderThemeManx::systemFont(int propId, FontDescription& fontDescription) const
{
    float fontSize = 16;

    switch (propId) {
    case CSSValueWebkitMiniControl:
    case CSSValueWebkitSmallControl:
    case CSSValueWebkitControl:
        // Why 2 points smaller? Because that's what Gecko does. Note that we
        // are assuming a 96dpi screen, which is the default that we use on
        // Windows.
        static const float pointsPerInch = 72.0f;
        static const float pixelsPerInch = 96.0f;
        fontSize -= (2.0f / pointsPerInch) * pixelsPerInch;
        break;
    }

    fontDescription.setOneFamily("sans-serif");
    fontDescription.setSpecifiedSize(fontSize);
    fontDescription.setIsAbsoluteSize(true);
    fontDescription.setGenericFamily(FontDescription::NoFamily);
    fontDescription.setWeight(FontWeightNormal);
    fontDescription.setItalic(false);
}

bool RenderThemeManx::paintCheckbox(RenderObject* obj, const PaintInfo& info, const IntRect& rect)
{
    static Image* const onNormal  = loadThemeImage("checkbox/checkbox_on_normal").leakRef();
    static Image* const onFocus   = loadThemeImage("checkbox/checkbox_on_focus").leakRef();
    static Image* const onHover   = loadThemeImage("checkbox/checkbox_on_hover").leakRef();
    static Image* const offNormal = loadThemeImage("checkbox/checkbox_off_normal").leakRef();
    static Image* const offFocus  = loadThemeImage("checkbox/checkbox_off_focus").leakRef();
    static Image* const offHover  = loadThemeImage("checkbox/checkbox_off_hover").leakRef();

    Image* image;

    if (this->isChecked(obj))
        image = isFocused(obj) ? onFocus : isHovered(obj) ? onHover : onNormal;
    else
        image = isFocused(obj) ? offFocus : isHovered(obj) ? offHover : offNormal;

    paintThemeImage(info.context, image, rect, obj->style()->colorSpace());

    return false;
}

void RenderThemeManx::setCheckboxSize(RenderStyle* style) const
{
    // width and height both specified so do nothing
    if (!style->width().isIntrinsicOrAuto() && !style->height().isAuto())
        return;

    // hardcode size to 13 to match Firefox
    if (style->width().isIntrinsicOrAuto())
        style->setWidth(Length(13, Fixed));

    if (style->height().isAuto())
        style->setHeight(Length(13, Fixed));
}

bool RenderThemeManx::paintRadio(RenderObject* obj, const PaintInfo& info, const IntRect& rect)
{
    static Image* const onNormal  = loadThemeImage("radio/radio_on_normal").leakRef();
    static Image* const onFocus   = loadThemeImage("radio/radio_on_focus").leakRef();
    static Image* const onHover   = loadThemeImage("radio/radio_on_hover").leakRef();
    static Image* const offNormal = loadThemeImage("radio/radio_off_normal").leakRef();
    static Image* const offFocus  = loadThemeImage("radio/radio_off_focus").leakRef();
    static Image* const offHover  = loadThemeImage("radio/radio_off_hover").leakRef();

    Image* image;

    if (isChecked(obj))
        image = isFocused(obj) ? onFocus : isHovered(obj) ? onHover : onNormal;
    else
        image = isFocused(obj) ? offFocus : isHovered(obj) ? offHover : offNormal;

    paintThemeImage(info.context, image, rect, obj->style()->colorSpace());

    return false;
}

void RenderThemeManx::setRadioSize(RenderStyle* style) const
{
    // width and height both specified so do nothing
    if (!style->width().isIntrinsicOrAuto() && !style->height().isAuto())
        return;

    // hardcode size to 13 to match Firefox
    if (style->width().isIntrinsicOrAuto())
        style->setWidth(Length(13, Fixed));

    if (style->height().isAuto())
        style->setHeight(Length(13, Fixed));
}

void RenderThemeManx::adjustButtonStyle(StyleResolver*, RenderStyle* style, Element*) const
{
    // ignore line-height
    if (style->appearance() == PushButtonPart)
        style->setLineHeight(RenderStyle::initialLineHeight());
}

bool RenderThemeManx::paintButton(RenderObject* obj, const PaintInfo& info, const IntRect& rect)
{
    static Image* const normal = loadThemeImage("button/button_normal").leakRef();
    static Image* const press  = loadThemeImage("button/button_press").leakRef();
    static Image* const focus  = loadThemeImage("button/button_focus").leakRef();
    static Image* const hover  = loadThemeImage("button/button_hover").leakRef();

    Image* image = isPressed(obj) ? press : isFocused(obj) ? focus : isHovered(obj) ? hover : normal;

    paintThemeImage3x3(info.context, image, rect, 6, 6, 6, 6, obj->style()->colorSpace());

    return false;
}

void RenderThemeManx::adjustMenuListStyle(StyleResolver*, RenderStyle* style, Element*) const
{
    // The tests check explicitly that select menu buttons ignore line height.
    style->setLineHeight(RenderStyle::initialLineHeight());

    // We cannot give a proper rendering when border radius is active, unfortunately.
    style->resetBorderRadius();
}

bool RenderThemeManx::paintMenuList(RenderObject* obj, const PaintInfo& info, const IntRect& rect)
{
    static Image* const normal = loadThemeImage("menulist/menulist_normal").leakRef();
    static Image* const press  = loadThemeImage("menulist/menulist_press").leakRef();
    static Image* const focus  = loadThemeImage("menulist/menulist_focus").leakRef();
    static Image* const hover  = loadThemeImage("menulist/menulist_hover").leakRef();

    Image* image = isPressed(obj) ? press : isFocused(obj) ? focus : isHovered(obj) ? hover : normal;

    paintThemeImage3x3(info.context, image, rect, 6, 20, 6, 6, obj->style()->colorSpace());

    return paintMenuListButton(obj, info, rect);
}

void RenderThemeManx::adjustMenuListButtonStyle(StyleResolver* selector, RenderStyle* style, Element* element) const
{
    adjustMenuListStyle(selector, style, element);
}

bool RenderThemeManx::paintMenuListButton(RenderObject* o, const PaintInfo& paintInfo, const IntRect& r)
{
    IntRect bounds = IntRect(r.x() + o->style()->borderLeftWidth(),
        r.y() + o->style()->borderTopWidth(),
        r.width() - o->style()->borderLeftWidth() - o->style()->borderRightWidth(),
        r.height() - o->style()->borderTopWidth() - o->style()->borderBottomWidth());
    const float arrowHeight = 5;
    const float arrowWidth = 7;
    const int arrowPaddingLeft = 5;
    const int arrowPaddingRight = 5;
    
    // Since we actually know the size of the control here, we restrict the font scale to make sure the arrow will fit vertically in the bounds
    float centerY = bounds.y() + bounds.height() / 2.0f;
    float leftEdge = bounds.maxX() - arrowPaddingRight - arrowWidth;
    float buttonWidth = arrowPaddingLeft + arrowWidth + arrowPaddingRight;

    if (bounds.width() < arrowWidth + arrowPaddingLeft)
        return false;

    GraphicsContext& context = *paintInfo.context;
    context.save();

    // Draw button background
    int fgGray = isPressed(o) ? 247 : isHovered(o) ? 84 : 94;
    int bgGray = isPressed(o) ? 84 : isHovered(o) ? 216 : 226;
    Color fgColor(fgGray, fgGray, fgGray);
    Color bgColor(bgGray, bgGray, bgGray);
    context.setStrokeStyle(SolidStroke);
    context.setStrokeColor(fgColor, ColorSpaceDeviceRGB);
    context.setFillColor(bgColor, ColorSpaceDeviceRGB);
    FloatRect button(bounds.maxX() - buttonWidth, bounds.y(), buttonWidth, bounds.height());
    button.inflate(-2);
    context.fillRect(button);
    context.strokeRect(button, 1);

    // Draw the arrow
    context.setFillColor(fgColor, ColorSpaceDeviceRGB);
    context.setStrokeColor(NoStroke, ColorSpaceDeviceRGB);

    FloatPoint arrow[3];
    arrow[0] = FloatPoint(leftEdge, centerY - arrowHeight / 2.0f);
    arrow[1] = FloatPoint(leftEdge + arrowWidth, centerY - arrowHeight / 2.0f);
    arrow[2] = FloatPoint(leftEdge + arrowWidth / 2.0f, centerY + arrowHeight / 2.0f);

    context.drawConvexPolygon(3, arrow, true);
    context.restore();
    return false;
}

bool RenderThemeManx::paintTextArea(RenderObject* obj, const PaintInfo& info, const IntRect& rect)
{
    return true;
}

int RenderThemeManx::popupInternalPaddingLeft(RenderStyle*) const
{
    return 5;
}

int RenderThemeManx::popupInternalPaddingRight(RenderStyle*) const
{
    return 5 + 20; // offset 20 for menulist button icon
}

int RenderThemeManx::popupInternalPaddingTop(RenderStyle*) const
{
    return 3;
}

int RenderThemeManx::popupInternalPaddingBottom(RenderStyle*) const
{
    return 3;
}

#if ENABLE(VIDEO)
static bool paintDummyButton(RenderObject* object, const PaintInfo& info, const IntRect& rect, Color color = Color(0, 255, 0))
{
    info.context->save();
    info.context->setFillColor(color, ColorSpaceDeviceRGB);
    info.context->drawRect(rect);
    info.context->restore();
    return true;
}

static bool paintDummyBackground(RenderObject* object, const PaintInfo& info, const IntRect& rect, Color color = Color(Color::darkGray))
{
    info.context->save();
    info.context->setStrokeStyle(NoStroke);
    info.context->setFillColor(color, ColorSpaceDeviceRGB);
    info.context->drawRect(rect);
    info.context->restore();
    return true;
}

static bool paintMediaSlider(RenderObject* object, const PaintInfo& paintInfo, const IntRect& rect)
{
    HTMLMediaElement* mediaElement = toParentMediaElement(object);
    if (!mediaElement)
        return false;

    GraphicsContext* context = paintInfo.context;
    int borderWidth = 1;

    // Draw the border of the time bar.
    // FIXME: this should be a rounded rect but need to fix GraphicsContextSkia first.
    // https://bugs.webkit.org/show_bug.cgi?id=30143
    context->save();
    context->setShouldAntialias(true);
    context->setStrokeStyle(SolidStroke);
    context->setStrokeColor(Color::white, ColorSpaceDeviceRGB);
    context->setStrokeThickness(borderWidth);
    context->setFillColor(Color::darkGray, ColorSpaceDeviceRGB);
    context->drawRect(rect);
    context->restore();

    // Draw the buffered ranges.
    // FIXME: Draw multiple ranges if there are multiple buffered ranges.
    IntRect bufferedRect = rect;
    bufferedRect.inflate(-borderWidth);

    double bufferedWidth = 0.0;
    if (mediaElement->percentLoaded() > 0.0) {
#if 0 // TODO:
        // Account for the width of the slider thumb.
        Image* mediaSliderThumb = getMediaSliderThumb();
        double thumbWidth = mediaSliderThumb->width() / 2.0 + 1.0;
#else
        double thumbWidth = 0;
#endif
        double rectWidth = bufferedRect.width() - thumbWidth;
        if (rectWidth < 0.0)
            rectWidth = 0.0;
        bufferedWidth = rectWidth * mediaElement->percentLoaded() + thumbWidth;
    }
    bufferedRect.setWidth(static_cast<int>(bufferedWidth));

    // Don't bother drawing an empty area.
    if (!bufferedRect.isEmpty()) {
        IntPoint sliderTopLeft = bufferedRect.location();
        IntPoint sliderTopRight = sliderTopLeft;
        sliderTopRight.move(0, bufferedRect.height());

        RefPtr<Gradient> gradient = Gradient::create(sliderTopLeft, sliderTopRight);
        Color startColor(255, 64, 64);
        gradient->addColorStop(0.0, startColor);
        gradient->addColorStop(1.0, Color(startColor.red() / 2, startColor.green() / 2, startColor.blue() / 2, startColor.alpha()));

        context->save();
        context->setStrokeStyle(NoStroke);
        context->setFillGradient(gradient);
        context->fillRect(bufferedRect);
        context->restore();
    }

    return true;
}

const int sliderThumbWidth = 7;
const int sliderThumbHeight = 15;

void RenderThemeManx::adjustSliderThumbSize(RenderStyle* style, Element*) const
{
    ControlPart part = style->appearance();
    switch (part) {
    case SliderThumbVerticalPart:
    case MediaVolumeSliderThumbPart:
        style->setWidth(Length(sliderThumbHeight, Fixed));
        style->setHeight(Length(sliderThumbWidth, Fixed));
        break;
    case SliderThumbHorizontalPart:
    case MediaSliderThumbPart:
        style->setWidth(Length(sliderThumbWidth, Fixed));
        style->setHeight(Length(sliderThumbHeight, Fixed));
        break;
    default:
        break;
    }
}

bool RenderThemeManx::paintMediaFullscreenButton(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    static Image* const image = loadThemeImage("media/media_fullscreen").leakRef();
    paintThemeImage(info.context, image, rect, object->style()->colorSpace());
    return true;
}
bool RenderThemeManx::paintMediaPlayButton(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    static Image* const playImage = loadThemeImage("media/media_play").leakRef();
    static Image* const pauseImage = loadThemeImage("media/media_pause").leakRef();

    Node* node = object->node();
    if (!node)
        return false;

    Image* image;
    if (mediaControlElementType(node) == MediaPlayButton)
        image = playImage;
    else
        image = pauseImage;
    paintThemeImage(info.context, image, rect, object->style()->colorSpace());
    return true;
}

bool RenderThemeManx::paintMediaMuteButton(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return paintDummyButton(object, info, rect);
}

bool RenderThemeManx::paintMediaSeekBackButton(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return paintDummyButton(object, info, rect);
}

bool RenderThemeManx::paintMediaSeekForwardButton(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return paintDummyButton(object, info, rect);
}

bool RenderThemeManx::paintMediaSliderTrack(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return paintMediaSlider(object, info, rect);
}

bool RenderThemeManx::paintMediaSliderThumb(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return paintDummyButton(object, info, rect);
}

bool RenderThemeManx::paintMediaVolumeSliderContainer(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return paintDummyBackground(object, info, rect);
}

bool RenderThemeManx::paintMediaVolumeSliderTrack(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return paintDummyButton(object, info, rect);
}

bool RenderThemeManx::paintMediaVolumeSliderThumb(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return paintDummyButton(object, info, rect);
}

bool RenderThemeManx::paintMediaRewindButton(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    static Image* const image = loadThemeImage("media/media_rewind").leakRef();
    paintThemeImage(info.context, image, rect, object->style()->colorSpace());
    return true;
}

bool RenderThemeManx::paintMediaReturnToRealtimeButton(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return paintDummyButton(object, info, rect);
}

bool RenderThemeManx::paintMediaToggleClosedCaptionsButton(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return paintDummyButton(object, info, rect);
}

bool RenderThemeManx::paintMediaControlsBackground(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return false;
}

bool RenderThemeManx::paintMediaCurrentTime(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return true;
}

bool RenderThemeManx::paintMediaTimeRemaining(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return true;
}

bool RenderThemeManx::paintMediaFullScreenVolumeSliderTrack(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return paintDummyButton(object, info, rect);
}

bool RenderThemeManx::paintMediaFullScreenVolumeSliderThumb(RenderObject* object, const PaintInfo& info, const IntRect& rect)
{
    return paintDummyButton(object, info, rect);
}

IntPoint RenderThemeManx::volumeSliderOffsetFromMuteButton(RenderBox* muteButtonBox, const IntSize& size) const
{
    return RenderMediaControls::volumeSliderOffsetFromMuteButton(muteButtonBox, size);
}
#endif

bool RenderThemeManx::delegatesMenuListRendering() const
{
    return false;
}

bool RenderThemeManx::popsMenuByArrowKeys() const
{
    return false;
}

bool RenderThemeManx::popsMenuBySpaceOrReturn() const
{
    return false;
}

}
