/*
    Copyright (C) 2012 Samsung Electronics
    Copyright (C) 2012 Intel Corporation. All rights reserved.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "config.h"
#include "GraphicsContext3DPrivate.h"

#if USE(3D_GRAPHICS) || USE(ACCELERATED_COMPOSITING)

#include "HostWindow.h"
#include "NotImplemented.h"

#if USE(ACCELERATED_COMPOSITING) && USE(TEXTURE_MAPPER) && USE(TEXTURE_MAPPER_GL)
#include <texmap/TextureMapperGL.h>
#endif

#if OS(ORBIS)
#include <canvas.h>
#endif

namespace WebCore {

PassOwnPtr<GraphicsContext3DPrivate> GraphicsContext3DPrivate::create(GraphicsContext3D* context, HostWindow* hostWindow)
{
    OwnPtr<GraphicsContext3DPrivate> platformLayer = adoptPtr(new GraphicsContext3DPrivate(context, hostWindow));

    if (platformLayer && platformLayer->initialize())
        return platformLayer.release();

    return nullptr;
}

GraphicsContext3DPrivate::GraphicsContext3DPrivate(GraphicsContext3D* context, HostWindow* hostWindow)
    : m_context(context)
    , m_hostWindow(hostWindow)
#if OS(ORBIS)
    , m_canvasID(sce::Canvas::kInvalidIdValue)
    , m_surfaceHandle(sce::Canvas::kInvalidHandle)
#endif
{
}

bool GraphicsContext3DPrivate::initialize()
{
    if (m_context->m_renderStyle == GraphicsContext3D::RenderDirectlyToHostWindow)
        return false;

    if (m_hostWindow && m_hostWindow->platformPageClient()) {
        // FIXME: Implement this code path for WebKit1.
        // Get Evas object from platformPageClient and set EvasGL related members.
        return false;
    }

    m_offScreenContext = GLPlatformContext::createContext(m_context->m_renderStyle);
    if (!m_offScreenContext)
        return false;

    if (m_context->m_renderStyle == GraphicsContext3D::RenderOffscreen) {
        m_offScreenSurface = GLPlatformSurface::createOffScreenSurface();

        if (!m_offScreenSurface)
            return false;

        if (!m_offScreenContext->initialize(m_offScreenSurface.get()))
            return false;

        if (!makeContextCurrent())
            return false;
#if USE(GRAPHICS_SURFACE)
        m_surfaceOperation = CreateSurface;
#endif
    }

    return true;
}

GraphicsContext3DPrivate::~GraphicsContext3DPrivate()
{
    releaseResources();
}

void GraphicsContext3DPrivate::releaseResources()
{
    if (m_context->m_renderStyle == GraphicsContext3D::RenderToCurrentGLContext)
        return;

    // Release the current context and drawable only after destroying any associated gl resources.
#if USE(GRAPHICS_SURFACE)
    m_surfaceHandle = releaseSurfaceHandle();
#endif

    if (m_offScreenContext) {
        m_offScreenContext->releaseCurrent();
        m_offScreenContext->destroy();
    }

    if (m_offScreenSurface)
        m_offScreenSurface->destroy();
}

void GraphicsContext3DPrivate::setContextLostCallback(PassOwnPtr<GraphicsContext3D::ContextLostCallback> callBack)
{
    m_contextLostCallback = callBack;
}

PlatformGraphicsContext3D GraphicsContext3DPrivate::platformGraphicsContext3D() const
{
    return m_offScreenContext->handle();
}

bool GraphicsContext3DPrivate::makeContextCurrent() const
{
    bool success = m_offScreenContext->makeCurrent(m_offScreenSurface.get());

    if (!m_offScreenContext->isValid()) {
        // FIXME: Restore context
        if (m_contextLostCallback)
            m_contextLostCallback->onContextLost();

        return false;
    }

    return success;
}

#if USE(TEXTURE_MAPPER_GL)
void GraphicsContext3DPrivate::paintToTextureMapper(TextureMapper* textureMapper, const FloatRect& targetRect, const TransformationMatrix& matrix, float opacity)
{
    if (!m_context)
        return;

    if (textureMapper->accelerationMode() != TextureMapper::OpenGLMode) {
        notImplemented();
        return;
    }
/*
    if (m_context->m_attrs.antialias && m_context->m_boundFBO == m_context->m_multisampleFBO) {
        GLContext* previousActiveContext = GLContext::getCurrent();

        if (previousActiveContext != m_glContext)
            m_context->makeContextCurrent();

        m_context->resolveMultisamplingIfNecessary();

        if (previousActiveContext && previousActiveContext != m_glContext)
            previousActiveContext->makeContextCurrent();
    }
*/
    TextureMapperGL* texmapGL = static_cast<TextureMapperGL*>(textureMapper);
    TextureMapperGL::Flags flags = TextureMapperGL::ShouldFlipTexture | (m_context->m_attrs.alpha ? TextureMapperGL::ShouldBlend : 0);
    IntSize textureSize(m_context->m_currentWidth, m_context->m_currentHeight);
    texmapGL->drawTexture(m_context->m_texture, flags, textureSize, targetRect, matrix, opacity);
}
#endif

#if USE(GRAPHICS_SURFACE)
#if OS(ORBIS)
GraphicsSurfaceToken GraphicsContext3DPrivate::bindSurfaceHandle()
{
    if (m_canvasID == sce::Canvas::kInvalidIdValue)
        m_canvasID = sce::Canvas::acquireId();
    
    if (m_canvasID == sce::Canvas::kInvalidIdValue)
        CRASH(); // Ran out of canvas ID pool.

    sce::CanvasUtil::bindTexture(m_canvasID, m_context->m_platformTexture.get());
    return GraphicsSurfaceToken(sce::Canvas::handle(m_canvasID));
}

GraphicsSurfaceToken GraphicsContext3DPrivate::releaseSurfaceHandle()
{
    if (m_canvasID != sce::Canvas::kInvalidIdValue) {
        sce::CanvasUtil::bindTexture(m_canvasID, 0);
        sce::Canvas::releaseId(m_canvasID);
        m_canvasID = sce::Canvas::kInvalidIdValue;
    }

    return GraphicsSurfaceToken(sce::Canvas::kInvalidHandle);
}
#endif

void GraphicsContext3DPrivate::didResizeCanvas(const IntSize& size)
{
    m_size = size;
    if (m_size.isEmpty())
        m_surfaceHandle = releaseSurfaceHandle();
    else
        m_surfaceHandle = bindSurfaceHandle();
}

uint32_t GraphicsContext3DPrivate::copyToGraphicsSurface()
{
    FUNCTION_MARKER;
    m_context->makeContextCurrent();
    m_surfaceHandle = bindSurfaceHandle();
#if OS(ORBIS) // bz133659
    if (!m_context->layerComposited())
        m_context->flush();
#else
    m_context->flush();
#endif
    return m_surfaceHandle.frontBufferHandle;
}

GraphicsSurfaceToken GraphicsContext3DPrivate::graphicsSurfaceToken() const
{
    return m_surfaceHandle;
}

IntSize GraphicsContext3DPrivate::platformLayerSize() const
{
    return IntSize(1, 1);
}

GraphicsSurface::Flags GraphicsContext3DPrivate::graphicsSurfaceFlags() const
{
    if (m_surfaceHandle.isValid())
        return GraphicsSurface::SupportsAlpha | GraphicsSurface::SupportsTextureSource | GraphicsSurface::SupportsSharing | GraphicsSurface::ShouldFlipTexture;

    return TextureMapperPlatformLayer::graphicsSurfaceFlags();
}

#endif

} // namespace WebCore

#endif // USE(3D_GRAPHICS) || USE(ACCELERATED_COMPOSITING)
