/*
 * Copyright (C) 2012 Sony Interactive Entertainment Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#include "config.h"
#include "FtStreamManx.h"

namespace WebCore {

  /*************************************************************************
   *
   * @functype:
   *   FT_Stream_IoFunc
   *
   * @description:
   *   A function used to seek and read data from a given input stream.
   *
   * @input:
   *   stream ::
   *     A handle to the source stream.
   *
   *   offset ::
   *     The offset of read in stream (always from start).
   *
   *   buffer ::
   *     The address of the read buffer.
   *
   *   count ::
   *     The number of bytes to read from the stream.
   *
   * @return:
   *   The number of bytes effectively read by the stream.
   *
   * @note:
   *   This function might be called to perform a seek or skip operation
   *   with a `count' of~0.  A non-zero return value then indicates an
   *   error.
   *
   */
static unsigned long FtStreamRead(FT_Stream stream, unsigned long offset, unsigned char* buffer, unsigned long count)
{
    ASSERT(stream);
    ASSERT(stream->descriptor.pointer);
    FontStreamBuffer* streamBuffer = reinterpret_cast<FontStreamBuffer*>(stream->descriptor.pointer);

    if (!count) {
        // Seek operation required. Check the offset value does not exceed the file size.
        if (offset < streamBuffer->fileSize())
            return 0;
        
        return -1;
    }

    // Do read operation.
    int readCount = streamBuffer->read(buffer, count, offset);
    if (readCount >= 0)
        return (unsigned long)readCount;
    
    return 0;
}

  /*************************************************************************
   *
   * @functype:
   *   FT_Stream_CloseFunc
   *
   * @description:
   *   A function used to close a given input stream.
   *
   * @input:
   *  stream ::
   *     A handle to the target stream.
   *
   */
static void FtStreamClose(FT_Stream stream)
{
    destroyFtStream(stream);
}

FT_Stream createFtStream(FontStreamBuffer* buffer)
{
    if (!buffer)
        return 0;

    FT_Stream stream = new FT_StreamRec_;
    if (!stream)
        return 0;

    buffer->attach();

    stream->base = 0;
    stream->size = buffer->fileSize();
    stream->pos = 0;
    stream->descriptor.pointer = buffer;
    stream->pathname.pointer = 0;
    stream->read = FtStreamRead;
    stream->close = FtStreamClose;
    return stream;
}

void destroyFtStream(FT_Stream stream)
{
    ASSERT(stream);
    ASSERT(stream->descriptor.pointer);
    FontStreamBuffer* streamBuffer = reinterpret_cast<FontStreamBuffer*>(stream->descriptor.pointer);
    streamBuffer->detach();

    delete stream;
}

}
