/*
 * Copyright (C) 2013 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "MediaKeys.h"

#if ENABLE(ENCRYPTED_MEDIA_V2)

#include "CDM.h"
#include "EventNames.h"
#include "HTMLMediaElement.h"
#include "MediaKeyMessageEvent.h"
#include "MediaKeySession.h"
#include "UUID.h"
#include <wtf/HashSet.h>

namespace WebCore {

PassRefPtr<MediaKeys> MediaKeys::create(const String& keySystem, ExceptionCode& ec)
{
    // From <http://dvcs.w3.org/hg/html-media/raw-file/tip/encrypted-media/encrypted-media.html#dom-media-keys-constructor>:
    // The MediaKeys(keySystem) constructor must run the following steps:

    // 1. If keySystem is null or an empty string, throw an INVALID_ACCESS_ERR exception and abort these steps.
    if (keySystem.isNull() || keySystem.isEmpty()) {
        ec = INVALID_ACCESS_ERR;
        return 0;
    }

    // 2. If keySystem is not one of the user agent's supported Key Systems, throw a NOT_SUPPORTED_ERR and abort these steps.
    if (!CDM::supportsKeySystem(keySystem)) {
        ec = NOT_SUPPORTED_ERR;
        return 0;
    }

    // 3. Let cdm be the content decryption module corresponding to keySystem.
    // 4. Load cdm if necessary.
    OwnPtr<CDM> cdm = CDM::create(keySystem);

    // 5. Create a new MediaKeys object.
    // 5.1 Let the keySystem attribute be keySystem.
    // 6. Return the new object to the caller.
    return adoptRef(new MediaKeys(keySystem, cdm.release()));
}

MediaKeys::MediaKeys(const String& keySystem, PassOwnPtr<CDM> cdm)
    : m_mediaElement(0)
    , m_keySystem(keySystem)
    , m_cdm(cdm)
{
    m_cdm->setClient(this);
}

MediaKeys::~MediaKeys()
{
    // From <http://dvcs.w3.org/hg/html-media/raw-file/tip/encrypted-media/encrypted-media.html#dom-media-keys-constructor>:
    // When destroying a MediaKeys object, follow the steps in close().
    for (size_t i = 0; i < m_sessions.size(); ++i) {
        m_sessions[i]->close();
        m_sessions[i]->setKeys(0);
    }
}

PassRefPtr<MediaKeySession> MediaKeys::createSession(ScriptExecutionContext* context, const String& type, Uint8Array* initData, ExceptionCode& ec)
{
    // From <http://dvcs.w3.org/hg/html-media/raw-file/tip/encrypted-media/encrypted-media.html#dom-createsession>:
    // The createSession(type, initData) method must run the following steps:
    // Note: The contents of initData are container-specific Initialization Data.

    // 1. If type is null or an empty string and initData is not null or an empty string, throw an
    // INVALID_ACCESS_ERR exception and abort these steps.
    if ((type.isNull() || type.isEmpty()) && (!initData || initData->length())) {
        ec = INVALID_ACCESS_ERR;
        return 0;
    }

    // 2. If type contains a MIME type that is not supported or is not supported by the keySystem, throw
    // a NOT_SUPPORTED_ERR exception and abort these steps.
    if (!type.isNull() && !type.isEmpty() && !m_cdm->supportsMIMEType(type)) {
        ec = NOT_SUPPORTED_ERR;
        return 0;
    }

    // 3. Create a new MediaKeySession object.
    // 3.1 Let the keySystem attribute be keySystem.
    // 3.2 Let the sessionId attribute be a unique Session ID string. It may be generated by cdm.
    RefPtr<MediaKeySession> session = MediaKeySession::create(context, this, keySystem());

    // 4. Add the new object to an internal list of session objects.
    m_sessions.append(session);

    // 5. Schedule a task to generate a key request, providing type, initData, and the new object.
    session->generateKeyRequest(type, initData);

    // 6. Return the new object to the caller.
    return session;
}

bool MediaKeys::isTypeSupported(const String& keySystem, const String& mimeType)
{
    // 1. If keySystem contains an unrecognized or unsupported Key System, return false and abort these steps.
    // Key system string comparison is case-sensitive.
    if (keySystem.isNull() || keySystem.isEmpty() || !CDM::supportsKeySystem(keySystem))
        return false;

    // 2. If type is null or an empty string, return true and abort these steps.
    if (mimeType.isNull() || mimeType.isEmpty())
        return true;

    // 3. If the Key System specified by keySystem does not support decrypting the container and/or codec
    // specified by type, return false and abort these steps.
    if (!CDM::keySystemSupportsMimeType(keySystem, mimeType))
        return false;

    // 4. Return true;
    return true;
}

void MediaKeys::setMediaElement(HTMLMediaElement* element)
{
    m_mediaElement = element;
}

MediaPlayer* MediaKeys::cdmMediaPlayer(const CDM*) const
{
    if (m_mediaElement)
        return m_mediaElement->player();
    return 0;
}

}

#endif
