/*
 * Copyright (C) 2010 Apple Inc. All rights reserved.
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef WebView_h
#define WebView_h

#include "APIObject.h"
#include "CanvasHole.h"
#include "Cursor.h"
#include "PageClientImpl.h"
#include "PageViewportController.h"
#include "PageViewportControllerClientManx.h"
#include "WebFullScreenManagerProxy.h"
#include "WebPageProxy.h"
#include "WebPopupMenuClient.h"
#include "WebPopupMenuProxyManx.h"
#include "WebViewAccessibilityClient.h"
#include "WebViewClient.h"
#include "WebViewContextMenuClient.h"
#include <WKView.h>
#include <WebCore/CoordinatedGraphicsScene.h>
#include <wtf/Forward.h>
#include <wtf/PassRefPtr.h>
#include <wtf/RefPtr.h>

namespace WebKit {

class WebView : public APIObject {
public:
    static PassRefPtr<WebView> create(WebContext* context, WebPageGroup* pageGroup, const WKViewClient* client)
    {
        RefPtr<WebView> webView = adoptRef(new WebView(context, pageGroup, client));
        webView->initialize();
        return webView;
    }
    ~WebView();

    void initialize();

    WebPageProxy* page() const { return m_page.get(); }
    void paint(cairo_t*, const WebCore::IntRect&);
    void setViewNeedsDisplay(const WebCore::IntRect&);
    void scrollBy(int x, int y);
    void handleKeyboardEvent(const Manx::KeyboardEvent&);
    void handleImeEvent(const Manx::ImeEvent&);
    void clearSelection();
    void handleMouseEvent(const Manx::MouseEvent&);
    void handleWheelEvent(const Manx::WheelEvent&);
    WebCore::IntSize viewSize();
    void setSize(int, int);
    void setPopupMenuClient(const WKViewPopupMenuClient*);
    void showPopupMenu(WebPopupMenuProxyManx*, const Vector<WebPopupItem>&, const WebCore::IntRect&, int32_t);
    void hidePopupMenu();
    void valueChangedForPopupMenu(int);
#if ENABLE(MANX_CURSOR_NAVIGATION)
    void setCursorPosition(const WebCore::IntPoint&);
#endif
    void setFocused(bool);
    void setActive(bool);
    void setIsVisible(bool);
    bool isFocused() const { return m_isFocused; } 
    bool isActive() const { return m_isActive; } 
    bool isVisible() const { return m_isVisible; } 
    void runJavaScriptAlert(WebFrameProxy*, const String&, PassRefPtr<Messages::WebPageProxy::RunJavaScriptAlert::DelayedReply>);
    void runJavaScriptComfirm(WebFrameProxy*, const String&, PassRefPtr<Messages::WebPageProxy::RunJavaScriptConfirm::DelayedReply>);
    void runJavaScriptPrompt(WebFrameProxy*, const String&, const String&, PassRefPtr<Messages::WebPageProxy::RunJavaScriptPrompt::DelayedReply>);
    void handleAuthenticationRequiredRequest(WebFrameProxy*, const String&, const String&, const String&, const String& prefilledPassword, PassRefPtr<Messages::WebPageProxy::AuthenticationRequiredRequest::DelayedReply>);
    void handleCertificateVerificationRequest(WebFrameProxy*, uint32_t error, const String& url, const Vector<CString>& certificates, PassRefPtr<Messages::WebPageProxy::CertificateVerificationRequest::DelayedReply>);
    void createNewPage(WebPageProxy*, const WebCore::ResourceRequest&, const WebCore::WindowFeatures&, WebEvent::Modifiers, WebMouseEvent::Button, PassRefPtr<Messages::WebPageProxy::CreateNewPage::DelayedReply>);
    void replyJavaScriptAlert();
    void replyJavaScriptConfirm(bool confirmResult);
    void replyJavaScriptPrompt(const String& promptResult);
    void replyAuthenticationChallenge(bool result, const String& username, const String& password);
    void replyCertificateVerificationRequest(bool result);
    void replyCreateNewPage(PassRefPtr<WebPageProxy>);
    void didChangeContentsSize(const WebCore::IntSize&);
    void didChangeEditorState(const WebKit::EditorState&);
    void didRelaunchProcess();
    void doneWithKeyEvent(const NativeWebKeyboardEvent&, bool wasEventHandled);
    void doneWithMouseDownEvent(bool wasEventHandled);
    void doneWithMouseUpEvent(bool wasEventHandled);

#if ENABLE(FULLSCREEN_API)
    void invalidateFullScreen();
    bool isFullScreen();
    void closeFullScreen();
    void enterFullScreen();
    void exitFullScreen();
    void beganEnterFullScreen(const WebCore::IntRect&, const WebCore::IntRect&);
    void beganExitFullScreen(const WebCore::IntRect&, const WebCore::IntRect&);
    void willEnterFullScreen();
    void didEnterFullScreen();
    void willExitFullScreen();
    void didExitFullScreen();
    void requestExitFullScreen();   
#endif // ENABLE(FULLSCREEN_API)

#if USE(ACCELERATED_COMPOSITING)
    void enterAcceleratedCompositingMode();
    void exitAcceleratedCompositingMode();
#endif // USE(ACCELERATED_COMPOSITING)

    void setCursor(const WebCore::Cursor&);

#if ENABLE(CONTEXT_MENUS)
    void setContextMenuClient(const WKViewContextMenuClient*);
    void showContextMenu(const WebCore::IntPoint&, const Vector<WebContextMenuItemData>&);
    void hideContextMenu();
    void contextMenuItemSelected(WebContextMenuItemData*);
#endif // ENABLE(CONTEXT_MENUS)

#if USE(TILED_BACKING_STORE)
    void pageDidRequestScroll(const WebCore::IntPoint&);
    void didRenderFrame(const WebCore::IntSize& contentsSize, const WebCore::IntRect& coveredRect);
    void pageTransitionViewportReady();
    void postFakeMouseMove();
#endif
    void didChangeViewportProperties(const WebCore::ViewportAttributes&);
    void didCommitLoad();

    void didChangeContentsVisibility(const WebCore::IntSize&, const WebCore::FloatPoint& position, float scale);

    void didUpdateBackingStoreState();
    void didCommitCoordinatedGraphicsState();

    void paintToCurrentGLContext(const WebCore::FloatRect&, WebCore::TextureMapper::PaintFlags);

    void clearBackingStores();
    void restoreBackingStores();
    void syncCoordinatedGraphicsState();

#if HAVE(ACCESSIBILITY)
    void setAccessibilityClient(const WKViewAccessibilityClient*);
    void handleAccessibilityNotification(WebAccessibilityObject*, WebCore::AXObjectCache::AXNotification);
    void handleAccessibilityTextChange(WebAccessibilityObject*, WebCore::AXObjectCache::AXTextChange, uint32_t, const String&);
    void handleAccessibilityLoadingEvent(WebAccessibilityObject*, WebCore::AXObjectCache::AXLoadingEvent);
    WebAccessibilityObject* accessibilityRootObject();
    WebAccessibilityObject* accessibilityFocusedObject();
    WebAccessibilityObject* accessibilityHitTest(const IntPoint&);
#endif

#if ENABLE(MANX_INDIRECT_COMPOSITING)
    WebCanvasHole canvasHole() const;
#endif

    void setPageScaleFactor(float newScale);
    void updateViewport();

private:
    WebView(WebContext*, WebPageGroup*, const WKViewClient*);

    bool useFixedLayout() const;

#if USE(COORDINATED_GRAPHICS)
    WebCore::CoordinatedGraphicsScene* coordinatedGraphicsScene();
#endif

    void didChangeCompositionState(const WebCore::IntRect& compositionRect);
    void didChangeSelectionState(const String& selectedText, const WebCore::IntRect& selectedRect);

    virtual Type type() const { return TypeView; }

    PageClientImpl m_pageClient;
    RefPtr<WebPageProxy> m_page;
    WebViewClient m_viewClient;
    WebPopupMenuProxyManx* m_popupMenu;
    WebPopupMenuClient m_popupMenuClient;
    WebViewContextMenuClient m_contextMenuClient;
#if HAVE(ACCESSIBILITY)
    WebViewAccessibilityClient m_axClient;
#endif

#if USE(ACCELERATED_COMPOSITING)
    OwnPtr<PageViewportControllerClientManx> m_viewportControllerClient;
    PageViewportController* m_viewportController;
#endif

    bool m_isFocused;
    bool m_isActive;
    bool m_isVisible;

    RefPtr<Messages::WebPageProxy::RunJavaScriptAlert::DelayedReply> m_alertReply;
    RefPtr<Messages::WebPageProxy::RunJavaScriptConfirm::DelayedReply> m_confirmReply;
    RefPtr<Messages::WebPageProxy::RunJavaScriptPrompt::DelayedReply> m_promptReply;
    RefPtr<Messages::WebPageProxy::AuthenticationRequiredRequest::DelayedReply> m_authReply;
    RefPtr<Messages::WebPageProxy::CertificateVerificationRequest::DelayedReply> m_certReply;
    RefPtr<Messages::WebPageProxy::CreateNewPage::DelayedReply> m_createNewPageReply;

#if ENABLE(FULLSCREEN_API)
    bool m_isFullScreen;
#endif

    WebCore::IntSize m_size;
    WebCore::IntSize m_contentsSize;
    WebCore::IntPoint m_lastMousePosition;

#if USE(COORDINATED_GRAPHICS) && OS(ORBIS)
    WebCore::GLMemoryInfo m_glMemoryInfo;
#endif
    EditorState m_editorState;
};

} // namespace WebKit

#endif // WebView_h
