/*
 * Copyright (C) 2008 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "InspectorClientManx.h"

#include "Frame.h"
#include "FrameView.h"
#include "InspectorController.h"
#include "InspectorServerManx.h"
#include "NotImplemented.h"
#include "Page.h"
#include "WebViewPrivate.h"
#include <wtf/text/WTFString.h>

using namespace WebKit;

namespace WebCore {

InspectorClientManx::InspectorClientManx(WebViewPrivate* webview)
    : m_webview(webview)
    , m_remoteFrontendClientId(0)
    , m_remoteFrontendConnected(false)
{
}

InspectorClientManx::~InspectorClientManx()
{
}

WebViewPrivate* InspectorClientManx::webview() const
{
    return m_webview;
}

Page* InspectorClientManx::page() const
{
    return m_webview->m_page.get();
}

void InspectorClientManx::setRemoteInspectionEnabled(bool enabled)
{
    if (enabled && !m_remoteFrontendClientId)
        m_remoteFrontendClientId = InspectorServerManx::shared().registerClient(this);
    else if (!enabled && m_remoteFrontendClientId)
        InspectorServerManx::shared().unregisterClient(m_remoteFrontendClientId);
}

bool InspectorClientManx::remoteInspectionEnabled() const
{
    return m_remoteFrontendClientId;
}

void InspectorClientManx::inspectorDestroyed()
{
    if (m_remoteFrontendClientId)
        InspectorServerManx::shared().unregisterClient(m_remoteFrontendClientId);

    delete this;
}

WebCore::InspectorFrontendChannel* InspectorClientManx::openInspectorFrontend(InspectorController* controller)
{
    notImplemented();
    return 0;
}

void InspectorClientManx::closeInspectorFrontend()
{
    notImplemented();
}

void InspectorClientManx::bringFrontendToFront()
{
    notImplemented();
}

void InspectorClientManx::highlight()
{
    Frame* mainFrame = m_webview->m_page->mainFrame();
    if (!mainFrame)
        return;

    m_webview->invalidate(IntRect(IntPoint(), mainFrame->view()->contentsSize()), false);
}

void InspectorClientManx::hideHighlight()
{
    highlight();
}

void InspectorClientManx::populateSetting(const WTF::String& key, WTF::String* value)
{
    notImplemented();
}

void InspectorClientManx::storeSetting(const WTF::String& key, const WTF::String& value)
{
    notImplemented();
}

bool InspectorClientManx::sendMessageToFrontend(const WTF::String& message)
{
    if (m_remoteFrontendClientId && m_remoteFrontendConnected) {
        InspectorServerManx::shared().sendMessageOverConnection(m_remoteFrontendClientId, message);
        return true;
    }

    notImplemented();
    return false;
}

void InspectorClientManx::resourceTrackingWasEnabled()
{
    notImplemented();
}

void InspectorClientManx::resourceTrackingWasDisabled()
{
    notImplemented();
}

void InspectorClientManx::timelineProfilerWasStarted()
{
    notImplemented();
}

void InspectorClientManx::timelineProfilerWasStopped()
{
    notImplemented();
}

void InspectorClientManx::dispatchMessageFromRemoteFrontend(const String& message)
{
    ASSERT(m_remoteFrontendConnected);
    if (m_webview->m_page)
        m_webview->m_page->inspectorController()->dispatchMessageFromFrontend(message);
}


void InspectorClientManx::remoteFrontendConnected()
{
    ASSERT(!m_remoteFrontendConnected);
    if (m_webview->m_page) {
        m_webview->m_page->inspectorController()->connectFrontend(this);
        m_remoteFrontendConnected = true;
    }
}

void InspectorClientManx::remoteFrontendDisconnected()
{
    ASSERT(m_remoteFrontendConnected);
    if (m_webview->m_page) {
        m_remoteFrontendConnected = false;
        m_webview->m_page->inspectorController()->disconnectFrontend();
    }
}

} // namespace WebKit
