/*
 * Copyright (C) 2009 Apple Inc. All rights reserved.
 * Copyright (C) 2009 Google Inc.  All rights reserved.
 * Copyright (C) 2012 Sony Network Entertainment Intl.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef SocketStreamHandle_h
#define SocketStreamHandle_h

#include "SocketStreamHandleBase.h"

#include <curl/curl.h>
#include <ntf/SslVerifier.h>
#include <wtf/PassRefPtr.h>
#include <wtf/ThreadSafeRefCounted.h>

namespace WebCore {

class AuthenticationChallenge;
class Credential;
class RunLoop;
class SocketStreamHandleClient;
class SocketStreamHandleManager;

class SocketStreamHandle : public ThreadSafeRefCounted<SocketStreamHandle>, public SocketStreamHandleBase {
    friend class SocketStreamHandleManager;
public:
    // These methods call initialize() automatically if the client is not null.
    // They MUST then be called on the client's thread.
    static PassRefPtr<SocketStreamHandle> create(const KURL&, SocketStreamHandleClient*);
    static PassRefPtr<SocketStreamHandle> create(int socketfd, SocketStreamHandleClient*);

    virtual ~SocketStreamHandle();

    // If the client is set after creation then this method MUST be called manually after calling setClient(),
    // and MUST be called on the client's thread.
    void initialize();

    bool isDisconnected() { return m_isDisconnected; }
    KURL& url();

    NTF::SslVerifier* sslVerifier() { return m_sslVerifier; }

protected:
    virtual int platformSend(const char* data, int length);
    virtual void platformClose();

private:
    SocketStreamHandle(const KURL&, SocketStreamHandleClient*);
    SocketStreamHandle(int socketfd, SocketStreamHandleClient*);

    void readyRead();
    void readyWrite();
    void remoteDisconnected();

    // No authentication for streams per se, but proxy may ask for credentials.
    void didReceiveAuthenticationChallenge(const AuthenticationChallenge&);
    void receivedCredential(const AuthenticationChallenge&, const Credential&);
    void receivedRequestToContinueWithoutCredential(const AuthenticationChallenge&);
    void receivedCancellation(const AuthenticationChallenge&);

    CURL* m_handle;
    curl_socket_t m_socket;
    NTF::SslVerifier* m_sslVerifier;

    bool m_hasPendingDataForSending;
    bool m_hasPendingReadOrDisconnect;
    bool m_isDisconnected;

    RunLoop* m_ownerRunLoop;
};

} // namespace WebCore

#endif // SocketStreamHandle_h
