/*
 * Copyright (C) 2014 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
#include "config.h"

#if ENABLE(MEDIA_SOURCE)

#include "MediaSourcePrivateManx.h"

#include "SourceBufferPrivateManx.h"

#include <wtf/PassOwnPtr.h>
#include <wtf/text/CString.h>
#include <wtf/text/WTFString.h>

// #define TRACE_FUNCTION printf("[WebCore::MediaPlayerPrivate] %s\n", __FUNCTION__)
#define TRACE_FUNCTION

namespace WebCore {

MediaSourcePrivateManx* MediaSourcePrivateManx::create()
{
    return new MediaSourcePrivateManx();
}

MediaSourcePrivateManx::MediaSourcePrivateManx()
    : m_platformMediaSource(0)
{
    TRACE_FUNCTION;
    m_platformMediaSource = Manx::MediaSource::create();
}

MediaSourcePrivateManx::~MediaSourcePrivateManx()
{
    TRACE_FUNCTION;
    delete m_platformMediaSource;
}

MediaSourcePrivate::AddStatus MediaSourcePrivateManx::addSourceBuffer(const String& type, const CodecsArray& codecs, OwnPtr<SourceBufferPrivate>* sourceBufferPrivate)
{
    TRACE_FUNCTION;

    // codecs -> char []
    char codecsStr[MaxCodecsLen];
    convertCodecsToString(codecs, codecsStr, MaxCodecsLen);

    // Private pointer -> MediaSource module
    SourceBufferPrivateManx* sourceBufferPrivateManx = SourceBufferPrivateManx::create();
    *sourceBufferPrivate = adoptPtr<SourceBufferPrivateManx>(sourceBufferPrivateManx);

    Manx::SourceBuffer* platformSourceBuffer = sourceBufferPrivateManx->getPlatformInstance();

    // addSourceBuffer
    Manx::MediaSource::AddStatus addStatus;
    addStatus = m_platformMediaSource->addSourceBuffer(type.utf8().data(), codecsStr, platformSourceBuffer);

    AddStatus status;
    switch (addStatus) {
    case Manx::MediaSource::Ok:
        status = Ok;
        break;
    case Manx::MediaSource::NotSupported:
        status = NotSupported;
        break;
    case Manx::MediaSource::ReachedIdLimit:
        status = ReachedIdLimit;
        break;
    }

    return status;
}

double MediaSourcePrivateManx::duration()
{
    TRACE_FUNCTION;
    return m_platformMediaSource->duration();
}

void MediaSourcePrivateManx::setDuration(double duration)
{
    TRACE_FUNCTION;
    m_platformMediaSource->setDuration(duration);
}

void MediaSourcePrivateManx::endOfStream(EndOfStreamStatus endOfStreamStatus)
{
    TRACE_FUNCTION;

    Manx::MediaSource::EndOfStreamStatus streamStatus;

    switch (endOfStreamStatus) {
    case EosNoError:
        streamStatus = Manx::MediaSource::EosNoError;
        break;
    case EosNetworkError:
        streamStatus = Manx::MediaSource::EosNetworkError;
        break;
    case EosDecodeError:
        streamStatus = Manx::MediaSource::EosDecodeError;
        break;
    }
    m_platformMediaSource->endOfStream(streamStatus);
}

void MediaSourcePrivateManx::convertCodecsToString(const CodecsArray& codecs, char* str, uint32_t len)
{
    str[0] = '\0';
    for (CodecsArray::const_iterator it = codecs.begin(); it != codecs.end(); it++) {
        if (it->isNull())
            continue;
        if (strlen(str) + strlen(it->utf8().data()) > len)
            break;  
        if (it != codecs.begin())
            strcat(str, ",");
        strcat(str, it->utf8().data());
    }
}

} // namespace WebCore

#endif // ENABLE(MEDIA_SOURCE)
