/*
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#ifndef FontStreamBufferManx_h
#define FontStreamBufferManx_h

#include <wtf/DoublyLinkedList.h>
#include <wtf/PassRefPtr.h>
#include <wtf/RefCounted.h>

namespace WebCore {

class FontStreamBuffer : public RefCounted<FontStreamBuffer> {
public:
    class BufferInfo : public DoublyLinkedListNode<BufferInfo> {
        friend DoublyLinkedListNode<BufferInfo>;
    public:
        BufferInfo()
            : m_data(0)
            , m_offset(0)
            , m_contentSize(0) { }

        void* data() const { return m_data; }
        unsigned offset() const { return m_offset; }
        unsigned contentSize() const { return m_contentSize; }

        void setData(void* data) { m_data = data; }
        void setOffset(unsigned offset) { m_offset = offset; }
        void setContentSize(unsigned contentSize) { m_contentSize = contentSize; }

    private:
        void* m_data;
        unsigned m_offset;
        unsigned m_contentSize;
        BufferInfo* m_prev;
        BufferInfo* m_next;
    };
    typedef DoublyLinkedList<BufferInfo> BufferInfoList;

    typedef int (*SeekFunc)(unsigned offset, void* handle);
    typedef int (*ReadFunc)(void* buffer, unsigned count, void* handle);
    typedef int (*CloseFunc)(void* handle);
    typedef void (*DestroyFunc)(void* handle);

    // If non-null DestroyFunc is passed, it is called to destroy handle object.
    // Otherwise caller must destroy handle object by its own.
    FontStreamBuffer(SeekFunc, ReadFunc, CloseFunc, DestroyFunc, size_t fileSize, void* handle);
    ~FontStreamBuffer();

    int read(void* buffer, size_t count, size_t offset);
    size_t fileSize() const { return m_fileSize; }

    void attach();
    void detach();
    bool isAttached() const { return m_attachCount; }

private:
    FontStreamBuffer() { }

    void close();
    int rawRead(void* buffer, size_t count, size_t offset);

    SeekFunc m_seekFunc;
    ReadFunc m_readFunc;
    CloseFunc m_closeFunc;
    DestroyFunc m_destroyFunc;
    void* m_handle;
    size_t m_fileSize;
    int m_attachCount;

    enum {
        NumOfBuffers = 32
    };
    BufferInfo m_bufferInfo[NumOfBuffers];

    BufferInfoList m_bufferInfoList;

    void* m_bufferBaseAddr;
};

WTF::PassRefPtr<FontStreamBuffer> createFontStreamBuffer(const char* fontPath);

namespace FontStreamBufferCache {

void init();
WTF::PassRefPtr<FontStreamBuffer> get(const char* fontPath);

}

}

#endif
