/*
 * Copyright (C) 2008 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebViewHitTestResult.h"

#include "IntRect.h"
#include "KURL.h"
#include <wtf/text/CString.h>

WebViewHitTestResult::WebViewHitTestResult()
    : m_imageUrl(new WebCore::KURL())
    , m_imageLength(0)
    , m_linkUrl(new WebCore::KURL())
    , m_linkLength(0)
    , m_rect(new RectInfo())
    , m_point(new PointInfo())
    , m_format(new WTF::CString(""))
    , m_formatLength(0)
    , m_imageSize(0)
{
}

WebViewHitTestResult::WebViewHitTestResult(const WebViewHitTestResult* base)
{
    m_imageUrl = new WebCore::KURL(WebCore::KURL(), WTF::String(base->imageUrl()));
    m_imageLength = base->imageUrlLength();

    m_linkUrl = new WebCore::KURL(WebCore::KURL(), WTF::String(base->linkUrl()));
    m_linkLength = base->linkUrlLength();

    m_format = new WTF::CString(base->format());
    m_formatLength = base->formatLength();

    m_rect = new RectInfo();
    RectInfo rect;
    base->getRect(rect);
    WebCore::IntRect intRect(rect.x, rect.y, rect.width, rect.height);
    this->setRect(intRect);

    m_point = new PointInfo();
    PointInfo point;
    base->getPoint(point);
    WebCore::IntPoint intPoint(point.x, point.y);
    this->setPoint(intPoint);

    m_imageSize = base->imageSize();
    m_canScroll = base->canScroll();
}

WebViewHitTestResult::~WebViewHitTestResult()
{
    delete m_imageUrl;
    delete m_linkUrl;
    delete m_rect;
    delete m_point;
    delete m_format;
}

void WebViewHitTestResult::setImageUrl(WebCore::KURL url)
{
    delete m_imageUrl;
    m_imageUrl = new WebCore::KURL(url, WTF::String(url.string().utf8().data()));
}

const char* WebViewHitTestResult::imageUrl() const
{
    return m_imageUrl->isEmpty() ? "" : (const char*)m_imageUrl->string().characters8();
}

void WebViewHitTestResult::setLinkUrl(WebCore::KURL url)
{
    delete m_linkUrl;
    m_linkUrl = new WebCore::KURL(url, WTF::String(url.string().utf8().data()));
}

const char* WebViewHitTestResult::linkUrl() const
{
    return m_linkUrl->isEmpty() ? "" : (const char*)m_linkUrl->string().characters8();
}

void WebViewHitTestResult::setRect(const WebCore::IntRect intRect)
{
    if (intRect.isEmpty()) {
        m_rect->x = 0;
        m_rect->y = 0;
        m_rect->width = 0;
        m_rect->height = 0;
    } else {
        m_rect->x = intRect.location().x();
        m_rect->y = intRect.location().y();
        m_rect->width = intRect.size().width();
        m_rect->height = intRect.size().height();
    }
}

void WebViewHitTestResult::getRect(RectInfo& rect) const
{
    rect.x = m_rect->x;
    rect.y = m_rect->y;
    rect.width = m_rect->width;
    rect.height = m_rect->height;
}

void WebViewHitTestResult::setPoint(const WebCore::IntPoint intPoint)
{
    m_point->x = intPoint.x();
    m_point->y = intPoint.y();
}

void WebViewHitTestResult::getPoint(PointInfo& point) const
{
    point.x = m_point->x;
    point.y = m_point->y;
}

void WebViewHitTestResult::setFormat(const char* format)
{
    delete m_format;
    m_format = new WTF::CString(format);
}

const char* WebViewHitTestResult::format() const
{
    return (const char*)m_format->data();
}
