/*
 * Copyright (C) 2009, 2010, 2011 Apple Inc. All rights reserved.
 * Copyright (C) 2013 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#ifndef MediaPlayerPrivateManx_h
#define MediaPlayerPrivateManx_h

#define USE_SELF_TIMER

#if ENABLE(VIDEO)
#include "MediaPlayerPrivate.h"
#if USE(ACCELERATED_COMPOSITING)
#include "VideoLayerManx.h"
#endif
#ifdef USE_SELF_TIMER
#include "Timer.h"
#endif
#include <manx/MediaPlayer.h>

namespace WebCore {

class MediaPlayerPrivate : public MediaPlayerPrivateInterface, public Manx::MediaPlayerClient {
public:
    static void registerMediaEngine(MediaEngineRegistrar);

    ~MediaPlayerPrivate();

#ifdef USE_SELF_TIMER
    void timerFired(Timer<MediaPlayerPrivate>*);
#endif

    void load(const String& url);
    void cancelLoad();
    
    void prepareToPlay();
    PlatformMedia platformMedia() const { return NoPlatformMedia; }
#if USE(ACCELERATED_COMPOSITING)
    PlatformLayer* platformLayer() const;
#endif

    void play();
    void pause();

    bool supportsFullscreen() const;

    IntSize naturalSize() const;

    bool hasVideo() const;
    bool hasAudio() const;

    void setVisible(bool);

    float duration() const;

    float currentTime() const;
    void seek(float time);
    bool seeking() const;

    void setRate(float);

    bool paused() const;

    void setVolume(float);

    bool supportsMuting() const;
    void setMuted(bool);

    MediaPlayer::NetworkState networkState() const;
    MediaPlayer::ReadyState readyState() const;

    float maxTimeSeekable() const;
    double minTimeSeekable() const;
    PassRefPtr<TimeRanges> buffered() const;

    bool didLoadingProgress() const;

    void setSize(const IntSize&);

    void paint(GraphicsContext*, const IntRect&);

    void setPreload(MediaPlayer::Preload);

#if ENABLE(PLUGIN_PROXY_FOR_VIDEO)
    void deliverNotification(MediaPlayerProxyNotificationType);
    void setMediaPlayerProxy(WebMediaPlayerProxy*);
    void setControls(bool) { }
#endif

#if USE(ACCELERATED_COMPOSITING)
    bool supportsAcceleratedRendering() const;
    void acceleratedRenderingStateChanged();
#endif

    MediaPlayer::MovieLoadType movieLoadType() const;

    bool hasSingleSecurityOrigin() const;

#if ENABLE(MANX_CANVAS_COMPOSITING)
    uint32_t canvasHandle() const;
#endif

#if OS(ORBIS)
    void restoreReleasedResource();
#endif

    // platform player interface
    void mediaPlayerNetworkStateChanged();
    void mediaPlayerReadyStateChanged();
    void mediaPlayerVolumeChanged(float);
    void mediaPlayerMuteChanged(bool);
    void mediaPlayerTimeChanged();
    void mediaPlayerDurationChanged();
    void mediaPlayerRateChanged();
    void mediaPlayerPlaybackStateChanged();
    void mediaPlayerRepaint();
    void mediaPlayerSizeChanged();
    void mediaPlayerEngineUpdated();
    void mediaPlayerFirstVideoFrameAvailable();
    void mediaPlayerCharacteristicChanged();
    void mediaPlayerUpdateCanvasHandle(bool isCanvasMode, uint32_t canvasHandle);
    bool mediaPlayerIsPaused() const;
    bool mediaPlayerAcceleratedCompositingEnabled();
    bool mediaPlayerIsActive() const;
    bool mediaPlayerIsVisible() const;

private:
    // constructor
    MediaPlayerPrivate(MediaPlayer*);

    // engine support
    static PassOwnPtr<MediaPlayerPrivateInterface> create(MediaPlayer*);
    static void getSupportedTypes(HashSet<String>& types);
    static MediaPlayer::SupportsType supportsType(const String& type, const String& codecs, const KURL&);
    static bool isAvailable();

    // platform support
    const String& mediaSourceTypeStringForUrl(const String& url);

#if ENABLE(VIDEO_TRACK)
    void setTracks();
#endif

    MediaPlayer* m_player;
    Manx::MediaPlayer* m_platformMediaPlayer;

#ifdef USE_SELF_TIMER
    OwnPtr<Timer<MediaPlayerPrivate> > m_timer;
#endif

#if USE(ACCELERATED_COMPOSITING)
#if USE(TEXTURE_MAPPER) && USE(TEXTURE_MAPPER_GL)
    VideoLayerManx m_platformLayer;
#elif USE(ACAGI)
    void* m_platformLayer;
#endif
#endif
};

}

#endif
#endif
