/*
 * Copyright (C) 2013 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY SCE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SCE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebTileManager.h"

#if ENABLE(MANX_HTMLTILE)

#include "DOMWrapperWorld.h"
#include "JSDOMWindow.h"
#include "JSHTMLTile.h"
#include "MessageID.h"
#include "ScriptState.h"
#include "ShareableBitmap.h"
#include "UpdateInfo.h"
#include "WebFrame.h"
#include "WebPage.h"
#include "WebProcess.h"
#include "WebTile.h"
#include "WebTileManagerProxyMessages.h"
#include <WebCore/HTMLTile.h>

using namespace WebCore;

namespace WebKit {

WebTileManager::WebTileManager(WebPage* page)
    : m_webPage(page)
{
}

WebTileManager::~WebTileManager()
{
}

bool WebTileManager::enableHTMLTileForFrame(WebFrame* frame, DOMWrapperWorld* world)
{
    if (frame->page() != m_webPage)
        return false;

    JSDOMWindow* window = toJSDOMWindow(frame->coreFrame(), world);
    ScriptState* exec = window->globalExec();

    window->putDirect(exec->globalData(), JSC::Identifier(exec, "HTMLTile"), JSHTMLTile::getConstructor(exec, window));

    return true;
}

void WebTileManager::tileUpdated(uint64_t tileID, const UpdateInfo& updateInfo)
{
    if (!m_tiles.contains(tileID)) {
        // We must still "consume" updateInfo.bitmapHandle to ensure the backing shared memory segment is released.
        // This happens automatically from ShareableBitmap::create():
        // - create() returns a PassRefPtr<ShareableBitmap> which is deleted right after.
        // - The PassRefPtr destructor derefs the ShareableBitmap object, which ends up deleted too since there was only one ref.
        // - The SharedMemory member gets deleted in turn, which is where the shared memory segment is released.
        ShareableBitmap::create(updateInfo.bitmapHandle);
        return;
    }

    m_tiles.get(tileID)->webTile->tileUpdated(updateInfo);
    m_tiles.get(tileID)->htmlTile->dispatchOnUpdateEvent(/*TODO updateInfo.updateRects*/);

    WebProcess::shared().connection()->send(Messages::WebTileManagerProxy::DidUpdateWebTile(tileID), m_webPage->pageID());
}

void WebTileManager::tileUnresponsive(uint64_t tileID)
{
    if (!m_tiles.contains(tileID))
        return;

    m_tiles.get(tileID)->htmlTile->dispatchOnUnresponsiveEvent();
}

void WebTileManager::tileCrashed(uint64_t tileID)
{
    if (!m_tiles.contains(tileID))
        return;

    m_tiles.get(tileID)->htmlTile->dispatchOnCrashEvent();
}

void WebTileManager::loadCompleted(uint64_t tileID)
{
    if (!m_tiles.contains(tileID))
        return;

    m_tiles.get(tileID)->htmlTile->dispatchOnLoadEvent();
}

void WebTileManager::loadFailed(uint64_t tileID, uint32_t errorCode)
{
    if (!m_tiles.contains(tileID))
        return;

    // Note: errorCode() is a CurlCode, see curl.h for the list of values
    m_tiles.get(tileID)->htmlTile->dispatchOnErrorEvent(errorCode);
}

void WebTileManager::locationChangeRequested(uint64_t tileID, const String& url)
{
    if (!m_tiles.contains(tileID))
        return;

    m_tiles.get(tileID)->htmlTile->dispatchOnLocationChangeEvent(url);
}

void WebTileManager::runJavaScriptResultCallback(uint64_t tileID, uint64_t callbackID, const String& result)
{
    if (!m_tiles.contains(tileID))
        return;

    m_tiles.get(tileID)->htmlTile->dispatchEvalCallback(callbackID, result);
}

void WebTileManager::didReceiveMessage(CoreIPC::Connection* connection, CoreIPC::MessageID messageID, CoreIPC::ArgumentDecoder* arguments)
{
    didReceiveWebTileManagerMessage(connection, messageID, arguments);
}

uint64_t WebTileManager::createTile(int width, int height, const unsigned char** tileData, HTMLTile* htmlTile)
{
    uint64_t tileID;
    WebProcess::shared().connection()->sendSync(Messages::WebTileManagerProxy::CreateWebTile(width, height), Messages::WebTileManagerProxy::CreateWebTile::Reply(tileID), m_webPage->pageID());

    if (!tileID)
        return 0;

    ASSERT(!m_tiles.contains(tileID));

    OwnPtr<WebTile> newTile = WebTile::create(width, height, this);
    *tileData = newTile->tileData();

    TileData* td = new TileData { newTile.release(), htmlTile };
    m_tiles.set(tileID, adoptPtr(td) );

    return tileID;
}

void WebTileManager::destroyTile(uint64_t tileID)
{
    WebProcess::shared().connection()->send(Messages::WebTileManagerProxy::DestroyWebTile(tileID), m_webPage->pageID());

    ASSERT(m_tiles.contains(tileID));

    m_tiles.remove(tileID);
}

void WebTileManager::loadURL(uint64_t tileID, const String &url)
{
    WebProcess::shared().connection()->send(Messages::WebTileManagerProxy::DidChangeWebTileURL(tileID, url), m_webPage->pageID());
}

void WebTileManager::sendEvent(uint64_t tileID, int eventType)
{
    WebProcess::shared().connection()->send(Messages::WebTileManagerProxy::SendWebTileEvent(tileID, eventType), m_webPage->pageID());
}

void WebTileManager::pause(uint64_t tileID)
{
    WebProcess::shared().connection()->send(Messages::WebTileManagerProxy::PauseWebTile(tileID), m_webPage->pageID());
}

void WebTileManager::resume(uint64_t tileID)
{
    WebProcess::shared().connection()->send(Messages::WebTileManagerProxy::ResumeWebTile(tileID), m_webPage->pageID());
}

void WebTileManager::eval(uint64_t tileID, const String& jsScript, uint64_t callbackID)
{
    WebProcess::shared().connection()->send(Messages::WebTileManagerProxy::RunJavaScriptInWebTile(tileID, jsScript, callbackID), m_webPage->pageID());
}

} // namespace WebKit

#endif // ENABLE(MANX_HTMLTILE)
