/*
 * Copyright (C) 2011 Apple Inc. All rights reserved.
 * Copyright (C) 2012 Igalia S.L.
 * Copyright (C) 2013 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "LayerTreeHostManx.h"

#include "DrawingAreaImpl.h"
#include "WebPage.h"
#include "WebProcess.h"
#include <WebCore/FocusController.h>
#include <WebCore/Frame.h>
#include <WebCore/FrameView.h>
#include <WebCore/Page.h>
#include <WebCore/Settings.h>

#if USE(TEXTURE_MAPPER)
#include <WebCore/GLContextEGL.h>
#include <WebCore/GraphicsLayerTextureMapper.h>
#include <WebCore/OpenGLManx.h>
#include <WebCore/TextureMapperGL.h>
#include <WebCore/TextureMapperLayer.h>
namespace WebCore {
typedef GraphicsLayerTextureMapper GraphicsLayerManx;
}
#endif

#if ENABLE(MANX_INDIRECT_COMPOSITING)
#define BEGIN_DIRECT_COMPOSITE() sce::IndirectCompositor::beginDirectComposite()
#define END_DIRECT_COMPOSITE() sce::IndirectCompositor::endDirectComposite()
#define SET_END_OF_FRAME() sce::IndirectCompositor::setEndOfFrame()
#else
#define BEGIN_DIRECT_COMPOSITE()
#define END_DIRECT_COMPOSITE()
#define SET_END_OF_FRAME()
#endif

// #define GL_CMD(...) do { __VA_ARGS__; GLenum err = glGetError(); if (err) { printf("L.%d: %08x\n", __LINE__, err); printf("\t%s\n", #__VA_ARGS__); } } while (0)
#define GL_CMD(...) __VA_ARGS__;

using namespace WebCore;

namespace WebKit {

struct CanvasBuffer {
    CanvasBuffer();

    void initialize(WebCore::GLContext*, int width, int height);
    void terminate(WebCore::GLContext*);

    void activateAsFrameBuffer();
    void activateAsCanvas(sce::Canvas::Id);

    int m_width;
    int m_height;
    GLuint m_fbo;
    GLuint m_tex;
    GLuint m_tex2;
    GLORBISTEXIMAGE2DMAPPEDDESCSCE m_desc;
};

CanvasBuffer::CanvasBuffer()
    : m_width(0)
    , m_height(0)
    , m_fbo(0)
    , m_tex(0)
    , m_tex2(0)
{
    memset(&m_desc, 0, sizeof(m_desc));
}


void CanvasBuffer::initialize(GLContext* context, int width, int height)
{
    terminate(context);

    m_width = width;
    m_height = height;

    GL_CMD(glGenTextures(1, &m_tex));
    GL_CMD(glBindTexture(GL_TEXTURE_2D, m_tex));
    GL_CMD(glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
    GL_CMD(glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
    GL_CMD(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP));
    GL_CMD(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP));
    GL_CMD(glTexImage2D(GL_TEXTURE_2D, 0, GL_BGRA_EXT, width, height, 0, GL_BGRA_EXT, GL_UNSIGNED_BYTE, 0));
    GL_CMD(glOrbisMapTexImage2DSCE(GL_TEXTURE_2D, 0, &m_desc));

#if 1
    GL_CMD(glGenTextures(1, &m_tex2));
    GL_CMD(glBindTexture(GL_TEXTURE_2D, m_tex2));
    GL_CMD(glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
    GL_CMD(glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
    GL_CMD(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP));
    GL_CMD(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP));
    GL_CMD(glTexImage2D(GL_TEXTURE_2D, 0, GL_BGRA_EXT, width, height, 0, GL_BGRA_EXT, GL_UNSIGNED_BYTE, 0));    
#endif
    
    GL_CMD(glBindTexture(GL_TEXTURE_2D, 0));

    GL_CMD(glGenFramebuffersEXT(1, &m_fbo));
}

void CanvasBuffer::terminate(GLContext* context)
{
    context->makeContextCurrent();

    if (m_fbo) {
        GL_CMD(glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0));
        GL_CMD(glDeleteFramebuffers(1, &m_fbo));
        m_fbo = 0;
    }
    if (m_tex) {
        GL_CMD(glBindTexture(GL_TEXTURE_2D, m_tex));
        GL_CMD(glOrbisUnmapTexImage2DSCE(GL_TEXTURE_2D));
        GL_CMD(glBindTexture(GL_TEXTURE_2D, 0));
        GL_CMD(glDeleteTextures(1, &m_tex));
    }
    if (m_tex2) {
        GL_CMD(glBindTexture(GL_TEXTURE_2D, m_tex2));
        GL_CMD(glBindTexture(GL_TEXTURE_2D, 0));
        GL_CMD(glDeleteTextures(1, &m_tex2));
    }
}

void CanvasBuffer::activateAsFrameBuffer()
{
    GL_CMD(glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, m_fbo));
    GL_CMD(glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, m_tex, 0));
    GL_CMD(glViewport(0, 0, m_width, m_height));
}

void CanvasBuffer::activateAsCanvas(sce::Canvas::Id id)
{
    sce::CanvasUtil::bindTexture(id, &m_desc);
}

PassRefPtr<LayerTreeHostManx> LayerTreeHostManx::create(WebPage* webPage)
{
    return adoptRef(new LayerTreeHostManx(webPage));
}

LayerTreeHostManx::LayerTreeHostManx(WebPage* webPage)
    : LayerTreeHost(webPage)
    , m_layerFlushTimer(this, &LayerTreeHostManx::layerFlushTimerFired)
    , m_layerFlushSchedulingEnabled(true)
    , m_isValid(true)
    , m_notifyAfterScheduledLayerFlush(false)
    , m_isActive(true)
    , m_context(0)
    , m_canvasID(sce::Canvas::kInvalidIdValue)
    , m_canvasBuffer(0)
#if ENABLE(MANX_INDIRECT_COMPOSITING)
    , m_indirectCanvas(0)
    , m_indirectCompositor(0)
#endif    
{
    m_canvasBuffer = new CanvasBuffer;
    initialize();
}

GLContext* LayerTreeHostManx::glContext()
{
    if (m_context)
        return m_context;
    m_context = GLContextEGL::createOffscreenContext();
    if (!m_context)
        CRASH();
    return m_context;
}

void LayerTreeHostManx::initialize()
{
    // Create a root layer.
    m_rootLayer = GraphicsLayerManx::create(this);
#ifndef NDEBUG
    m_rootLayer->setName("LayerTreeHost root layer");
#endif
    m_rootLayer->setDrawsContent(false);
    m_rootLayer->setSize(m_webPage->size());
    // static_cast<GraphicsLayerManx*>(m_rootLayer.get())->platformLayer()->setGeometryFlipped(true);

    m_nonCompositedContentLayer = GraphicsLayerManx::create(this);
    // static_cast<GraphicsLayerManx*>(m_nonCompositedContentLayer.get())->setAllowTiledLayer(false);
#ifndef NDEBUG
    m_nonCompositedContentLayer->setName("LayerTreeHost non-composited content");
#endif
    m_nonCompositedContentLayer->setDrawsContent(true);
    m_nonCompositedContentLayer->setContentsOpaque(m_webPage->drawsBackground() && !m_webPage->drawsTransparentBackground());
    m_nonCompositedContentLayer->setSize(m_webPage->size());
    if (m_webPage->corePage()->settings()->acceleratedDrawingEnabled())
        m_nonCompositedContentLayer->setAcceleratesDrawing(true);

    m_rootLayer->addChild(m_nonCompositedContentLayer.get());

    if (m_webPage->hasPageOverlay())
        createPageOverlayLayer();

    GLContext* context = glContext();
    if (!context) {
        m_isValid = false;
        return;
    }

    // Acqurire Canvas ID and set its handle to LayerTreeContext.
    sce::Canvas::releaseId(m_canvasID);
    m_canvasID = sce::Canvas::kInvalidIdValue;
    uint32_t canvasID = sce::Canvas::acquireId();
    m_canvasID = canvasID;
    // Force Canvas handle update by setting kInvalidHandle.
    m_layerTreeContext.setCanvasHandle(sce::Canvas::handle(m_canvasID));

    // Initialize canvas buffer with current width and height.
    int width = m_webPage->size().width();
    int height = m_webPage->size().height();
    m_canvasBuffer->initialize(context, width, height);

    // The creation of the TextureMapper needs an active OpenGL context.
    context->makeContextCurrent();

    m_textureMapper = TextureMapper::create(TextureMapper::OpenGLMode);
    toTextureMapperLayer(m_rootLayer.get())->setTextureMapper(m_textureMapper.get());

    setLayerFlushSchedulingEnabled(!m_webPage->drawingArea() || !m_webPage->drawingArea()->layerTreeStateIsFrozen());
    scheduleLayerFlush();
}

LayerTreeHostManx::~LayerTreeHostManx()
{
    ASSERT(!m_isValid);
    ASSERT(!m_rootLayer);
    if (m_context && m_canvasBuffer)
        m_canvasBuffer->terminate(m_context);
    delete m_context;
    delete m_canvasBuffer;
#if ENABLE(MANX_INDIRECT_COMPOSITING)
    if (m_indirectCanvas)
        sce::IndirectCanvas::destroy(m_indirectCanvas);
    delete m_indirectCompositor;
#endif
    sce::Canvas::releaseId(m_canvasID);
}

void LayerTreeHostManx::layerFlushTimerFired(Timer<LayerTreeHostManx>*)
{
    performScheduledLayerFlush();

    if (toTextureMapperLayer(m_rootLayer.get())->descendantsOrSelfHaveRunningAnimations() && !m_layerFlushTimer.isActive())
        m_layerFlushTimer.startOneShot(1.0 / 60.0);
}

const LayerTreeContext& LayerTreeHostManx::layerTreeContext()
{
    return m_layerTreeContext;
}

void LayerTreeHostManx::setLayerFlushSchedulingEnabled(bool layerFlushingEnabled)
{
    if (m_layerFlushSchedulingEnabled == layerFlushingEnabled)
        return;

    m_layerFlushSchedulingEnabled = layerFlushingEnabled;
        
    if (m_layerFlushSchedulingEnabled) {
        scheduleLayerFlush();
        return;
    }

    cancelPendingLayerFlush();
}

void LayerTreeHostManx::scheduleLayerFlush()
{
    if (!m_layerFlushSchedulingEnabled)
        return;

    if (!m_layerFlushTimer.isActive())
        m_layerFlushTimer.startOneShot(0);
}

void LayerTreeHostManx::setShouldNotifyAfterNextScheduledLayerFlush(bool notifyAfterScheduledLayerFlush)
{
    m_notifyAfterScheduledLayerFlush = notifyAfterScheduledLayerFlush;
}

void LayerTreeHostManx::setRootCompositingLayer(GraphicsLayer* graphicsLayer)
{
    m_nonCompositedContentLayer->removeAllChildren();

    // Add the accelerated layer tree hierarchy.
    if (graphicsLayer)
        m_nonCompositedContentLayer->addChild(graphicsLayer);
}

void LayerTreeHostManx::invalidate()
{
    ASSERT(m_isValid);
    m_rootLayer = nullptr;
    m_isValid = false;
}

void LayerTreeHostManx::setNonCompositedContentsNeedDisplay(const IntRect& rect)
{
    m_nonCompositedContentLayer->setNeedsDisplayInRect(rect);
    if (m_pageOverlayLayer)
        m_pageOverlayLayer->setNeedsDisplayInRect(rect);

    scheduleLayerFlush();
}

void LayerTreeHostManx::scrollNonCompositedContents(const IntRect& scrollRect, const IntSize& scrollOffset)
{
    setNonCompositedContentsNeedDisplay(scrollRect);
}

void LayerTreeHostManx::sizeDidChange(const IntSize& newSize)
{
    if (m_rootLayer->size() == newSize)
        return;

    // Recreate canvas buffer.
    m_canvasBuffer->initialize(glContext(), newSize.width(), newSize.height());
    // Force canvas handle update by setting kInvalidHandle.
    m_layerTreeContext.setCanvasHandle(sce::Canvas::kInvalidHandle);
    
    m_rootLayer->setSize(newSize);

    // If the newSize exposes new areas of the non-composited content a setNeedsDisplay is needed
    // for those newly exposed areas.
    FloatSize oldSize = m_nonCompositedContentLayer->size();
    m_nonCompositedContentLayer->setSize(newSize);

    if (newSize.width() > oldSize.width()) {
        float height = std::min(static_cast<float>(newSize.height()), oldSize.height());
        m_nonCompositedContentLayer->setNeedsDisplayInRect(FloatRect(oldSize.width(), 0, newSize.width() - oldSize.width(), height));
    }

    if (newSize.height() > oldSize.height())
        m_nonCompositedContentLayer->setNeedsDisplayInRect(FloatRect(0, oldSize.height(), newSize.width(), newSize.height() - oldSize.height()));
    m_nonCompositedContentLayer->setNeedsDisplay();

    if (m_pageOverlayLayer)
        m_pageOverlayLayer->setSize(newSize);

    scheduleLayerFlush();
    flushPendingLayerChanges();
}

void LayerTreeHostManx::deviceScaleFactorDidChange()
{
    // Other layers learn of the scale factor change via WebPage::setDeviceScaleFactor.
    m_nonCompositedContentLayer->deviceOrPageScaleFactorChanged();
}

void LayerTreeHostManx::forceRepaint()
{
    scheduleLayerFlush();
    flushPendingLayerChanges();
}    

void LayerTreeHostManx::didInstallPageOverlay()
{
    createPageOverlayLayer();
    scheduleLayerFlush();
}

void LayerTreeHostManx::didUninstallPageOverlay()
{
    destroyPageOverlayLayer();
    scheduleLayerFlush();
}

void LayerTreeHostManx::setPageOverlayNeedsDisplay(const IntRect& rect)
{
    ASSERT(m_pageOverlayLayer);
    m_pageOverlayLayer->setNeedsDisplayInRect(rect);
    scheduleLayerFlush();
}

void LayerTreeHostManx::notifyAnimationStarted(const WebCore::GraphicsLayer*, double time)
{
}

void LayerTreeHostManx::notifySyncRequired(const WebCore::GraphicsLayer*)
{
}

void LayerTreeHostManx::paintContents(const GraphicsLayer* graphicsLayer, GraphicsContext& graphicsContext, GraphicsLayerPaintingPhase, const IntRect& clipRect)
{
    if (graphicsLayer == m_nonCompositedContentLayer) {
        m_webPage->drawRect(graphicsContext, clipRect);
        return;
    }

    if (graphicsLayer == m_pageOverlayLayer) {
        m_webPage->drawPageOverlay(graphicsContext, clipRect);
        return;
    }
}

bool LayerTreeHostManx::showDebugBorders(const GraphicsLayer*) const
{
    return m_webPage->corePage()->settings()->showDebugBorders();
}

bool LayerTreeHostManx::showRepaintCounter(const GraphicsLayer*) const
{
    return m_webPage->corePage()->settings()->showRepaintCounter();
}

float LayerTreeHostManx::deviceScaleFactor() const
{
    return m_webPage->corePage()->deviceScaleFactor();
}

void LayerTreeHostManx::performScheduledLayerFlush()
{
    {
        RefPtr<LayerTreeHostManx> protect(this);
        m_webPage->layoutIfNeeded();

        if (!m_isValid)
            return;
    }

    if (!flushPendingLayerChanges())
        return;

    compositeLayersToContext();

    didPerformScheduledLayerFlush();
}

void LayerTreeHostManx::didPerformScheduledLayerFlush()
{
    if (m_notifyAfterScheduledLayerFlush) {
        // Let the drawing area know that we've done a flush of the layer changes.
        static_cast<DrawingAreaImpl*>(m_webPage->drawingArea())->layerHostDidFlushLayers();
        m_notifyAfterScheduledLayerFlush = false;
    }
}

bool LayerTreeHostManx::flushPendingLayerChanges()
{
    GLContext* context = glContext();
    if (!context)
        return false;
    if (!context->makeContextCurrent())
        return false;

    m_rootLayer->syncCompositingStateForThisLayerOnly();
    m_nonCompositedContentLayer->syncCompositingStateForThisLayerOnly();
    if (m_pageOverlayLayer)
        m_pageOverlayLayer->syncCompositingStateForThisLayerOnly();

    return m_webPage->corePage()->mainFrame()->view()->syncCompositingStateIncludingSubframes();
}

void LayerTreeHostManx::cancelPendingLayerFlush()
{
    m_layerFlushTimer.stop();
}

void LayerTreeHostManx::createPageOverlayLayer()
{
    ASSERT(!m_pageOverlayLayer);

    m_pageOverlayLayer = GraphicsLayer::create(this);
#ifndef NDEBUG
    m_pageOverlayLayer->setName("LayerTreeHost page overlay content");
#endif

    m_pageOverlayLayer->setDrawsContent(true);
    m_pageOverlayLayer->setSize(m_webPage->size());

    m_rootLayer->addChild(m_pageOverlayLayer.get());
}

void LayerTreeHostManx::destroyPageOverlayLayer()
{
    ASSERT(m_pageOverlayLayer);
    m_pageOverlayLayer->removeFromParent();
    m_pageOverlayLayer = nullptr;
}

void LayerTreeHostManx::setActive(bool)
{
    // Schedule layer flush to update check and update 'active' status.
    scheduleLayerFlush();
}

void LayerTreeHostManx::compositeLayersToContext(CompositePurpose purpose)
{
    WebCore::Settings* settings = m_webPage->corePage()->settings();
    bool isWebGLEnabled = settings ? settings->webGLEnabled() : false;
    bool isActive = m_webPage->corePage()->focusController()->isActive();
    if (!m_isActive && !isActive && !isWebGLEnabled) {
        if (m_layerFlushSchedulingEnabled)
            m_layerFlushTimer.startOneShot(1.0 / 10);
        return;
    }
    m_isActive = isActive;

    GLContext* context = glContext();
    if (!context)
        return;
    if (!context->makeContextCurrent())
        return;

    // Since the Y axis goes up in OpenGL's texture coodinate, vertical flip is required.
    const bool shouldFlipY = true;

    // Store "self" canvas handle.
    sce::Canvas::Handle selfCanvas = sce::Canvas::handle(m_canvasID);
    sce::Canvas::Handle canvasHandle = selfCanvas;

#if 0 // FIXME:
    // The window size may be out of sync with the page size at this point, and getting
    // the viewport parameters incorrect, means that the content will be misplaced. Thus
    // we set the viewport parameters directly from the window size.
    IntSize contextSize = m_context->defaultFrameBufferSize();
    GL_CMD(glViewport(0, 0, contextSize.width(), contextSize.height()));
#endif

    bool drawsTransparentBackground = m_webPage->drawsTransparentBackground();
    if (purpose == ForResize || drawsTransparentBackground) { // FIXME: Opaque background color may be set.
        GL_CMD(glClearColor(0, 0, 0, 0));
        GL_CMD(glClear(GL_COLOR_BUFFER_BIT));
    }

    BEGIN_DIRECT_COMPOSITE();
    {
        m_canvasBuffer->activateAsFrameBuffer();

        TextureMapper::PaintFlags flags = 0;
        if (shouldFlipY)
            flags |= TextureMapperGL::PaintingMirrored;
#if ENABLE(MANX_INDIRECT_COMPOSITING)
        if (isActive)
            flags |= TextureMapper::PaintingForIndirectCompositing;
#endif
        m_textureMapper->beginPainting(flags);
        toTextureMapperLayer(m_rootLayer.get())->paint();
        m_textureMapper->endPainting();
        GL_CMD(glFlush());

        m_canvasBuffer->activateAsCanvas(m_canvasID);
    }
    END_DIRECT_COMPOSITE();    

#if ENABLE(MANX_INDIRECT_COMPOSITING)
    if (m_textureMapper->accelerationMode() == TextureMapper::OpenGLMode) {
        TextureMapperGL* texmapGL = static_cast<TextureMapperGL*>(m_textureMapper.get());
        const Vector<TextureMapperGL::IndirectCompositionRecord>& records = texmapGL->indirectCompositionRecords();

        if (isActive && records.size() > 0) { // Needs to create IndirectCanvas and IndirectCompositor.
            if (!m_indirectCanvas)
                m_indirectCanvas = sce::IndirectCanvas::create();
            if (m_indirectCanvas)
                m_indirectCanvas->setSize(m_webPage->size().width(), m_webPage->size().height());
            if (m_indirectCanvas && !m_indirectCompositor)
                m_indirectCompositor = new sce::IndirectCompositor();

            ASSERT(m_indirectCanvas);
            ASSERT(m_indirectCompositor);
        }

        if (isActive && m_indirectCanvas && m_indirectCompositor && records.size() > 0) {
            m_indirectCompositor->beginComposite();
            // We activate the base canvas here to tell its ID to the IndirectCompositor.
            m_canvasBuffer->activateAsCanvas(m_canvasID);
            m_indirectCompositor->drawBaseCanvas(selfCanvas);
            for (Vector<TextureMapperGL::IndirectCompositionRecord>::const_reverse_iterator it = records.rbegin(); it != records.rend(); ++it) {
                // it->debugPrint();

                int i;
                const TransformationMatrix& m = it->transformation;
                sce::Vec4 vertices[4];

                // (0, 0, 0, 1)
                i = 0;
                vertices[i].f[0] = m.m41();
                vertices[i].f[1] = m.m42();
                vertices[i].f[2] = 0;
                vertices[i].f[3] = m.m44();

                // (0, 1, 0, 1)
                i = 1;
                vertices[i].f[0] = m.m41() + m.m21();
                vertices[i].f[1] = m.m42() + m.m22();
                vertices[i].f[2] = 0;
                vertices[i].f[3] = m.m44() + m.m24();

                // (1, 1, 0, 1)
                i = 3;
                vertices[i].f[0] = m.m41() + m.m11() + m.m21();
                vertices[i].f[1] = m.m42() + m.m12() + m.m22();
                vertices[i].f[2] = 0;
                vertices[i].f[3] = m.m44() + m.m14() + m.m24();

                // (1, 0, 0, 1)
                i = 2;
                vertices[i].f[0] = m.m41() + m.m11();
                vertices[i].f[1] = m.m42() + m.m12();
                vertices[i].f[2] = 0;
                vertices[i].f[3] = m.m44() + m.m14();

                if (shouldFlipY) {
                    vertices[0].f[1] *= -1;
                    vertices[1].f[1] *= -1;
                    vertices[2].f[1] *= -1;
                    vertices[3].f[1] *= -1;
                }

                m_indirectCompositor->drawCanvas(it->texture, vertices, it->opacity);
            }
            m_indirectCompositor->endComposite(m_indirectCanvas, m_canvasID);
        }
    }
#endif
    SET_END_OF_FRAME();

    // Update LayerTreeContext if needed.
    if (canvasHandle != m_layerTreeContext.canvasHandle()) {
        m_layerTreeContext.setCanvasHandle(canvasHandle);
        m_webPage->updateAcceleratedCompositingMode();
    }
}

} // namespace WebKit
