/*
 * Copyright (C) 2008 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef WebViewQueue_h
#define WebViewQueue_h

#include <pthread.h>

class WebViewQueueItem {
public:
    friend class WebViewQueue;

    WebViewQueueItem(uintptr_t id = 0) :
        m_id(id),
        m_prev(0),
        m_next(0) { }
    void setId(uintptr_t id)    { m_id = id; }
    uintptr_t getId()           { return m_id; }
    WebViewQueueItem *prev()    { return m_prev; }
    WebViewQueueItem *next()    { return m_next; }

private:
    uintptr_t m_id;
    WebViewQueueItem *m_prev;
    WebViewQueueItem *m_next;
};

class WebViewQueue {
public:
    WebViewQueue() :
        m_count(0),
        m_head(0),
        m_tail(0)            { pthread_mutex_init(&m_mutex, 0); }
    ~WebViewQueue()             { pthread_mutex_destroy(&m_mutex); }
    void lock()                 { pthread_mutex_lock(&m_mutex); }
    void unlock()               { pthread_mutex_unlock(&m_mutex); }
    int count()                 { return m_count; }
    WebViewQueueItem *head()    { return m_head; }
    WebViewQueueItem *tail()    { return m_tail; }

    void remove(WebViewQueueItem *item, bool shouldLock = true)
    {
        if (shouldLock)
            lock();

        if (item->m_prev)
            item->m_prev->m_next = item->m_next;
        else
            m_head = item->m_next;

        if (item->m_next)
            item->m_next->m_prev = item->m_prev;
        else
            m_tail = item->m_prev;

        delete item;

        m_count--;

        if (shouldLock)
            unlock();
    }

    void push(WebViewQueueItem *item, int limit = 0, bool shouldLock = true)
    {
        if (shouldLock)
            lock();

        /* if 'limit' is set, and there are 'limit' number of
           same id items already in the queue, pop the oldest */
        if (limit) {
            int count = 0;
            WebViewQueueItem *ptr = m_head;
            WebViewQueueItem *old = 0;
            while (ptr) {
                if (ptr->m_id == item->m_id) {
                    if (!old)
                        old = ptr;
                    count++;
                }
                ptr = ptr->m_next;
            }
            if (count >= limit)
                remove(old, false);
        }

        if (!m_head)
            m_head = item;
        if (m_tail)
            m_tail->m_next = item;

        item->m_prev = m_tail;
        item->m_next = 0;

        m_tail = item;

        m_count++;

        if (shouldLock)
            unlock();
    }

    WebViewQueueItem *pop(bool shouldLock = true)
    {
        if (shouldLock)
            lock();

        WebViewQueueItem *item = m_head;
        if (m_head) {
            m_head = m_head->m_next;
            m_count--;
        }
        if (m_head)
            m_head->m_prev = 0;
        else
            m_tail = 0;

        if (shouldLock)
            unlock();

        return item;
    }

private:
    int m_count;
    WebViewQueueItem *m_head, *m_tail;
    pthread_mutex_t m_mutex;
};

#endif // WebViewQueue_h
