/*
 * Copyright (C) 2008 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebView.h"

#include "MGWebViewInternal.h"
#include "WebViewHitTestResult.h"
#include <cassert>
#include <stdlib.h>
#include <string.h>
#include <webkit/WebCString.h>

namespace MXGlue {

using WebKit::WebString;
using WebKit::WebCString;

// WebViewClientImpl

class WebViewClientImpl : public WebKit::WebViewClient {
public:
    WebViewClientImpl(WebView* v) : m_webView(v) { }

    virtual WebKit::WebView* createWindow(const char* url, int width, int height, bool isScript);
    virtual void closeWindowSoon();
    virtual void runJavaScriptAlert(WebString text);
    virtual bool runJavaScriptConfirm(WebString text);
    virtual bool runJavaScriptPrompt(WebString text, WebString defaultValue, WebString& result);
    virtual void didViewportChanged(float initialScale, float maximumScale, float minimumScale, bool userScalable);
    virtual void didContentSizeChanged(int width, int height);
    virtual void didUpdateDrawing(const unsigned char* buffer, int size);
    virtual void doCallbackDeprecated(CallbackType);
    virtual void didFocusedNodeChanged();
    virtual void didCommitLoad();
    
    WebView* m_webView;
};

WebKit::WebView* WebViewClientImpl::createWindow(const char* url, int width, int height, bool isScript)
{
    WebViewManager::s_callbackIn[0].type_const_char_ptr = url;
    WebViewManager::s_callbackIn[1].type_int = width;
    WebViewManager::s_callbackIn[2].type_int = height;
    WebViewManager::s_callbackIn[3].type_bool = isScript;
    WebViewManager::processCallback(m_webView, CALLBACK_CREATE_WINDOW);
    WebView* webview = (WebView*)WebViewManager::s_callbackOut[0].type_void_ptr;
    if (!webview)
        return 0;
    return webview->m_impl;
}

void WebViewClientImpl::closeWindowSoon()
{
    WebViewManager::processCallback(m_webView, CALLBACK_CLOSE_WINDOW);
}

void WebViewClientImpl::runJavaScriptAlert(WebString text)
{
    WebCString cText = text.utf8();
    WebViewManager::s_callbackIn[0].type_const_char_ptr = cText.isNull() ? "" : cText.data();
    WebViewManager::processCallback(m_webView, CALLBACK_JAVASCRIPT_ALERT);
}

bool WebViewClientImpl::runJavaScriptConfirm(WebString text)
{
    WebCString cText = text.utf8();
    WebViewManager::s_callbackIn[0].type_const_char_ptr = cText.isNull() ? "" : cText.data();
    WebViewManager::processCallback(m_webView, CALLBACK_JAVASCRIPT_CONFIRM);
    return WebViewManager::s_callbackOut[0].type_bool;
}

bool WebViewClientImpl::runJavaScriptPrompt(WebString text, WebString defaultValue, WebString& result)
{
    WebCString cText = text.utf8();
    WebCString cDefaultValue = defaultValue.utf8();

    WebViewManager::s_callbackOut[0].type_bool = false;
    WebViewManager::s_callbackOut[1].type_char_ptr = new char[VARDATA_CHAR_OUT_SIZE];
    strncpy(WebViewManager::s_callbackOut[1].type_char_ptr, cDefaultValue.data(), cDefaultValue.length());

    WebViewManager::s_callbackIn[0].type_const_char_ptr = !cText.isNull() ? cText.data() : "";
    WebViewManager::s_callbackIn[1].type_const_char_ptr = cDefaultValue.data();
    WebViewManager::processCallback(m_webView, CALLBACK_JAVASCRIPT_PROMPT);

    result = WebString::fromUTF8(WebViewManager::s_callbackOut[1].type_char_ptr);

    delete[] WebViewManager::s_callbackOut[1].type_char_ptr;
    return WebViewManager::s_callbackOut[0].type_bool;
}

void WebViewClientImpl::didViewportChanged(float initialScale, float maximumScale, float minimumScale, bool userScalable)
{
    WebViewManager::s_callbackIn[0].type_float = initialScale;
    WebViewManager::s_callbackIn[1].type_float = maximumScale;
    WebViewManager::s_callbackIn[2].type_float = minimumScale;
    WebViewManager::s_callbackIn[3].type_bool  = userScalable;
    WebViewManager::processCallback(m_webView, CALLBACK_VIEWPORT_CHANGED);
}

void WebViewClientImpl::didContentSizeChanged(int width, int height)
{
    WebViewManager::s_callbackIn[0].type_int = width;
    WebViewManager::s_callbackIn[1].type_int = height;

    WebViewManager::processCallback(m_webView, CALLBACK_CONTENTS_SIZE_CHANGED);
}

void WebViewClientImpl::didUpdateDrawing(const unsigned char* buffer, int size)
{
    if (buffer) {
        WebViewManager::s_callbackIn[0].type_const_uchar_ptr = buffer;
        WebViewManager::s_callbackIn[1].type_int = size;
        WebViewManager::processCallback(m_webView, CALLBACK_SET_TEXTURE);
    } else
        WebViewManager::processCallback(m_webView, CALLBACK_SET_BACKINGSTORE);
}

void WebViewClientImpl::doCallbackDeprecated(CallbackType type)
{
    WebViewManager::processCallback(m_webView, type);
}

void WebViewClientImpl::didFocusedNodeChanged()
{
    m_webView->didFocusedNodeChangedWT();
}

void WebViewClientImpl::didCommitLoad()
{
    m_webView->clearInputMethodInfoWT();
}

/* class WebView */
bool WebView::isWebThread()
{
    return WebViewManager::isWebThread();
}

WebView::WebView(int width, int height, void* pool, void* painter)
{
    assert(isWebThread());

    m_webViewClient = new WebViewClientImpl(this);
    m_impl = WebKit::WebView::create(*m_webViewClient);
    ASSERT(m_impl);
    m_impl->create(width, height, pool, painter);
}

WebView::~WebView()
{
    assert(isWebThread());

    delete m_impl;
    delete m_webViewClient;
}

void WebView::drawRect(float x, float y, float w, float h, void* data, unsigned int dataSize)
{
    m_impl->drawRect(x, y, w, h, data, dataSize);
}

void WebView::setClipRect(int x, int y, int w, int h, bool pending)
{
    {
        if (m_impl->updateClipRect(x, y, w, h, pending)) {
            WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_SET_CLIP_RECT);
            WebViewManager::pushCommand(cmd);
        }
    }
}

void WebView::sendScrollEvent(ScrollEventType event, int x, int y)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_SCROLL_EVENT);
        cmd->m_data[0].typeScrollEvent = event;
        cmd->m_data[1].typeInt = x;
        cmd->m_data[2].typeInt = y;
        WebViewManager::pushCommand(cmd, 1);
    }
}

void WebView::sendMouseEvent(MouseEventType event, MouseButtonType button, int x, int y)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, (event == MOUSE_EVENT_MOVED) ? WEBVIEW_COMMAND_MOUSE_EVENT_MOVE : WEBVIEW_COMMAND_MOUSE_EVENT);
        cmd->m_data[0].typeMouseEvent = event;
        cmd->m_data[1].typeMouseButton = button;
        cmd->m_data[2].typeInt = x;
        cmd->m_data[3].typeInt = y;
        WebViewManager::pushCommand(cmd, (event == MOUSE_EVENT_MOVED) ? 10 : 0);
    }
}

void WebView::sendKeyEvent(KeyEventType event, KeyType key, char ascii, bool shift, bool ctrl, bool alt, bool win)
{
    sendKeyEvent2(event, key, ascii, shift, ctrl, alt, win, KeyDefaultBehaviorCursorNavigation);
}

void WebView::sendKeyEvent2(KeyEventType event, KeyType key, char ascii, bool shift, bool ctrl, bool alt, bool win, KeyDefaultBehavior behavior)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_KEY_EVENT);
        cmd->m_data[0].typeKeyEvent = event;
        cmd->m_data[1].typeKey = key;
        cmd->m_data[2].typeChar = ascii;
        cmd->m_data[3].typeBool = shift;
        cmd->m_data[4].typeBool = ctrl;
        cmd->m_data[5].typeBool = alt;
        cmd->m_data[6].typeBool = win;
        cmd->m_data[7].typeInt = behavior;
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::sendHtmlEvent(HtmlEventType event, const HtmlEventData* data)
{
    {
        int size;
        switch (event) {
        case HTML_INPUT_ELEMENT_SET_VALUE:  size = 1; break;
        default:                            size = 0; break;
        }

        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_HTML_EVENT);
        cmd->m_data[0].typeHtmlEvent = event;
        cmd->m_data[1].typeHtmlEventDataPtr = 0;
        if (size) {
            cmd->m_data[1].typeHtmlEventDataPtr = new HtmlEventData[size];
            memcpy(cmd->m_data[1].typeHtmlEventDataPtr, data, size * sizeof(HtmlEventData));
        }
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::sendTouchEvent(const Manx::TouchEvent* event)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_TOUCH_EVENT);
        cmd->m_data[0].typeTouchEventDataPtr = new Manx::TouchEvent();
        memcpy(cmd->m_data[0].typeTouchEventDataPtr, event, sizeof(Manx::TouchEvent));
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::sendGestureEvent(unsigned type, int x, int y)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_GESTURE_EVENT);
        cmd->m_data[0].typeChar = type;
        cmd->m_data[0].typeInt = x;
        cmd->m_data[1].typeInt = y;
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::loadUrl(const char* url)
{
    if (!url)
        return;

    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_LOAD_URL);
        cmd->m_data[0].typeCharPtr = new char[strlen(url) + 1];
        strncpy(cmd->m_data[0].typeCharPtr, url, strlen(url) + 1);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::loadHtml(const char *html, const char *baseUrl)
{
    if (!html || !baseUrl)
        return;

    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_LOAD_HTML);
        cmd->m_data[0].typeCharPtr = new char[strlen(html) + 1];
        strncpy(cmd->m_data[0].typeCharPtr, html, strlen(html) + 1);
        cmd->m_data[1].typeCharPtr = new char[strlen(baseUrl) + 1];
        strncpy(cmd->m_data[1].typeCharPtr, baseUrl, strlen(baseUrl) + 1);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::loadData(const char *data, const char *mimetype, const char *encoding, const char *baseUrl)
{
    if (!data || !mimetype || !encoding || !baseUrl)
        return;

    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_LOAD_DATA);
        cmd->m_data[0].typeCharPtr = new char[strlen(data) + 1];
        strncpy(cmd->m_data[0].typeCharPtr, data, strlen(data) + 1);
        cmd->m_data[1].typeCharPtr = new char[strlen(mimetype) + 1];
        strncpy(cmd->m_data[1].typeCharPtr, mimetype, strlen(mimetype) + 1);
        cmd->m_data[2].typeCharPtr = new char[strlen(encoding) + 1];
        strncpy(cmd->m_data[2].typeCharPtr, encoding, strlen(encoding) + 1);
        cmd->m_data[3].typeCharPtr = new char[strlen(baseUrl) + 1];
        strncpy(cmd->m_data[3].typeCharPtr, baseUrl, strlen(baseUrl) + 1);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::loadArchive(const char *buf, const char *url)
{
    if (!url)
        return;

    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_LOAD_ARCHIVE);
        cmd->m_data[0].typeCharPtr = new char[strlen(buf) + 1];
        strncpy(cmd->m_data[0].typeCharPtr, buf, strlen(buf) + 1);
        cmd->m_data[1].typeCharPtr = new char[strlen(url) + 1];
        strncpy(cmd->m_data[1].typeCharPtr, url, strlen(url) + 1);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::reload()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_RELOAD);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::stop()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_STOP);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::goBack()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_GO_BACK);
        WebViewManager::pushCommand(cmd, 1);
    }
}

void WebView::goForward()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_GO_FORWARD);
        WebViewManager::pushCommand(cmd, 1);
    }
}

void WebView::increaseTextSize()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_INCREASE_TEXT_SIZE);
        WebViewManager::pushCommand(cmd, 1);
    }
}

void WebView::decreaseTextSize()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_DECREASE_TEXT_SIZE);
        WebViewManager::pushCommand(cmd, 1);
    }
}

void WebView::resetTextSize()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_RESET_TEXT_SIZE);
        WebViewManager::pushCommand(cmd, 1);
    }
}

void WebView::setComposition(const char *compositionText, unsigned int underlineFrom, unsigned int underlineTo)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_SET_COMPOSITION);
        cmd->m_data[0].typeCharPtr = new char[strlen(compositionText) + 1];
        strncpy(cmd->m_data[0].typeCharPtr, compositionText, strlen(compositionText) + 1);
        cmd->m_data[1].typeInt = underlineFrom;
        cmd->m_data[2].typeInt = underlineTo;
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::confirmComposition(const char *compositionText)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_CONFIRM_COMPOSITION);
        cmd->m_data[0].typeCharPtr = new char[strlen(compositionText) + 1];
        strncpy(cmd->m_data[0].typeCharPtr, compositionText, strlen(compositionText) + 1);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::cancelComposition()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_CANCEL_COMPOSITION);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::exitComposition()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_EXIT_COMPOSITION);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::setSelectedItemsPopupMenu(int index, bool isEnd)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_SELECT_POPUPMENU);
        cmd->m_data[0].typeInt = index;
        cmd->m_data[1].typeBool = isEnd;

        WebViewManager::pushCommand(cmd);
    }
}

void WebView::getItemPopupMenu(int index)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_GETITEM_POPUPMENU);
        cmd->m_data[0].typeInt = index;

        WebViewManager::pushCommand(cmd);
    }
}

void WebView::setWebViewSettingsScriptEnabled(bool enable)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_SET_SETTINGS_JS);
        cmd->m_data[0].typeBool = enable;
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::setWebGLEnabled(bool enabled)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_SET_SETTINGS_WEBGL);
        cmd->m_data[0].typeBool = enabled;
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::setUserAgent(const char* userAgent)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_SET_USERAGENT);
        cmd->m_data[0].typeCharPtr = new char[strlen(userAgent) + 1];
        strncpy(cmd->m_data[0].typeCharPtr, userAgent, strlen(userAgent) + 1);
        WebViewManager::pushCommand(cmd);
    }
}

const char* WebView::userAgent()
{
    return m_impl->userAgent();
}

void WebView::setJavaScriptObj(WebViewJavaScriptObject* window)
{
    WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_SET_JAVA_SCRIPT_OBJ);
    cmd->m_data[0].typeVoidPtr = window;
    WebViewManager::pushCommand(cmd);
}

void WebView::setWebViewSettingsCookieEnabled(bool enable)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_SET_SETTINGS_COOKIE);
        cmd->m_data[0].typeBool = enable;
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::deleteAllCookies()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_DELETE_ALL_COOKIES);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::hitTest(Manx::Point& hitPoint, int type)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_HITTEST);
        cmd->m_data[0].typeInt = hitPoint.x;
        cmd->m_data[1].typeInt = hitPoint.y;
        cmd->m_data[2].typeInt = type;
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::imageBuffer()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_IMAGEBUF);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::setUploadFile(const char* path, const char* name)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_UPLOAD);
        cmd->m_data[0].typeCharPtr = new char[strlen(path) + 1];
        strncpy(cmd->m_data[0].typeCharPtr, path, strlen(path) + 1);
        cmd->m_data[1].typeCharPtr = new char[strlen(name) + 1];
        strncpy(cmd->m_data[1].typeCharPtr, name, strlen(name) + 1);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::setCookies(const char* url, const char* cookie)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_SET_COOKIES);
        cmd->m_data[0].typeCharPtr = new char[strlen(url) + 1];
        strncpy(cmd->m_data[0].typeCharPtr, url, strlen(url) + 1);
        cmd->m_data[1].typeCharPtr = new char[strlen(cookie) + 1];
        strncpy(cmd->m_data[1].typeCharPtr, cookie, strlen(cookie) + 1);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::setWebSecurityEnabled(bool enabled)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_SET_SETTINGS_WEBSECURITY);
        cmd->m_data[0].typeBool = enabled;
        WebViewManager::pushCommand(cmd);
    }
}


void WebView::releaseTiles()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_RELEASE_TILES);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::overflowScroll(Manx::Point &hitPoint, int deltaX, int deltaY)
{
    m_impl->overflowScroll(deltaX, deltaY);
}

void WebView::setDefaultLayoutWidth(int width)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_SET_LAYOUT_WIDTH);
        cmd->m_data[0].typeInt = width;
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::requestDownloadImage()
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_REQUEST_DL_IMAGE);
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::requestDownloadImageData(int pos, int size)
{
    {
        WebViewCommand* cmd = new WebViewCommand(this, WEBVIEW_COMMAND_REQUEST_DL_IMAGE_DATA);
        cmd->m_data[0].typeInt = pos;
        cmd->m_data[1].typeInt = size;
        WebViewManager::pushCommand(cmd);
    }
}

void WebView::didFocusedNodeChangedWT()
{
    using namespace WebKit;

    assert(isWebThread());

    m_impl->getInputMethodInfo(m_inputMethodInfo);
    if (m_impl->isProcessingUserGesture())
        return;
    bool shouldUseInputMethod = m_inputMethodInfo.type != FocusedNodeUnknown;
    if (shouldUseInputMethod)
        m_impl->setCaretVisible(false);
    else
        didInputTextBluredWT();
}

void WebView::checkInputMethodStateWT()
{
    using namespace WebKit;

    assert(isWebThread());

    InputMethodInfo& info = m_inputMethodInfo;
    bool shouldUseInputMethod = info.type != FocusedNodeUnknown;
    if (shouldUseInputMethod) {
        WebCString title = info.title.utf8();
        WebCString value = info.value.utf8();
        const char* type = "";
        switch (info.type) {
        case FocusedNodeText:
            type = "text";
            break;
        case FocusedNodePassword:
            type = "password";
            break;
        case FocusedNodeNumber:
            type = "number";
            break;
        case FocusedNodeTel:
            type = "tel";
            break;
        case FocusedNodeEMail:
            type = "email";
            break;
        case FocusedNodeSearch:
            type = "search";
            break;
        }
        WebViewManager::s_callbackIn[0].type_bool = true;
        WebViewManager::s_callbackIn[1].type_const_char_ptr = !title.isNull() ? title.data() : "";
        WebViewManager::s_callbackIn[2].type_const_char_ptr = !value.isNull() ? value.data() : "";
        WebViewManager::s_callbackIn[3].type_const_char_ptr = type;
        WebViewManager::s_callbackIn[4].type_int = info.rect.x;
        WebViewManager::s_callbackIn[5].type_int = info.rect.y;
        WebViewManager::s_callbackIn[6].type_int = info.rect.width;
        WebViewManager::s_callbackIn[7].type_int = info.rect.height;
        WebViewManager::s_callbackIn[8].type_bool = info.isMultiple;
        WebViewManager::processCallback(this, CALLBACK_FOCUS_INPUT_TEXT);

        m_impl->setCaretVisible(true);
    } else
        didInputTextBluredWT();
    
}

void WebView::didInputTextBluredWT()
{
    assert(isWebThread());

    WebViewManager::s_callbackIn[0].type_bool = false;
    WebViewManager::processCallback(this, CALLBACK_FOCUS_INPUT_TEXT);
}

void WebView::clearInputMethodInfoWT()
{
    using namespace WebKit;

    assert(isWebThread());

    m_inputMethodInfo.type = FocusedNodeUnknown;
}

} // namespace MXGlue

