/*
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"
#include "AboutData.h"

#include "Base64.h"
#include "MemoryCache.h"
#include "TextEncoding.h"

#if OS(PSP2)
extern "C" size_t malloc_max_inuse_size();
extern "C" size_t malloc_current_inuse_size();
#endif

using namespace WebCore;

static String htmlHeader(const KURL& aboutUrl);
static String htmlBody(const KURL& aboutUrl);

static String aboutManxBody();
static String aboutMemoryBody();
static String aboutTileBody();

static String css();

KURL createAboudDataUri(const KURL& aboutUrl)
{
    if (aboutUrl.path() == "blank")
        return WebCore::KURL();
    
    String html;
    html += "<html><head>";
    html += htmlHeader(aboutUrl);
    html += "<head/><body>";
    html += htmlBody(aboutUrl);
    html += "</body></html>";

    String url = "data:text/html;charset=utf-8;base64,";    
    url += base64Encode(html.ascii(), html.length());
    return KURL(KURL(), url, UTF8Encoding());
}

static String htmlHeader(const KURL& aboutUrl)
{
    String header;
    header += "<title>About</title>";
    header += css();
    return header;
}

static String htmlBody(const KURL& aboutUrl)
{
    if (aboutUrl.path() == "memory")
        return aboutMemoryBody();

    return aboutManxBody();
}

static String aboutManxBody()
{
    String body = "";
    return body;
}

static String mallocStatisticsTable()
{
    size_t max = 0;
    size_t current = 0;
#if OS(PSP2)
    max = malloc_max_inuse_size();
    current = malloc_current_inuse_size();
#endif
    
    String table = "<table><tr><th>maxInUse</th><th>currentInUse</th></tr>";
    table += "<tr><td>" + String::number(max) + "</td>";
    table += "<td>" + String::number(current) + "</td></tr></table>";
    return table;
}

static String memoryCacheStatisticsHeader()
{
    String row;
    row += "<tr>";
    row += "<th></th>";
    row += "<th>count</th>";
    row += "<th>size</th>";
    row += "<th>liveSize</th>";
    row += "<th>decodedSize</th>";
    row += "<th>purgeableSize</th>";
    row += "<th>purgedSize</th>";
    row += "</tr>";
    return row;
}

static String memoryCacheStatisticsRow(const MemoryCache::TypeStatistic& statistic, const String& description)
{
    String row;
    row += "<tr>";
    row += "<td>" + description + "</td>";
    row += "<td>" + String::number(statistic.count) + "</td>";
    row += "<td>" + String::number(statistic.size) + "</td>";
    row += "<td>" + String::number(statistic.liveSize) + "</td>";
    row += "<td>" + String::number(statistic.decodedSize) + "</td>";
    row += "<td>" + String::number(statistic.purgeableSize) + "</td>";
    row += "<td>" + String::number(statistic.purgedSize) + "</td>";
    row += "</tr>";
    return row;
}

#define ADD_MEMORY_STATISTICS_ROW(x) body += memoryCacheStatisticsRow(statistic.x, #x);
static String aboutMemoryBody()
{
    String body;

    // FastMalloc
    body += "<h3>malloc</h3>";
    body += mallocStatisticsTable();

    // MemoryCache
    body += "<h3>MemoryCache</h3>";
    body +="<table>";
    body += memoryCacheStatisticsHeader();
    MemoryCache::Statistics statistic = memoryCache()->getStatistics();
    ADD_MEMORY_STATISTICS_ROW(images);
    ADD_MEMORY_STATISTICS_ROW(cssStyleSheets);
    ADD_MEMORY_STATISTICS_ROW(scripts);
    ADD_MEMORY_STATISTICS_ROW(xslStyleSheets);
    ADD_MEMORY_STATISTICS_ROW(fonts);
    body += "</table>";
    return body;
}

static String css()
{
    String css;
    css += "<style type=\"text/css\">";
    css += "* { font-family: sans-serif }";
    css += "table { border: 1px White solid; border-collapse: collapse; }";
    css += "tr:nth-child(odd) { background-color:WhiteSmoke }";
    css += "tr:nth-child(even) { background-color:White }";
    css += "th { background-color:Gainsboro }";
    css += "td { text-align:right; width:20em; padding:2px }";
    css += "</style>";
    return css;
}
