/*
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"
#include "TiledBuffer.h"

namespace WebCore {

IntRect TiledBuffer::tileRect(const IntSize& size, const IntPoint& coordinate)
{
    return IntRect(
        coordinate.x() * size.width(),
        coordinate.y() * size.height(),
        size.width(),
        size.height());
}

FloatRect TiledBuffer::scaledRect(const IntSize& size, const IntPoint& coordinate, float scale)
{
    return FloatRect(
        coordinate.x() * size.width() / scale,
        coordinate.y() * size.height() / scale,
        size.width() / scale,
        size.height() / scale);
}

IntRect TiledBuffer::enclosingScaledRect(const IntSize& size, const IntPoint& coordinate, float scale)
{
    return enclosingIntRect(scaledRect(size, coordinate, scale));
}

IntRect TiledBuffer::tileRect() const
{
    return tileRect(tileSize(), m_coordinate);
}

FloatRect TiledBuffer::scaledRect() const
{
    return scaledRect(tileSize(), m_coordinate, m_contentsScale);
}

IntRect TiledBuffer::enclosingScaledRect() const
{
    return enclosingScaledRect(tileSize(), m_coordinate, m_contentsScale);
}

PassRefPtr<TiledBuffer> TiledBuffer::create(const Coordinate& coordinate, TilePool* pool)
{
    TiledBuffer* tile = 0;
    uint32_t* buffer = (uint32_t*)pool->allocTileBuffer();
    if (buffer)
        tile = new TiledBuffer(coordinate, buffer, pool);
    else
        pool->freeTileBuffer(buffer);

    if (tile)
        tile->setValid(false);
    return adoptRef(tile);
}

TiledBuffer::TiledBuffer(const Coordinate& coordinate, uint32_t* buffer, TilePool* pool)
    : m_refCount(1)
    , m_coordinate(coordinate)
    , m_contentsScale(1.0f)
    , m_buffer(buffer)
    , m_pool(pool)
{
    TileExtent extent = pool->getTileExtent();
    m_metrics.width = extent.width;
    m_metrics.height = extent.height;
    m_metrics.stride = extent.stride;

    m_contentsScale = 1.0;
}

TiledBuffer::~TiledBuffer()
{
    m_pool->freeTileBuffer(m_buffer);
}

void TiledBuffer::duplicate(TiledBuffer* tile)
{
    m_contentsScale = tile->m_contentsScale;
    m_isValid = tile->m_isValid;
    m_dirtyRect = tile->m_dirtyRect;
    memcpy(m_buffer, tile->m_buffer, m_metrics.memSize());
}


} // namespace WebCore
