/*
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"
#include "HfRasterizer.h"

#include "CairoUtilities.h"
#include "FontStreamBufferManx.h"
#include "HfWidthIterator.h"
#include "PlatformContextCairo.h"
#include "TextRun.h"
#include <hf/hafnium.h>
#include <manx/FontData.h>
#include <pixman.h>
#include <wtf/text/AtomicString.h>

namespace WebCore {

/// @param filePath [in] file path to open
/// @param usrdata [out] pointer to user data
/// @return file size in bytes or 0 for error
static size_t HfOpenStream(const char *filePath, void **usrdata)
{
    RefPtr<FontStreamBuffer> streamBuffer = FontStreamBufferCache::get(filePath);
    if (!streamBuffer)
        return 0;

    size_t fileSize = streamBuffer->fileSize();
    if (!fileSize)
        return 0;

    streamBuffer->attach();
    *usrdata = streamBuffer.get();
    return fileSize;
}

/// @param usrdata [in] usrdata
static void HfCloseStream(void *usrdata)
{
    FontStreamBuffer* streamBuffer = (FontStreamBuffer*)usrdata;
    streamBuffer->detach();
}

/// @param buf [in] buffer to write data
/// @param offset [in] file offset in bytes
/// @param count [in] bytes to read
/// @param usrdata [in] usrdata
static size_t HfReadStream(void *buf, size_t offset, size_t count, void *usrdata)
{
    FontStreamBuffer* streamBuffer = (FontStreamBuffer*)usrdata;
    int result = streamBuffer->read(buf, count, offset);
    if (result > 0)
        return (size_t)result;
    return 0;    
}

static bool s_isEnabled = false;
static bool s_isInitialized = false;

static Hafnium::Interface *s_hfSerif;
static Hafnium::Interface *s_hfSansSerif;

static bool isSerif(const Font* font)
{
    static const AtomicString serif(L"serif");
    AtomicString family = font->family().family();
    AtomicString genericFamily = Manx::getGenericFamily(family.characters(), family.length());
    return genericFamily == serif;
}

static void writeToPng(cairo_surface_t* surf)
{
    char path[256];
    static int cnt = 0;
    sprintf(path, "host0:png/%d.png", cnt++);
    if (surf)
        cairo_status_t stat = cairo_surface_write_to_png(surf, path);
}

static void drawGlyph(cairo_t* cr, const Hafnium::Glyph* glyph, bool isRotated)
{
    Hafnium::Bitmap* bitmap = glyph->m_bitmap;
    if (bitmap) {
        cairo_surface_t* surf = cairo_image_surface_create_for_data(
            bitmap->m_data,
            CAIRO_FORMAT_A8,
            bitmap->m_width,
            bitmap->m_height,
            bitmap->m_stride);

        // writeToPng(surf);

        // Save CTM
        cairo_matrix_t ctm;
        cairo_get_matrix(cr, &ctm);

        // Adjust position
        cairo_translate(cr, (float)glyph->m_lsb, -(float)glyph->m_tsb);
        
        if (!isRotated) {
            // Align position
            cairo_matrix_t matrix;
            cairo_get_matrix(cr, &matrix);
            matrix.xx = 1;
            matrix.xy = 0;
            matrix.yx = 0;
            matrix.yy = 1;
            matrix.x0 = floor(matrix.x0);
            matrix.y0 = floor(matrix.y0);
            cairo_set_matrix(cr, &matrix);
        }

        // Draw glyph
        cairo_mask_surface(cr, surf, 0, 0);

        // Restore CTM
        cairo_set_matrix(cr, &ctm);

        cairo_surface_destroy(surf);
    }
}

struct FastDrawContext {
    pixman_image_t* m_dst;
    pixman_image_t* m_src;

    FastDrawContext() : m_dst(0), m_src(0) { }
    ~FastDrawContext()
    {
        if (m_dst)
            pixman_image_unref(m_dst);
        if (m_src)
            pixman_image_unref(m_src);
    }
};

static void drawGlyphFast(cairo_t* cr, const Hafnium::Glyph* glyph, FastDrawContext& fastDrawContext)
{
    Hafnium::Bitmap* bitmap = glyph->m_bitmap;
    if (bitmap) {
        // Save CTM
        cairo_matrix_t ctm;
        cairo_get_matrix(cr, &ctm);

        // Adjust position
        cairo_translate(cr, (float)glyph->m_lsb, -(float)glyph->m_tsb);
        
        // Align position
        cairo_matrix_t matrix;
        cairo_get_matrix(cr, &matrix);
        IntPoint position((int)floor(matrix.x0), (int)floor(matrix.y0));

        // Draw glyph
        pixman_image_t* mask = pixman_image_create_bits(
                PIXMAN_a8,
                bitmap->m_width,
                bitmap->m_height,
                (uint32_t*)bitmap->m_data,
                bitmap->m_stride);

        pixman_image_composite32(
            PIXMAN_OP_OVER,
            fastDrawContext.m_src,
            mask,
            fastDrawContext.m_dst,
            0, 0, 0, 0, position.x(), position.y(),
            bitmap->m_width,
            bitmap->m_height
            );

        pixman_image_unref(mask);

        // Restore CTM
        cairo_set_matrix(cr, &ctm);

    }
}

static void drawText(const Font* font, Hafnium::Interface *hf, cairo_t* cr, const TextRun& run, const FloatPoint& point, int from, int to)
{
    const SimpleFontData* simpleFontData = font->primaryFont();
    const FontPlatformData& fontPlatformData = simpleFontData->platformData();

    // Get cairo context matrix.
    cairo_matrix_t ctm;
    cairo_get_matrix(cr, &ctm);
    double contextScaleX;
    double contextScaleY;
    bool isRotated = ctm.xy || ctm.yx;
    if (isRotated) {
        contextScaleX = sqrt(ctm.xx * ctm.xx + ctm.yx * ctm.yx);
        contextScaleY = sqrt(ctm.xy * ctm.xy + ctm.yy * ctm.yy);
    } else {
        contextScaleX = ctm.xx;
        contextScaleY = ctm.yy;
    }

    // Use fast path if source is solid color.
    bool fastPath = false;
    FastDrawContext fastDrawContext;
    cairo_pattern_t* source = cairo_get_source(cr);
    cairo_pattern_type_t sourceType = cairo_pattern_get_type(source);
    if (!isRotated && sourceType == CAIRO_PATTERN_TYPE_SOLID) {
        double r;
        double g;
        double b;
        double a;
        cairo_surface_t* target = cairo_get_group_target(cr);
        if (cairo_pattern_get_rgba(source, &r, &g, &b, &a) == CAIRO_STATUS_SUCCESS && target == cairo_get_target(cr)) {
            fastPath = true;
            fastDrawContext.m_dst = pixman_image_create_bits(
                PIXMAN_a8r8g8b8,
                cairo_image_surface_get_width(target),
                cairo_image_surface_get_height(target),
                (unsigned*)cairo_image_surface_get_data(target),
                cairo_image_surface_get_stride(target));
            pixman_color_t color = {
                clampTo<unsigned>(r * 65535, 0, 65535),
                clampTo<unsigned>(g * 65535, 0, 65535),
                clampTo<unsigned>(b * 65535, 0, 65535),
                clampTo<unsigned>(a * 65535, 0, 65535)
            };
            fastDrawContext.m_src = pixman_image_create_solid_fill(&color);
        }
    }

    // Get font matrix
    cairo_matrix_t ftm;
    cairo_scaled_font_t* scaledFont = fontPlatformData.scaledFont();
    cairo_scaled_font_get_scale_matrix(scaledFont, &ftm);
    double fontScaleX = ftm.xx;
    double fontScaleY = ftm.yy;

    float compositeScaleX = fontScaleX * contextScaleX;
    float compositeScaleY = fontScaleY * contextScaleY;

    // Set font scale
    Hafnium::FixedSize hfScale;
    hfScale.m_x = compositeScaleX;
    hfScale.m_y = compositeScaleY;
    hf->setScale(hfScale);

    // Move to pen position
    cairo_save(cr);
    cairo_translate(cr, point.x(), point.y());
    cairo_scale(cr, 1 / contextScaleX, 1 / contextScaleY);

    // Set filter
    cairo_pattern_set_filter(cairo_get_source(cr), CAIRO_FILTER_FAST);

    // Get clip extents
    FloatRect clipRect;
    do {
        double x1;
        double y1;
        double x2;
        double y2;
        cairo_clip_extents(cr, &x1, &y1, &x2, &y2);
        clipRect.setLocation(FloatPoint(x1, y1));
        clipRect.setSize(FloatSize(x2 - x1, y2 - y1));
    } while (0);

    // Prepare HfWidthIterator
    HfWidthIterator widthIterator(font, run);
    widthIterator.advance(from);
    float runWidthSoFar = widthIterator.m_runWidthSoFar;

    for (int i = from; i < to; ++i) {
        UChar charCode = *(run.data(i));
        // Skip ZERO_WIDTH_SPACE
        if (charCode == 0x200B)
            continue;
        // Calculate bbox
        Hafnium::GlyphMetrics metrics = hf->getGlyphMetrics(charCode);
        FloatRect bbox(FloatPoint((float)metrics.m_lsb, -(float)metrics.m_tsb), FloatSize((float)metrics.m_width, (float)metrics.m_height));
        bbox.scale(compositeScaleX, compositeScaleY);

        // Align bbox to pixel grid.
        if (!isRotated)
            bbox.setLocation(flooredIntPoint(bbox.location()));

        if (clipRect.intersects(bbox)) {
            const Hafnium::Glyph* glyph = hf->getGlyph(charCode);
            if (glyph) {
                if (fastPath && clipRect.contains(bbox))
                    drawGlyphFast(cr, glyph, fastDrawContext);
                else
                    drawGlyph(cr, glyph, isRotated);
            }
        }
        
        // Advance one character.
        if (i + 1 < to) {
            widthIterator.advance(i + 1, (float)metrics.m_advance * fontScaleX);
            float advance = widthIterator.m_runWidthSoFar - runWidthSoFar;
            runWidthSoFar = widthIterator.m_runWidthSoFar;
            advance *= contextScaleX;
            cairo_translate(cr, advance, 0);
            clipRect.moveBy(FloatPoint(-advance, 0));
        }
    }

    cairo_restore(cr);
}

/// @return Bounding box in document coordinate.
static FloatRect layoutText(const Font* font, Hafnium::Interface *hf, const TextRun& run, int from, int to)
{
    FloatPoint pen;
    FloatRect bbox;

    const SimpleFontData* simpleFontData = font->primaryFont();
    const FontPlatformData& fontPlatformData = simpleFontData->platformData();

    // Get font matrix
    cairo_matrix_t ftm;
    cairo_scaled_font_t* scaledFont = fontPlatformData.scaledFont();
    cairo_scaled_font_get_scale_matrix(scaledFont, &ftm);
    double fontScaleX = ftm.xx;
    double fontScaleY = ftm.yy;

    // Prepare HfWidthIterator
    HfWidthIterator widthIterator(font, run);
    widthIterator.advance(from);
    float runWidthSoFar = widthIterator.m_runWidthSoFar;

    for (int i = from; i < to; ++i) {
        UChar charCode = *(run.data(i));
        // Skip ZERO_WIDTH_SPACE
        if (charCode == 0x200B)
            continue;
        Hafnium::GlyphMetrics metrics = hf->getGlyphMetrics(charCode);
        FloatPoint location((float)metrics.m_lsb, (float)metrics.m_tsb);
        FloatSize size((float)metrics.m_width, (float)metrics.m_height);
        FloatRect glyphBbox(location, size);
        glyphBbox.scale(fontScaleX, fontScaleY);
        glyphBbox.moveBy(pen);

        bbox.uniteIfNonZero(glyphBbox);

        // Advance one character.
        if (i + 1 < to) {
            widthIterator.advance(i + 1, (float)metrics.m_advance * fontScaleX);
            float advance = widthIterator.m_runWidthSoFar - runWidthSoFar;
            runWidthSoFar = widthIterator.m_runWidthSoFar;
            pen.moveBy(FloatPoint(advance, 0));
        }
    }

    return bbox;
}

bool HfRasterizer::init()
{
    if (s_isInitialized)
        return true;

    const char *env = getenv("ENABLE_HF");
    if (env && !strcmp(env, "true"))
        s_isEnabled = true;
    else
        s_isEnabled = false;

    if (!s_isEnabled)
        return true;

    Hafnium::StreamFuncs streamFuncs = {
        &HfOpenStream,
        &HfCloseStream,
        &HfReadStream
    };
    Hafnium::init(&streamFuncs);

    s_hfSerif = Hafnium::createInterface();
    s_hfSansSerif = Hafnium::createInterface();
    if (!s_hfSerif || !s_hfSansSerif)
        return false;

    s_hfSerif->bindFont(L"serif");
    s_hfSansSerif->bindFont(L"sans-serif");

    s_isInitialized = true;
    return true;
}

bool HfRasterizer::isEnabled()
{
    return s_isEnabled;
}

bool HfRasterizer::drawSimpleText(const Font* font, GraphicsContext* graphicsContext, const TextRun& run, const FloatPoint& point, int from, int to)
{
    // NOTE:
    // 'point' is the PEN position in the document coordinate.
    // 'from' and 'to' are not clipped by the target rect. 

    if (!s_isEnabled || !s_isInitialized)
        return false;

    if (graphicsContext->textDrawingMode() != TextModeFill)
        return false;

    const SimpleFontData* simpleFontData = font->primaryFont();
    if (simpleFontData->isCustomFont())
        return false;

    // Select font
    Hafnium::Interface *hf = isSerif(font) ? s_hfSerif : s_hfSansSerif;

    // Set attribute
    const FontPlatformData& fontPlatformData = simpleFontData->platformData();
    uint32_t attribute = 0;
    if (fontPlatformData.syntheticOblique())
        attribute |= Hafnium::Interface::OBLIQUE;
    if (fontPlatformData.shouldEmbolden())
        attribute |= Hafnium::Interface::BOLD;
    hf->setAttribute(attribute);

    // Get cairo context
    PlatformContextCairo* platformContext = graphicsContext->platformContext();
    cairo_t* cr = platformContext->cr();

    // Draw text shadow
    ShadowBlur& shadow = platformContext->shadowBlur();
    if (shadow.type() != ShadowBlur::NoShadow) {
        if (!shadow.mustUseShadowBlur(graphicsContext)) {
            // Optimize non-blurry shadows, by just drawing text without the ShadowBlur.
            cairo_save(cr);

            FloatSize shadowOffset(graphicsContext->state().shadowOffset);
            cairo_translate(cr, shadowOffset.width(), shadowOffset.height());
            setSourceRGBAFromColor(cr, graphicsContext->state().shadowColor);
            drawText(font, hf, cr, run, point, from, to);
            cairo_restore(cr);
        } else {
            FloatRect fontExtentsRect = layoutText(font, hf, run, from, to);

            if (GraphicsContext* shadowContext = shadow.beginShadowLayer(graphicsContext, fontExtentsRect)) {
                drawText(font, hf, shadowContext->platformContext()->cr(), run, point, from, to);
                shadow.endShadowLayer(graphicsContext);
            }
        }
    }

    platformContext->prepareForFilling(graphicsContext->state(), PlatformContextCairo::AdjustPatternForGlobalAlpha);
    drawText(font, hf, cr, run, point, from, to);

    return true;
}

}
