/*
 * Copyright (C) 2011 Apple Inc. All rights reserved.
 * Copyright (C) 2012 Igalia S.L.
 * Copyright (C) 2013 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef LayerTreeHostManx_h
#define LayerTreeHostManx_h

#include "LayerTreeContext.h"
#include "LayerTreeHost.h"
#include "Timer.h"
#include <WebCore/GraphicsLayerClient.h>
#include <canvas.h>
#include <wtf/OwnPtr.h>
#if USE(TEXTURE_MAPPER)
#include <WebCore/GLContext.h>
#include <WebCore/TextureMapper.h>
#endif

namespace WebKit {

class LayerTreeHostManx : public LayerTreeHost, WebCore::GraphicsLayerClient {
public:
    virtual ~LayerTreeHostManx();

public:
    static PassRefPtr<LayerTreeHostManx> create(WebPage*);

    virtual void setActive(bool);

protected:
    explicit LayerTreeHostManx(WebPage*);

    WebCore::GraphicsLayer* rootLayer() const { return m_rootLayer.get(); }

    void initialize();
    void performScheduledLayerFlush();

    // LayerTreeHost.
    virtual void invalidate();
    virtual void sizeDidChange(const WebCore::IntSize& newSize);
    virtual void deviceScaleFactorDidChange();
    virtual void forceRepaint();
    virtual void setRootCompositingLayer(WebCore::GraphicsLayer*);
    virtual void setLayerFlushSchedulingEnabled(bool);
    virtual void scheduleLayerFlush();

    // LayerTreeHostManx
    void didPerformScheduledLayerFlush();
    bool flushPendingLayerChanges();
    void cancelPendingLayerFlush();

    void layerFlushTimerFired(WebCore::Timer<LayerTreeHostManx>*);
    WebCore::Timer<LayerTreeHostManx> m_layerFlushTimer;
    bool m_layerFlushSchedulingEnabled;

    LayerTreeContext m_layerTreeContext;

private:
    // LayerTreeHost.
    virtual const LayerTreeContext& layerTreeContext();
    virtual void setShouldNotifyAfterNextScheduledLayerFlush(bool);

    virtual void setNonCompositedContentsNeedDisplay(const WebCore::IntRect&);
    virtual void scrollNonCompositedContents(const WebCore::IntRect& scrollRect, const WebCore::IntSize& scrollOffset);

    virtual void didInstallPageOverlay();
    virtual void didUninstallPageOverlay();
    virtual void setPageOverlayNeedsDisplay(const WebCore::IntRect&);

    // GraphicsLayerClient
    virtual void notifyAnimationStarted(const WebCore::GraphicsLayer*, double time);
    virtual void notifySyncRequired(const WebCore::GraphicsLayer*);
    virtual void paintContents(const WebCore::GraphicsLayer*, WebCore::GraphicsContext&, WebCore::GraphicsLayerPaintingPhase, const WebCore::IntRect& clipRect);
    virtual bool showDebugBorders(const WebCore::GraphicsLayer*) const;
    virtual bool showRepaintCounter(const WebCore::GraphicsLayer*) const;
    virtual float deviceScaleFactor() const;
    virtual void didCommitChangesForLayer(const WebCore::GraphicsLayer*) const { }

    // LayerTreeHostManx
    void createPageOverlayLayer();
    void destroyPageOverlayLayer();

    enum CompositePurpose { ForResize, NotForResize };
    void compositeLayersToContext(CompositePurpose = NotForResize);

    WebCore::GLContext* glContext();

    // Whether the layer tree host is valid or not.
    bool m_isValid;    

    // Whether we should let the drawing area know the next time we've flushed
    // layer tree changes.
    bool m_notifyAfterScheduledLayerFlush;

    bool m_isActive;
    
    // The root layer.
    OwnPtr<WebCore::GraphicsLayer> m_rootLayer;

    // The layer which contains all non-composited content.
    OwnPtr<WebCore::GraphicsLayer> m_nonCompositedContentLayer;

    // The page overlay layer. Will be null if there's no page overlay.
    OwnPtr<WebCore::GraphicsLayer> m_pageOverlayLayer;

#if USE(TEXTURE_MAPPER)
    OwnPtr<WebCore::TextureMapper> m_textureMapper;
    WebCore::GLContext *m_context;
#endif

    sce::Canvas::Id m_canvasID;
    struct CanvasBuffer* m_canvasBuffer;
#if ENABLE(MANX_INDIRECT_COMPOSITING)
    sce::IndirectCanvas* m_indirectCanvas;
    sce::IndirectCompositor* m_indirectCompositor;
#endif
};

} // namespace WebKit

#endif // LayerTreeHostManx_h
