/*
 * Copyright (C) 2013 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY SCE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SCE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef WebTileProxy_h
#define WebTileProxy_h

#if ENABLE(MANX_HTMLTILE)

#include "PageClient.h"

#include <WebCore/IntSize.h>
#include <WebCore/KURL.h>
#include <wtf/PassRefPtr.h>
#include <wtf/RefCounted.h>
#include <wtf/RefPtr.h>

namespace WebKit {

class WebContext;
class WebPageGroup;
class WebPageProxy;
class WebTileManagerProxy;

class WebTileProxy : public RefCounted<WebTileProxy>, public PageClient {
public:
    static PassRefPtr<WebTileProxy> create(PassRefPtr<WebTileManagerProxy>, WebPageGroup*);
    ~WebTileProxy();

    WebTileManagerProxy* tileManager() const { return m_tileManager.get(); }
    WebPageProxy* page() const { return m_page.get(); }

    uint64_t tileID() const;
    void setTileID(uint64_t);

    WebCore::IntSize tileSize() const;
    void setTileSize(int, int);

    bool isFocused() const;
    void setFocused(bool);
    bool isClicked() const;
    void setClicked(bool);

    void pause();
    void resume();

    void evalScript(const String& jsScript, uint64_t callbackID);

    void setTileURL(const String&);
    const WebCore::KURL& tileURL() const;
    String actualURL() const;

    void didUpdate();

private:
    WebTileProxy(PassRefPtr<WebTileManagerProxy>, WebPageGroup*);

    // PageClient interface
    virtual PassOwnPtr<DrawingAreaProxy> createDrawingAreaProxy();
    virtual void setViewNeedsDisplay(const WebCore::IntRect&);
    virtual void displayView();
    virtual void scrollView(const WebCore::IntRect&, const WebCore::IntSize&);
    virtual WebCore::IntSize viewSize();
    virtual bool isViewWindowActive();
    virtual bool isViewFocused();
    virtual bool isViewVisible();
    virtual bool isViewInWindow();

    virtual void processDidCrash();
    virtual void didRelaunchProcess();
    virtual void pageClosed();

    virtual void toolTipChanged(const String&, const String&);

#if USE(TILED_BACKING_STORE)
    virtual void pageDidRequestScroll(const WebCore::IntPoint&);
#endif
    virtual void didChangeContentEditable(bool contentEditable, int fieldType, int fieldLang, const WebCore::IntRect& fieldRect, const String& fieldText, int caretOffset);
    virtual void didChangeComposition(const WebCore::IntRect& compositionRect);
    virtual void didChangeSelection(const String& selectedText, const WebCore::IntRect& selectedRect);
    virtual void didMoveCaret(const String& fieldText, int caretOffset);

#if ENABLE(MANX_CURSOR_NAVIGATION)
    virtual void setCursorPosition(const WebCore::IntPoint&);
#endif
    virtual void runJavaScriptAlert(WebFrameProxy*, const String&, PassRefPtr<Messages::WebPageProxy::RunJavaScriptAlert::DelayedReply>);
    virtual void runJavaScriptComfirm(WebFrameProxy*, const String&,  PassRefPtr<Messages::WebPageProxy::RunJavaScriptConfirm::DelayedReply>);
    virtual void runJavaScriptPrompt(WebFrameProxy*, const String&, const String&, PassRefPtr<Messages::WebPageProxy::RunJavaScriptPrompt::DelayedReply>);
    virtual void handleAuthenticationRequiredRequest(WebFrameProxy*, const String&, const String&, const String&, const String&, PassRefPtr<Messages::WebPageProxy::AuthenticationRequiredRequest::DelayedReply>);
    virtual void handleCertificateVerificationRequest(WebFrameProxy*, uint32_t error, const String& url, const Vector<CString>& certificates, PassRefPtr<Messages::WebPageProxy::CertificateVerificationRequest::DelayedReply>);
    virtual void createNewPage(WebPageProxy*, const WebCore::ResourceRequest&, const WebCore::WindowFeatures&, WebEvent::Modifiers, WebMouseEvent::Button, PassRefPtr<Messages::WebPageProxy::CreateNewPage::DelayedReply>);
    virtual void setCursor(const WebCore::Cursor&);
    virtual void setCursorHiddenUntilMouseMoves(bool);
    virtual void didChangeViewportProperties(const WebCore::ViewportAttributes&);

    virtual void registerEditCommand(PassRefPtr<WebEditCommandProxy>, WebPageProxy::UndoOrRedo);
    virtual void clearAllEditCommands();
    virtual bool canUndoRedo(WebPageProxy::UndoOrRedo);
    virtual void executeUndoRedo(WebPageProxy::UndoOrRedo);
    virtual WebCore::FloatRect convertToDeviceSpace(const WebCore::FloatRect&);
    virtual WebCore::FloatRect convertToUserSpace(const WebCore::FloatRect&);
    virtual WebCore::IntPoint screenToWindow(const WebCore::IntPoint&);
    virtual WebCore::IntRect windowToScreen(const WebCore::IntRect&);

    virtual void doneWithKeyEvent(const NativeWebKeyboardEvent&, bool);
#if ENABLE(GESTURE_EVENTS)
    virtual void doneWithGestureEvent(const WebGestureEvent&, bool wasEventHandled);
#endif
#if ENABLE(TOUCH_EVENTS)
    virtual void doneWithTouchEvent(const NativeWebTouchEvent&, bool wasEventHandled);
#endif
    virtual void doneWithMouseDownEvent(bool wasEventHandled);
    virtual void doneWithMouseUpEvent(bool wasEventHandled);

    virtual PassRefPtr<WebPopupMenuProxy> createPopupMenuProxy(WebPageProxy*);
    virtual PassRefPtr<WebContextMenuProxy> createContextMenuProxy(WebPageProxy*);

    virtual void setFindIndicator(PassRefPtr<FindIndicator>, bool, bool);
#if USE(ACCELERATED_COMPOSITING)
    virtual void enterAcceleratedCompositingMode(const LayerTreeContext&);
    virtual void exitAcceleratedCompositingMode();
    virtual void updateAcceleratedCompositingMode(const LayerTreeContext&);
#endif

    virtual void didChangeScrollbarsForMainFrame() const;

    virtual void didCommitLoadForMainFrame(bool);
    virtual void didFinishLoadingDataForCustomRepresentation(const String&, const CoreIPC::DataReference&);
    virtual double customRepresentationZoomFactor();
    virtual void setCustomRepresentationZoomFactor(double);

    virtual void flashBackingStoreUpdates(const Vector<WebCore::IntRect>&);
    virtual void findStringInCustomRepresentation(const String&, FindOptions, unsigned);
    virtual void countStringMatchesInCustomRepresentation(const String&, FindOptions, unsigned);

    // End PageClient interface

    // WKPageLoaderClient callbacks
    static void didFinishLoadForFrame(WKPageRef, WKFrameRef, WKTypeRef userData, const void *clientInfo);
    static void didFailProvisionalLoadWithErrorForFrame(WKPageRef, WKFrameRef, WKErrorRef, WKTypeRef userData, const void *clientInfo);

    // WKPagePolicyClient callbacks
    static void decidePolicyForNavigationAction(WKPageRef, WKFrameRef, WKFrameNavigationType, WKEventModifiers, WKEventMouseButton, WKURLRequestRef, WKFramePolicyListenerRef, WKTypeRef userData, const void* clientInfo);

    // WKPageUIClient callbacks
    static bool shouldInterruptJavaScript(WKPageRef, const void *clientInfo);
    static unsigned long long exceededDatabaseQuota(WKPageRef, WKFrameRef, WKSecurityOriginRef, WKStringRef databaseName, WKStringRef displayName, unsigned long long currentQuota, unsigned long long currentOriginUsage, unsigned long long currentDatabaseUsage, unsigned long long expectedUsage, const void *clientInfo);

    // evalScript callback
    struct EvalScriptCallbackData {
        WebTileProxy* thiz;
        uint64_t callbackID;
    };
    static void evalScriptCallback(WKStringRef, WKErrorRef, void* context);

    RefPtr<WebTileManagerProxy> m_tileManager;
    RefPtr<WebPageProxy> m_page;

    uint64_t m_tileID;
    WebCore::IntSize m_tileSize;
    WebCore::KURL m_tileURL;

    bool m_isFocused;
    bool m_isClicked;
};

} // namespace WebKit

#endif // ENABLE(MANX_HTMLTILE)

#endif // WebTileProxy_h
