/*
 * Copyright (C) 2010 Apple Inc. All rights reserved.
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebContext.h"

#include "WebProcessCreationParameters.h"
#include "WebProcessMessages.h"
#include <WebCore/FileSystem.h>
#include <manx/NetworkUtility.h>
#include <manx/StoragePath.h>

using namespace WebCore;

namespace WebKit {

void WebContext::setUserID(int userID)
{
    m_userID = userID;
}

void WebContext::setWebProcessPath(const String& path)
{
    m_webProcessPath = path;
}

void WebContext::setApplicationCacheDirectory(const String& applicationCacheDirectory)
{
    m_applicationCacheDirectory = applicationCacheDirectory;
}

void WebContext::setCookieStorageDirectory(const String& cookieStorageDirectory)
{
    m_cookieStorageDirectory = cookieStorageDirectory;
}

void WebContext::setUserStorageDirectory(const String& userStorageDirectory)
{
    m_userStorageDirectory = userStorageDirectory;
}

void WebContext::setSystemStorageDirectory(const String& systemStorageDirectory)
{
    m_systemStorageDirectory = systemStorageDirectory;
}

void WebContext::platformInitializeWebProcess(WebProcessCreationParameters& parameters)
{
    parameters.userID = m_userID;
    parameters.cookieStorageDirectory = cookieStorageDirectory();
}

void WebContext::platformInvalidateContext()
{
}

String WebContext::applicationCacheDirectory()
{
    if (!m_applicationCacheDirectory.isEmpty())
        return m_applicationCacheDirectory;

    return platformDefaultApplicationCacheDirectory();
}

String WebContext::cookieStorageDirectory() const
{
    if (!m_cookieStorageDirectory.isEmpty())
        return m_cookieStorageDirectory;

    return platformDefaultCookieStorageDirectory();
}

String WebContext::platformDefaultApplicationCacheDirectory() const
{
    if (!m_systemStorageDirectory.isEmpty())
        Manx::StoragePath::setSystemStorageBaseDir(m_systemStorageDirectory.utf8().data());

    return String(Manx::StoragePath::appCache());
}

String WebContext::platformDefaultCookieStorageDirectory() const
{
    if (!m_userStorageDirectory.isEmpty())
        Manx::StoragePath::setUserStorageBaseDir(m_userStorageDirectory.utf8().data());

    return String(Manx::StoragePath::cookieStorage());
}

String WebContext::platformDefaultDatabaseDirectory() const
{
    if (!m_userStorageDirectory.isEmpty())
        Manx::StoragePath::setUserStorageBaseDir(m_userStorageDirectory.utf8().data());

    return String(Manx::StoragePath::sqlDatabase());
}

String WebContext::platformDefaultIconDatabasePath() const
{
    if (!m_systemStorageDirectory.isEmpty())
        Manx::StoragePath::setSystemStorageBaseDir(m_systemStorageDirectory.utf8().data());

    return String(Manx::StoragePath::iconDatabase());
}

String WebContext::platformDefaultLocalStorageDirectory() const
{
    if (!m_userStorageDirectory.isEmpty())
        Manx::StoragePath::setUserStorageBaseDir(m_userStorageDirectory.utf8().data());

    return String(Manx::StoragePath::localStorage());
}

void WebContext::setPrivilegedNetworkBandwidth(uint32_t mode)
{
    if (!m_process || !m_process->canSendMessage()) {
        m_pendingMessageToSetPrivilegedNetworkBandwidth = mode;
        return;
    }

    m_process->send(Messages::WebProcess::SetPrivilegedNetworkBandwidth(mode), 0);
}

void WebContext::setIFilterHTTPProxy(bool enable, const String& server, int port, const String& userId, const String& password)
{
    if (!m_process || !m_process->canSendMessage()) {
        m_pendingMessageTosetIFilterHTTPProxyEnable = enable;
        m_pendingMessageTosetIFilterHTTPProxyServer = server;
        m_pendingMessageTosetIFilterHTTPProxyPort = port;
        m_pendingMessageTosetIFilterHTTPProxyUserId = userId;
        m_pendingMessageTosetIFilterHTTPProxyPassword = password;
        return;
    }

    m_process->send(Messages::WebProcess::SetIFilterHTTPProxy(enable, server, port, userId, password), 0);
}

} // namespace WebKit

