/*
 * Copyright (C) 2012 Sony Network Entertainment Intl. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY SONY NETWORK ENTERTAINMENT INTL. ``AS IS''
 * ANY ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SONY NETWORK ENTERTAINMENT INTL.
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#include "config.h"

#if ENABLE(INSPECTOR_SERVER)
#include "WebInspectorServer.h"

#include "FileSystem.h"
#include "InspectorValues.h"
#include "MIMETypeRegistry.h"
#include "NotImplemented.h"
#include "WebContext.h"
#include "WebInspectorProxy.h"
#include "WebProcessProxy.h"
#include <errno.h>
#include <wtf/Assertions.h>
#include <wtf/text/CString.h>

using namespace WebCore;

namespace WebKit {

bool WebInspectorServer::platformResourceForPath(const String& path, Vector<char>& data, String& contentType)
{
    if (m_clientMap.isEmpty())
        return false;

    WebInspectorProxy* inspector = m_clientMap.begin()->second;
    ASSERT(inspector);

    if (path == "/json") {
        RefPtr<InspectorArray> message = InspectorArray::create();

        for (HashMap<unsigned, WebInspectorProxy*>::const_iterator it = m_clientMap.begin(); it != m_clientMap.end(); ++it) {
            WebInspectorProxy* client = it->second;
            WebPageProxy* page = client->page();
            WebProcessProxy* process = page->process();
            WebContext* context = process->context();
            RefPtr<InspectorObject> tabInfo = InspectorObject::create();

            tabInfo->setNumber("pageID", it->first);
            tabInfo->setNumber("processPageID", page->pageID());
            tabInfo->setString("title", page->pageTitle());
            tabInfo->setString("activeURL", page->activeURL());
            if (context->processModel() == ProcessModelSecondaryProcess) {
                tabInfo->setNumber("processID", process->processIdentifier());

                if (context->webProcessPath().endsWith("SecureWebProcess.self"))
                    tabInfo->setString("processDisplayName", "Live Area / Store / RegCAM");
                else if (context->webProcessPath().endsWith("WebProcessHTMLTile.self"))
                    tabInfo->setString("processDisplayName", "HTML Live Tiles");
                else
                    tabInfo->setString("processDisplayName", context->userStorageDirectory().endsWith("webbrowser") ? "Web Browser" : "Miscellaneous");
            }
            message->pushObject(tabInfo);
        }

        CString tabInfo = message->toJSONString().utf8();
        data.append(tabInfo.data(), tabInfo.length());
        contentType = "application/json; charset=utf-8";
        return true;
    }

    // Point the default path to display the inspector page directly, no multi-page support needed yet.
    // All other paths are mapped directly to a resource, if possible.
    const String resourceURL = (path == "/") ? inspector->inspectorPageURL() : inspector->inspectorBaseURL() + path;

    if (!resourceURL.startsWith("file://")) {
        notImplemented();
        return false;
    }

    const String localPath = resourceURL.substring(7);
    PlatformFileHandle handle = openFile(localPath, OpenForRead);
    if (!isHandleValid(handle)) {
        LOG_ERROR("WebInspectorServer: couldn't access platform resource '%s' for reading! (%d)", localPath.utf8().data(), errno);
        return false;
    }

    long long fileSize;
    if (!getFileSize(localPath, fileSize)) {
        LOG_ERROR("WebInspectorServer: couldn't get file size for '%s'! (%d)", localPath.utf8().data(), errno);
        closeFile(handle);
        return false;
    }
    data.grow(fileSize);
    if (readFromFile(handle, data.data(), data.size()) < fileSize) {
        LOG_ERROR("WebInspectorServer: didn't read all contents of file '%s'! (%d)", localPath.utf8().data(), errno);
        closeFile(handle);
        return false;
    }
    closeFile(handle);

    contentType = MIMETypeRegistry::getMIMETypeForPath(localPath);

    return true;
}

void WebInspectorServer::updateServerState()
{
    char* envValue = getenv("MANX_INSPECTOR_SERVER_PORT");
    if (!envValue)
        return;

    if (!m_clientMap.isEmpty() && serverState() == Closed) {
        const int inspectorServerPort = atoi(envValue);
        if (inspectorServerPort <= 0) {
            LOG_ERROR("Invalid Inspector Server port!");
            return;
        }

        if (!listen("", inspectorServerPort))
            LOG_ERROR("Couldn't start the Inspector Server!");

    } else if (m_clientMap.isEmpty() && serverState() == Listening) {
        close();
        ASSERT(serverState() == Closed);
    }
}

}

#endif // ENABLE(INSPECTOR_SERVER)
