/*
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"
#include "TiledBufferPainter.h"

#include "FrameView.h"
#include "GraphicsContext.h"
#include <wtf/StringExtras.h>

static bool gShowPaintTime = false;
static bool gShowDirtyRects = false;

static void showPaintTime(cairo_t *cr, int paintTime);
static void showDirtyRect(PassRefPtr<WebCore::TiledBuffer> , Vector<WebCore::IntRect>& dirtyRects, float contentsScale);

namespace {

class FallbackPainterCairo : public tilebackend::PainterInterface {
public:
    FallbackPainterCairo();

    virtual int beginPaint(tilebackend::Target*);
    virtual int endPaint();
    virtual int saveContext();
    virtual int restoreContext();
    virtual int translate(float x, float y);
    virtual int scale(float x, float y);
    virtual int paintTile(const tilebackend::Tile* , const tilebackend::IntRect*);
    virtual int setBackgroundColor(const float *);

private:
    void doPaint(const tilebackend::Tile* , const tilebackend::IntRect *);
    void doPaintFast(const tilebackend::Tile* , const tilebackend::IntRect *);
    void doPaintBackground(const tilebackend::IntRect*);

    cairo_t* m_cr;
    float m_backgroundColor[4];
};

FallbackPainterCairo::FallbackPainterCairo()
    : m_cr(0)
{
    m_backgroundColor[0] = 
    m_backgroundColor[1] = 
    m_backgroundColor[2] = 
    m_backgroundColor[3] = 1.0f;
}

int FallbackPainterCairo::beginPaint(tilebackend::Target* target)
{
    if (!target)
        CRASH();

    // create target surface and cairo context
    cairo_surface_t* surface =
        cairo_image_surface_create_for_data(
            (unsigned char*)target->buffer,
            (cairo_format_t)target->format,
            target->width,
            target->height,
            target->stride);
    
    cairo_t* cr = cairo_create(surface);
    if (cairo_status(cr) != CAIRO_STATUS_SUCCESS)
        CRASH();
    
    m_cr = cr;
    return 0;
}

int FallbackPainterCairo::endPaint()
{
    if (cairo_status(m_cr) != CAIRO_STATUS_SUCCESS)
        CRASH();

    cairo_surface_t* surface = cairo_get_target(m_cr);
    cairo_destroy(m_cr);
    cairo_surface_destroy(surface);

    m_cr = 0;
    return 0;
}

int FallbackPainterCairo::saveContext()
{
    cairo_save(m_cr);
    return 0;
}

int FallbackPainterCairo::restoreContext()
{
    cairo_restore(m_cr);
    return 0;
}

int FallbackPainterCairo::translate(float x, float y)
{
    cairo_translate(m_cr, x, y);

    cairo_matrix_t matrix;
    cairo_get_matrix(m_cr, &matrix);
    matrix.x0 = (int)matrix.x0;
    matrix.y0 = (int)matrix.y0;
    cairo_set_matrix(m_cr, &matrix);

    return 0;
}

int FallbackPainterCairo::scale(float x, float y)
{
    cairo_scale(m_cr, x, y);
    return 0;
}

int FallbackPainterCairo::paintTile(const tilebackend::Tile *tile, const tilebackend::IntRect *rect)
{
    if (rect->w <= 0 || rect->h <= 0)
        return 0;

    cairo_matrix_t matrix;
    cairo_get_matrix(m_cr, &matrix);
    if (tile) {
        if (matrix.xx == 1.0f && matrix.yy == 1.0f && !matrix.xy && !matrix.yx)
            doPaintFast(tile, rect);
        else
            doPaint(tile, rect);
    } else
        doPaintBackground(rect);
    return 0;
}

void FallbackPainterCairo::doPaint(const tilebackend::Tile* tile, const tilebackend::IntRect *rect)
{
    // create a image surface of tile buffer
    cairo_surface_t* surface =
        cairo_image_surface_create_for_data(
        (unsigned char*)tile->buffer,
        (cairo_format_t)tile->format,
        tile->width,
        tile->height,
        tile->stride);

    if (surface) {
        cairo_set_source_surface(m_cr, surface, rect->x, rect->y);
        cairo_pattern_set_filter(cairo_get_source(m_cr), CAIRO_FILTER_FAST);
        cairo_set_operator(m_cr, CAIRO_OPERATOR_OVER);

        cairo_rectangle(m_cr, rect->x, rect->y, rect->w, rect->h);
        cairo_fill(m_cr);

        cairo_surface_finish(surface);
        cairo_surface_destroy(surface);
    }

    // draw tile frame
    cairo_set_source_rgb(m_cr, 1.0, 0.0, 0.0);
    cairo_rectangle(m_cr, rect->x, rect->y, rect->w, rect->h);
    cairo_stroke(m_cr);
}

void FallbackPainterCairo::doPaintFast(const tilebackend::Tile* tile, const tilebackend::IntRect *rect)
{
    cairo_matrix_t matrix;
    cairo_get_matrix(m_cr, &matrix);
    
    cairo_surface_t* target = cairo_get_target(m_cr);
    int dstStride = cairo_image_surface_get_stride(target);
    int dstWidth  = cairo_image_surface_get_width(target);
    int dstHeight = cairo_image_surface_get_height(target);
    WebCore::IntRect dstRect(-matrix.x0, -matrix.y0, dstWidth, dstHeight);

    WebCore::IntRect paintRect = dstRect;
    paintRect.intersect(WebCore::IntRect(rect->x, rect->y, rect->w, rect->h));
    if (paintRect.isEmpty())
        return;

    // src
    int srcX = paintRect.x() - rect->x;
    int srcY = paintRect.y() - rect->y;
    uint32_t* src = (uint32_t*)tile->buffer;
    src += srcX + srcY * tile->width;

    // dst
    int dstX = paintRect.x() - dstRect.x();
    int dstY = paintRect.y() - dstRect.y();
    uint8_t* dst = (uint8_t*)cairo_image_surface_get_data(target);
    dst += dstX * sizeof(uint32_t) + dstY * dstStride;

    // do copy
    for (int y = 0; y < paintRect.height(); ++y) {
        memcpy(dst, src, paintRect.width()*sizeof(uint32_t));
        src += tile->width;
        dst += dstStride;
    }
}

void FallbackPainterCairo::doPaintBackground(const tilebackend::IntRect* rect)
{
    // fill
    cairo_set_source_rgb(m_cr, m_backgroundColor[0], m_backgroundColor[1], m_backgroundColor[2]);
    cairo_set_operator(m_cr, CAIRO_OPERATOR_SOURCE);
    cairo_rectangle(m_cr, rect->x, rect->y, rect->w, rect->h);
    cairo_fill(m_cr);

    // draw tile frame
    cairo_set_source_rgb(m_cr, 1.0, 0.0, 1.0);
    cairo_rectangle(m_cr, rect->x, rect->y, rect->w, rect->h);
    cairo_stroke(m_cr);
}

int FallbackPainterCairo::setBackgroundColor(const float *color)
{
    m_backgroundColor[0] = color[0];
    m_backgroundColor[1] = color[1];
    m_backgroundColor[2] = color[2];
    m_backgroundColor[3] = color[3];

    return 0;
}

FallbackPainterCairo s_fallbackPainter;

}

namespace {

class FallbackPool : public tilebackend::PoolInterface {
public:
    FallbackPool()
        : m_count(0)
        , m_tileExtent() { }

    virtual tilebackend::TileExtent getTileExtent() const
    {
        return m_tileExtent;
    }

    virtual int getMaxTileCount() const
    {
        return 256;
    }

    virtual int getCurrentTileCount() const
    {
        return m_count;
    }

    virtual void *allocTileBuffer()
    {
        int memSize = m_tileExtent.memSize();
        void* addr = malloc(memSize);
        if (addr)
            WTF::atomicIncrement(&m_count);
        return addr;
    }

    virtual void freeTileBuffer(void *addr)
    {
        if (addr)
            WTF::atomicDecrement(&m_count);
        free(addr);
    }

private:
    volatile int m_count;
    tilebackend::TileExtent m_tileExtent;
};

FallbackPool s_fallbackPool;

}

namespace WebCore {

TiledBufferPainter::TiledBufferPainter()
    : m_drawBuffer(0)
{
}

TiledBufferPainter::~TiledBufferPainter()
{
    free(m_drawBuffer);
}

TiledBufferPainter* TiledBufferPainter::create(const IntSize& screenSize, tilebackend::PoolInterface* pool, tilebackend::PainterInterface* painter)
{
    // create instance
    TiledBufferPainter* that = new TiledBufferPainter;
    if (that) {
        if (!pool)
            pool = &s_fallbackPool;

        tilebackend::TileExtent tileExtent = pool->getTileExtent();

        that->m_backgroundColor = Color::white;
        that->m_contentsScale = 1.0f;
        that->m_committedScale = 1.0f;
        that->m_screenSize = screenSize;
        that->m_pool = pool;
        that->m_painter = painter;
        that->m_tileExtent = tileExtent;

        int bufferSize = tileExtent.memSize();
        that->m_drawBuffer = (unsigned char*)malloc(bufferSize);
    }

    return that;
}

void TiledBufferPainter::setContentsScale(float contentsScale)
{
    m_contentsScale = contentsScale;
}

bool TiledBufferPainter::updateTiles(FrameView* view)
{
    tilebackend::PoolInterface* pool = m_pool;
    if (!pool)
        CRASH();

    bool needsRepaint = false;

    view->setPaintsEntireContents(true);

    IntRect viewRect = view->visibleContentRect();

    IntSize tileSize = this->tileSize();
    FloatRect rect = (FloatRect)viewRect;
    if (m_contentsScale > 0) {
        FloatSize margin = tileSize;
        margin.scale(1 / m_contentsScale);
        rect.expand(margin);
        margin.scale(-0.5);
        rect.move(margin);
    }
    rect.scale(m_contentsScale / tileSize.width(), m_contentsScale / tileSize.height());

    // rect in tile coord to cover VisibleContent
    IntRect coverRect = enclosingIntRect(rect);

    // expand cover rect
    FloatRect contentsRect(FloatPoint(), view->contentsSize());
    contentsRect.scale(m_contentsScale / tileSize.width(), m_contentsScale / tileSize.height());
    coverRect.intersect(enclosingIntRect(contentsRect));

    TileMap tiles; // A TileMap which update tiles are stored.

    if (m_committedScale == m_contentsScale) {

        // get dirty tiles
        Vector<TiledBuffer::Coordinate> dirtyTileCoords;
        updateDirtyRects();
        getDirtyTileCoords(coverRect, dirtyTileCoords);

        // for each dirty tile coords...
        unsigned size = dirtyTileCoords.size();
        for (unsigned i = 0; i < size; ++i) {
            TiledBuffer::Coordinate coord = dirtyTileCoords[i];

            // crate tile
            RefPtr<TiledBuffer> tile = TiledBuffer::create(coord, pool);

            // in case tile pool is empty, release tile and retry.
            if (!tile && releaseMostDistantTile(viewRect))
                tile = TiledBuffer::create(coord, pool);

            if (tile) {
                RefPtr<TiledBuffer> original = m_tiles.get(coord);
                if (original && original->isReadyForPartialRepaint(m_contentsScale)) {
                    // Copy already painted tile for partial painting.
                    tile->duplicate(original.get());
                }
                
                updateTile(view, tile);
                tiles.set(coord, tile);

                if (gShowDirtyRects)
                    showDirtyRect(tile, m_dirtyRects, m_contentsScale);
            } else
                needsRepaint = true;
        }
    } else {
        // ContentsScale changed. Update all tiles within coverRect.
        int left   = coverRect.x();
        int right  = coverRect.maxX();
        int top    = coverRect.y();
        int bottom = coverRect.maxY();

        for (int y = top; y < bottom; ++y) {
            for (int x = left; x < right; ++x) {
                TiledBuffer::Coordinate coord(x, y);

                // crate tile
                RefPtr<TiledBuffer> tile = TiledBuffer::create(coord, pool);

                // in case tile pool is empty, release tile and retry.
                if (!tile && releaseMostDistantTile(viewRect))
                    tile = TiledBuffer::create(coord, pool);

                if (tile) {
                    updateTile(view, tile);
                    tiles.set(coord, tile);

                    if (gShowDirtyRects)
                        showDirtyRect(tile, m_dirtyRects, m_contentsScale);
                } else
                    needsRepaint = true;
            }
        }
    }

    // commit
    commitTiles(tiles, m_contentsScale);

    Color backgroundColor = view->documentBackgroundColor();
    if (backgroundColor.isValid())
        m_backgroundColor.setRGB(backgroundColor.rgb() | 0xFF000000U);

    m_dirtyRects.clear();

    return needsRepaint;
}

void TiledBufferPainter::updateTile(FrameView* view, PassRefPtr<TiledBuffer> tile)
{
    bool isPaintingPartially = false;
    IntSize tileSize = this->tileSize();
    TileExtent tileExtent = m_tileExtent;
    TiledBuffer::Coordinate coord = tile->coordinate();

    // rect in the document coord to paint
    IntRect paintRect;
    FloatRect rect(coord.x(), coord.y(), 1, 1);
    rect.scale(tileSize.width() / m_contentsScale, tileSize.height() / m_contentsScale);
    paintRect = enclosingIntRect(rect);

    // Target surface rect in device coordinate.
    IntRect targetRect(coord.x() * tileSize.width(), coord.y() * tileSize.height(), tileExtent.width, tileExtent.height);

    if (tile->isReadyForPartialRepaint(m_contentsScale)) {
        // Paint partially
        isPaintingPartially = true;
        paintRect = tile->dirtyRect();
        FloatRect rect = paintRect;
        rect.scale(m_contentsScale);
        targetRect.intersect(enclosingIntRect(rect));

        rect = targetRect;
        rect.scale(1 / m_contentsScale);
        paintRect = enclosingIntRect(rect);
    }

    // create surface and context
    cairo_surface_t* surface = cairo_image_surface_create_for_data(
            m_drawBuffer,
            (cairo_format_t)CAIRO_FORMAT_ARGB32,
            targetRect.width(),
            targetRect.height(),
            tileExtent.stride);
    cairo_t* cr = cairo_create(surface);
    if (cairo_status(cr) != CAIRO_STATUS_SUCCESS)
        CRASH();

    // paint to a tile
    if (gShowPaintTime) {
        WebCore::GraphicsContext ctx(cr);
        ctx.save();
        ctx.translate(-targetRect.x(), -targetRect.y());
        ctx.scale(FloatSize(m_contentsScale, m_contentsScale));
        double paintTime = WTF::currentTime();
        view->paintContents(&ctx, paintRect);
        paintTime = WTF::currentTime() - paintTime;
        ctx.restore();

        // show paint time
        showPaintTime(cr, (int)(paintTime * 1000.0));
    } else {
        WebCore::GraphicsContext ctx(cr);
        ctx.save();
        ctx.translate(-targetRect.x(), -targetRect.y());
        ctx.scale(FloatSize(m_contentsScale, m_contentsScale));
        view->paintContents(&ctx, paintRect);
        ctx.restore();
    }

    // destroy context and surface
    cairo_destroy(cr);
    cairo_surface_destroy(surface);

    if (isPaintingPartially) {
        uint8_t* src = (uint8_t*)m_drawBuffer;
        uint8_t* dst = (uint8_t*)tile->buffer();
        dst += tileExtent.stride * (targetRect.y() - coord.y() * tileSize.height());
        dst += sizeof(uint32_t) * (targetRect.x() - coord.x() * tileSize.width());
        for (int y = 0; y < targetRect.height(); ++y) {
            memcpy(dst, src, sizeof(uint32_t) * targetRect.width());
            src += tileExtent.stride;
            dst += tileExtent.stride;
        }
    } else
        memcpy(tile->buffer(), m_drawBuffer, tileExtent.memSize());

    tile->setContentsScale(m_contentsScale);
    tile->setValid(true);
    tile->clearDirtyRect();
}

void TiledBufferPainter::commitTiles(TileMap &tiles, float contentsScale)
{
    WTF::MutexLocker locker(m_mutex);

    // release tiles which contentsScales are different from current contentsScale
    for (TileMap::iterator it = m_tiles.begin(); it != m_tiles.end(); ++it) {
        if (it->second && it->second->contentsScale() != contentsScale)
            it->second.release();
    }

    // set new tiles
    for (TileMap::iterator it = tiles.begin(); it != tiles.end(); ++it) {
        if (it->second)
            m_tiles.set(it->first, it->second);
    }

    // removed empty tiles
    bool done;
    do {
        done = true;
        for (TileMap::iterator it = m_tiles.begin(); it != m_tiles.end(); ++it) {
            if (!it->second) {
                m_tiles.remove(it);
                done = false;
                break;
            }
        }
    } while (!done);

    m_committedScale = contentsScale;
}

void TiledBufferPainter::updateDirtyRects()
{
    for (TileMap::iterator it = m_tiles.begin(); it != m_tiles.end(); ++it) {
        RefPtr<TiledBuffer> tile = it->second;
        if (!tile || tile->contentsScale() != m_contentsScale)
            continue;

        // check dirtyRects
        IntRect rect = tile->enclosingScaledRect();
        unsigned size = m_dirtyRects.size();
        for (unsigned i = 0; i < size; ++i) {
            IntRect dirtyRect = m_dirtyRects[i];
            dirtyRect.intersect(rect);
            if (!dirtyRect.isEmpty())
                tile->uniteDirtyRect(dirtyRect);
        }
    }
}

void TiledBufferPainter::getDirtyTileCoords(const IntRect& coverRect, Vector<TiledBuffer::Coordinate>& dirtyTileCoords)
{
    int left   = coverRect.x();
    int right  = coverRect.maxX();
    int top    = coverRect.y();
    int bottom = coverRect.maxY();

    for (int y = top; y < bottom; ++y) {
        for (int x = left; x < right; ++x) {
            TiledBuffer::Coordinate coord(x, y);
            IntRect tileRect = TiledBuffer::enclosingScaledRect(tileSize(), coord, m_contentsScale);

            RefPtr<TiledBuffer> tile = m_tiles.get(coord);
            if (!tile || tile->contentsScale() != m_contentsScale || tile->isDirty()) {
                dirtyTileCoords.append(coord);
                continue;
            }
        }
    }
}

bool TiledBufferPainter::releaseMostDistantTile(const IntRect& viewRect)
{
    WTF::MutexLocker locker(m_mutex);

    IntPoint center = viewRect.center();

    // find most distant tile
    TileMap::iterator candidate = m_tiles.end();
    int maxDistance = 0;
    for (TileMap::iterator it = m_tiles.begin(); it != m_tiles.end(); ++it) {
        IntRect rect = it->second->enclosingScaledRect();
        int distance = rect.distanceSquaredToPoint(center);
        if (distance > maxDistance) {
            maxDistance = distance;
            candidate = it;
        }
    }

    if (candidate != m_tiles.end()) {
        m_tiles.remove(candidate);
        return true;
    }

    return false;
}

void TiledBufferPainter::paint(tilebackend::Target* target, const IntRect& viewRect)
{
    if (!target)
        return;

    tilebackend::PainterInterface* painter = m_painter;

    IntSize tileSize = this->tileSize();

    // acquire lock to copy tile map
    m_mutex.lock();
    TileMap tiles;
    float committedScale = m_committedScale;
    for (TileMap::iterator it = m_tiles.begin(); it != m_tiles.end(); ++it)
        tiles.set(it->first, it->second);
    m_mutex.unlock();

    if (!painter)
        painter = &s_fallbackPainter;

    float color[4] = {
        m_backgroundColor.red()   / 255.0f,
        m_backgroundColor.green() / 255.0f,
        m_backgroundColor.blue()  / 255.0f,
        m_backgroundColor.alpha() / 255.0f
    };
    painter->setBackgroundColor(color);

    // begin paint
    painter->beginPaint(target);

    // scale
//    float contentsScale = m_contentsScale;
    float contentsScale = (float)target->height / viewRect.height();
    float scale = 1.0f;

    // translate
    painter->translate(-viewRect.x() * contentsScale, -viewRect.y() * contentsScale);

    if (contentsScale != committedScale)
        scale = contentsScale / committedScale;
    if (scale != 1.0f)
        painter->scale(scale, scale);

    // calculate required tile coord
    FloatRect floatViewRect(viewRect);
    floatViewRect.scale(committedScale);
    floatViewRect.scale(1.0f / tileSize.width(), 1.0f / tileSize.height());
    IntRect coverRect = enclosingIntRect(floatViewRect);
    
    for (int y = coverRect.y(); y < coverRect.maxY(); y++) {
        for (int x = coverRect.x(); x < coverRect.maxX(); x++) {
            TiledBuffer::Coordinate coord(x, y);
            // check intersection
            IntRect tileRect = TiledBuffer::enclosingScaledRect(tileSize, coord, committedScale);
            if (!tileRect.intersects(viewRect)) {
                // CRASH();
                continue;
            }

            // do paint
            painter->saveContext();

            // get TiledBuffer
            RefPtr<TiledBuffer> tilePtr = tiles.get(coord);

            if (tilePtr && tilePtr->isValid()) {
                tileRect = tilePtr->tileRect();
                tilebackend::IntRect rect;
                rect.x = tileRect.x();
                rect.y = tileRect.y();
                rect.w = tileRect.width();
                rect.h = tileRect.height();

                tilebackend::Tile tile;
                tile.buffer = tilePtr->buffer();
                tile.width = m_tileExtent.width;
                tile.height = m_tileExtent.height;
                tile.stride = m_tileExtent.stride;
                tile.format = CAIRO_FORMAT_ARGB32;

                painter->paintTile(&tile, &rect);
            } else {
                tileRect = TiledBuffer::tileRect(tileSize, coord);
                tilebackend::IntRect rect;
                rect.x = tileRect.x();
                rect.y = tileRect.y();
                rect.w = tileRect.width();
                rect.h = tileRect.height();

                painter->paintTile(0, &rect);
            }

            painter->restoreContext();
        }
    }

    // end paint
    painter->endPaint();
}

void TiledBufferPainter::invalidate(const IntRect& updateRect, bool immediate)
{
    // check if updateRect is contained by other rects
    unsigned size = m_dirtyRects.size();
    for (unsigned i = 0; i < size; ++i) {
        if (m_dirtyRects[i].contains(updateRect))
            return;
    }

    m_dirtyRects.append(updateRect);
}

void TiledBufferPainter::clear()
{
    for (TileMap::iterator it = m_tiles.begin(); it != m_tiles.end(); ++it)
        it->second->setValid(false);

    WTF::MutexLocker locker(m_mutex);
    m_tiles.clear();
}

} // namespace WebCore

static void showPaintTime(cairo_t *cr, int paintTime)
{
    char buf[64];
    snprintf(buf, 64, "%d ms", paintTime);
    cairo_set_source_rgb(cr, 1.0, 0.0, 0.0);
    cairo_set_font_size(cr, 30.0);
    cairo_move_to(cr, 0.0, 30.0);
    cairo_show_text(cr, buf);
}

static void showDirtyRect(PassRefPtr<WebCore::TiledBuffer> tile, Vector<WebCore::IntRect>& dirtyRects, float contentsScale)
{
    tilebackend::TileExtent tileExtent = tile->tileExtent();

    // create surface and context
    cairo_surface_t* surface =
        cairo_image_surface_create_for_data(
            (unsigned char*)tile->buffer(),
            (cairo_format_t)CAIRO_FORMAT_ARGB32,
            tileExtent.width,
            tileExtent.height,
            tileExtent.stride);
    cairo_t* cr = cairo_create(surface);
    if (cairo_status(cr) != CAIRO_STATUS_SUCCESS)
        CRASH();

    cairo_set_source_rgba(cr, 1, 1, 0, 0.5);

    WebCore::IntPoint pos = tile->tileRect().location();
    cairo_translate(cr, -(double)pos.x(), -(double)pos.y());
    cairo_scale(cr, contentsScale, contentsScale);

    // check dirtyRects
    unsigned size = dirtyRects.size();
    for (unsigned i = 0; i < size; ++i) {
        WebCore::FloatRect rect = dirtyRects[i];

        cairo_rectangle(cr, rect.x(), rect.y(), rect.width(), rect.height());
        cairo_stroke(cr);
    }

    // destroy context and surface
    cairo_destroy(cr);
    cairo_surface_destroy(surface);
}

static JSValueRef getTileProperty(JSContextRef ctx, JSObjectRef object, JSStringRef propertyName, JSValueRef* exception)
{
    if (JSStringIsEqualToUTF8CString(propertyName, "showDirtyRect"))
        return JSValueMakeBoolean(ctx, gShowDirtyRects);

    if (JSStringIsEqualToUTF8CString(propertyName, "showPaintTime"))
        return JSValueMakeBoolean(ctx, gShowPaintTime);

    return 0;
}

static bool setTileProperty(JSContextRef ctx, JSObjectRef object, JSStringRef propertyName, JSValueRef value, JSValueRef* exception)
{
    if (JSStringIsEqualToUTF8CString(propertyName, "showDirtyRect") && JSValueIsBoolean(ctx, value)) {
        gShowDirtyRects = JSValueToBoolean(ctx, value);
        return true;
    }

    if (JSStringIsEqualToUTF8CString(propertyName, "showPaintTime") && JSValueIsBoolean(ctx, value)) {
        gShowPaintTime = JSValueToBoolean(ctx, value);
        return true;
    }

    return false;
}
