/*
 * Copyright (C) 2006 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2006 Michael Emmel mike.emmel@gmail.com
 * All rights reserved.
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GO188ODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef ResourceHandleManager_h
#define ResourceHandleManager_h

#include "FormDataStreamCurl.h"
#include "Frame.h"
#include "PlatformString.h"
#include "ResourceHandleClient.h"
#include "Timer.h"
#include <wtf/Threading.h>
#include <wtf/threads/BinarySemaphore.h>

#if PLATFORM(WIN) || PLATFORM(MANX) && OS(WINDOWS)
#include <windows.h>
#include <winsock2.h>
#endif

#include <curl/curl.h>
#include <manx/NetworkProfile.h>
#include <manx/Ssl.h>
#include <manx/X509.h>
#include <wtf/Vector.h>
#include <wtf/text/CString.h>

#if OS(ORBIS)
#define EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
// #define DISABLE_RUNNINGJOBS_COUNTING
#endif

namespace WebCore {

enum HTTPMethodType {
    HttpMethod_Get,
    HttpMethod_Head,
    HttpMethod_Post,
    HttpMethod_Put,
    HttpMethod_Delete,
    HttpMethod_Options,
};

class UserDefinedThreadContext {
    WTF_MAKE_FAST_ALLOCATED;
 public:
    UserDefinedThreadContext(WTF::Mutex* threadCondMutex,
                             WTF::Mutex* threadCreatedCondMutex,
                             WTF::ThreadCondition* threadCond,
                             WTF::ThreadCondition* threadCreatedCond)
        : m_threadCondMutex(threadCondMutex)
        , m_threadCreatedCondMutex(threadCreatedCondMutex)
        , m_threadCond(threadCond)
        , m_threadCreatedCond(threadCreatedCond)
    {
    }

    WTF::Mutex* m_threadCondMutex;
    WTF::Mutex* m_threadCreatedCondMutex;
    WTF::ThreadCondition* m_threadCond;
    WTF::ThreadCondition* m_threadCreatedCond;

};

class ResourceHandleContext : public Manx::Ssl::SslContext {
    WTF_MAKE_FAST_ALLOCATED;
 public:

    ResourceHandleContext(ResourceHandle* job)
        : m_url(0)
        , m_handle(0) 
        , job(job)
        , m_customHeaders(0)
        , m_isHttpsScheme(false)
        , m_cancelled(false)
        , m_defersLoading(false)
        , hasLocation(false)
        , m_formDataStream(job)
    {
        m_filteringhandle = 0;
        m_filteringCustomHeaders = 0;
        m_filteringHeadersSlist = 0;
        m_isPost = false;
        m_redirAfterPost = false;
        m_isFinishedContext = false;
        m_hasHttpRealm = false;
        m_hasProxyRealm = false;
        m_blockedLocation = false;
        m_curlResult = CURLE_OK;
        m_filteringActivatedStatus = 0;
        m_httpAuthType = Manx::HttpAuthNone;
        m_proxyAuthType = Manx::HttpAuthNone;
        m_hasHttpCredential = false;
        m_hasProxyCredential = false;
        m_authCancelled = false;
        m_proxyAuthMaybeFailed = false;
        m_authModeFromHeader = Manx::AuthModeNone;
        m_authModeInProcess = Manx::AuthModeNone;
        m_httpChallenge.nullify();
        m_proxyChallenge.nullify();
        if ("POST" == job->firstRequest().httpMethod())
            m_method = HttpMethod_Post;
        else if ("PUT" == job->firstRequest().httpMethod())
            m_method = HttpMethod_Put;
        else if ("HEAD" == job->firstRequest().httpMethod())
            m_method = HttpMethod_Head;
        else if ("DELETE" == job->firstRequest().httpMethod())
            m_method = HttpMethod_Delete;
        else if ("OPTIONS" == job->firstRequest().httpMethod())
            m_method = HttpMethod_Options;
        else 
            m_method = HttpMethod_Get;
    }

    ~ResourceHandleContext()
    {
        if (m_host)
            fastFree(m_host);
        if (m_url)
            fastFree(m_url);
        if (m_customHeaders) {
            curl_slist_free_all(m_customHeaders);
            m_customHeaders = 0;
        }
        if (m_filteringHeadersSlist) {
            curl_slist_free_all(m_filteringHeadersSlist);
            m_filteringHeadersSlist = 0;
        }
        if (m_isPost)
            Manx::NetworkProfile::unMountUploadDir();
    }

    char* m_url;
    CURL* m_handle;
    CURLcode m_curlResult;
    RefPtr<NetworkingContext> m_nwcontext;
    ResourceHandle* job;
    struct curl_slist* m_customHeaders;

    WebCore::HTTPMethodType m_method;
    bool m_isHttpsScheme;
    bool m_cancelled;
    bool m_defersLoading;
    bool hasLocation;
    bool m_isPost;
    bool m_redirAfterPost;
    bool m_isFinishedContext;

    // authenticate
    bool m_hasHttpRealm;
    bool m_hasProxyRealm;
    bool m_hasHttpCredential;
    bool m_hasProxyCredential;
    bool m_proxyAuthMaybeFailed;
    bool m_authCancelled;
    long m_httpAuthType; // Basic(1) or Digest(2) (set to curl_easy_setopt)
    long m_proxyAuthType; // Basic(1) or Digest(2) (set to curl_easy_setopt)
    int m_authModeFromHeader; // http or/and proxy 
    int m_authModeInProcess; // http or/and proxy 
    String m_httpRealm;
    String m_proxyRealm;
    String m_httpUserPass;
    String m_proxyUserPass;
    String m_preSetProxyUserPass;
    ProtectionSpaceServerType m_httpProtectType;
    ProtectionSpaceServerType m_proxyProtectType;
    AuthenticationChallenge m_httpChallenge;
    AuthenticationChallenge m_proxyChallenge;
    ProtectionSpace m_httpSpace;
    ProtectionSpace m_proxySpace;

    FormDataStream m_formDataStream;
    RefPtr<FormData> m_formData;
    KURL m_kurl;
    KURL m_proxyurl;

    Vector<char> m_postBytes;

    // members for filtering
    String m_filteringurl;
    CURL* m_filteringhandle;
    struct curl_slist* m_filteringCustomHeaders;
    ResourceRequest m_filteringRequest;
    KURL m_evaluatedkurl;
    bool m_blockedLocation;
    int m_filteringActivatedStatus;
    struct curl_slist* m_filteringHeadersSlist;

    Vector<char> m_filteringRespHeaders;
    Vector<char> m_filteringBody;

};

class ResourceHandleManager {
public:
#if PLATFORM(MANX)
    static ResourceHandleManager *s_sharedInstance;
    static void cleanup();
#endif
    static ResourceHandleManager* sharedInstance();
    void add(ResourceHandle*, NetworkingContext*);
    void cancel(ResourceHandle*);
    void setCookieJarFileName(const char* cookiejarFileName);
#if PLATFORM(MANX)
    void setCookies(const KURL&, const String&);
    String cookies(const KURL &);
    int deleteAllCookies();
    void cleanupSessionCookies();
    void saveCookies();
#endif

    void dispatchSynchronousJob(ResourceHandle*, NetworkingContext*);

    void setupPOST(ResourceHandle*, struct curl_slist**);
    void setupPUT(ResourceHandle*, struct curl_slist**);

    void setupPOSTAsync(ResourceHandleContext*, struct curl_slist**);
    void setupPUTAsync(ResourceHandleContext*, struct curl_slist**);

    static void signalWorkerThread()
    {
        WTF::MutexLocker lock(*s_workerThreadConditionMutex);
        s_workerThreadCondition->signal();
    }
    static void signalFilteringThread()
    {
        WTF::MutexLocker lock(*s_filteringThreadConditionMutex);
        s_filteringThreadCondition->signal();
    }
    static void createAuthenticateThread(void* data);
    static void handleSslBadCertificate(void* context);
    static CURL* curlShareHandle()
    {
        return sharedInstance()->m_curlShareHandle;
    }

    static void lockCurlSharedResource(curl_lock_data);
    static void unlockCurlSharedResource(curl_lock_data);

    void initializeCurlHandle(CURL*, ResourceHandleContext*, curl_write_callback, curl_write_callback);
    void initializeHandleForSync(ResourceHandleContext*);
    void initializeHandleForAsync(ResourceHandleContext*);
    void initializeHandleForFiltering(ResourceHandleContext*);
    
private:
    ResourceHandleManager();
    ~ResourceHandleManager();
    bool processDownload();
    bool processFiltering();

    static void saveHttpCredential(ResourceHandleContext*);
    static void saveProxyCredential(ResourceHandleContext*);

    static size_t prepareHandleAuthenticate(ResourceHandleContext*, long, long, size_t, void*);
    static size_t headerCallbackAsync(char*, size_t, size_t, void*);
    static size_t headerCallback(char*, size_t, size_t, void*);
    static size_t writeCallbackAsync(char*, size_t, size_t, void*);
    static size_t readCallbackAsync(void*, size_t, size_t, void*);

    static void postNotifyBadCertSynchronously(void* context);
    static void notifyBadCertSynchronously(void* cmd);

    void removeFromCurl(ResourceHandleContext*);
    void removeFromCurlForFiltering(ResourceHandleContext*);

    bool removeScheduledJob(ResourceHandle*);
    void startJob(ResourceHandleContext*);
    void startFilteringJob(ResourceHandleContext*);
    void checkAndCreateFilteringThread();

    bool startScheduledJobs();
    bool startScheduledFilteringJobs();

    void saveAuthCredential(ResourceHandleContext*);
    void setCustomHeaders(const HTTPHeaderMap&, ResourceHandleContext*);

    CURLSH* m_curlShareHandle;
    char* m_cookieJarFileName;
    char m_curlErrorBuffer[CURL_ERROR_SIZE];
    Vector<ResourceHandleContext*> m_resourceHandleContextList;
    Vector<ResourceHandleContext*> m_filteringResourceHandleContextList;

#ifndef DISABLE_RUNNINGJOBS_COUNTING
    int m_runningJobs;
    int m_runningFilteringJobs;
#endif

    static WTF::ThreadIdentifier s_userThreadId;
    static WTF::ThreadIdentifier s_workerThreadId;
    static WTF::ThreadIdentifier s_filteringThreadId;

    static WTF::Mutex* s_userThreadMutex;
    static WTF::Mutex* s_workerThreadMutex;
    static WTF::Mutex* s_filteringThreadMutex;
    static WTF::Mutex* s_userThreadConditionMutex;
    static WTF::Mutex* s_workerThreadConditionMutex;
    static WTF::Mutex* s_filteringThreadConditionMutex;
    static WTF::Mutex* s_notifyBadCertConditionMutex;

    static WTF::ThreadCondition* s_userThreadCondition;
    static WTF::ThreadCondition* s_workerThreadCondition;
    static WTF::ThreadCondition* s_filteringThreadCondition;
    static WTF::ThreadCondition* s_notifyBadCertCondition;

    static void threadWaitFunction(UserDefinedThreadContext* threadctx, bool& threadCreated);
    static WTF::ThreadIdentifier createUserDefinedThread(WTF::ThreadFunction threadfunc, void* ctxt, const char* threadName, WTF::Mutex*& threadCondMutex, WTF::ThreadCondition*& thrCond, WTF::Mutex*& threadCreatedCondMutex, WTF::ThreadCondition*& threadCreatedCond);

    static void userCallbackThread(void* thrcontext);
    static void workerThread(void *context);
    static void filteringThread(void *context);
    static void handleBadCertificate();

    static void selectSockets(CURLM* multi, long timeoutMilliSecond);
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
    static void selectSocketsForFiltering(CURLM* multi, long timeoutMilliSecond);
#endif

};

}

#endif
