/*
 * Copyright (C) 2004, 2006 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2006 Michael Emmel mike.emmel@gmail.com
 * Copyright (C) 2007 Alp Toker <alp@atoker.com>
 * Copyright (C) 2007 Holger Hans Peter Freyther
 * Copyright (C) 2008 Collabora Ltd.
 * Copyright (C) 2008 Nuanti Ltd.
 * Copyright (C) 2009 Appcelerator Inc.
 * Copyright (C) 2009 Brent Fulgham <bfulgham@webkit.org>
 * All rights reserved.
 * Copyright (C) 2012 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "ResourceHandleManager.h"

#include "CredentialBackingStoreManx.h"
#include "CredentialTransformDataManx.h"
#include "DataURL.h"
#include "DocumentLoader.h"
#include "FrameLoaderClient.h"
#include "HTTPParsers.h"
#include "MIMETypeRegistry.h"
#include "MainResourceLoader.h"
#include "NetworkStateNotifier.h"
#include "NotImplemented.h"
#include "ResourceError.h"
#include "ResourceHandle.h"
#include "ResourceHandleInternal.h"
#include "RunLoop.h"

#include <errno.h>
#include <manx/CookieJar.h>
#include <manx/Date.h>
#include <manx/Locale.h>
#include <manx/NetworkMemory.h>
#include <manx/NetworkProfile.h>
#include <manx/WebSecurity.h>
#include <stdio.h>
#include <wtf/HashSet.h>
#include <wtf/ThreadingPrimitives.h>

#if OS(PSP2)
#define USE_SCRACHPAD_COOKIEJAR 1
#endif

#if USE_SCRACHPAD_COOKIEJAR
extern "C" {
bool scrachpadmatch(const char *path);
char* makescrachpadpath(const char *path);
void freescrachpadpath(char *path);
void clearscrachpad();
}
#endif

// Reuse curl easy handle to keep connections in synchronous operations. In asynchronous operations,
// connection pool is used by 'multi' handle. Note that to keep connection use curl_easy_reset()
// instead of curl_easy_cleanup().
#if OS(ORBIS)
#define ENABLE_PERSISTENT_CONNECTION_FOR_SYNCHRONOUS_COMMUNICATION
#endif
#ifdef ENABLE_PERSISTENT_CONNECTION_FOR_SYNCHRONOUS_COMMUNICATION

static pthread_key_t curlEasyHandleForSyncKey;
static pthread_once_t curlEasyHandleForSyncKeyOnce = PTHREAD_ONCE_INIT;

static void finalizeCurlEasyHandleForSync(void* tsd)
{
    curl_easy_cleanup(reinterpret_cast<CURL*>(tsd));
}

static void initializeCurlEasyHandleForSyncKey()
{
    pthread_key_create(&curlEasyHandleForSyncKey, finalizeCurlEasyHandleForSync);
}

static CURL* curlEasyHandleForSync()
{
    pthread_once(&curlEasyHandleForSyncKeyOnce, initializeCurlEasyHandleForSyncKey);
    CURL* handle = reinterpret_cast<CURL*>(pthread_getspecific(curlEasyHandleForSyncKey));
    if (!handle) {
        handle = curl_easy_init();
        pthread_setspecific(curlEasyHandleForSyncKey, handle);
    }
    return handle;
}
#endif

#define RESOURCEHANDLE_COMMAND_DATA_MAX    7

class ResourceHandleCommand {
public:
    union data {
        int typeInt;
        bool typeBool;
        char typeChar;
        char* typeCharPtr;
        void* typeVoidPtr;
        const char* typeConstCharPtr;
        const void* typeConstVoidPtr;
    };

    ResourceHandleCommand()
    {
        memset(m_data, 0x00, sizeof(union data) * RESOURCEHANDLE_COMMAND_DATA_MAX);
    }

    ~ResourceHandleCommand()
    {
    }

    union data m_data[RESOURCEHANDLE_COMMAND_DATA_MAX];
};

#define ASYNC_REQUEST_DEBUG 0
// #define DISABLE_X_FRAME_OPTIONS
// #define DEBUG_CURL 0

#if ASYNC_REQUEST_DEBUG
#define DEBUG_LOG(...) printf(__VA_ARGS__)
static int jobscount = 0;
#else
#define DEBUG_LOG(...)
#endif

namespace WebCore {

/*
 * Functions declaration
 */

// functions running on webkit thread and receiving and translating network message
// from worker thread.
static void didRecvData(void* cmd);
static void didRecvResponse(void* cmd);
static void willSendRequest(void* cmd);
static void didFinish(void* cmd);
static void didFail(void* cmd);
static void addResponseHeader(void* cmd);
static void handleCancel(void* cmd);
static void handleResponseHeaders(void* cmd);
static void notifyOriginFrame(void* cmd);
#ifndef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
static void postWorkerThreadPumpMessage(void* cmd);
#endif
static CURLcode sslctxfun(CURL* curlhandle, void* sslctx, void* param);
static void replaceUrlOfMainResource(void* data);

// functions running on worker thread and posting network message to webkit thread.
void postDidRecvData(ResourceHandleContext*, void* ptr, const char* hdr, size_t totalSize);
void postDidRecvResponse(ResourceHandleContext*);
void postWillSendRequest(ResourceHandleContext*);
void postHandleCancel(ResourceHandleContext*);
void postDidFinish(ResourceHandleContext*);
void postDidFail(ResourceHandleContext*);
void postAddResponseHeader(ResourceHandleContext*, char* ptr, size_t totalSize);
void postHandleResponseHeaders(ResourceHandleContext*);
void postNotifyOriginFrame(void* context);

// functions for url filtering
static bool evaluateFilteringResult(ResourceHandleContext* , bool&);
static bool convertToFilteringUrl(ResourceHandleContext* , bool&);

// resources which are used to synchronize thread messages
// between webkit-thread and worker-thread.
WTF::ThreadIdentifier ResourceHandleManager::s_userThreadId = 0;
WTF::ThreadIdentifier ResourceHandleManager::s_workerThreadId = 0;
WTF::ThreadIdentifier ResourceHandleManager::s_filteringThreadId = 0;
static WTF::Mutex* s_authContextListMutex = 0;
static WTF::Mutex* s_authProcessMutex = 0;
WTF::Mutex* ResourceHandleManager::s_userThreadMutex = 0;
WTF::Mutex* ResourceHandleManager::s_userThreadConditionMutex = 0;
WTF::Mutex* ResourceHandleManager::s_workerThreadMutex = 0;
WTF::Mutex* ResourceHandleManager::s_workerThreadConditionMutex = 0;
WTF::Mutex* ResourceHandleManager::s_filteringThreadMutex = 0;
WTF::Mutex* ResourceHandleManager::s_filteringThreadConditionMutex = 0;
WTF::Mutex* ResourceHandleManager::s_notifyBadCertConditionMutex = 0;
WTF::ThreadCondition* ResourceHandleManager::s_userThreadCondition = 0;
WTF::ThreadCondition* ResourceHandleManager::s_workerThreadCondition = 0;
WTF::ThreadCondition* ResourceHandleManager::s_filteringThreadCondition = 0;
WTF::ThreadCondition* ResourceHandleManager::s_notifyBadCertCondition = 0;
static bool s_userThreadCreated = false;
static bool s_workerThreadCreated = false;
static bool s_filteringThreadCreated = false;
static WTF::ThreadCondition* s_userThreadCreatedCondition = 0;
static WTF::Mutex* s_userThreadCreatedConditionMutex = 0;
static WTF::ThreadCondition* s_workerThreadCreatedCondition = 0;
static WTF::Mutex* s_workerThreadCreatedConditionMutex = 0;
static WTF::ThreadCondition* s_filteringThreadCreatedCondition = 0;
static WTF::Mutex* s_filteringThreadCreatedConditionMutex = 0;

// curl multi handles which retains live network context including 
// various information about url, header, request body and etc.
static CURLM* s_workerMultiHandle = 0;
static WTF::Mutex* s_workerThreadMultiMutex = 0;
static CURLM* s_filteringMultiHandle = 0;

// default timeouts
static long s_transferTimeout = 0;
static long s_connectTimeout = 60;
static long s_dnsCacheTimeout = 60 * 5;

#ifndef NDEBUG
static ThreadIdentifier workerThreadIdentifier;
static bool isWorkerThread()
{
    return currentThread() == workerThreadIdentifier;
}
#endif

#if defined(EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION) || OS(PSP2)
static void resetSslConn(ResourceHandleContext* context)
{
    ResourceHandleManager::sharedInstance()->initializeHandleForAsync(context);
    curl_easy_setopt(context->m_handle, CURLOPT_SSL_VERIFYHOST, false);
    curl_easy_setopt(context->m_handle, CURLOPT_SSL_VERIFYPEER, false);

    // set auth credential if it exists.
    if (!context->m_httpUserPass.isEmpty()) {
        curl_easy_setopt(context->m_handle, CURLOPT_HTTPAUTH, context->m_httpAuthType);
        curl_easy_setopt(context->m_handle, CURLOPT_USERPWD, context->m_httpUserPass.latin1().data());
    }
    if (!context->m_proxyUserPass.isEmpty()) {
        curl_easy_setopt(context->m_handle, CURLOPT_PROXYAUTH, context->m_proxyAuthType);
        curl_easy_setopt(context->m_handle, CURLOPT_PROXYUSERPWD, context->m_proxyUserPass.latin1().data());
        context->m_proxyAuthMaybeFailed = false;
    }
}

// map between ResourceHandle and ResourceHandleContext
static HashMap<ResourceHandle*, ResourceHandleContext*> s_jobToContextMap;
static Vector<ResourceHandleContext*> s_cancelContextList;
#endif
static bool s_exit = false;
//
//  handle bad certificate
//  handle authentication
//
static WTF::Vector<ResourceHandleContext*> m_authContextList;
static WTF::HashSet<unsigned> s_userAcceptedHttpsDomains;
static Vector<ResourceHandleContext*> m_sslBadcertResourceHandleContextList;
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
static Vector<ResourceHandleContext*> m_sslProcessedBadCertRHCList;
static WTF::Vector<ResourceHandleContext*> m_authProcessedContextList;
#define RHM_SELECT_PIPE_BUF 32
static int s_RPIPE[4] = { 0, 0, 0, 0 };
static int s_WPIPE[4] = { 0, 0, 0, 0 };
static int s_maxPipeFd = 0;
static void restart(ResourceHandleContext* context)
{
    DEBUG_LOG("-> restart() url = [%s]\n", context->m_url);
    CURLMcode ret = curl_multi_add_handle(s_workerMultiHandle, context->m_handle);
    if (ret && ret != CURLM_CALL_MULTI_PERFORM && !context->m_isFinishedContext)
        postHandleCancel(context);
}

// This function breaks select() in asynchronous operation.
static void wakeupSelect(int fd)
{
    ssize_t size = 0;
labelWritePipe:
    size = write(fd, "W", 1);
    if (size == -1) {
        int errsv = errno;
        switch (errsv) {
        case EAGAIN:
            goto labelWritePipe;
        case EPIPE:
            s_exit = true;
            break;
        }
    }
}
#endif

void ResourceHandleManager::handleBadCertificate()
{

    bool continueProcess = true;

    do {

        ResourceHandleContext* context = 0;

        {
            WTF::MutexLocker lock(*s_userThreadMutex);

            if (!m_sslBadcertResourceHandleContextList.isEmpty()) {
                context = m_sslBadcertResourceHandleContextList[0];
                m_sslBadcertResourceHandleContextList.remove(0);
            } else
                continueProcess = false;
        }
        if (context && continueProcess) {

            DEBUG_LOG("### handleBadCertificate() url = [%s]\n", context->m_url);

            if (!s_userAcceptedHttpsDomains.contains(StringHash::hash(context->m_kurl.host()))) {
                if (context->m_isMainResource) {
                    DEBUG_LOG("### handleBadCertificate() Main Resource\n");

                    if (context->m_nwcontext) {
                        if (context->m_chainForBadCertNotify) {

#if OS(PSP2)
                            ResourceError error(context->m_sslError, String());
                            error.setCert(context->m_chainForBadCertNotify);
                            context->m_nwcontext->onCertificateVerificationRequest(error);
                            context->m_confirm = error.confirmCert();
#else
                            postNotifyBadCertSynchronously(context);
#endif
                        }
                    }
                }
            } else
                context->m_confirm = true;

            if (context->m_confirm) {
                s_userAcceptedHttpsDomains.add(StringHash::hash(context->m_kurl.host()));
                if (context->m_url) {
                    fastFree(context->m_url); 
                    context->m_url = 0;
                }
                context->m_curlResult = CURLE_OK;

#if OS(PSP2)
                resetSslConn(context);
#endif
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
                WTF::MutexLocker workerMutexlock(*s_workerThreadMultiMutex);
                m_sslProcessedBadCertRHCList.append(reinterpret_cast<ResourceHandleContext*>(context));
                wakeupSelect(s_WPIPE[3]);
#else
                WTF::MutexLocker workerMutexlock(*s_workerThreadMultiMutex);
#ifndef DISABLE_RUNNINGJOBS_COUNTING
                ResourceHandleManager::sharedInstance()->m_runningJobs++;
#endif
                CURLMcode ret = curl_multi_add_handle(s_workerMultiHandle, context->m_handle);
                if (ret && ret != CURLM_CALL_MULTI_PERFORM)
                    postHandleCancel(context);

                callOnMainThread(postWorkerThreadPumpMessage, 0);
#endif
            } else
                postHandleCancel(context);
        }
    } while (continueProcess);

}

static void notifyAuthenticationChallenge(ResourceHandleContext* context, long AuthAvail, WebCore::AuthenticationChallenge& challenge, WebCore::ProtectionSpace& space, WebCore::ResourceHandleInternal* d, WebCore::ResourceHandle* job)
{
    ASSERT(d && job);

    if (AuthAvail == CURLAUTH_BASIC || AuthAvail == CURLAUTH_DIGEST) {

        if (context->m_authModeFromHeader & Manx::AuthModeHttp) {
            context->m_httpSpace = WebCore::ProtectionSpace(
                context->m_kurl.host(), 
                context->m_kurl.port(), 
                context->m_httpProtectType,
                context->m_httpRealm, 
                (AuthAvail == CURLAUTH_BASIC) ? ProtectionSpaceAuthenticationSchemeHTTPBasic : ProtectionSpaceAuthenticationSchemeHTTPDigest);
        }
        if (context->m_authModeFromHeader & Manx::AuthModeProxy) {
            context->m_proxySpace = WebCore::ProtectionSpace(
                context->m_proxyurl.host(),
                context->m_proxyurl.port(),
                context->m_proxyProtectType,
                context->m_proxyRealm, 
                (AuthAvail == CURLAUTH_BASIC) ? ProtectionSpaceAuthenticationSchemeHTTPBasic : ProtectionSpaceAuthenticationSchemeHTTPDigest);
        }

        // query application username and password
        WebCore::Credential cred;
        WebCore::ResourceResponse resp;
        WebCore::ResourceError err;
        challenge = WebCore::AuthenticationChallenge(space, cred, 0, resp, err);
        if (context->m_nwcontext)
            context->m_nwcontext->onAuthenticationRequired(challenge);

    }
}

static void getCredentialFromUserOrCache(
    ResourceHandleContext* context,
    WebCore::ResourceHandleInternal* d,
    WebCore::ResourceHandle* job,
    long httpAvailAuth, // Basic or Digest
    String host, 
    int port, 
    ProtectionSpaceServerType type, 
    String realm, 
    ProtectionSpace& space,
    AuthenticationChallenge& challenge,
    bool& hasCredential,
    CURLoption opt)
{
    DEBUG_LOG("->getCredentialFromUserOrCache() url = [%s]\n", context->m_url);

    String userpass("");

    space = ProtectionSpace(
        host, 
        port,
        type,
        realm,
        (httpAvailAuth == CURLAUTH_BASIC)?ProtectionSpaceAuthenticationSchemeHTTPBasic:ProtectionSpaceAuthenticationSchemeHTTPDigest);

    WebCore::Credential savedCredential = CredentialBackingStore::instance()->getLogin(space);

    if (!savedCredential.isEmpty()) {
        userpass.append(savedCredential.user());
        userpass.append(":");
        userpass.append(savedCredential.password());
        curl_easy_setopt(context->m_handle, opt == CURLOPT_USERPWD?CURLOPT_HTTPAUTH:CURLOPT_PROXYAUTH, httpAvailAuth);
        curl_easy_setopt(context->m_handle, opt, userpass.latin1().data());
        hasCredential = true;
        challenge.m_user = savedCredential.user();
        challenge.m_pass = savedCredential.password();
        challenge.activate();
        context->m_proxyAuthMaybeFailed = false;
    } else {
        if (Manx::NetworkProfile::proxyEnabled() && !context->m_preSetProxyUserPass.isEmpty()) {
            userpass.append(context->m_preSetProxyUserPass);
            challenge.m_ok = true;
        } else {
            notifyAuthenticationChallenge(context, httpAvailAuth, challenge, space, d, job);

            userpass.append(challenge.m_user);
            userpass.append(":");
            userpass.append(challenge.m_pass);
        }

        if (challenge.m_ok) {
            curl_easy_setopt(context->m_handle, opt == CURLOPT_USERPWD?CURLOPT_HTTPAUTH:CURLOPT_PROXYAUTH, httpAvailAuth);
            curl_easy_setopt(context->m_handle, opt, userpass.latin1().data());
            hasCredential = true;
            context->m_proxyAuthMaybeFailed = false;
        } else
            context->m_authCancelled = true;
    }

    // reset realm status
    // maintain userpass cache
    if (!context->m_authCancelled) {
        if (opt == CURLOPT_USERPWD) {
            context->m_hasHttpRealm = false;
            context->m_httpUserPass = userpass;
            context->m_authModeInProcess |= Manx::AuthModeHttp;
        } else if (opt == CURLOPT_PROXYUSERPWD) {
            context->m_hasProxyRealm = false;
            context->m_proxyUserPass = userpass;
            context->m_authModeInProcess |= Manx::AuthModeProxy;
        }
    }
}

static void setAuthType(ResourceHandleContext* context)
{
    if (context->m_authModeFromHeader & Manx::AuthModeHttp)
        curl_easy_getinfo(context->m_handle, CURLINFO_HTTPAUTH_AVAIL, &context->m_httpAuthType);
    if (context->m_authModeFromHeader & Manx::AuthModeProxy)
        curl_easy_getinfo(context->m_handle, CURLINFO_PROXYAUTH_AVAIL, &context->m_proxyAuthType);
    DEBUG_LOG("handleAuthentication() httpAuthAvail = [%ld] proxyAuthAvail = [%ld]\n", context->m_httpAuthType, context->m_proxyAuthType);
}

static void handleAuthentication(void* data)
{

    WTF::MutexLocker lock(*s_authProcessMutex);

    WebCore::ResourceHandleContext* context = reinterpret_cast<WebCore::ResourceHandleContext*>(data);
    WebCore::ResourceHandle* job = reinterpret_cast<WebCore::ResourceHandle*>(context->job);
    ASSERT(job);
    WebCore::ResourceHandleInternal* d = context->job->getInternal();
    ASSERT(d);

    DEBUG_LOG("### handleAuthentication() url = [%s]\n", context->m_url);

    // discard old curl handle.
    {
#ifndef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
        // curl_multi_remove_handle will be moved to RHM::workerThread.
        // RHM::workerThread calls curl_multi_remove_handle() then handleAuthentication() is called.
        WTF::MutexLocker workerMutexlock(*s_workerThreadMultiMutex);
        curl_multi_remove_handle(s_workerMultiHandle, context->m_handle);
#endif
#ifdef ENABLE_PERSISTENT_CONNECTION_FOR_SYNCHRONOUS_COMMUNICATION
        if (context->m_handle == curlEasyHandleForSync()) {
            curl_easy_reset(context->m_handle);
            context->m_handle = 0;
            if (context->m_url) {
                fastFree(context->m_url); 
                context->m_url = 0;
            }
        }
#endif
    }
    // create new curl handle.
    if (context->m_syncReq)
        ResourceHandleManager::sharedInstance()->initializeHandleForSync(context);

    if (context->m_isHttpsScheme && s_userAcceptedHttpsDomains.contains(StringHash::hash(context->m_kurl.host()))) {
        curl_easy_setopt(context->m_handle, CURLOPT_SSL_VERIFYHOST, false);
        curl_easy_setopt(context->m_handle, CURLOPT_SSL_VERIFYPEER, false);
    }

    if (context->m_authModeFromHeader & Manx::AuthModeHttp)
        getCredentialFromUserOrCache(context, d, job, context->m_httpAuthType, context->m_kurl.host(), context->m_kurl.port(), context->m_httpProtectType, context->m_httpRealm, context->m_httpSpace, context->m_httpChallenge, context->m_hasHttpCredential, CURLOPT_USERPWD); 

    if (context->m_authModeFromHeader & Manx::AuthModeProxy)
        getCredentialFromUserOrCache(context, d, job, context->m_proxyAuthType, context->m_proxyurl.host(), context->m_proxyurl.port(), context->m_proxyProtectType, context->m_proxyRealm, context->m_proxySpace, context->m_proxyChallenge, context->m_hasProxyCredential, CURLOPT_PROXYUSERPWD);

    if (context->m_syncReq) {
        CURLcode err =  curl_easy_perform(context->m_handle);
        if (err) {
            ResourceError error(String(d->m_url), err, String(d->m_url), String(curl_easy_strerror(err)));
            job->client()->didFail(job, error);
        } else
            d->client()->didFinishLoading(job, 0);

#ifdef ENABLE_PERSISTENT_CONNECTION_FOR_SYNCHRONOUS_COMMUNICATION
        curl_easy_reset(context->m_handle);
#else
        curl_easy_cleanup(context->m_handle);
#endif
        fastFree(context->m_url); 
        context->m_url = 0;
    } else {
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
        WTF::MutexLocker workerMutexlock(*s_workerThreadMultiMutex);
        m_authProcessedContextList.append(reinterpret_cast<ResourceHandleContext*>(context));
        wakeupSelect(s_WPIPE[2]);
#else
        WTF::MutexLocker workerMutexlock(*s_workerThreadMultiMutex);
        CURLMcode ret = curl_multi_add_handle(s_workerMultiHandle, context->m_handle);
        if (ret && ret != CURLM_CALL_MULTI_PERFORM)
            postHandleCancel(context);

        callOnMainThread(postWorkerThreadPumpMessage, 0);
#endif
    }
}

static bool extractRealmFromAuthHeader(const ProtectionSpaceServerType space, const String& header, long& httpAuthType, String& realm)
{
    if (header.isEmpty())
        return false;

    // Extract the auth scheme and realm from the header.
    const String strBasic("Basic");
    const String strDigest("Digest");

    int authTypeLen = 0;
    size_t authTypePos = header.find(strBasic);
    authTypeLen = strBasic.length();
    httpAuthType = CURLAUTH_BASIC;
    if (authTypePos == notFound) {
        authTypePos = header.find(strDigest);
        authTypeLen = strDigest.length();
        httpAuthType = CURLAUTH_DIGEST;
        if (authTypePos == notFound)
            return false;
    }

    size_t spacePos = header.find(' ');
    if (spacePos == notFound) {
        DEBUG_LOG("%s-Authenticate field '%s' badly formatted: missing scheme.", space == ProtectionSpaceServerHTTP ? "WWW" : "Proxy", header.utf8().data());
        return false;
    }
    spacePos = header.find(' ', spacePos + 1); // find 2nd space which is located after auth type.
    if (spacePos == notFound) {
        DEBUG_LOG("AuthType '%s' badly formatted: missing scheme.", authTypeLen == 5 ? "Basic" : "Digest");
        return false;
    }

    size_t realmPos = header.findIgnoringCase("realm=", spacePos);
    if (realmPos == notFound) {
        DEBUG_LOG("%s-Authenticate field '%s' badly formatted: missing realm.", space == ProtectionSpaceServerHTTP ? "WWW" : "Proxy", header.utf8().data());
        return false;
    }
    size_t beginPos = realmPos + 6;
    realm  = header.right(header.length() - beginPos);
    if (realm.startsWith("\"")) {
        beginPos += 1;
        size_t endPos = header.find("\"", beginPos);
        if (endPos == notFound) {
            DEBUG_LOG("%s-Authenticate field '%s' badly formatted: invalid realm.", space == ProtectionSpaceServerHTTP ? "WWW" : "Proxy", header.utf8().data());
            return false;
        }
        realm = header.substring(beginPos, endPos - beginPos);
    }

    return true;
}

void ResourceHandleManager::createAuthenticateThread(void* data)
{

    if (!s_userThreadCreated) {
        s_userThreadMutex = new WTF::Mutex();
        ASSERT(s_userThreadMutex);

        s_userThreadId = createUserDefinedThread(&userCallbackThread, data, "userCallbackThread", s_userThreadConditionMutex, s_userThreadCondition, s_userThreadCreatedConditionMutex, s_userThreadCreatedCondition);
    }
}

void ResourceHandleManager::handleSslBadCertificate(void* context)
{

    if (!s_userThreadCreated) {
        s_userThreadMutex = new WTF::Mutex();
        ASSERT(s_userThreadMutex);

        s_userThreadId = createUserDefinedThread(&userCallbackThread, 0, "userCallbackThread", s_userThreadConditionMutex, s_userThreadCondition, s_userThreadCreatedConditionMutex, s_userThreadCreatedCondition);
    }

    WTF::MutexLocker userMutexlock(*s_userThreadMutex);
    m_sslBadcertResourceHandleContextList.append(reinterpret_cast<ResourceHandleContext*>(context));

    s_userThreadCondition->signal();

}

void ResourceHandleManager::userCallbackThread(void* thrcontext)
{
    UserDefinedThreadContext* userCtxt = reinterpret_cast<UserDefinedThreadContext*>(thrcontext);
    bool toContinue = false;
    while (!s_exit) {
        if (!toContinue)    
            threadWaitFunction(userCtxt, s_userThreadCreated);

        toContinue = false;
        DEBUG_LOG("<- s_userCallbackThreadSemaphore->wait() time = [%f]\n", currentTime());

        if (!m_authContextList.isEmpty()) {
            WTF::MutexLocker lock(*s_authContextListMutex);
            ResourceHandleContext* context = m_authContextList[0];
            m_authContextList.remove(0);

            if (context)
                callOnMainThread(handleAuthentication, context);
            if (m_authContextList.size() > 0)
                toContinue = true;
        }

        handleBadCertificate();
    }
}
///////////////////////////////////////////////////////////////////////////////////////////

const int maxRunningJobs = 20;
const int maxConnections = 32;

static Mutex* sharedResourceMutex(curl_lock_data data)
{
    DEFINE_STATIC_LOCAL(Mutex, cookieMutex, ());
    DEFINE_STATIC_LOCAL(Mutex, dnsMutex, ());
    DEFINE_STATIC_LOCAL(Mutex, shareMutex, ());

    switch (data) {
    case CURL_LOCK_DATA_COOKIE:
        return &cookieMutex;
    case CURL_LOCK_DATA_DNS:
        return &dnsMutex;
    case CURL_LOCK_DATA_SHARE:
        return &shareMutex;
    default:
        ASSERT_NOT_REACHED();
        return 0;
    }
}

static void lockCurlSharedResource(curl_lock_data data)
{
    if (Mutex* mutex = sharedResourceMutex(data))
        mutex->lock();
}

static void unlockCurlSharedResource(curl_lock_data data)
{
    if (Mutex* mutex = sharedResourceMutex(data))
        mutex->unlock();
}

// libcurl does not implement its own thread synchronization primitives.
// these two functions provide mutexes for cookies, and for the global DNS
// cache.
static void curlLockCallback(CURL* handle, curl_lock_data data, curl_lock_access access, void* userPtr)
{
    lockCurlSharedResource(data);
}

static void curlUnlockCallback(CURL* handle, curl_lock_data data, void* userPtr)
{
    unlockCurlSharedResource(data);
}

void ResourceHandleManager::lockCurlSharedResource(curl_lock_data data)
{
    WebCore::lockCurlSharedResource(data);
}

void ResourceHandleManager::unlockCurlSharedResource(curl_lock_data data)
{
    WebCore::unlockCurlSharedResource(data);
}

static void networkStateChanged(bool isOnLine)
{
    networkStateNotifier().setOnLine(isOnLine);
}

ResourceHandleManager::ResourceHandleManager()
    : m_cookieJarFileName(0)
#ifndef DISABLE_RUNNINGJOBS_COUNTING
    , m_runningJobs(0)
    , m_runningFilteringJobs(0)
#endif
{
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
    // initialize pipe.
    for (int i = 0; i < 4; i++) {
        int pipeFds[2];
        if (!pipe(pipeFds)) {
            fcntl(pipeFds[0], F_SETFD, FD_CLOEXEC);
            fcntl(pipeFds[1], F_SETFD, FD_CLOEXEC);
            s_RPIPE[i] = pipeFds[0];
            if (s_maxPipeFd < s_RPIPE[i])
                s_maxPipeFd = s_RPIPE[i];
            s_WPIPE[i] = pipeFds[1];
            if (s_maxPipeFd < s_WPIPE[i])
                s_maxPipeFd = s_WPIPE[i];
            int flags = fcntl(s_RPIPE[i], F_GETFL, 0);
            fcntl(s_RPIPE[i], F_SETFL, flags | O_NONBLOCK);
            flags = fcntl(s_WPIPE[i], F_GETFL, 0);
            fcntl(s_WPIPE[i], F_SETFL, flags | O_NONBLOCK);
        } else
            ASSERT(false);
    }
#endif

    Manx::NetworkProfile::initialize();

    Manx::Ssl::initialize();
    Manx::Ssl::SslContext::postNotifyOriginFrame = postNotifyOriginFrame;

    networkStateNotifier().setOnLine(Manx::NetworkProfile::isOnLine());
    Manx::NetworkProfile::setOnLineStateCallback(networkStateChanged);

    s_authContextListMutex = new WTF::Mutex();
    s_authProcessMutex = new WTF::Mutex();

#if 1
    curl_global_init(CURL_GLOBAL_ALL);
#else
    curl_global_init_mem(CURL_GLOBAL_ALL, Manx::Network::malloc, Manx::Network::free, Manx::Network::realloc, Manx::Network::strdup, Manx::Network::calloc);
#endif
    m_curlShareHandle = curl_share_init();
    curl_share_setopt(m_curlShareHandle, CURLSHOPT_SHARE, CURL_LOCK_DATA_COOKIE);
    curl_share_setopt(m_curlShareHandle, CURLSHOPT_SHARE, CURL_LOCK_DATA_DNS);
    curl_share_setopt(m_curlShareHandle, CURLSHOPT_LOCKFUNC, curlLockCallback);
    curl_share_setopt(m_curlShareHandle, CURLSHOPT_UNLOCKFUNC, curlUnlockCallback);
    cleanupSessionCookies();
}

ResourceHandleManager::~ResourceHandleManager()
{
    curl_share_cleanup(m_curlShareHandle);
    if (m_cookieJarFileName)
        fastFree(m_cookieJarFileName);
    curl_global_cleanup();

    Manx::Ssl::finalize();
    Manx::NetworkProfile::finalize();

#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
    // cleanup pipe for monitoring select().
    for (int i = 0; i < 4; i++) {
        close(s_RPIPE[i]);
        close(s_WPIPE[i]);
    }
#endif
}

void ResourceHandleManager::setCookieJarFileName(const char* cookiejarFileName)
{
    m_cookieJarFileName = fastStrDup(cookiejarFileName);
}

ResourceHandleManager* ResourceHandleManager::s_sharedInstance = 0;
void ResourceHandleManager::cleanup()
{
    s_exit = true;

    if (s_workerThreadCondition) {
        s_workerThreadCondition->signal();
        WTF::waitForThreadCompletion(s_workerThreadId);
    }

    if (s_workerThreadMutex) { 
        delete s_workerThreadMutex; 
        s_workerThreadMutex = 0;
    }
    if (s_workerThreadConditionMutex) {
        delete s_workerThreadConditionMutex; 
        s_workerThreadConditionMutex = 0;
    }
    if (s_workerThreadCreatedConditionMutex) {
        delete s_workerThreadCreatedConditionMutex; 
        s_workerThreadCreatedConditionMutex = 0;
    }
    if (s_workerThreadCondition) {
        delete s_workerThreadCondition; 
        s_workerThreadCondition= 0;
    }
    if (s_workerThreadCreatedCondition) {
        delete s_workerThreadCreatedCondition; 
        s_workerThreadCreatedCondition = 0;
    }

    if (s_userThreadCondition) {
        s_userThreadCondition->signal();
        WTF::waitForThreadCompletion(s_userThreadId);
    }

    if (s_userThreadMutex) { 
        delete s_userThreadMutex; 
        s_userThreadMutex = 0;
    }
    if (s_userThreadConditionMutex) {
        delete s_userThreadConditionMutex; 
        s_userThreadConditionMutex = 0;
    }
    if (s_userThreadCreatedConditionMutex) {
        delete s_userThreadCreatedConditionMutex; 
        s_userThreadCreatedConditionMutex = 0;
    }
    if (s_userThreadCondition) {
        delete s_userThreadCondition; 
        s_userThreadCondition= 0;
    }
    if (s_userThreadCreatedCondition) {
        delete s_userThreadCreatedCondition; 
        s_userThreadCreatedCondition = 0;
    }

    if (s_filteringThreadId) {
        s_filteringThreadCondition->signal();
        WTF::waitForThreadCompletion(s_filteringThreadId);
    }

    if (s_filteringThreadMutex) { 
        delete s_filteringThreadMutex; 
        s_filteringThreadMutex = 0;
    }
    if (s_filteringThreadConditionMutex) {
        delete s_filteringThreadConditionMutex; 
        s_filteringThreadConditionMutex = 0;
    }
    if (s_filteringThreadCreatedConditionMutex) {
        delete s_filteringThreadCreatedConditionMutex; 
        s_filteringThreadCreatedConditionMutex = 0;
    }
    if (s_filteringThreadCondition) {
        delete s_filteringThreadCondition; 
        s_filteringThreadCondition= 0;
    }
    if (s_filteringThreadCreatedCondition) {
        delete s_filteringThreadCreatedCondition; 
        s_filteringThreadCreatedCondition = 0;
    }

    if (s_authContextListMutex) {
        delete s_authContextListMutex;
        s_authContextListMutex = 0;
    }
    if (s_authProcessMutex) {
        delete s_authProcessMutex;
        s_authProcessMutex = 0;
    }

    if (s_workerThreadMultiMutex) { 
        delete s_workerThreadMultiMutex; 
        s_workerThreadMultiMutex = 0;
    }

    if (s_sharedInstance) {
        delete s_sharedInstance;
        s_sharedInstance = 0;
    }
}

void ResourceHandleManager::selectSockets(CURLM* multi, long timeoutMilliSecond)
{
    ASSERT(multi);

    fd_set fdread;
    fd_set fdwrite;
    fd_set fdexcep;
    int maxfd = 0;

    struct timeval timeout;
    timeout.tv_sec = timeoutMilliSecond / 1000;
    timeout.tv_usec = (timeoutMilliSecond % 1000) * 1000; // select waits microseconds

    int rc = 0;
    do {
        FD_ZERO(&fdread);
        FD_ZERO(&fdwrite);
        FD_ZERO(&fdexcep);
        curl_multi_fdset(multi, &fdread, &fdwrite, &fdexcep, &maxfd);
        DEBUG_LOG("<- curl_multi_fdset() maxfd = [%d] time = [%f]\n", maxfd, currentTime());
#ifndef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
        if (maxfd >= 0) {
#endif
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
            for (int i = 0; i < 4; i++)
                FD_SET(s_RPIPE[i], &fdread); // cancel,add,auth,badcert
            if (maxfd < s_maxPipeFd)
                maxfd = s_maxPipeFd;
#endif

            if (timeoutMilliSecond > 0)
                rc = ::select(maxfd + 1, &fdread, &fdwrite, &fdexcep, &timeout);
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
            else
                rc = ::select(maxfd + 1, &fdread, 0, 0, 0);
            if (rc > 0) {
                char buff[RHM_SELECT_PIPE_BUF];
                memset(buff, 0x0, RHM_SELECT_PIPE_BUF);
                ssize_t size = 0;
                int savedErrno = 0;
                for (int i = 0; i < 4; i++) {
                    if (FD_ISSET(s_RPIPE[i], &fdread)) {
labelRetryRead:
                        size = read(s_RPIPE[i], buff, RHM_SELECT_PIPE_BUF);
                        if (size > 0) {
                            if (!i) { // cancelled
                                DEBUG_LOG("<- cancelled [%s]\n", buff);
                                WTF::MutexLocker workerMultiMutexLock(*s_workerThreadMultiMutex);
                                // check cancel
                                int size = s_cancelContextList.size();
                                for (int i = 0; i < size; i++) {
                                    ResourceHandleContext* context = s_cancelContextList[0];
                                    s_cancelContextList.remove(0);
                                    context->m_cancelled = true;
                                }
                            } else if (i == 2) { // handle authentication
                                WTF::MutexLocker workerMutexlock(*s_workerThreadMultiMutex);
                                if (!m_authProcessedContextList.isEmpty()) {
                                    int size = m_authProcessedContextList.size();
                                    for (int i = 0; i < size; i++) {
                                        ResourceHandleContext* context = m_authProcessedContextList[0];
                                        m_authProcessedContextList.remove(0);
                                        restart(context);
                                    }
                                }
                            } else if (i == 3) { // handle bad certificate
                                WTF::MutexLocker workerMutexlock(*s_workerThreadMultiMutex);
                                if (!m_sslProcessedBadCertRHCList.isEmpty()) {
                                    ResourceHandleContext* context = m_sslProcessedBadCertRHCList[0];
                                    m_sslProcessedBadCertRHCList.remove(0);
#ifndef DISABLE_RUNNINGJOBS_COUNTING
                                    ResourceHandleManager::sharedInstance()->m_runningJobs++;
#endif
                                    resetSslConn(context);
                                    restart(context);
                                }
                            }
                        } else if (!size) {
                            // pipe is closed.
                            // consider it is a shutting down.
                            DEBUG_LOG("read size == 0 pipe is broken?\n");
                            s_exit = true;
                            break;
                        } else if (size == -1) {
                            savedErrno = errno;
                            if (savedErrno == EAGAIN || savedErrno == EWOULDBLOCK) {
                                DEBUG_LOG("<- pipe EAGAIN\n");
                                goto labelRetryRead;
                            }
                        }
                    }
                }
            }
#endif
            DEBUG_LOG("<- ::select() time = [%f]\n", currentTime());
#ifndef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
        }
#endif
    } while (rc == -1 && errno == EINTR);
}

#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
void ResourceHandleManager::selectSocketsForFiltering(CURLM* multi, long timeoutMilliSecond)
{
    ASSERT(multi);

    fd_set fdread;
    fd_set fdwrite;
    fd_set fdexcep;
    int maxfd = 0;

    struct timeval timeout;
    timeout.tv_sec = timeoutMilliSecond / 1000;
    timeout.tv_usec = (timeoutMilliSecond % 1000) * 1000; // select waits microseconds

    int rc = 0;
    do {
        FD_ZERO(&fdread);
        FD_ZERO(&fdwrite);
        FD_ZERO(&fdexcep);
        curl_multi_fdset(multi, &fdread, &fdwrite, &fdexcep, &maxfd);
        DEBUG_LOG("<- curl_multi_fdset() maxfd = [%d] time = [%f]\n", maxfd, currentTime());
        if (maxfd >= 0) {
            if (timeoutMilliSecond > 0)
                rc = ::select(maxfd + 1, &fdread, &fdwrite, &fdexcep, &timeout);
            DEBUG_LOG("<- ::select() time = [%f]\n", currentTime());
        }
    } while (rc == -1 && errno == EINTR);

}
#endif

void ResourceHandleManager::filteringThread(void* context)
{
    DEBUG_LOG("-> ResourceHandleManager::filteringThread()\n");
    s_filteringMultiHandle = curl_multi_init();

    curl_multi_setopt(s_filteringMultiHandle, CURLMOPT_MAXCONNECTS, maxConnections);

    bool continueProcess = false;

    while (!s_exit) {
        if (!continueProcess) {
            DEBUG_LOG("-> s_filteringThreadCondition->wait() time = [%f]\n", currentTime());
            threadWaitFunction(reinterpret_cast<UserDefinedThreadContext*>(context), s_filteringThreadCreated);
            DEBUG_LOG("<- s_filteringThreadCondition->wait() time = [%f]\n", currentTime());
        }

        continueProcess = false;

        continueProcess = ResourceHandleManager::s_sharedInstance->processFiltering();

        long timeout = 0;
        CURLMcode ret = curl_multi_timeout(s_filteringMultiHandle, &timeout);
        if (ret)
            ASSERT(false);

        DEBUG_LOG("curl_multi_timeout timeout time value = %ld\n", timeout);
        
        if (!timeout) {
            continueProcess = true;
            continue;
        }

#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
        selectSocketsForFiltering(s_filteringMultiHandle, timeout);
#else
        selectSockets(s_filteringMultiHandle, timeout);
#endif
    }

    curl_multi_cleanup(s_filteringMultiHandle);

    return;
}

void ResourceHandleManager::workerThread(void* argument)
{
#ifndef NDEBUG
    workerThreadIdentifier = WTF::currentThread();
#endif
    UserDefinedThreadContext* context = reinterpret_cast<UserDefinedThreadContext*>(argument);

    DEBUG_LOG("-> ResourceHandleManager::workerThread()\n");
    ASSERT(context);

    s_workerMultiHandle = curl_multi_init();

    curl_multi_setopt(s_workerMultiHandle, CURLMOPT_MAXCONNECTS, maxConnections);

#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
    threadWaitFunction(context, s_workerThreadCreated);
#endif
    bool continueProcess = false;

    while (!s_exit) {
#ifndef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
        if (!continueProcess)
            threadWaitFunction(context, s_workerThreadCreated);

        continueProcess = false;
#endif

        DEBUG_LOG("-> s_sharedInstance->processDownload() time = [%f]\n", currentTime());
        continueProcess = ResourceHandleManager::s_sharedInstance->processDownload();
        DEBUG_LOG("<- s_sharedInstance->processDownload() time = [%f]\n", currentTime());

        long timeout = 0;
        CURLMcode ret = curl_multi_timeout(s_workerMultiHandle, &timeout);
        if (ret)
            ASSERT(false);

        DEBUG_LOG("curl_multi_timeout timeout time value = %ld\n", timeout);

        if (!timeout) {
            continueProcess = true;
            continue;
        }

        selectSockets(s_workerMultiHandle, timeout);
    }

    if (context)
        delete context;

    curl_multi_cleanup(s_workerMultiHandle);

    return;

}

void ResourceHandleManager::threadWaitFunction(UserDefinedThreadContext* threadctx, bool& threadCreated)
{
    ASSERT(threadctx && threadctx->m_threadCondMutex && threadctx->m_threadCreatedCondMutex && threadctx->m_threadCond && threadctx->m_threadCreatedCond);

    WTF::MutexLocker lock(*threadctx->m_threadCondMutex);
    if (!threadCreated) {
        WTF::MutexLocker lock(*threadctx->m_threadCreatedCondMutex);
        threadctx->m_threadCreatedCond->signal();
        threadCreated = true;
    }
    threadctx->m_threadCond->wait(*threadctx->m_threadCondMutex);
}

WTF::ThreadIdentifier ResourceHandleManager::createUserDefinedThread(WTF::ThreadFunction threadfunc, void* ctxt, const char* threadName, WTF::Mutex*& threadCondMutex, WTF::ThreadCondition*& thrCond, WTF::Mutex*& threadCreatedCondMutex, WTF::ThreadCondition*& threadCreatedCond)
{
    threadCondMutex = new WTF::Mutex();
#if USE(MANX_COND_INIT)
    thrCond = new WTF::ThreadCondition(*threadCondMutex);
#else
    thrCond = new WTF::ThreadCondition();
#endif

    threadCreatedCondMutex = new WTF::Mutex();
#if USE(MANX_COND_INIT)
    threadCreatedCond = new WTF::ThreadCondition(*threadCreatedCondMutex);
#else
    threadCreatedCond = new WTF::ThreadCondition();
#endif
    ASSERT(threadCondMutex && thrCond && threadCreatedCondMutex && threadCreatedCond);

    UserDefinedThreadContext* threadcon = new UserDefinedThreadContext(threadCondMutex, threadCreatedCondMutex, thrCond, threadCreatedCond);

    threadCreatedCondMutex->lock();
    WTF::ThreadIdentifier threadId = createThread(threadfunc, reinterpret_cast<void*>(threadcon), threadName);
    threadCreatedCond->wait(*threadCreatedCondMutex);
    threadCreatedCondMutex->unlock();

    return threadId;
}

void ResourceHandleManager::checkAndCreateFilteringThread()
{
    if (!s_filteringThreadId) {

        s_filteringThreadMutex = new WTF::Mutex();
        ASSERT(s_filteringThreadMutex);

        s_filteringThreadId = createUserDefinedThread(&filteringThread, 0, "RscHdlMan:Filtering", s_filteringThreadConditionMutex, s_filteringThreadCondition, s_filteringThreadCreatedConditionMutex, s_filteringThreadCreatedCondition);
    }
}

ResourceHandleManager* ResourceHandleManager::sharedInstance()
{
    if (!s_sharedInstance) {
        s_sharedInstance = new ResourceHandleManager();

        if (s_sharedInstance) {

#if OS(ORBIS)
            s_notifyBadCertConditionMutex = new WTF::Mutex();
            ASSERT(s_notifyBadCertConditionMutex);
#if USE(MANX_COND_INIT)
            s_notifyBadCertCondition = new WTF::ThreadCondition(*s_notifyBadCertConditionMutex);
#else
            s_notifyBadCertCondition = new WTF::ThreadCondition();
#endif
            ASSERT(s_notifyBadCertCondition);
#endif

            s_workerThreadMutex = new WTF::Mutex();
            ASSERT(s_workerThreadMutex);
            s_workerThreadMultiMutex = new WTF::Mutex();
            ASSERT(s_workerThreadMultiMutex);

            s_workerThreadId = createUserDefinedThread(&workerThread, 0, "RscHdlMan:Worker", s_workerThreadConditionMutex, s_workerThreadCondition, s_workerThreadCreatedConditionMutex, s_workerThreadCreatedCondition);

#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
            s_workerThreadConditionMutex->lock();
            s_workerThreadCondition->signal();
            DEBUG_LOG("<- s_workerThreadCondition->signal() time = [%f]\n", currentTime());
            s_workerThreadConditionMutex->unlock();
#endif

            if (Manx::WebSecurity::activatedStatus())
                s_sharedInstance->checkAndCreateFilteringThread();

            if (!s_workerThreadId) {
                DEBUG_LOG("ResourceHandleManager::sharedInstance: createThread() failed!\n");
                delete s_sharedInstance;
                s_sharedInstance = 0;
            }

        }

    }
    return s_sharedInstance;
}


/*
  Functions which post to main thread

  - postDidRecvData
  - postDidRecvResponse
  - postWillSendRequest
  - postHandleCancel
  - postDidFinish
  - postDidFail
  - postAddResponseHeader
  - postHandleResponseHeaders
  - postNotifyOriginFrame

 */
void postDidRecvData(ResourceHandleContext* context, void* ptr, const char* hdr, size_t totalSize)
{
    ResourceHandleCommand* cmd = new ResourceHandleCommand();
    if (cmd) {
        cmd->m_data[0].typeVoidPtr = context->job;
        cmd->m_data[1].typeCharPtr = new char[totalSize];
        if (cmd->m_data[1].typeCharPtr)
            memcpy(cmd->m_data[1].typeCharPtr, ptr, totalSize);
        else
            ASSERT(false);
        cmd->m_data[2].typeInt = totalSize;

        size_t hdrlen = strlen(hdr);
        cmd->m_data[3].typeCharPtr = new char[hdrlen + 1];
        if (cmd->m_data[3].typeCharPtr) {
            strncpy(cmd->m_data[3].typeCharPtr, hdr, hdrlen);
            cmd->m_data[3].typeCharPtr[hdrlen] = '\0';
        } else
            ASSERT(false);
        callOnMainThread(didRecvData, cmd);
    }
}

void postDidRecvResponse(ResourceHandleContext* context)
{
    CURL* h = context->m_handle;
    CURLcode err;

    const char* hdr = 0;
    err = curl_easy_getinfo(h, CURLINFO_EFFECTIVE_URL, &hdr);

    if (err == CURLE_OK) {
        ResourceHandleCommand* cmd = new ResourceHandleCommand();
        if (cmd) {
            cmd->m_data[0].typeVoidPtr = static_cast<ResourceHandle*>(context->job);
            if (hdr) {
                size_t hdrlen = strlen(hdr);
                cmd->m_data[1].typeCharPtr = new char[hdrlen + 1];
                if (cmd->m_data[1].typeCharPtr) {
                    strncpy(cmd->m_data[1].typeCharPtr, hdr, hdrlen);
                    cmd->m_data[1].typeCharPtr[hdrlen] = '\0';
                } else
                    ASSERT(false);
            }

            if (!context->m_evaluatedkurl.isEmpty()) {
                size_t evaluatedurllen = strlen(context->m_evaluatedkurl.string().latin1().data());
                cmd->m_data[2].typeCharPtr = new char[evaluatedurllen + 1];
                if (cmd->m_data[2].typeCharPtr) {
                    strncpy(cmd->m_data[2].typeCharPtr, context->m_evaluatedkurl.string().latin1().data(), evaluatedurllen);
                    cmd->m_data[2].typeCharPtr[evaluatedurllen] = '\0';
                }
            }
            callOnMainThread(didRecvResponse, cmd);
            callOnMainThread(replaceUrlOfMainResource, context);
        }
    } else
        ASSERT(false);
}

void postWillSendRequest(ResourceHandleContext* context)
{
    ResourceHandleCommand* cmd = new ResourceHandleCommand();
    if (cmd) {
        cmd->m_data[0].typeVoidPtr = static_cast<ResourceHandle*>(context->job);
        cmd->m_data[1].typeVoidPtr = context;
        callOnMainThread(willSendRequest, cmd);
    }
}

void postHandleCancel(ResourceHandleContext* context)
{
    ResourceHandleCommand* cmd = new ResourceHandleCommand();
    if (cmd) {
        cmd->m_data[0].typeVoidPtr = context->job;
        cmd->m_data[1].typeVoidPtr = context;
        callOnMainThread(handleCancel, cmd);
    }
}

void postDidFinish(ResourceHandleContext* context)
{
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
    s_workerThreadMultiMutex->lock();
    s_jobToContextMap.remove(context->job);
    s_workerThreadMultiMutex->unlock();
#endif
    ResourceHandleCommand* cmd = new ResourceHandleCommand();
    if (cmd) {
        size_t urllen = strlen(context->m_url);
        cmd->m_data[0].typeVoidPtr = static_cast<ResourceHandle*>(context->job);
        cmd->m_data[1].typeCharPtr = new char[urllen + 1];
        if (cmd->m_data[1].typeCharPtr) {
            strncpy(cmd->m_data[1].typeCharPtr, context->m_url, urllen);
            cmd->m_data[1].typeCharPtr[urllen] = '\0';
            DEBUG_LOG("postDidFinish [%s]\n", context->m_url);
        } else
            ASSERT(false);
        cmd->m_data[2].typeVoidPtr = context;
        callOnMainThread(didFinish, cmd);
    }
}

void postDidFail(ResourceHandleContext* context)
{
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
    s_workerThreadMultiMutex->lock();
    s_jobToContextMap.remove(context->job);
    s_workerThreadMultiMutex->unlock();
#endif
    ResourceHandleCommand* cmd = new ResourceHandleCommand();
    if (cmd) {
        cmd->m_data[0].typeVoidPtr = static_cast<ResourceHandle*>(context->job);
        cmd->m_data[1].typeInt = context->m_curlResult;

        if (context->m_url) {
            size_t urllen = strlen(context->m_url);
            cmd->m_data[2].typeCharPtr = new char[urllen + 1];
            if (cmd->m_data[2].typeCharPtr) {
                strncpy(cmd->m_data[2].typeCharPtr, context->m_url, urllen);
                cmd->m_data[2].typeCharPtr[urllen] = '\0';
                DEBUG_LOG("postDidFail [%s]\n", context->m_url);
            } else
                ASSERT(false);
        }

        size_t errlen = strlen(curl_easy_strerror(context->m_curlResult));
        cmd->m_data[3].typeCharPtr = new char[errlen + 1];
        if (cmd->m_data[3].typeCharPtr)
            strncpy(cmd->m_data[3].typeCharPtr, curl_easy_strerror(context->m_curlResult), errlen);
        cmd->m_data[4].typeVoidPtr = context;
        callOnMainThread(didFail, cmd);
    }
}

void postAddResponseHeader(ResourceHandleContext* context, char* ptr, size_t totalSize)
{
    ResourceHandleCommand* cmd = new ResourceHandleCommand();
    if (cmd) {
        cmd->m_data[0].typeVoidPtr = static_cast<ResourceHandle*>(context->job);
        cmd->m_data[1].typeCharPtr = new char[totalSize + 1];
        if (cmd->m_data[1].typeCharPtr) {
            strncpy(cmd->m_data[1].typeCharPtr, ptr, totalSize);
            cmd->m_data[1].typeCharPtr[totalSize] = '\0';
        } else
            ASSERT(false);
        cmd->m_data[2].typeInt = totalSize;
        callOnMainThread(addResponseHeader, cmd);
    }
}

void postHandleResponseHeaders(ResourceHandleContext* context)
{
    CURL* h = context->m_handle;

    double contentLength = 0;
    curl_easy_getinfo(h, CURLINFO_CONTENT_LENGTH_DOWNLOAD, &contentLength);

    const char* hdr;
    curl_easy_getinfo(h, CURLINFO_EFFECTIVE_URL, &hdr);

    long httpCode = 0;
    curl_easy_getinfo(h, CURLINFO_RESPONSE_CODE, &httpCode);

    ResourceHandleCommand* cmd = new ResourceHandleCommand();
    if (cmd) {
        size_t hdrlen = strlen(hdr);
        cmd->m_data[0].typeVoidPtr = static_cast<ResourceHandle*>(context->job);
        cmd->m_data[1].typeInt = static_cast<long long int>(contentLength);
        cmd->m_data[2].typeCharPtr = new char[hdrlen + 1];
        if (cmd->m_data[2].typeCharPtr) {
            strncpy(cmd->m_data[2].typeCharPtr, hdr, hdrlen);
            cmd->m_data[2].typeCharPtr[hdrlen] = '\0';
        } else 
            ASSERT(false);

        cmd->m_data[3].typeInt = httpCode;
        callOnMainThread(handleResponseHeaders, cmd);
    }
}

void postNotifyOriginFrame(void* context)
{
    ResourceHandleCommand* cmd = new ResourceHandleCommand();
    if (cmd) {
        cmd->m_data[0].typeVoidPtr = context;
        callOnMainThread(notifyOriginFrame, cmd);
    }
}

void ResourceHandleManager::postNotifyBadCertSynchronously(void* context)
{
    ASSERT(context);

    if (context) {
        ResourceHandleCommand* cmd = new ResourceHandleCommand();
        if (cmd) {
            ResourceHandleContext* con = reinterpret_cast<ResourceHandleContext*>(context);

            cmd->m_data[0].typeVoidPtr = context;
            cmd->m_data[1].typeInt = con->m_sslError;
            if (con->m_url) {
                size_t urllen = strlen(con->m_url);
                cmd->m_data[2].typeCharPtr = new char[urllen + 1];
                if (cmd->m_data[2].typeCharPtr) {
                    strncpy(cmd->m_data[2].typeCharPtr, con->m_url, urllen);
                    cmd->m_data[2].typeCharPtr[urllen] = '\0';
                    DEBUG_LOG("postNotifyBadCertSynchronously [%s]\n", con->m_url);
                } else
                    ASSERT(false);
            }

            WTF::MutexLocker locker(*s_notifyBadCertConditionMutex);
            callOnMainThread(notifyBadCertSynchronously, cmd);
            s_notifyBadCertCondition->wait(*s_notifyBadCertConditionMutex);
        }
    }
}

void extractAuthInfo(ResourceHandleContext* context, String& header, CURL* handle)
{
    ASSERT(context && handle);

    DEBUG_LOG("### extractAuthInfo()");
    DEBUG_LOG("    url = [%s]\n", context->m_url);
    DEBUG_LOG("    header = [%s]\n", header.latin1().data());

    if ((!context->m_hasHttpRealm || !context->m_hasProxyRealm)
        && (header.find((const LChar*)"Authenticate") != -1)) {
        // To handle "WWW" or "Proxy" Authenticate,
        // this browser has to retrieve "realm" from
        // Authenticate header in this libcurl header callback.
        // "realm" value gotten in this method will be used in
        // handleAuthentication() function afterwards to show in
        // authenticate dialog.
        long httpCode = 0;
        long connectCode = 0;
        curl_easy_getinfo(handle, CURLINFO_RESPONSE_CODE, &httpCode);
        curl_easy_getinfo(handle, CURLINFO_HTTP_CONNECTCODE, &connectCode);

        if (httpCode == 401) {
            if (context->m_isHttpsScheme)
                context->m_httpProtectType = ProtectionSpaceServerHTTPS;
            else 
                context->m_httpProtectType = ProtectionSpaceServerHTTP;
            context->m_hasHttpRealm = extractRealmFromAuthHeader(context->m_httpProtectType, header, context->m_httpAuthType, context->m_httpRealm);
            context->m_authModeFromHeader |= Manx::AuthModeHttp;
        } else if (httpCode == 407 || connectCode == 407) {
            context->m_proxyProtectType = ProtectionSpaceProxyHTTP;
            context->m_hasProxyRealm = extractRealmFromAuthHeader(context->m_proxyProtectType, header, context->m_proxyAuthType, context->m_proxyRealm);
            context->m_authModeFromHeader |= Manx::AuthModeProxy;
        }
    }
}

/*
  Functions registered to curl handle which are called in worker thread.
  Classes in WebCore were removed to avoid race condition with main thread.

  - headerCallbackAsync
  - writeCallbackAsync
  - readCallbackAsync

*/
size_t ResourceHandleManager::headerCallbackAsync(char* ptr, size_t size, size_t nmemb, void* data)
{
    ASSERT(data);
    ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(data);
    if (context->m_cancelled)
        return 0;

#if LIBCURL_VERSION_NUM > 0x071200
    // We should never be called when deferred loading is activated.
    ASSERT(!context->m_defersLoading);
#endif

    CURL* h = context->m_handle;

    long connectCode = 0;
    long httpCode = 0;
    curl_easy_getinfo(h, CURLINFO_HTTP_CONNECTCODE, &connectCode);
    curl_easy_getinfo(h, CURLINFO_RESPONSE_CODE, &httpCode);

    if (!httpCode && connectCode == 200) {
        if (context->m_authModeInProcess & Manx::AuthModeProxy)
            saveProxyCredential(context);

        return size * nmemb;
    }

    if (context->m_isPost && !context->m_redirAfterPost) {
        // To prevent WebCore confising by telling status code 100,
        // return nothing.
        if (httpCode == 100)
            return size * nmemb;
    }

    size_t totalSize = size * nmemb;

    String header(static_cast<const char*>(ptr), totalSize);

    extractAuthInfo(context, header, context->m_handle);

    /*
     * a) We can finish and send the ResourceResponse
     * b) We will add the current header to the HTTPHeaderMap of the ResourceResponse
     *
     * The HTTP standard requires to use \r\n but for compatibility it recommends to
     * accept also \n.
     */
    if (header == "\r\n" || header == "\n") {
        /*
        I should use variable "err" to do cancel operation later.
        */
        postHandleResponseHeaders(context);

        if (httpCode == 200)
            context->m_isFinishedContext = true;

        if (httpCode == 401 || httpCode == 407 || connectCode == 407)
            return prepareHandleAuthenticate(context, httpCode, connectCode, totalSize, data);

        // HTTP redirection
        if (httpCode >= 300 && httpCode < 400) {

            if (context->hasLocation) {

                if (context->m_filteringActivatedStatus) {
                    bool skipEval = false;
                    convertToFilteringUrl(context, skipEval);
                    if (skipEval) {
                        if (!context->m_evaluatedkurl.isEmpty())
                            context->m_evaluatedkurl = context->m_kurl;
                        postWillSendRequest(context);
                        return totalSize;
                    }
                    ResourceHandleManager::sharedInstance()->initializeHandleForFiltering(context);
                    context->m_filteringRespHeaders.clear();
                    context->m_filteringBody.clear();
                    CURLcode err = curl_easy_perform(context->m_filteringhandle);
                    curl_easy_cleanup(context->m_filteringhandle);
                    if (!err) {
                        bool showPage = false;
                        evaluateFilteringResult(context, showPage);
                        if (showPage) {

                            if (context->m_kurl == context->m_evaluatedkurl) {
                                context->m_kurl = context->m_evaluatedkurl;

                                postWillSendRequest(context);
                                return totalSize;

                            }
                            context->m_blockedLocation = true;
                            return 0; // make this request failure and will goto block page.

                        }
                        // TODO:
                        // do second url conversion.
                        return 0;

                    }
                    return 0; // network error occurred
                }
                postWillSendRequest(context);
                return totalSize;
            }
        }

        postDidRecvResponse(context);

    } else {

        int splitPos = header.find(':');
        if (splitPos != -1) {
            if (header.lower().startsWith("location:")) {
                String location = header.substring(splitPos + 1).stripWhiteSpace();
                if (location.startsWith((const LChar*)"https://", false) || location.startsWith((const LChar*)"http://", false)) {
                    DEBUG_LOG("Original Location = [%s]\n", location.latin1().data());
                    context->m_kurl = KURL(KURL(), location);

                    if (!context->m_kurl.path().isEmpty() && !context->m_kurl.path().contains('%')) {
                        String escapedPath = WebCore::encodeWithURLEscapeSequences(context->m_kurl.path());
                        context->m_kurl.setPath(escapedPath);
                    }
                    if (!context->m_kurl.query().isEmpty() && !context->m_kurl.query().contains('%')) {
                        DEBUG_LOG("Location Query = [%s]\n", context->m_kurl.query().latin1().data());
                        String escapedQuery = WebCore::encodeWithURLEscapeSequences(context->m_kurl.query());
                        DEBUG_LOG("Location escapedQuery = [%s]\n", escapedQuery.latin1().data());
                        context->m_kurl.setQuery(escapedQuery);
                    }
                    if (!context->m_kurl.fragmentIdentifier().isEmpty() && !context->m_kurl.fragmentIdentifier().contains('%')) {
                        String escapedFragment = WebCore::encodeWithURLEscapeSequences(context->m_kurl.fragmentIdentifier());
                        context->m_kurl.setFragmentIdentifier(escapedFragment);
                    }
                    DEBUG_LOG("Location URL = [%s]\n", context->m_kurl.string().latin1().data());
                } else {
                    DEBUG_LOG("Location baseURL = [%s]\n", context->m_kurl.string().latin1().data());
                    context->m_kurl = KURL(context->m_kurl, location);
                }

                context->hasLocation = true;
            }
#ifdef DISABLE_X_FRAME_OPTIONS // disable x-frame-options for debug
            String xFrameOptions = response.httpHeaderField("x-frame-options");
            if (!xFrameOptions.isEmpty())
                return totalSize;
#endif
        }

        postAddResponseHeader(context, ptr, totalSize);
    }

    return totalSize;
}

size_t ResourceHandleManager::writeCallbackAsync(char* ptr, size_t size, size_t nmemb, void* data)
{
    ASSERT(data);
    ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(data);

    if (context->m_cancelled)
        return 0;

#if LIBCURL_VERSION_NUM > 0x071200
    // We should never be called when deferred loading is activated.
    ASSERT(!context->m_defersLoading);
#endif

    size_t totalSize = size * nmemb;

    // this shouldn't be necessary but apparently is. CURL writes the data
    // of html page even if it is a redirect that was handled internally
    // can be observed e.g. on gmail.com
    CURL* h = context->m_handle;
    long httpCode = 0;
    long connectCode = 0;
    CURLcode err = curl_easy_getinfo(h, CURLINFO_RESPONSE_CODE, &httpCode);
    if (CURLE_OK == err)
        err = curl_easy_getinfo(h, CURLINFO_HTTP_CONNECTCODE, &connectCode);
    if ((CURLE_OK == err && httpCode >= 300 && httpCode < 400)
        || ((CURLE_OK == err) && ((httpCode == 401) || (httpCode == 407) || (connectCode == 407)) && context->m_authModeInProcess && !context->m_authCancelled))
        return totalSize;

    const char* hdr;
    err = curl_easy_getinfo(h, CURLINFO_EFFECTIVE_URL, &hdr);
    ASSERT_UNUSED(err, CURLE_OK == err);

    postDidRecvData(context, ptr, hdr, totalSize);

    return totalSize;
}

size_t ResourceHandleManager::readCallbackAsync(void* ptr, size_t size, size_t nmemb, void* data)
{

    ASSERT(data);
    ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(data);

    if (context->m_cancelled)
        return 0;

#if LIBCURL_VERSION_NUM > 0x071200
    // We should never be called when deferred loading is activated.
    ASSERT(!context->m_defersLoading);
#endif

    if (!size || !nmemb)
        return 0;

    if (!context->m_formDataStream.hasMoreElementsAsync(context->m_formData))
        return 0;

    size_t sent = context->m_formDataStream.readAsync(ptr, size, nmemb, context->m_formData);

    // Something went wrong so cancel the job.
    if (!sent)
        postHandleCancel(context);

    return sent;
}

/*
  Functions registerd to curl handle which are called in filtering thread.

  - headerCallbackFiltering
  - writeCallbackFiltering

*/
static size_t headerCallbackFiltering(char* ptr, size_t size, size_t nmemb, void* data)
{
    ASSERT(data);
    ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(data);
    if (context->m_cancelled)
        return 0;

#if LIBCURL_VERSION_NUM > 0x071200
    // We should never be called when deferred loading is activated.
    ASSERT(!context->m_defersLoading);
#endif

    context->m_filteringRespHeaders.append(reinterpret_cast<const char*>(ptr), size * nmemb);

    return size * nmemb;
}

static size_t writeCallbackFiltering(char* ptr, size_t size, size_t nmemb, void* data)
{
    ASSERT(data);
    ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(data);
    if (context->m_cancelled)
        return 0;

#if LIBCURL_VERSION_NUM > 0x071200
    // We should never be called when deferred loading is activated.
    ASSERT(!context->m_defersLoading);
#endif

    context->m_filteringBody.append(reinterpret_cast<char*>(ptr), size * nmemb);

    return size * nmemb;
}

#ifndef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
/*
  Functions which are called to pump thread to which commands are thrown.
*/
static void postWorkerThreadPumpMessage(void* cmd)
{
    DEBUG_LOG("-> postWorkerThreadPumpMessage() time = [%f]\n", currentTime());
    ResourceHandleManager::sharedInstance()->signalWorkerThread();
}
#endif

/*
  Functions which are called in main(webkit) thread via callOnMainThread function
  called in worker thread.

  - didRecvResponse
  - didRecvData
  - willSendRequest
  - didFinish
  - didFail
  - handleCancel
  - addResponseHeader
  - handleResponseHeaders
  - handleSetMimeType
  - handleDataUrl
  - notifyOriginFrame

*/
static void didRecvResponse(void* cmd)
{
    DEBUG_LOG("-> didRecvResponse()\n");
    ASSERT(cmd);

    ResourceHandleCommand* command = reinterpret_cast<ResourceHandleCommand*>(cmd);
    WebCore::ResourceHandle* handle = reinterpret_cast<WebCore::ResourceHandle*>(command->m_data[0].typeVoidPtr);

    if (handle) {
        WebCore::ResourceHandleInternal* d = handle->getInternal();
        if (d) {
            if (!d->m_cancelled) {
                if (command->m_data[1].typeCharPtr)
                    d->m_response.setURL(WebCore::KURL(
                                                       KURL(),
                                                       command->m_data[1].typeCharPtr));

                if (command->m_data[2].typeCharPtr) {
                    d->m_response.setReplaceUrl(WebCore::KURL(
                                                              KURL(),
                                                              command->m_data[2].typeCharPtr));
                }

                WebCore::ResourceHandleClient* client = d->client();
                if (client)
                    client->didReceiveResponse(handle, d->m_response);

                d->m_response.setResponseFired(true);
            }
        }
    }
    if (command->m_data[1].typeCharPtr)
        delete [] command->m_data[1].typeCharPtr;

    if (command->m_data[2].typeCharPtr)
        delete [] command->m_data[2].typeCharPtr;

    if (command)
        delete command;

}

static void didRecvData(void* cmd)
{
    DEBUG_LOG("-> didRecvData()\n");
    ASSERT(cmd);

    ResourceHandleCommand* command = reinterpret_cast<ResourceHandleCommand*>(cmd);
    WebCore::ResourceHandle* handle = reinterpret_cast<WebCore::ResourceHandle*>(command->m_data[0].typeVoidPtr);

    if (handle) {
        WebCore::ResourceHandleInternal* d = handle->getInternal();
        if (d) {
            if (!d->m_cancelled) {
                if (!d->m_response.responseFired()) {
                    d->m_response.setURL(KURL(KURL(), command->m_data[3].typeCharPtr));
                    if (d->client())
                        d->client()->didReceiveResponse(handle, d->m_response);
                    d->m_response.setResponseFired(true);
                    if (d->m_cancelled)
                        goto cleancmd;
                }

                WebCore::ResourceHandleClient* client = d->client();
                if (client) {
                    client->didReceiveData(
                                           handle, 
                                           static_cast<char*>(command->m_data[1].typeCharPtr), 
                                           command->m_data[2].typeInt, 
                                           0);
                }
            }
        }
    }

 cleancmd:
    if (command->m_data[1].typeCharPtr)
        delete [] command->m_data[1].typeCharPtr;
    if (command->m_data[3].typeCharPtr)
        delete [] command->m_data[3].typeCharPtr;

    if (command)
        delete command;

}

static void willSendRequest(void* cmd)
{

    DEBUG_LOG("-> willSendRequest()\n");
    ASSERT(cmd);

    ResourceHandleCommand* command = reinterpret_cast<ResourceHandleCommand*>(cmd);
    WebCore::ResourceHandle* handle = reinterpret_cast<WebCore::ResourceHandle*>(command->m_data[0].typeVoidPtr);
    ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(command->m_data[1].typeVoidPtr);

    if (handle && context) {
        WebCore::ResourceHandleInternal* d = handle->getInternal();
        if (d) {
            if (!d->m_cancelled) {
                WebCore::ResourceHandleClient* client = d->client();
                if (client) {

                    WTF::String location = d->m_response.httpHeaderField("location");
                    if (!location.isEmpty()) {
                        WebCore::KURL newURL = context->m_kurl;

                        WebCore::ResourceRequest redirectedRequest = handle->firstRequest();
                        redirectedRequest.setURL(newURL);
                        redirectedRequest.setRedirected(true);

                        if (client)
                            client->willSendRequest(handle, redirectedRequest, d->m_response);

                        redirectedRequest.setRedirected(false);
                        d->m_firstRequest.setURL(newURL);

                    }
                }
            }
        }
    }

    if (command)
        delete command;

}

static void didFinish(void* cmd)
{
    DEBUG_LOG("-> didFinish()\n");

    ASSERT(cmd);

    ResourceHandleCommand* command = reinterpret_cast<ResourceHandleCommand*>(cmd);
    WebCore::ResourceHandle* handle = reinterpret_cast<WebCore::ResourceHandle*>(command->m_data[0].typeVoidPtr);

    if (handle) {
        WebCore::ResourceHandleInternal* d = handle->getInternal();
        if (d) {
            if (!d->m_cancelled) {
                if (!d->m_response.responseFired()) {

                    d->m_response.setURL(KURL(KURL(), command->m_data[1].typeCharPtr));
                    if (d->client())
                        d->client()->didReceiveResponse(handle, d->m_response);
                    d->m_response.setResponseFired(true);

                }

                if (d->client()) {
                    d->client()->didFinishLoading(handle, 0);
                    handle->deref();
                    handle = 0;
                }
            }
        }
    }

    if (command->m_data[1].typeCharPtr)
        delete [] command->m_data[1].typeCharPtr;
    if (command->m_data[2].typeVoidPtr) {
        ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(command->m_data[2].typeVoidPtr);
        if (context)
            delete context;
    }

    if (command)
        delete command;

#if ASYNC_REQUEST_DEBUG
    jobscount--;
    printf("didfinish() count = [%d]\n", jobscount);
#endif
}

static void didFail(void* cmd)
{
    DEBUG_LOG("-> didFail()\n");
    ASSERT(cmd);

    ResourceHandleCommand* command = reinterpret_cast<ResourceHandleCommand*>(cmd);
    WebCore::ResourceHandle* handle = reinterpret_cast<WebCore::ResourceHandle*>(command->m_data[0].typeVoidPtr);

    if (handle) {
        WebCore::ResourceHandleInternal* d = handle->getInternal();
        if (d) {
            if (!d->m_cancelled) {
                WebCore::ResourceHandleClient* client = d->client();
                if (client)
                    client->didFail(handle, 
                    (WebCore::ResourceError(String(),
                    command->m_data[1].typeInt,
                    WTF::String(command->m_data[2].typeCharPtr),
                    WTF::String(command->m_data[3].typeCharPtr)
                    )
                    )
                    );
                handle->deref();
                handle = 0;
            }
        }
    }
    if (command->m_data[2].typeCharPtr)
        delete [] command->m_data[2].typeCharPtr;
    if (command->m_data[3].typeCharPtr)
        delete [] command->m_data[3].typeCharPtr;
    if (command->m_data[4].typeVoidPtr) {
        ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(command->m_data[4].typeVoidPtr);
        if (context)
            delete context;
    }

    if (command)
        delete command;

#if ASYNC_REQUEST_DEBUG
    jobscount--;
    printf("didfail() count = [%d]\n", jobscount);
#endif
}

static void handleCancel(void* cmd)
{
    DEBUG_LOG("-> handleCancel()\n");

    ResourceHandleCommand* command = reinterpret_cast<ResourceHandleCommand*>(cmd);
    ASSERT(command);
    WebCore::ResourceHandle* handle = reinterpret_cast<WebCore::ResourceHandle*>(command->m_data[0].typeVoidPtr);
    ASSERT(handle);
    ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(command->m_data[1].typeVoidPtr);
    ASSERT(context);

    if (handle && context) {
        WebCore::ResourceHandleClient* client = handle->client();
        ASSERT(client);
        if (client) {
            WebCore::ResourceError error(String(),
                                         0,
                                         WTF::String(),
                                         WTF::String());
            error.setIsCancellation(true);
            client->didFail(handle, error);
        }
        handle->cancel();
        handle->deref();
        handle = 0;
        if (context)
            delete context;
    }
    
    if (command)
        delete command;

#if ASYNC_REQUEST_DEBUG
    jobscount--;
    printf("handleCancel() count = [%d]\n", jobscount);
#endif
}

static void addResponseHeader(void *cmd)
{
    DEBUG_LOG("-> addResponseHeader()\n");

    ASSERT(cmd);

    ResourceHandleCommand* command = reinterpret_cast<ResourceHandleCommand*>(cmd);
    WebCore::ResourceHandle* handle = reinterpret_cast<WebCore::ResourceHandle*>(command->m_data[0].typeVoidPtr);

    if (handle) {
        WebCore::ResourceHandleInternal* d = handle->getInternal();
        if (d) {
            if (!d->m_cancelled) {
                String header(static_cast<const char*>(command->m_data[1].typeCharPtr), command->m_data[2].typeInt);

                int splitPos = header.find(":");
                if (splitPos != -1)
                    d->m_response.setHTTPHeaderField(header.left(splitPos), header.substring(splitPos+1).stripWhiteSpace());
            }
        }
    }
    if (command->m_data[1].typeCharPtr)
        delete [] command->m_data[1].typeCharPtr;

    if (command)
        delete command;

}

static void handleResponseHeaders(void* cmd)
{
    DEBUG_LOG("-> handleResponseHeaders()\n");

    ASSERT(cmd);

    ResourceHandleCommand* command = reinterpret_cast<ResourceHandleCommand*>(cmd);
    WebCore::ResourceHandle* handle = reinterpret_cast<WebCore::ResourceHandle*>(command->m_data[0].typeVoidPtr);


    if (handle) {
        WebCore::ResourceHandleInternal* d = handle->getInternal();
        if (d) {
            if (!d->m_cancelled) {
                d->m_response.setExpectedContentLength(command->m_data[1].typeInt);

                if (command->m_data[2].typeCharPtr)
                    d->m_response.setURL(KURL(KURL(), command->m_data[2].typeCharPtr));

                d->m_response.setHTTPStatusCode(command->m_data[3].typeInt);

                d->m_response.setMimeType(extractMIMETypeFromMediaType(d->m_response.httpHeaderField("Content-Type")));
                d->m_response.setTextEncodingName(extractCharsetFromMediaType(d->m_response.httpHeaderField("Content-Type")));
                d->m_response.setSuggestedFilename(filenameFromHTTPContentDisposition(d->m_response.httpHeaderField("Content-Disposition")));
            }
        }
    }
    if (command->m_data[2].typeCharPtr)
        delete [] command->m_data[2].typeCharPtr;

    if (command)
        delete command;

}

static void handleSetMimeType(void *cmd)
{
    DEBUG_LOG("-> handleSetMimeType()\n");

    ASSERT(cmd);

    ResourceHandleCommand* command = reinterpret_cast<ResourceHandleCommand*>(cmd);
    WebCore::ResourceHandle* handle = reinterpret_cast<WebCore::ResourceHandle*>(command->m_data[0].typeVoidPtr);

    if (handle) {
        WebCore::ResourceHandleInternal* d = handle->getInternal();
        if (d)
            d->m_response.setMimeType(MIMETypeRegistry::getMIMETypeForPath(command->m_data[1].typeCharPtr));

    }

    if (command->m_data[1].typeCharPtr)
        delete [] command->m_data[1].typeCharPtr;

    
    if (command)
        delete command;

}

static void handleDataUrl(void* cmd)
{
    DEBUG_LOG("-> handleDataUrl()\n");

    ASSERT(cmd);

    ResourceHandleCommand* command = reinterpret_cast<ResourceHandleCommand*>(cmd);
    WebCore::ResourceHandle* handle = reinterpret_cast<WebCore::ResourceHandle*>(command->m_data[0].typeVoidPtr);

    if (handle) {
        WebCore::ResourceHandleInternal* d = handle->getInternal();
        if (d) {
            if (!d->m_cancelled)
                handleDataURL(handle);
        }
    }

    if (command)
        delete command;

}

static void notifyOriginFrame(void* cmd)
{
    ASSERT(cmd);

    ResourceHandleCommand* command = reinterpret_cast<ResourceHandleCommand*>(cmd);
    ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(command->m_data[0].typeVoidPtr);
    WebCore::ResourceHandle* handle = reinterpret_cast<WebCore::ResourceHandle*>(context->job);

    if (handle) {
        WebCore::ResourceHandleInternal* d = handle->getInternal();
        if (d) {
            if (!d->m_cancelled) {
                if (context->m_isMainResource && context->m_chainForJSExt)
                    d->m_response.setCert(context->m_chainForJSExt);
            }
        }
    }

    if (command)
        delete command;

}

void ResourceHandleManager::notifyBadCertSynchronously(void* cmd)
{
    DEBUG_LOG("-> notifyBadCertSynchronously\n");

    ASSERT(cmd);

    if (cmd) {
        ResourceHandleCommand* command = reinterpret_cast<ResourceHandleCommand*>(cmd);
        ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(command->m_data[0].typeVoidPtr);

        if (context) {
            ASSERT(context->m_nwcontext);
            ResourceError error(command->m_data[1].typeInt, (const LChar*)(command->m_data[2].typeCharPtr));
            Vector<CString> pems;
            for (int i = 0; i < context->m_pemlen; i++)
                pems.insert(i, (const char*)(context->m_pems[i]));

            error.setCerts(pems);

            if (context->m_nwcontext) {
                context->m_nwcontext->onCertificateVerificationRequest(error);
                context->m_confirm = error.confirmCert();
                DEBUG_LOG("<- notifyBadCertSynchronously\n");
            }

            if (command->m_data[2].typeCharPtr)
                delete [] command->m_data[2].typeCharPtr;
        }

        if (command)
            delete command;
    }
    // signal to resume
    WTF::MutexLocker locker(*s_notifyBadCertConditionMutex);
    s_notifyBadCertCondition->signal();
}


/*
  Functions which are used from dispatchSynchronousJob().
  Classes in WebCore were not removed since it is safe.

  - handleLocalReceiveResponse
  - writeCallback
  - headerCallback
  - readCallback
 */
static void handleLocalReceiveResponse(CURL* handle, ResourceHandle* job, ResourceHandleInternal* d)
{
    // since the code in headerCallback will not have run for local files
    // the code to set the URL and fire didReceiveResponse is never run,
    // which means the ResourceLoader's response does not contain the URL.
    // Run the code here for local files to resolve the issue.
    // TODO: See if there is a better approach for handling this.
    const char* hdr;
    CURLcode err = curl_easy_getinfo(handle, CURLINFO_EFFECTIVE_URL, &hdr);
    ASSERT_UNUSED(err, CURLE_OK == err);

    d->m_response.setURL(KURL(KURL(), hdr));
    if (d->client())
        d->client()->didReceiveResponse(job, d->m_response);
    d->m_response.setResponseFired(true);

}


// called with data after all headers have been processed via headerCallback
static size_t writeCallback(char* ptr, size_t size, size_t nmemb, void* data)
{
    if (!data)
        return 0;

    ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(data);
    ResourceHandle* job = context->job;
    ResourceHandleInternal* d = job->getInternal();
    
    if (d->m_cancelled)
        return 0;

#if LIBCURL_VERSION_NUM > 0x071200
    // We should never be called when deferred loading is activated.
    ASSERT(!d->m_defersLoading);
#endif

    size_t totalSize = size * nmemb;

    // this shouldn't be necessary but apparently is. CURL writes the data
    // of html page even if it is a redirect that was handled internally
    // can be observed e.g. on gmail.com
    CURL* h = d->m_handle;
    long httpCode = 0;
    CURLcode err = curl_easy_getinfo(h, CURLINFO_RESPONSE_CODE, &httpCode);
    if ((CURLE_OK == err && httpCode >= 300 && httpCode < 400)
        || ((CURLE_OK == err) && ((httpCode == 401) || (httpCode == 407)) && context->m_authModeInProcess && !context->m_authCancelled))
        return totalSize;

    if (!d->m_response.responseFired()) {
        handleLocalReceiveResponse(h, job, d);
        if (d->m_cancelled)
            return 0;
    }

    if (d->client())
        d->client()->didReceiveData(job, static_cast<char*>(ptr), totalSize, 0);
    return totalSize;
}

/*
 * This is being called for each HTTP header in the response. This includes '\r\n'
 * for the last line of the header.
 *
 * We will add each HTTP Header to the ResourceResponse and on the termination
 * of the header (\r\n) we will parse Content-Type and Content-Disposition and
 * update the ResourceResponse and then send it away.
 *
 */
size_t ResourceHandleManager::headerCallback(char* ptr, size_t size, size_t nmemb, void* data)
{
    if (!data)
        return 0;

    ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(data);
    ResourceHandle* job = context->job;
    ResourceHandleInternal* d = job->getInternal();

    if (d->m_cancelled)
        return 0;

#if LIBCURL_VERSION_NUM > 0x071200
    // We should never be called when deferred loading is activated.
    ASSERT(!d->m_defersLoading);
#endif

    size_t totalSize = size * nmemb;
    ResourceHandleClient* client = d->client();

    String header(static_cast<const char*>(ptr), totalSize);

    extractAuthInfo(context, header, context->m_handle);

    /*
     * a) We can finish and send the ResourceResponse
     * b) We will add the current header to the HTTPHeaderMap of the ResourceResponse
     *
     * The HTTP standard requires to use \r\n but for compatibility it recommends to
     * accept also \n.
     */
    if (header == "\r\n" || header == "\n") {
        CURL* h = d->m_handle;
        CURLcode err;

        long httpCode = 0;
        long connectCode = 0;
        curl_easy_getinfo(h, CURLINFO_RESPONSE_CODE, &httpCode);
        curl_easy_getinfo(h, CURLINFO_HTTP_CONNECTCODE, &connectCode);

        if (httpCode == 401 || httpCode == 407 || connectCode == 407)
            return prepareHandleAuthenticate(context, httpCode, connectCode, totalSize, data);

        // HTTP redirection
        if (httpCode >= 300 && httpCode < 400) {
            String location = d->m_response.httpHeaderField("location");
            if (!location.isEmpty()) {
                KURL newURL = KURL(job->firstRequest().url(), location);

                ResourceRequest redirectedRequest = job->firstRequest();
                redirectedRequest.setURL(newURL);
                if (client)
                    client->willSendRequest(job, redirectedRequest, d->m_response);

                d->m_firstRequest.setURL(newURL);

                return totalSize;
            }
        }

        double contentLength = 0;
        err = curl_easy_getinfo(h, CURLINFO_CONTENT_LENGTH_DOWNLOAD, &contentLength);
        d->m_response.setExpectedContentLength(static_cast<long long int>(contentLength));

        const char* hdr;
        err = curl_easy_getinfo(h, CURLINFO_EFFECTIVE_URL, &hdr);
        d->m_response.setURL(KURL(KURL(), hdr));

        d->m_response.setHTTPStatusCode(httpCode);
        d->m_response.setMimeType(extractMIMETypeFromMediaType(d->m_response.httpHeaderField("Content-Type")));
        d->m_response.setTextEncodingName(extractCharsetFromMediaType(d->m_response.httpHeaderField("Content-Type")));
        d->m_response.setSuggestedFilename(filenameFromHTTPContentDisposition(d->m_response.httpHeaderField("Content-Disposition")));

        if (client)
            client->didReceiveResponse(job, d->m_response);
        d->m_response.setResponseFired(true);

    } else {
        int splitPos = header.find(':');
        if (splitPos != -1)
            d->m_response.setHTTPHeaderField(header.left(splitPos), header.substring(splitPos+1).stripWhiteSpace());
    }

    return totalSize;
}

/* This is called to obtain HTTP POST or PUT data.
   Iterate through FormData elements and upload files.
   Carefully respect the given buffer size and fill the rest of the data at the next calls.
*/
size_t readCallback(void* ptr, size_t size, size_t nmemb, void* data)
{
    if (!data)
        return 0;

    ResourceHandleContext* context = reinterpret_cast<ResourceHandleContext*>(data);
    ResourceHandle* job = context->job;
    ResourceHandleInternal* d = job->getInternal();

    if (d->m_cancelled)
        return 0;

#if LIBCURL_VERSION_NUM > 0x071200
    // We should never be called when deferred loading is activated.
    ASSERT(!d->m_defersLoading);
#endif

    if (!size || !nmemb)
        return 0;

    if (!d->m_formDataStream.hasMoreElements())
        return 0;

    size_t sent = d->m_formDataStream.read(ptr, size, nmemb);

    // Something went wrong so cancel the job.
    if (!sent)
        job->cancel();

    return sent;
}

size_t ResourceHandleManager::prepareHandleAuthenticate(ResourceHandleContext* context, long httpCode, long connectCode, size_t totalSize, void* data)
{
    //
    //  handle authentication
    //
    if (context->m_authCancelled)
        return totalSize;

    if (httpCode == 401) {

        // comming 401 after 407 means that proxy auth was passed.
        if (context->m_authModeInProcess & Manx::AuthModeProxy)
            saveProxyCredential(context);

        // if there isn't a realm, we cancel authentication process.
        if (!context->m_hasHttpRealm)
            return totalSize;

        // if threre isn't a credential yet, we prepare to handle authenticate.
        if (!context->m_hasHttpCredential && !(context->m_authModeInProcess & Manx::AuthModeHttp)) {
            ResourceHandleManager::createAuthenticateThread(data);

            // This(returning zero) will cause CURLE_WRITE_ERROR.
            return 0;
        }
    }
    if (httpCode == 407 || connectCode == 407) {
        if (context->m_proxyAuthMaybeFailed) {
            context->m_proxyAuthMaybeFailed = false;
            if (context->m_authModeInProcess & Manx::AuthModeProxy)
                context->m_authModeInProcess -= Manx::AuthModeProxy;

            // libcurl tells 407 only once when user:pass is correct per one request.
            return 0;
        }

        context->m_proxyAuthMaybeFailed = true;

        // if there isn't a realm, we cancel authentication process.
        if (!context->m_hasProxyRealm)
            return totalSize;

        if (!context->m_hasProxyCredential && !(context->m_authModeInProcess & Manx::AuthModeProxy)) {
            ResourceHandleManager::createAuthenticateThread(data);

            // This(returning zero) will cause CURLE_WRITE_ERROR.
            return 0;
        }
    }
    return totalSize;
}

/*
  Functions which are called when filtering is enabled.

  - convertToFilteringUrl
  - evaluateFilteringResult
*/
bool convertToFilteringUrl(ResourceHandleContext* context, bool& skipEval)
{
    char* convertedUrlBuf = 0;
    int convertedUrlLen = 0;
    const int defaultLen = 8192;
    convertedUrlBuf = reinterpret_cast<char*>(malloc(defaultLen));

    if (convertedUrlBuf)
        Manx::WebSecurity::convertURL(context->m_isMainResource, context->m_kurl.string().latin1().data(), convertedUrlBuf, defaultLen, &convertedUrlLen, skipEval);
    else
        return false;
    if (convertedUrlLen > defaultLen-1) {
        convertedUrlBuf = reinterpret_cast<char*>(realloc(reinterpret_cast<void*>(convertedUrlBuf), convertedUrlLen + 1));

        if (convertedUrlBuf)
            Manx::WebSecurity::convertURL(context->m_isMainResource, context->m_kurl.string().latin1().data(), convertedUrlBuf, convertedUrlLen + 1, &convertedUrlLen, skipEval);
        else
            return false;
    }
    String convertedUrlStr(convertedUrlBuf, convertedUrlLen);
    context->m_filteringurl = convertedUrlStr;
    free(convertedUrlBuf);
    return true;
}

/*
     if showPage == true, we will finish filtering process and prepare for requesting
     original url.
     if showPage == false, we will convert original url again

     then we will send request again.
 */
bool evaluateFilteringResult(ResourceHandleContext* context, bool& showPage)
{
    char* generatedUrlBuf = 0;
    int generatedUrlLen = 0;
    const int defaultLen = 8192;

    generatedUrlBuf = reinterpret_cast<char*>(malloc(defaultLen));
    if (generatedUrlBuf)
        Manx::WebSecurity::filterReturnURL(context->m_filteringRespHeaders.data(), context->m_filteringBody.data(), context->m_kurl.string().latin1().data(), generatedUrlBuf, defaultLen, &generatedUrlLen, showPage);
    else
        return false;

    if (showPage && generatedUrlLen > defaultLen-1) {
        generatedUrlBuf = reinterpret_cast<char*>(realloc(reinterpret_cast<void*>(generatedUrlBuf), generatedUrlLen + 1));
        if (generatedUrlBuf)
            Manx::WebSecurity::filterReturnURL(context->m_filteringRespHeaders.data(), context->m_filteringBody.data(), context->m_kurl.string().latin1().data(), generatedUrlBuf, generatedUrlLen+1, &generatedUrlLen, showPage);
        else
            return false;
    }

    String generatedUrlStr = String(generatedUrlBuf, generatedUrlLen);
    context->m_evaluatedkurl = KURL(KURL(), generatedUrlStr);
    if (generatedUrlBuf)
        free(generatedUrlBuf);

    return true;
}

/*
  Functions to process HTTP request in worker threads.

  - processFiltering (for filtering request)
  - processDownload (for normal request)
*/
bool ResourceHandleManager::processFiltering()
{
    /*
      1) look up filtering-queue if there are any contexts
         to handle filtering.
      2) verify url if it needs to be filtered using libfiltering API.
      3) if the url needs to be filtered, set flag to do curl_multi_perform 
         for filtering-multi-handle.
      4) if the url doesn't need to be filtered, context will be removed then inserted to
         worker-queue for normal requests.
      5) if the flag for executing curl_multi_perform is set,
         call curl_multi_perform to execute requests.
      6) before calling didFinish(), evaluate filtering respnse 
         to check if original request should be sent or blocking url should be sent.
    */


    // handle 1),2),3),4) here.
    startScheduledFilteringJobs();

    // handle 5) here.
    // call curl_multi_perform to process curl handles which are added in startScheduledFilteringJobs().

    int runningHandles = 0;
    while (curl_multi_perform(s_filteringMultiHandle, &runningHandles) == CURLM_CALL_MULTI_PERFORM) {
    }

    // check the curl messages indicating completed transfers
    // and free their resources
    while (true) {
        int messagesInQueue;
        CURLMsg* msg = curl_multi_info_read(s_filteringMultiHandle, &messagesInQueue);
        if (!msg) {
            if (runningHandles > messagesInQueue) {
                DEBUG_LOG("ResourceHandleManager::processFiltering() continue processing filtering\n");

                // this direct to look up filtering-queue again.
                return true;
            }
            break;
        }

        CURL* handle = msg->easy_handle;
        ASSERT(handle);
        ResourceHandleContext* context = 0;
        CURLcode err = curl_easy_getinfo(handle, CURLINFO_PRIVATE, &context);
        ASSERT_UNUSED(err, CURLE_OK == err);
        ASSERT(context);

        if (!context->job)
            continue;
        ASSERT(context->m_filteringhandle == handle);

        if (context->m_cancelled) {
            removeFromCurlForFiltering(context);
            // move context to worker-queue
            continue;
        }

        if (CURLMSG_DONE != msg->msg)
            continue;

        if (CURLE_OK == msg->data.result) {

            if (context->m_cancelled) {
                removeFromCurlForFiltering(context);
                // move context to worker-queue
                continue;
            }

            // handle 6) here.
            // before didFinish, 
            // I should evaluate a response returned from a rating server using libwebfiltering.
            String generatedUrl;
            bool showPage = false;
            evaluateFilteringResult(context, showPage);
            if (showPage) {
                // move ResourceHandleContext from filtering-queue to worker-queue.
                context->m_kurl = context->m_evaluatedkurl;

                WTF::MutexLocker workerMutexlock(*s_workerThreadMutex);
                m_resourceHandleContextList.append(context);

#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
                WTF::MutexLocker workerMultiMutexlock(*s_workerThreadMultiMutex);
                wakeupSelect(s_WPIPE[1]);
#else
                WTF::MutexLocker workerConditionlock(*s_workerThreadConditionMutex);
                s_workerThreadCondition->signal();
#endif
            } else {
                bool skipEval = false;
                convertToFilteringUrl(context, skipEval);
                if (skipEval) {
                    WTF::MutexLocker workerMutexlock(*s_workerThreadMutex);
                    m_resourceHandleContextList.append(context);
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
                    WTF::MutexLocker workerMultiMutexlock(*s_workerThreadMultiMutex);
                    wakeupSelect(s_WPIPE[1]);
#else
                    WTF::MutexLocker workerConditionlock(*s_workerThreadConditionMutex);
                    s_workerThreadCondition->signal();
#endif
                } else {
                    removeFromCurlForFiltering(context);
                    context->m_filteringRespHeaders.clear();
                    context->m_filteringBody.clear();
                    startFilteringJob(context);
                    return true;
                }
            }

        } else {
            WTF::MutexLocker workerMutexlock(*s_workerThreadMutex);
            m_resourceHandleContextList.append(context);
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
            WTF::MutexLocker workerMultiMutexlock(*s_workerThreadMultiMutex);
            wakeupSelect(s_WPIPE[1]);
#else
            WTF::MutexLocker workerConditionlock(*s_workerThreadConditionMutex);
            s_workerThreadCondition->signal();
#endif

        }
        removeFromCurlForFiltering(context);
    }

    return startScheduledFilteringJobs();
}

void ResourceHandleManager::saveHttpCredential(ResourceHandleContext* context)
{
    if (!context->m_httpChallenge.isNull()) {
        Credential inputCredential(context->m_httpChallenge.m_user, context->m_httpChallenge.m_pass, CredentialPersistencePermanent);
        CredentialTransformData data = CredentialTransformData(context->m_kurl, context->m_httpSpace, inputCredential);

        Credential newSavedCredential = CredentialBackingStore::instance()->getLogin(data.url());
        if (newSavedCredential != data.credential()) {
            if (newSavedCredential.isEmpty())
                CredentialBackingStore::instance()->addLogin(data.url(), data.protectionSpace(), data.credential());
            else
                CredentialBackingStore::instance()->updateLogin(data.url(), data.protectionSpace(), data.credential());
        }
    }
}

void ResourceHandleManager::saveProxyCredential(ResourceHandleContext* context)
{
    if (!context->m_proxyChallenge.isNull()) {
        Credential inputCredential(context->m_proxyChallenge.m_user, context->m_proxyChallenge.m_pass, CredentialPersistencePermanent);
        CredentialTransformData data = CredentialTransformData(context->m_proxyurl, context->m_proxySpace, inputCredential);

        Credential newSavedCredential = CredentialBackingStore::instance()->getLogin(context->m_proxyurl);
        if (newSavedCredential != data.credential()) {
            if (newSavedCredential.isEmpty())
                CredentialBackingStore::instance()->addLogin(context->m_proxyurl, data.protectionSpace(), data.credential());
            else
                CredentialBackingStore::instance()->updateLogin(context->m_proxyurl, data.protectionSpace(), data.credential());
        }
    }
}

void ResourceHandleManager::saveAuthCredential(ResourceHandleContext* context)
{
    if (context->m_authModeInProcess) {

        if (context->m_authModeFromHeader & Manx::AuthModeHttp)
            saveHttpCredential(context);

        if (context->m_authModeFromHeader & Manx::AuthModeProxy)
            saveProxyCredential(context);

        context->m_authModeInProcess = Manx::AuthModeNone;
        context->m_authModeFromHeader = Manx::AuthModeNone;
        context->m_httpAuthType = context->m_proxyAuthType = Manx::HttpAuthNone;
    }
}

bool ResourceHandleManager::processDownload()
{
    startScheduledJobs();

    int runningHandles = 0;
    while (curl_multi_perform(s_workerMultiHandle, &runningHandles) == CURLM_CALL_MULTI_PERFORM) {
    }

    // check the curl messages indicating completed transfers
    // and free their resources
    while (true) {
        int messagesInQueue;
        CURLMsg* msg = curl_multi_info_read(s_workerMultiHandle, &messagesInQueue);
        if (!msg) {
            if (runningHandles > messagesInQueue)
                return true;

            break;
        }

        // find the node which has same d->m_handle as completed transfer
        CURL* handle = msg->easy_handle;
        ASSERT(handle);
        ResourceHandleContext* context = 0;
        CURLcode err = curl_easy_getinfo(handle, CURLINFO_PRIVATE, &context);
        ASSERT_UNUSED(err, CURLE_OK == err);
        ASSERT(context);
        if (!context->job)
            continue;
        ASSERT(context->m_handle == handle);

        if (context->m_cancelled) {
            removeFromCurl(context);
            continue;
        }

        if (CURLMSG_DONE != msg->msg)
            continue;

        if (CURLE_WRITE_ERROR == msg->data.result) {
            long httpCode = 0;
            long connectCode = 0;
            curl_easy_getinfo(handle, CURLINFO_RESPONSE_CODE, &httpCode);
            curl_easy_getinfo(handle, CURLINFO_HTTP_CONNECTCODE, &connectCode);

            //
            //  URL is blocked by filtering while redirection.
            //
            if (context->m_filteringActivatedStatus) {
                if (httpCode >= 300 && httpCode < 400 && context->m_blockedLocation) {
                    context->m_blockedLocation = false;
                    s_workerThreadMultiMutex->lock();
                    curl_multi_remove_handle(s_workerMultiHandle, context->m_handle);
                    curl_easy_cleanup(context->m_handle);
                
                    context->m_kurl = context->m_evaluatedkurl;
                    postWillSendRequest(context);
                    if (context->m_url) {
                        fastFree(context->m_url);
                        context->m_url = 0;
                    }
                    initializeHandleForAsync(context);
                    curl_multi_add_handle(s_workerMultiHandle, context->m_handle);
                    s_workerThreadMultiMutex->unlock();

                    while (curl_multi_perform(s_workerMultiHandle, &runningHandles) == CURLM_CALL_MULTI_PERFORM) {
                    }
                    continue;
                }
            }
            //
            //  handle authentication
            //
            if ((httpCode == 401 && context->m_hasHttpRealm && !(context->m_authModeInProcess & Manx::AuthModeHttp)) 
                || ((connectCode == 407 || httpCode == 407) && context->m_hasProxyRealm  && !(context->m_authModeInProcess & Manx::AuthModeProxy))) {

                // delegate the process to worker thread for authentication.
                WTF::MutexLocker lock(*s_authContextListMutex);
                m_authContextList.append(context);

#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
                // This will be moved from handleAuthentication()
                // to make sure that multi handle is used only in one thread.
                // Just removing easy handle from multi handle is safe.
                curl_multi_remove_handle(s_workerMultiHandle, context->m_handle);
#endif
                setAuthType(context);
                curl_easy_reset(context->m_handle);
                if (context->m_url) {
                    fastFree(context->m_url); 
                    context->m_url = 0;
                }
                initializeHandleForAsync(context);
                WTF::MutexLocker userMutexlock(*s_userThreadCreatedConditionMutex);
                s_userThreadCondition->signal();
                DEBUG_LOG("s_userThreadCondition->signal() httpCode = [%ld] connectCode = [%ld]\n", httpCode, connectCode);

                continue;
            }
        }

        //
        //  handle bad certificate
        //
        if (CURLE_SSL_CACERT == msg->data.result || CURLE_PEER_FAILED_VERIFICATION == msg->data.result) {
            if (!context->m_chainForBadCertNotify) {
                if (context->m_url) {
                    fastFree(context->m_url); 
                    context->m_url = 0;
                }

                s_workerThreadMultiMutex->lock();
                curl_multi_remove_handle(s_workerMultiHandle, context->m_handle);
                s_workerThreadMultiMutex->unlock();
                curl_easy_cleanup(context->m_handle);
 
                ResourceHandleManager::sharedInstance()->initializeHandleForAsync(context);
                curl_easy_setopt(context->m_handle, CURLOPT_SSL_VERIFYHOST, false);
                curl_easy_setopt(context->m_handle, CURLOPT_SSL_VERIFYPEER, false);
                s_workerThreadMultiMutex->lock();
                curl_multi_add_handle(s_workerMultiHandle, context->m_handle);
                s_workerThreadMultiMutex->unlock();
                while (curl_multi_perform(s_workerMultiHandle, &runningHandles) == CURLM_CALL_MULTI_PERFORM) { }
            } else {
                removeFromCurl(context);
                context->m_curlResult = msg->data.result;

                /*
                  Once Application(User) confirmed the bad certificate,
                  browser automatically cofirms the domain.
                */
                handleSslBadCertificate(context);
            }

            continue;
            
        }

        if (CURLE_OK == msg->data.result) {

            long httpCode;
            curl_easy_getinfo(context->m_handle, CURLINFO_RESPONSE_CODE, &httpCode);

            if (httpCode != 401 && httpCode != 407)
                saveAuthCredential(context);
            else {
                if (!context->m_authCancelled && ((httpCode == 401 && context->m_hasHttpRealm) || (httpCode == 407 && context->m_hasProxyRealm))) {
                    // delegate the process to worker thread for authentication.
                    WTF::MutexLocker lock(*s_authContextListMutex);
                    m_authContextList.append(context);
                    
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
                    // This will be moved from handleAuthentication()
                    // to make sure that multi handle is used only in one thread.
                    // Just removing easy handle from multi handle is safe.
                    curl_multi_remove_handle(s_workerMultiHandle, context->m_handle);
#endif
                    setAuthType(context);
                    curl_easy_reset(context->m_handle);
                    if (context->m_url) {
                        fastFree(context->m_url); 
                        context->m_url = 0;
                    }
                    initializeHandleForAsync(context);
                    WTF::MutexLocker userMutexlock(*s_userThreadCreatedConditionMutex);
                    s_userThreadCondition->signal();
                    DEBUG_LOG("s_userThreadCondition->signal()\n");

                    continue;
                }
            }

            if (context->m_cancelled) {
                removeFromCurl(context);
                continue;
            }

            removeFromCurl(context);
            postDidFinish(context);

        } else {
            char* url = 0;
            CURLcode err = curl_easy_getinfo(context->m_handle, CURLINFO_EFFECTIVE_URL, &url);
            if (err == CURLE_OK && url) {
                if (context->m_url)
                    fastFree(context->m_url);

                context->m_url = fastStrDup(url);
            }
            context->m_curlResult = msg->data.result;
#ifndef NDEBUG
            fprintf(stderr, "Curl ERROR for url='%s', error: '%s'\n", url, curl_easy_strerror(msg->data.result));
#endif

            long httpCode = 0;
            long connectCode = 0;
            curl_easy_getinfo(handle, CURLINFO_RESPONSE_CODE, &httpCode);
            curl_easy_getinfo(handle, CURLINFO_HTTP_CONNECTCODE, &connectCode);

            if (Manx::WebSecurity::isBlockedPageOnFilterProxy(httpCode, connectCode, msg->data.result)) {
                s_workerThreadMultiMutex->lock();
                curl_multi_remove_handle(s_workerMultiHandle, context->m_handle);
                curl_easy_cleanup(context->m_handle);
                        
                WTF::String url = Manx::WebSecurity::getBlockedPageOnFilterProxyURL();
                context->m_kurl = KURL(context->m_kurl, url);
                postWillSendRequest(context);
                if (context->m_url) {
                    fastFree(context->m_url);
                    context->m_url = 0;
                }
                initializeHandleForAsync(context);
                curl_multi_add_handle(s_workerMultiHandle, context->m_handle);
                s_workerThreadMultiMutex->unlock();

                while (curl_multi_perform(s_workerMultiHandle, &runningHandles) == CURLM_CALL_MULTI_PERFORM) {
                }
                continue;
            } 

            removeFromCurl(context);
            postDidFail(context);

        }
    }

    return startScheduledJobs(); // new jobs might have been added in the meantime
}

void ResourceHandleManager::removeFromCurl(ResourceHandleContext* context)
{
    ASSERT(context->m_handle);
    if (!context->m_handle)
        return;
#ifndef DISABLE_RUNNINGJOBS_COUNTING
    m_runningJobs--;
#endif
    s_workerThreadMultiMutex->lock();
    curl_multi_remove_handle(s_workerMultiHandle, context->m_handle);
    s_workerThreadMultiMutex->unlock();
    curl_easy_cleanup(context->m_handle);

    context->m_handle = 0;

}

void ResourceHandleManager::removeFromCurlForFiltering(ResourceHandleContext* context)
{
    ASSERT(context->m_filteringhandle);
    if (!context->m_filteringhandle)
        return;
#ifndef DISABLE_RUNNINGJOBS_COUNTING
    m_runningJobs--;
#endif
    s_workerThreadMultiMutex->lock();
    curl_multi_remove_handle(s_filteringMultiHandle, context->m_filteringhandle);
    s_workerThreadMultiMutex->unlock();
    curl_easy_cleanup(context->m_filteringhandle);

    context->m_filteringhandle = 0;

}

void ResourceHandleManager::setupPUT(ResourceHandle*, struct curl_slist**)
{
    notImplemented();
}

void ResourceHandleManager::setupPUTAsync(ResourceHandleContext*, struct curl_slist**)
{
    notImplemented();
}

/* Calculate the length of the POST.
   Force chunked data transfer if size of files can't be obtained.
 */
void ResourceHandleManager::setupPOST(ResourceHandle* job, struct curl_slist** headers)
{
    ResourceHandleInternal* d = job->getInternal();
    // CURL_REDIR_POST_302 and CURL_REDIR_POST_303 don't change the HTTP method,
    // so if the first request's method were POST and then redirected, the second request's
    // method would be set to POST when using CURL_REDIR_POST_302 or CURL_REDIR_POST_303.
    // Although the behavior of CURL_REDIR_POST_302 is conformant to RFC,
    // defact standard's behavior is different and changes to GET method, 
    // and the behavior of CURL_REDIR_POST_303 isn't conformant to RFC.
    curl_easy_setopt(d->m_handle, CURLOPT_POSTREDIR, CURL_REDIR_POST_301);
    curl_easy_setopt(d->m_handle, CURLOPT_POST, TRUE);
    curl_easy_setopt(d->m_handle, CURLOPT_POSTFIELDSIZE, 0);

    if (!job->firstRequest().httpBody())
        return;

    Vector<FormDataElement> elements = job->firstRequest().httpBody()->elements();
    size_t numElements = elements.size();
    if (!numElements)
        return;

    // Do not stream for simple POST data
    if (numElements == 1) {
        job->firstRequest().httpBody()->flatten(d->m_postBytes);
        if (d->m_postBytes.size()) {
            curl_easy_setopt(d->m_handle, CURLOPT_POSTFIELDSIZE, d->m_postBytes.size());
            curl_easy_setopt(d->m_handle, CURLOPT_POSTFIELDS, d->m_postBytes.data());
        }
        return;
    }

    // Obtain the total size of the POST
    // The size of a curl_off_t could be different in WebKit and in cURL depending on
    // compilation flags of both. For CURLOPT_POSTFIELDSIZE_LARGE we have to pass the
    // right size or random data will be used as the size.
    static int expectedSizeOfCurlOffT = 0;
    if (!expectedSizeOfCurlOffT) {
        curl_version_info_data* infoData = curl_version_info(CURLVERSION_NOW);
        if (infoData->features & CURL_VERSION_LARGEFILE)
            expectedSizeOfCurlOffT = sizeof(long long);
        else
            expectedSizeOfCurlOffT = sizeof(int);
    }

#if COMPILER(MSVC)
    // work around compiler error in Visual Studio 2005. It can't properly
    // handle math with 64-bit constant declarations.
#pragma warning(disable: 4307)
#endif
    static const long long maxCurlOffT = (1LL << (expectedSizeOfCurlOffT * 8 - 1)) - 1;
    curl_off_t size = 0;
    bool chunkedTransfer = false;
    for (size_t i = 0; i < numElements; i++) {
        FormDataElement element = elements[i];
        if (element.m_type == FormDataElement::encodedFile) {
            long long fileSizeResult;
            if (getFileSize(element.m_filename, fileSizeResult)) {
                if (fileSizeResult > maxCurlOffT) {
                    // File size is too big for specifying it to cURL
                    chunkedTransfer = true;
                    break;
                }
                size += fileSizeResult;
            } else {
                chunkedTransfer = true;
                break;
            }
        } else
            size += elements[i].m_data.size();
    }

    // cURL guesses that we want chunked encoding as long as we specify the header
    if (chunkedTransfer)
        *headers = curl_slist_append(*headers, "Transfer-Encoding: chunked");
    else {
        if (sizeof(long long) == expectedSizeOfCurlOffT)
          curl_easy_setopt(d->m_handle, CURLOPT_POSTFIELDSIZE_LARGE, (long long)size);
        else
          curl_easy_setopt(d->m_handle, CURLOPT_POSTFIELDSIZE_LARGE, (int)size);
    }

    curl_easy_setopt(d->m_handle, CURLOPT_READFUNCTION, readCallback);
    curl_easy_setopt(d->m_handle, CURLOPT_READDATA, job);
}

void ResourceHandleManager::setupPOSTAsync(ResourceHandleContext* context, struct curl_slist** headers)
{
    ASSERT(context);

    // CURL_REDIR_POST_302 and CURL_REDIR_POST_303 don't change the HTTP method,
    // so if the first request's method were POST and then redirected, the second request's
    // method would be set to POST when using CURL_REDIR_POST_302 or CURL_REDIR_POST_303.
    // Although the behavior of CURL_REDIR_POST_302 is conformant to RFC,
    // defact standard's behavior is different and changes to GET method, 
    // and the behavior of CURL_REDIR_POST_303 isn't conformant to RFC.
    curl_easy_setopt(context->m_handle, CURLOPT_POSTREDIR, CURL_REDIR_POST_301);
    curl_easy_setopt(context->m_handle, CURLOPT_POST, TRUE);
    curl_easy_setopt(context->m_handle, CURLOPT_POSTFIELDSIZE, 0);

    if (!context->m_formData)
        return;

    Vector<FormDataElement> elements = context->m_formData->elements();
    size_t numElements = elements.size();
    if (!numElements)
        return;

    // Do not stream for simple POST data
    if (numElements == 1) {
        context->m_formData->flatten(context->m_postBytes);
        if (context->m_postBytes.size()) {
            curl_easy_setopt(context->m_handle, CURLOPT_POSTFIELDSIZE, context->m_postBytes.size());
            curl_easy_setopt(context->m_handle, CURLOPT_POSTFIELDS, context->m_postBytes.data());
        }
        return;
    }

    // Obtain the total size of the POST
    // The size of a curl_off_t could be different in WebKit and in cURL depending on
    // compilation flags of both. For CURLOPT_POSTFIELDSIZE_LARGE we have to pass the
    // right size or random data will be used as the size.
    static int expectedSizeOfCurlOffT = 0;
    if (!expectedSizeOfCurlOffT) {
        curl_version_info_data* infoData = curl_version_info(CURLVERSION_NOW);
        if (infoData->features & CURL_VERSION_LARGEFILE)
            expectedSizeOfCurlOffT = sizeof(long long);
        else
            expectedSizeOfCurlOffT = sizeof(int);
    }

#if COMPILER(MSVC)
    // work around compiler error in Visual Studio 2005. It can't properly
    // handle math with 64-bit constant declarations.
#pragma warning(disable: 4307)
#endif
    static const long long maxCurlOffT = (1LL << (expectedSizeOfCurlOffT * 8 - 1)) - 1;
    curl_off_t size = 0;
    bool chunkedTransfer = false;
    for (size_t i = 0; i < numElements; i++) {
        FormDataElement element = elements[i];
        if (element.m_type == FormDataElement::encodedFile) {
            long long fileSizeResult;
            if (getFileSize(element.m_filename, fileSizeResult)) {
                if (fileSizeResult > maxCurlOffT) {
                    // File size is too big for specifying it to cURL
                    chunkedTransfer = true;
                    break;
                }
                size += fileSizeResult;
            } else {
                chunkedTransfer = true;
                break;
            }
        } else
            size += elements[i].m_data.size();
    }

    // cURL guesses that we want chunked encoding as long as we specify the header
    if (chunkedTransfer)
        *headers = curl_slist_append(*headers, "Transfer-Encoding: chunked");
    else {
        if (sizeof(long long) == expectedSizeOfCurlOffT) {
          curl_easy_setopt(context->m_handle, CURLOPT_POSTFIELDSIZE_LARGE, (long long)size);
          DEBUG_LOG("set CURLOPT_POSTFIELDSIZE_LARGE: (long)size=0x%llx\n", (long long)size);
        } else {
          curl_easy_setopt(context->m_handle, CURLOPT_POSTFIELDSIZE_LARGE, (int)size);
          DEBUG_LOG("set CURLOPT_POSTFIELDSIZE_LARGE: (int)size=0x%llx\n", (long long)size);
        }
    }

    curl_easy_setopt(context->m_handle, CURLOPT_READFUNCTION, readCallbackAsync);
    curl_easy_setopt(context->m_handle, CURLOPT_READDATA, context);
}

void ResourceHandleManager::add(ResourceHandle* job, NetworkingContext* nwcontext)
{
    // we can be called from within curl, so to avoid re-entrancy issues
    // schedule this job to be added the next time we enter curl download loop
    job->ref();

#if ASYNC_REQUEST_DEBUG
    jobscount++;
    printf("ResourceHandleManager::add() count = [%d]\n", jobscount);
#endif

    ResourceHandleContext* context = new ResourceHandleContext(job);
    if (context) {
        context->m_kurl = job->firstRequest().url().copy();
        context->m_postBytes = job->getInternal()->m_postBytes;
        context->m_formDataStream = job->getInternal()->m_formDataStream;
        context->m_isHttpsScheme = context->m_kurl.protocolIs("https");
        context->m_filteringActivatedStatus = Manx::WebSecurity::activatedStatus();
        context->m_host = fastStrDup(context->m_kurl.host().latin1().data());
        context->m_nwcontext = nwcontext;
        if (job->firstRequest().httpBody())
            context->m_formData = job->firstRequest().httpBody()->deepCopy();

        const HTTPHeaderMap headerFields =  job->getInternal()->m_firstRequest.httpHeaderFields(); 
        setCustomHeaders(headerFields, context);

        ResourceHandleInternal* handle = job->getInternal();
        if (handle) {
            context->m_isMainResource = nwcontext->isMainResource(handle->client());
            job->getInternal()->m_response.setIsMainResource(context->m_isMainResource);
        }
       
        DEBUG_LOG("ResourceHandleManager::add url = [%s]\n", context->m_kurl.string().latin1().data());

        if (context->m_filteringActivatedStatus) {
            checkAndCreateFilteringThread();

            s_filteringThreadMutex->lock();
            m_filteringResourceHandleContextList.append(context);
            s_filteringThreadMutex->unlock();

            s_filteringThreadConditionMutex->lock();
            s_filteringThreadCondition->signal();
            DEBUG_LOG("<- s_filteringThreadCondition->signal() time = [%f]\n", currentTime());
            s_filteringThreadConditionMutex->unlock();

        } else {
            s_workerThreadMutex->lock();
            m_resourceHandleContextList.append(context);
            s_workerThreadMutex->unlock();

#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
            WTF::MutexLocker workerMultiMutexlock(*s_workerThreadMultiMutex);
            s_jobToContextMap.add(context->job, context);
            wakeupSelect(s_WPIPE[1]);
#else
            s_workerThreadConditionMutex->lock();
            s_workerThreadCondition->signal();
            DEBUG_LOG("<- s_workerThreadCondition->signal() time = [%f]\n", currentTime());
            s_workerThreadConditionMutex->unlock();
#endif
        }
    }
}

bool ResourceHandleManager::removeScheduledJob(ResourceHandle* job)
{
    bool removed = false;

    s_workerThreadMutex->lock();
    int size = m_resourceHandleContextList.size();

    for (int i = 0; i < size; i++) {
        if (job == m_resourceHandleContextList[i]->job) {

            m_resourceHandleContextList[i]->m_cancelled = true;
            m_resourceHandleContextList.remove(i);
            job->deref();
            removed = true;
            break;
        }
    }
    s_workerThreadMutex->unlock();

    return removed;
}

bool ResourceHandleManager::startScheduledFilteringJobs()
{
    bool started = false;
    bool workerStart = false;

    WTF::MutexLocker lock(*s_filteringThreadMutex);

#ifndef DISABLE_RUNNINGJOBS_COUNTING
    while (!m_filteringResourceHandleContextList.isEmpty() && m_runningFilteringJobs < maxRunningJobs) {
#else
        while (!m_filteringResourceHandleContextList.isEmpty()) {
#endif
        ResourceHandleContext* context = m_filteringResourceHandleContextList[0];
        m_filteringResourceHandleContextList.remove(0);

        DEBUG_LOG("ResourceHandleManager::startScheduledFilteringJobs url = [%s]\n", context->m_kurl.string().latin1().data());

        bool skipEval = false;
        convertToFilteringUrl(context, skipEval);
        if (skipEval) {
            WTF::MutexLocker lock(*s_workerThreadMutex);
            m_resourceHandleContextList.append(context);

            workerStart = true;

        } else {
            // add curl handle to multi interface for filtering url.
            startFilteringJob(context);
            started = true;
        }
    }

    if (workerStart) {
        // request original url
        WTF::MutexLocker lock(*s_workerThreadConditionMutex);
        s_workerThreadCondition->signal();
#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
        WTF::MutexLocker workerMutexlock(*s_workerThreadMultiMutex);
        wakeupSelect(s_WPIPE[1]);
#endif
    }

    return started;
}

bool ResourceHandleManager::startScheduledJobs()
{
    bool started = false;

    s_workerThreadMutex->lock();

#ifndef DISABLE_RUNNINGJOBS_COUNTING
    while (!m_resourceHandleContextList.isEmpty() && m_runningJobs < maxRunningJobs) {
#else
    while (!m_resourceHandleContextList.isEmpty()) {
#endif
        ResourceHandleContext* context = m_resourceHandleContextList[0];
        m_resourceHandleContextList.remove(0);

        DEBUG_LOG("ResourceHandleManager::startScheduledJobs url = [%s]\n", context->m_kurl.string().latin1().data());
        startJob(context);
        started = true;
    }

    s_workerThreadMutex->unlock();
    return started;
}

void ResourceHandleManager::dispatchSynchronousJob(ResourceHandle* job, NetworkingContext* nwcontext)
{

#if ASYNC_REQUEST_DEBUG
    jobscount++;
    printf("ResourceHandleManager::dispatchSynchronousJob() count = [%d]\n", jobscount);
#endif

    KURL kurl = job->firstRequest().url();

    if (kurl.protocolIsData()) {
        handleDataURL(job);
        return;
    }

    ResourceHandleInternal* handle = job->getInternal();

#if LIBCURL_VERSION_NUM > 0x071200
    // If defersLoading is true and we call curl_easy_perform
    // on a paused handle, libcURL would do the transfer anyway
    // and we would assert so force defersLoading to be false.
    handle->m_defersLoading = false;
#endif

    /*
      MANX:

      Needs to handle filtering here using easy interface
      before sending original url.
     */

    ResourceHandleContext* context = new ResourceHandleContext(job);
    if (context) {
        context->m_syncReq = true;

        context->m_kurl = job->firstRequest().url();
        context->m_postBytes = job->getInternal()->m_postBytes;
        context->m_formDataStream = job->getInternal()->m_formDataStream;
        context->m_isHttpsScheme = context->m_kurl.protocolIs("https");
        context->m_filteringActivatedStatus = Manx::WebSecurity::activatedStatus();
        context->m_host = fastStrDup(context->m_kurl.host().latin1().data());
        context->m_nwcontext = nwcontext;

        DEBUG_LOG("ResourceHandleManager::dispatchSynchronousJob url = [%s]\n", context->m_kurl.string().latin1().data());
        initializeHandleForSync(context);

        // curl_easy_perform blocks until the transfert is finished.
        CURLcode ret =  curl_easy_perform(handle->m_handle);

        long httpCode = 0;
        curl_easy_getinfo(handle->m_handle, CURLINFO_RESPONSE_CODE, &httpCode);

        if (ret == CURLE_WRITE_ERROR && (httpCode == 401 || httpCode == 407)) {
            setAuthType(context);
            handleAuthentication(context);
        } else {
            if (ret) {
                ResourceError error(String(handle->m_url), ret, String(handle->m_url), String(curl_easy_strerror(ret)));
                handle->client()->didFail(job, error);
            }

#ifdef ENABLE_PERSISTENT_CONNECTION_FOR_SYNCHRONOUS_COMMUNICATION
            curl_easy_reset(context->m_handle);
#else
            curl_easy_cleanup(handle->m_handle);
#endif
        }

        delete context;

    } else 
        ASSERT(false);

#if ASYNC_REQUEST_DEBUG
    jobscount--;
    printf("ResourceHandleManager::dispatchSynchronousJob() count = [%d]\n", jobscount);
#endif
}

void ResourceHandleManager::startJob(ResourceHandleContext* context)
{
    ASSERT(context);

    if (context->m_kurl.protocolIsData()) {
        ResourceHandleCommand* cmd = new ResourceHandleCommand();
        if (cmd) {
            cmd->m_data[0].typeVoidPtr = context->job;
            callOnMainThread(handleDataUrl, cmd);
        }
        return;
    }

    initializeHandleForAsync(context);

#ifndef DISABLE_RUNNINGJOBS_COUNTING
    m_runningJobs++;
#endif
    s_workerThreadMultiMutex->lock();
    CURLMcode ret = curl_multi_add_handle(s_workerMultiHandle, context->m_handle);
    s_workerThreadMultiMutex->unlock();
    // don't call perform, because events must be async
    // timeout will occur and do curl_multi_perform
    if (ret && ret != CURLM_CALL_MULTI_PERFORM) {
#ifndef NDEBUG
        fprintf(stderr, "Error %d starting job %s\n", ret, encodeWithURLEscapeSequences(context->m_kurl.string()).latin1().data());
#endif
        postHandleCancel(context);
        return;
    }
}

void ResourceHandleManager::startFilteringJob(ResourceHandleContext* context)
{
    ASSERT(context);

    // initialize easy_handle to set specific settings for filtering.

    initializeHandleForFiltering(context);

#ifndef DISABLE_RUNNINGJOBS_COUNTING
    m_runningJobs++;
#endif
    s_workerThreadMultiMutex->lock();
    CURLMcode ret = curl_multi_add_handle(s_filteringMultiHandle, context->m_filteringhandle);
    s_workerThreadMultiMutex->unlock();
    if (ret && ret != CURLM_CALL_MULTI_PERFORM) {
        ASSERT(false);
        /*
          I should cope with this case as "Network Error".
         */
        return;
    }
}

void ResourceHandleManager::setCustomHeaders(const HTTPHeaderMap& headerFields, ResourceHandleContext* context)
{
    if (headerFields.size() > 0) {
        HTTPHeaderMap::const_iterator end = headerFields.end();
        for (HTTPHeaderMap::const_iterator it = headerFields.begin(); it != end; ++it) {
            String key = it->first;
            String value = it->second;
            String headerString(key);
            headerString.append(": ");
            headerString.append(value);
            CString headerLatin1 = headerString.latin1();
            context->m_customHeaders = curl_slist_append(context->m_customHeaders, headerLatin1.data());
        }
        CString expectString("Expect:");
        context->m_customHeaders = curl_slist_append(context->m_customHeaders, expectString.data());

        String acceptLanguageString("Accept-Language: ");
        acceptLanguageString.append(Manx::Locale::getLanguage());
        context->m_customHeaders = curl_slist_append(context->m_customHeaders, acceptLanguageString.latin1().data());
    }
}

void ResourceHandleManager::initializeHandleForSync(ResourceHandleContext* context)
{
    ResourceHandle* job = context->job;

    KURL kurl = job->firstRequest().url();

    // Remove any fragment part, otherwise curl will send it as part of the request.
    kurl.removeFragmentIdentifier();

    ResourceHandleInternal* d = job->getInternal();
    String url = kurl.string();

    if (kurl.isLocalFile()) {
        String query = kurl.query();
        // Remove any query part sent to a local file.
        if (!query.isEmpty()) {
            int queryIndex = url.find(query);
            if (queryIndex != -1)
                url = url.left(queryIndex - 1);
        }
        // Determine the MIME type based on the path.
        d->m_response.setMimeType(MIMETypeRegistry::getMIMETypeForPath(url));
    }

#ifdef ENABLE_PERSISTENT_CONNECTION_FOR_SYNCHRONOUS_COMMUNICATION
    d->m_handle = curlEasyHandleForSync();
#else
    d->m_handle = curl_easy_init();
#endif
    context->m_handle = d->m_handle;

#if LIBCURL_VERSION_NUM > 0x071200
    if (d->m_defersLoading) {
        CURLcode error = curl_easy_pause(d->m_handle, CURLPAUSE_ALL);
        // If we did not pause the handle, we would ASSERT in the
        // header callback. So just assert here.
        ASSERT_UNUSED(error, error == CURLE_OK);
    }
#endif
    const HTTPHeaderMap headerFields =  job->firstRequest().httpHeaderFields(); 
    setCustomHeaders(headerFields, context);
    initializeCurlHandle(d->m_handle, context, headerCallback, writeCallback);

    if (Manx::NetworkProfile::get(Manx::NetPropsSslVerifyPeer)) {
        curl_easy_setopt(d->m_handle, CURLOPT_SSL_CTX_FUNCTION, sslctxfun);
        curl_easy_setopt(d->m_handle, CURLOPT_SSL_CTX_DATA, context);
    }

    // url must remain valid through the request
    ASSERT(!d->m_url);

    // url is in ASCII so latin1() will only convert it to char* without character translation.
    d->m_url = fastStrDup(url.latin1().data());
    curl_easy_setopt(d->m_handle, CURLOPT_URL, d->m_url);

    DEBUG_LOG("-> initializeCurlHandle(sync) url [%s]\n", context->m_url);

    if (HttpMethod_Get == context->m_method)
        curl_easy_setopt(d->m_handle, CURLOPT_HTTPGET, TRUE);
    else if (HttpMethod_Post == context->m_method) {
        Manx::NetworkProfile::mountUploadDir();
        context->m_isPost = true;
        setupPOST(job, &context->m_customHeaders);
    } else if (HttpMethod_Put == context->m_method)
        setupPUT(job, &context->m_customHeaders);
    else if (HttpMethod_Head == context->m_method)
        curl_easy_setopt(d->m_handle, CURLOPT_NOBODY, TRUE);
    else if (HttpMethod_Delete == context->m_method)
        curl_easy_setopt(d->m_handle, CURLOPT_CUSTOMREQUEST, "DELETE");
    else if (HttpMethod_Options == context->m_method)
        curl_easy_setopt(context->m_handle, CURLOPT_CUSTOMREQUEST, "OPTIONS");

    if (context->m_customHeaders)
        curl_easy_setopt(d->m_handle, CURLOPT_HTTPHEADER, context->m_customHeaders);
}

void ResourceHandleManager::initializeHandleForAsync(ResourceHandleContext* context)
{
#if OS(ORBIS)
    ASSERT(isWorkerThread());
#endif
    ASSERT(context);

    KURL kurl = context->m_kurl;

    // Remove any fragment part, otherwise curl will send it as part of the request.
    kurl.removeFragmentIdentifier();

    String url = kurl.string();

    if (kurl.isLocalFile()) {
        String query = kurl.query();
        // Remove any query part sent to a local file.
        if (!query.isEmpty()) {
            int queryIndex = url.find(query);
            if (queryIndex != -1)
                url = url.left(queryIndex - 1);
        }

        // Determine the MIME type based on the path.
        ResourceHandleCommand* cmd = new ResourceHandleCommand();
        if (cmd) {
            cmd->m_data[0].typeVoidPtr = context->job;

            size_t datalen = strlen(url.ascii().data());
            cmd->m_data[1].typeCharPtr = new char[datalen + 1];
            if (cmd->m_data[1].typeCharPtr) {
                strncpy(cmd->m_data[1].typeCharPtr, url.ascii().data(), datalen);
                cmd->m_data[1].typeCharPtr[datalen] = '\0';
            } else
                ASSERT(false);
            callOnMainThread(handleSetMimeType, cmd);
        }
    }

    context->m_handle = curl_easy_init();

    if (kurl.host() == "psvitareg.trendmicro.com") {
        curl_easy_setopt(context->m_handle, CURLOPT_SSL_SESSIONID_CACHE, 0L);
        curl_easy_setopt(context->m_handle, CURLOPT_FRESH_CONNECT, 1L);
        curl_easy_setopt(context->m_handle, CURLOPT_FORBID_REUSE, 1L);
    }

#if LIBCURL_VERSION_NUM > 0x071200
    if (context->m_defersLoading) {
        CURLcode error = curl_easy_pause(context->m_handle, CURLPAUSE_ALL);
        // If we did not pause the handle, we would ASSERT in the
        // header callback. So just assert here.
        ASSERT_UNUSED(error, error == CURLE_OK);
    }
#endif
    initializeCurlHandle(context->m_handle, context, headerCallbackAsync, writeCallbackAsync);

    if (Manx::NetworkProfile::get(Manx::NetPropsSslVerifyPeer)) {
        curl_easy_setopt(context->m_handle, CURLOPT_SSL_CTX_FUNCTION, sslctxfun);
        curl_easy_setopt(context->m_handle, CURLOPT_SSL_CTX_DATA, context);
    }

    // url must remain valid through the request
    ASSERT(!context->m_url);

    // url is in ASCII so latin1() will only convert it to char* without character translation.
    context->m_url = fastStrDup(url.latin1().data());
    curl_easy_setopt(context->m_handle, CURLOPT_URL, context->m_url);

    DEBUG_LOG("-> initializeCurlHandle url [%s]\n", context->m_url);

    if (HttpMethod_Get == context->m_method)
        curl_easy_setopt(context->m_handle, CURLOPT_HTTPGET, TRUE);
    else if (HttpMethod_Post == context->m_method) {
        Manx::NetworkProfile::mountUploadDir();
        context->m_isPost = true;
        setupPOSTAsync(context, &context->m_customHeaders);
    } else if (HttpMethod_Put == context->m_method)
        setupPUTAsync(context, &context->m_customHeaders);
    else if (HttpMethod_Head == context->m_method)
        curl_easy_setopt(context->m_handle, CURLOPT_NOBODY, TRUE);
    else if (HttpMethod_Delete == context->m_method)
        curl_easy_setopt(context->m_handle, CURLOPT_CUSTOMREQUEST, "DELETE");
    else if (HttpMethod_Options == context->m_method)
        curl_easy_setopt(context->m_handle, CURLOPT_CUSTOMREQUEST, "OPTIONS");

    if (context->m_customHeaders)
        curl_easy_setopt(context->m_handle, CURLOPT_HTTPHEADER, context->m_customHeaders);
}

void ResourceHandleManager::initializeHandleForFiltering(ResourceHandleContext* context)
{
    ASSERT(context);

    context->m_filteringhandle = curl_easy_init();

    KURL kurl = context->m_kurl;
    if (kurl.host() == "psvitareg.trendmicro.com") {
        curl_easy_setopt(context->m_filteringhandle, CURLOPT_SSL_SESSIONID_CACHE, 0L);
        curl_easy_setopt(context->m_filteringhandle, CURLOPT_FRESH_CONNECT, 1L);
        curl_easy_setopt(context->m_filteringhandle, CURLOPT_FORBID_REUSE, 1L);
    }

    initializeCurlHandle(context->m_filteringhandle, context, headerCallbackFiltering, writeCallbackFiltering);

    String filteringUserAgentString("User-Agent: ");
    filteringUserAgentString.append(Manx::WebSecurity::userAgent());
    context->m_customHeaders = curl_slist_append(context->m_customHeaders, filteringUserAgentString.latin1().data());

    if (Manx::NetworkProfile::get(Manx::NetPropsSslVerifyPeer)) {
        curl_easy_setopt(context->m_filteringhandle, CURLOPT_SSL_CTX_FUNCTION, sslctxfun);
        curl_easy_setopt(context->m_filteringhandle, CURLOPT_SSL_CTX_DATA, context);
    }

    curl_easy_setopt(context->m_filteringhandle, CURLOPT_URL, context->m_filteringurl.latin1().data());

    // Should we support other methods than GET?
    curl_easy_setopt(context->m_filteringhandle, CURLOPT_HTTPGET, TRUE);

    if (context->m_customHeaders)
        curl_easy_setopt(context->m_filteringhandle, CURLOPT_HTTPHEADER, context->m_customHeaders);
}

void ResourceHandleManager::setCookies(const KURL& url, const String& value)
{
    if (!Manx::NetworkProfile::cookieEnabled())
        return;

    String host = url.host();
    String path = url.path();

    DEBUG_LOG("setCookies(): url.host=%s\n", host.utf8().data());
    DEBUG_LOG("setCookies(): url.path=%s\n", path.utf8().data());
    DEBUG_LOG("setCookies(): value=%s\n", value.utf8().data());

    Manx::Cookie *cookie = Manx::Cookie::create(value.utf8().data(), host.utf8().data(), path.utf8().data());
    if (!cookie) {
        DEBUG_LOG("setCookies(): BAD COOKIE\n");
        return;
    }

#if 0
    DEBUG_LOG("NAME    : %s\n", cookie->name());
    DEBUG_LOG("VALUE   : %s\n", cookie->value());
    DEBUG_LOG("DOMAIN  : %s\n", cookie->domain());
    DEBUG_LOG("PATH    : %s\n", cookie->path());
    DEBUG_LOG("EXPIRE  : %s\n", cookie->expire());
    DEBUG_LOG("MAXAGE  : %s\n", cookie->maxAge());
    DEBUG_LOG("SECURE  : %s\n", cookie->secure() ? "true" : "false");
    DEBUG_LOG("HTTPONLY: %s\n", cookie->httpOnly() ? "true" : "false");
#endif

    if (cookie->httpOnly()) {
        DEBUG_LOG("setCookies(): HTTP ONLY COOKIE\n");
        delete cookie;
        return;
    }

    const char* netscapeCookie = cookie->netscapeCookie();
    if (netscapeCookie) {
        // set the cookie in curl
        CURL* handle = curl_easy_init();
#if DEBUG_CURL
            curl_easy_setopt(handle, CURLOPT_VERBOSE, 1);
#endif
        curl_easy_setopt(handle, CURLOPT_SHARE, m_curlShareHandle);
        curl_easy_setopt(handle, CURLOPT_URL, url.string().latin1().data());
        lockCurlSharedResource(CURL_LOCK_DATA_COOKIE);
        curl_easy_setopt(handle, CURLOPT_COOKIELIST, netscapeCookie);
        unlockCurlSharedResource(CURL_LOCK_DATA_COOKIE);
        curl_easy_cleanup(handle);
        DEBUG_LOG("<ResourceHandleManager::setCookies(): %s>\n", netscapeCookie);
    }

    delete cookie;
}

void ResourceHandleManager::saveCookies()
{
    char cookieJarFileNameBuffer[256] = {""}; 
    char* cookieJarFileName = m_cookieJarFileName;
    if (!cookieJarFileName) {
        Manx::NetworkProfile::get(Manx::NetPropsCookieFilename, cookieJarFileNameBuffer, sizeof(cookieJarFileNameBuffer));
        cookieJarFileName = cookieJarFileNameBuffer;
    }

#if USE_SCRACHPAD_COOKIEJAR
    char* scrachpadFileName = 0;
    if (scrachpadmatch(cookieJarFileName)) {
        scrachpadFileName = makescrachpadpath(cookieJarFileName);
        cookieJarFileName = scrachpadFileName;
    }
#endif

    long long size = 0;
    if (getFileSize(cookieJarFileName, size)) {
        if (size > MAX_COOKIE_JAR_FILE_SIZE) {
            DEBUG_LOG("<ResourceHandleManager::saveCookies() file size over. :%s: size:%lld>\n", cookieJarFileName, size);
            if (!deleteAllCookies())
                abort();
            return;
        }
    }
    DEBUG_LOG("<ResourceHandleManager::saveCookies file:%s :size:%lld>\n", cookieJarFileName, size);

    // save cookie list to file 
    CURL* handle = curl_easy_init();

#if DEBUG_CURL
        curl_easy_setopt(handle, CURLOPT_VERBOSE, 1);
#endif

    cookieJarFileName = m_cookieJarFileName;
    if (!cookieJarFileName) {
        Manx::NetworkProfile::get(Manx::NetPropsCookieFilename, cookieJarFileNameBuffer, sizeof(cookieJarFileNameBuffer));
        cookieJarFileName = cookieJarFileNameBuffer;
    }

    if (Manx::NetworkProfile::cookieEnabled()) {
        curl_easy_setopt(handle, CURLOPT_SHARE, m_curlShareHandle);
        curl_easy_setopt(handle, CURLOPT_COOKIEJAR, cookieJarFileName);
        DEBUG_LOG("<ResourceHandleManager::saveCookies file:%s>\n", cookieJarFileName);
    } else {
        curl_easy_setopt(handle, CURLOPT_SHARE, m_curlShareHandle);
        curl_easy_setopt(handle, CURLOPT_COOKIELIST, "ALL");
    }
    curl_easy_cleanup(handle);

#if USE_SCRACHPAD_COOKIEJAR
    freescrachpadpath(scrachpadFileName);
#endif

}

void ResourceHandleManager::initializeCurlHandle(CURL* handle, ResourceHandleContext* context, curl_write_callback headerCallback, curl_write_callback writecallback)
{
    curl_easy_setopt(handle, CURLOPT_SSLVERSION, Manx::NetworkProfile::get(Manx::NetPropsSslVersion));
    curl_easy_setopt(handle, CURLOPT_PRIVATE, context);
    curl_easy_setopt(handle, CURLOPT_ERRORBUFFER, m_curlErrorBuffer);
    curl_easy_setopt(handle, CURLOPT_WRITEFUNCTION, writecallback);
    curl_easy_setopt(handle, CURLOPT_WRITEDATA, context);
    curl_easy_setopt(handle, CURLOPT_HEADERFUNCTION, headerCallback);
    curl_easy_setopt(handle, CURLOPT_WRITEHEADER, context);
    curl_easy_setopt(handle, CURLOPT_AUTOREFERER, Manx::NetworkProfile::get(Manx::NetPropsAutoReferer));
    curl_easy_setopt(handle, CURLOPT_FOLLOWLOCATION, Manx::NetworkProfile::get(Manx::NetPropsFollowLocation));
    curl_easy_setopt(handle, CURLOPT_MAXREDIRS, Manx::NetworkProfile::get(Manx::NetPropsMaxRedirs));
    curl_easy_setopt(handle, CURLOPT_SHARE, m_curlShareHandle);
    curl_easy_setopt(handle, CURLOPT_DNS_CACHE_TIMEOUT, s_dnsCacheTimeout); // 5 minutes
    curl_easy_setopt(handle, CURLOPT_TIMEOUT, s_transferTimeout); // libcurl default transfer timeout is 0 which means it never times out.
    curl_easy_setopt(handle, CURLOPT_CONNECTTIMEOUT, s_connectTimeout); // libcurl default connection timeout is 300 seconds.
    // FIXME: Enable SSL verification when we have a way of shipping certs
    // and/or reporting SSL errors to the user.
    curl_easy_setopt(handle, CURLOPT_SSL_VERIFYHOST, Manx::NetworkProfile::get(Manx::NetPropsSslVerifyHost));
    curl_easy_setopt(handle, CURLOPT_SSL_VERIFYPEER, Manx::NetworkProfile::get(Manx::NetPropsSslVerifyPeer));
    curl_easy_setopt(handle, CURLOPT_SSL_CIPHER_LIST, Manx::NetworkProfile::getString(Manx::NetPropsSslCipherList));
    curl_easy_setopt(handle, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);

    // If you want to use file type CAINFO, you can activate following codes.
    // CAINFO is loaded from memory every creating curl handle on MANX port.
    // (see also loadRootServerCerts())
    // char caPath[256] = {""};
    // Manx::NetworkProfile::get(Manx::NetPropsServerCertPath, caPath, sizeof(caPath));
    // if (strlen(caPath) > 0)
    //    curl_easy_setopt(handle, CURLOPT_CAINFO, caPath);

    // enable gzip and deflate through Accept-Encoding:
    char acceptEncoding[256] = {""}; 
    Manx::NetworkProfile::get(Manx::NetPropsEncoding, acceptEncoding, sizeof(acceptEncoding));
    curl_easy_setopt(handle, CURLOPT_ACCEPT_ENCODING, acceptEncoding);

    // Set cookie options 
    if (!Manx::NetworkProfile::cookieEnabled())
        curl_easy_setopt(handle, CURLOPT_COOKIELIST, "ALL");

    // check network 
    if (context->m_isMainResource)
        Manx::NetworkProfile::netCtlCheck();

    // Set proxy options if we have them.
    if (Manx::NetworkProfile::proxyEnabled() &&
        !Manx::WebSecurity::isBlockPageResourceURL(context->m_kurl.string().latin1().data()) &&
        !Manx::WebSecurity::isUnblockedDomain(context->m_kurl.string().latin1().data())) {
         
        char proxyServer[256];
        Manx::NetworkProfile::get(Manx::NetPropsProxy, proxyServer, sizeof(proxyServer));
        curl_easy_setopt(handle, CURLOPT_PROXY, proxyServer);
        curl_easy_setopt(handle, CURLOPT_PROXYTYPE, Manx::NetworkProfile::get(Manx::NetPropsProxyType));

        char proxyUserPass[256];
        Manx::NetworkProfile::get(Manx::NetPropsUserPass, proxyUserPass, sizeof(proxyUserPass));
        context->m_preSetProxyUserPass = "";
        context->m_preSetProxyUserPass.append((const LChar*)proxyUserPass);

        // set proxy auth credential if it exsits.
        ProtectionSpaceServerType type;
        ProtectionSpaceAuthenticationScheme scheme;
        KURL proxyUrl(KURL(), (const LChar*)proxyServer);
        context->m_proxyurl = proxyUrl.copy();
        WebCore::Credential savedProxyCredential = CredentialBackingStore::instance()->getLogin(proxyUrl, type, scheme);
        if (!savedProxyCredential.isEmpty()) {
            context->m_proxyUserPass = "";
            context->m_proxyUserPass.append(savedProxyCredential.user());
            context->m_proxyUserPass.append(":");
            context->m_proxyUserPass.append(savedProxyCredential.password());
            context->m_proxyAuthType = (scheme == ProtectionSpaceAuthenticationSchemeHTTPBasic)?CURLAUTH_BASIC:CURLAUTH_DIGEST;

            curl_easy_setopt(handle, CURLOPT_PROXYAUTH, context->m_proxyAuthType);
            curl_easy_setopt(handle, CURLOPT_PROXYUSERPWD, context->m_proxyUserPass.latin1().data());
            context->m_hasProxyCredential = true;
        }

    }

    // set auth credential if it exists.
    ProtectionSpaceServerType type;
    ProtectionSpaceAuthenticationScheme scheme;
    WebCore::Credential savedHttpCredential = CredentialBackingStore::instance()->getLogin(context->m_kurl, type, scheme);
    if (!savedHttpCredential.isEmpty()) {
        context->m_httpUserPass = "";
        context->m_httpUserPass.append(savedHttpCredential.user());
        context->m_httpUserPass.append(":");
        context->m_httpUserPass.append(savedHttpCredential.password());
        context->m_httpAuthType = (scheme == ProtectionSpaceAuthenticationSchemeHTTPBasic)?CURLAUTH_BASIC:CURLAUTH_DIGEST;

        curl_easy_setopt(handle, CURLOPT_HTTPAUTH, context->m_httpAuthType);
        curl_easy_setopt(handle, CURLOPT_USERPWD, context->m_httpUserPass.latin1().data());
        context->m_hasHttpCredential = true;

        context->m_authModeInProcess |= Manx::AuthModeHttp;
    }

    // allow bad ssl domain if it is already allowed.
    if (context->m_isHttpsScheme && s_userAcceptedHttpsDomains.contains(StringHash::hash(context->m_kurl.host()))) {
        curl_easy_setopt(handle, CURLOPT_SSL_VERIFYHOST, false);
        curl_easy_setopt(handle, CURLOPT_SSL_VERIFYPEER, false);
    }

}

String ResourceHandleManager::cookies(const KURL& url)
{
    String cookie;
    struct curl_slist *list;

    if (!Manx::NetworkProfile::cookieEnabled()) 
        return cookie;

    // get the cookie list in curl
    CURL* handle = curl_easy_init();
#if DEBUG_CURL
        curl_easy_setopt(handle, CURLOPT_VERBOSE, 1);
#endif
    curl_easy_setopt(handle, CURLOPT_SHARE, m_curlShareHandle);
    curl_easy_setopt(handle, CURLOPT_URL, url.string().latin1().data());
    lockCurlSharedResource(CURL_LOCK_DATA_COOKIE);
    curl_easy_getinfo(handle, CURLINFO_COOKIELIST, &list);
    unlockCurlSharedResource(CURL_LOCK_DATA_COOKIE);
    curl_easy_cleanup(handle);

    // curl cookie format (netscape/mozilla cookie file format)
    // 7 tab separated fields in the following specific order:
    //
    // 0: domain    - domain that set & can subsequently read the cookie
    // 1: tailmatch - TRUE or FALSE, if machines under that domain can read the cookie
    // 2: path      - root path under domain where the cookie is valid. If '/', cookie is valid for entire domain.
    // 3: secure    - TRUE or FALSE, if a secure connection (HTTPS) is required to read the cookie.
    // 4: expires   - Unix Time in seconds when the cookie is set to expire.
    // 5: name      - name of the value that the cookie is storing/saving.
    // 6: value     - value of the cookie

    double now = currentTime();
    // iterate through the cookie list to look for a match
    for (struct curl_slist *cur = list; cur; cur = cur->next) {
        String data(cur->data);
        Vector<String> bits;
        data.split("\t", bits);

        // http only cookies shouldn't go to javascript
        if (bits[0].startsWith("#HttpOnly_"))
            continue;

        // domain
        if (!bits[0].isEmpty()) {
            // tailmatch
            if (equalIgnoringCase(bits[1], "TRUE")) {
                // if (!url.host().endsWith(bits[0])) {
                // curl prepends '.' to the domain so make sure handle case where domain is http://domain.com
                if (!url.host().endsWith(bits[0]) && String(bits[0]) != String(String(".") + url.host()))
                    continue;
            } else {
                if (url.host() != bits[0])
                    continue;
            }
        }

        // path
        if (!bits[2].isEmpty()) {
            if (!url.path().startsWith(bits[2]))
                continue;
        }

        // secure
        if (equalIgnoringCase(bits[3], "TRUE")) {
            if (!(url.protocolIs("https")))
                continue;
        }

        // expires
        double exp = bits[4].toDouble();
        if (exp && exp < now)
            continue;

        // append to existing string seperated with semi-colon
        if (!cookie.isEmpty())
            cookie.append("; ");

        // sometimes name=value pair has empty value so handle accordingly
        cookie.append(bits[5] + "=" + (bits.size() == 7 ? bits[6] : ""));
    }

    // free up allocated resources
    curl_slist_free_all(list);

    DEBUG_LOG("<ResourceHandleManager::cookies getcookie= %s> \n", cookie.latin1().data());
    return cookie;
}

int ResourceHandleManager::deleteAllCookies()
{
    int ret = 0;
    CURL* handle = curl_easy_init();
    curl_easy_setopt(handle, CURLOPT_SHARE, m_curlShareHandle);

    char cookieJarFileNameBuffer[256] = {""}; 
    char* cookieJarFileName = m_cookieJarFileName;
    if (!cookieJarFileName) {
        Manx::NetworkProfile::get(Manx::NetPropsCookieFilename, cookieJarFileNameBuffer, sizeof(cookieJarFileNameBuffer));
        cookieJarFileName = cookieJarFileNameBuffer;
    }

#if USE_SCRACHPAD_COOKIEJAR
    char* scrachpadFileName = 0;
    if (scrachpadmatch(cookieJarFileName)) {
        scrachpadFileName = makescrachpadpath(cookieJarFileName);
        cookieJarFileName = scrachpadFileName;
        clearscrachpad();
    }
#endif

    DEBUG_LOG("<ResourceHandleManager::deleteAllCookies() FileName=%s> \n", cookieJarFileName);
    ret = remove(cookieJarFileName);
    curl_easy_setopt(handle, CURLOPT_COOKIELIST, "ALL");
    curl_easy_cleanup(handle);

#if USE_SCRACHPAD_COOKIEJAR
    freescrachpadpath(scrachpadFileName);
#endif
    return ret;
}

void ResourceHandleManager::cleanupSessionCookies()
{
    if (!Manx::NetworkProfile::cookieEnabled()) 
        return;

    CURL* handle = curl_easy_init();

#if DEBUG_CURL
        curl_easy_setopt(handle, CURLOPT_VERBOSE, 1);
#endif

    curl_easy_setopt(handle, CURLOPT_SHARE, m_curlShareHandle);

    char cookieJarFileName[256] = {""}; 
    Manx::NetworkProfile::get(Manx::NetPropsCookieFilename, cookieJarFileName, sizeof(cookieJarFileName));
    setCookieJarFileName(cookieJarFileName);
    
    curl_easy_setopt(handle, CURLOPT_COOKIEFILE, m_cookieJarFileName);
    curl_easy_setopt(handle, CURLOPT_COOKIEJAR, m_cookieJarFileName);

    if (getenv("WEBKIT_NET_SKIP_CLEAR_SESSCOOKIE")) {
        curl_easy_setopt(handle, CURLOPT_COOKIELIST, "FLUSH");
        curl_easy_setopt(handle, CURLOPT_COOKIELIST, "SESS");
        remove(m_cookieJarFileName);
        
        curl_easy_cleanup(handle);
        LOG(Loading, "<cleanup SessionCookies from CookieJarFile>\n");
    } else {
        curl_easy_cleanup(handle);
        DEBUG_LOG("<Skip cleanupSessionCookies...>\n");
    }
}

void ResourceHandleManager::cancel(ResourceHandle* job)
{
    DEBUG_LOG("ResourceHandleManager::cancel url = [%s]\n", job->firstRequest().url().string().latin1().data());

    if (removeScheduledJob(job))
        return;

#ifdef EXPERIMENTAL_SELECT_IMPROVEMENT_FOR_ASYNCHRONOUS_COMMUNICATION
    s_workerThreadMultiMutex->lock();
    ResourceHandleContext* context = s_jobToContextMap.get(job);
    if (context) {
        s_cancelContextList.append(context);
        DEBUG_LOG("-> RHM::cancel wakeupSelect()\n");
        wakeupSelect(s_WPIPE[0]);
    }
    s_workerThreadMultiMutex->unlock();
#endif
    ResourceHandleInternal* d = job->getInternal();
    d->m_cancelled = true;

}

static CURLcode sslctxfun(CURL* curlhandle, void* sslctx, void* param)
{
    Manx::Ssl::sslctxfun(sslctx, param);
    return CURLE_OK;
}

static void replaceUrlOfMainResource(void* data)
{
    if (WebCore::s_filteringThreadCreated && Manx::WebSecurity::activatedStatus()) {
        WebCore::ResourceHandleContext* context = reinterpret_cast<WebCore::ResourceHandleContext*>(data);
        if (context) {
            if (context->m_nwcontext) {
                DEBUG_LOG("-> replaceUrlOfMainResource(context->m_kurl)\n");
                context->m_nwcontext->replaceUrlOfMainResource(context->m_kurl);
            }
        }
    } else if (Manx::NetworkProfile::proxyEnabled()) {
        WebCore::ResourceHandleContext* context = reinterpret_cast<WebCore::ResourceHandleContext*>(data);
        if (Manx::WebSecurity::isBlockPageResourceURL(context->m_url)) {
            WebCore::ResourceHandleContext* context = reinterpret_cast<WebCore::ResourceHandleContext*>(data);
            if (context) {
                if (context->m_nwcontext) {
                    DEBUG_LOG("-> replaceUrlOfMainResource(context->m_kurl):[%s]\n", context->m_kurl.string().latin1().data());
                    context->m_nwcontext->replaceUrlOfMainResource(context->m_kurl);
                }
            }
        }
    }
}

} // namespace WebCore
