/*
 * Copyright (C) 2013 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"

#if USE(ACCELERATED_COMPOSITING) && USE(TEXTURE_MAPPER) && USE(TEXTURE_MAPPER_GL) && ENABLE(VIDEO)
#include "VideoLayerManx.h"

#include "NotImplemented.h"
#include "OpenGLShims.h"
#include "TextureMapperGL.h"

#define GL_CMD(...) do { __VA_ARGS__; ASSERT_ARG(__VA_ARGS__, !glGetError()); } while (0)

namespace WebCore {

#define INVALID_CANVAS_HANDLE_VALUE 0xFFFFFFFFU

VideoLayerManx::VideoLayerManx()
    : m_isCanvasMode(false)
    , m_canvasHandle(INVALID_CANVAS_HANDLE_VALUE)
    , m_texture(0)
    , m_needsUpdateTexture(false)
{
}

VideoLayerManx::~VideoLayerManx()
{
    deleteTexture();
}

void VideoLayerManx::deleteTexture()
{
    if (m_texture)
        GL_CMD(glDeleteTextures(1, &m_texture));
    m_texture = 0;
}

void VideoLayerManx::updateCanvasHandle(bool isCanvasMode, uint32_t canvasHandle)
{
    if (isCanvasMode == m_isCanvasMode && canvasHandle == m_canvasHandle)
        return;

    if (isCanvasMode && canvasHandle != INVALID_CANVAS_HANDLE_VALUE) {
        m_isCanvasMode = true;
        m_canvasHandle = canvasHandle;
    } else {
        m_isCanvasMode = false;
        m_canvasHandle = INVALID_CANVAS_HANDLE_VALUE;
    }
    m_needsUpdateTexture = true;
}

void VideoLayerManx::updateTexture()
{
#if ENABLE(MANX_CANVAS_COMPOSITING)
    if (!m_needsUpdateTexture)
        return;

    if (m_texture && !m_isCanvasMode) {
        ASSERT(m_canvasHandle == INVALID_CANVAS_HANDLE_VALUE);
        deleteTexture();
        return;
    }

    ASSERT(m_canvasHandle != INVALID_CANVAS_HANDLE_VALUE);
    if (!m_texture)
        GL_CMD(glGenTextures(1, &m_texture));

    GL_CMD(glBindTexture(GL_TEXTURE_2D, m_texture));
    GL_CMD(glTexImageCanvas2D(GL_TEXTURE_2D, m_canvasHandle));
    GL_CMD(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST));
    GL_CMD(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST));
    GL_CMD(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE));
    GL_CMD(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE));
    GL_CMD(glBindTexture(GL_TEXTURE_2D, 0));
#endif
}

void VideoLayerManx::paintToTextureMapper(TextureMapper* textureMapper, const FloatRect& targetRect, const TransformationMatrix& matrix, float opacity, BitmapTexture* mask)
{
    if (textureMapper->accelerationMode() != TextureMapper::OpenGLMode) {
        notImplemented();
        return;
    }

    if (!isCanvasMode())
        return;

    updateTexture();

    TextureMapperGL* texmapGL = static_cast<TextureMapperGL*>(textureMapper);
    TextureMapperGL::Flags flags = 0; // TextureMapperGL::ShouldFlipTexture;

    IntSize textureSize = IntSize(); // textureSize parameter of drawTexture() is not used.

#if ENABLE(MANX_INDIRECT_COMPOSITING)
    texmapGL->drawIndirectTexture(m_canvasHandle, flags, textureSize, targetRect, matrix, opacity);
#else
    texmapGL->drawTexture(m_texture, flags, textureSize, targetRect, matrix, opacity, mask);
#endif
}

void VideoLayerManx::swapBuffers()
{
    printf("%s\n", __FUNCTION__);
}

}
#endif
