/*
 * Copyright (C) 2010 Apple Inc. All rights reserved.
 * Copyright (C) 2013 Sony Computer Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"

#if USE(REQUEST_ANIMATION_FRAME_DISPLAY_MONITOR)

#include "DisplayRefreshMonitor.h"

#include <wtf/CurrentTime.h>
#include <wtf/MainThread.h>

namespace WebCore {

DisplayRefreshMonitor::~DisplayRefreshMonitor()
{
    m_active = false;
    if (m_timerThread)
        waitForThreadCompletion(m_timerThread);

    delete m_monitor;

    cancelCallOnMainThread(DisplayRefreshMonitor::refreshDisplayOnMainThread, this);
}

void DisplayRefreshMonitor::timerThreadFunc(void* data)
{
    DisplayRefreshMonitor* that = (DisplayRefreshMonitor*)data;
    volatile bool* active = &that->m_active;

    Manx::DisplayRefreshMonitor* monitor = that->m_monitor;
    if (!monitor)
        return;

    monitor->start();
    while (*active) {
            monitor->waitVblank();
        FUNCTION_MARKER;
        that->displayLinkFired();
    }
    monitor->stop();
}

bool DisplayRefreshMonitor::requestRefreshCallback()
{
    if (!m_active)
        return false;

    if (!m_monitor)
        m_monitor = new Manx::DisplayRefreshMonitor;

    if (!m_monitor) {
        m_active = false;
        return false;
    }

    if (!m_timerThread) {
        m_active = true;
        m_timerThread = createThread(timerThreadFunc, this, "SceDisplayRefreshMonitor");
        if (!m_timerThread) {
            delete m_monitor;
            m_monitor = 0;
            m_active = false;
            return false;
        }
    }
    
    MutexLocker lock(m_mutex);
    m_scheduled = true;
    return true;
}

void DisplayRefreshMonitor::displayLinkFired()
{
    MutexLocker lock(m_mutex);
    if (!m_scheduled || !m_previousFrameDone)
        return;

    m_previousFrameDone = false;

    m_timestamp = currentTime();
    
    callOnMainThread(refreshDisplayOnMainThread, this);
}

}

#endif // USE(REQUEST_ANIMATION_FRAME_DISPLAY_MONITOR)
