///////////////////////////////////////////////////////////////////////////////
// Copyright 2012 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef URL_h
#define URL_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"
#include "ceurl.h"
#include "ICEUString.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------


//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEURL ICEURL
 * @{
 */

/*!
 * ID of ICEURL
 */
#define CEComIID_ICEURL 0xa3f1fb44

/*!
 * ICEURL
 */
typedef struct ICEURL
{
	const struct ICEURL_vtbl* _vtbl;
} ICEURL;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iURL	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEURL_queryInterface) (ICEURL* iURL, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEURL_addRef) (ICEURL* iURL);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEURL_release) (ICEURL* iURL);

typedef CEHResult (*iCEURL_init) (ICEURL* iURL, struct ICEURL* base, const char* uri);

typedef CEHResult (*iCEURL_clone) (ICEURL* iURL, const struct ICEURL* other);

typedef CEHResult (*iCEURL_parse) (ICEURL* iURL, const struct ICEURL* base, struct ICEUString* url);

typedef CEHResult (*iCEURL_equals) (ICEURL* iURL, const struct ICEURL* other, CEURIProperty2 name, bool ignoreCase, bool* equals);

typedef CEHResult (*iCEURL_hasHttpQuery) (ICEURL* iURL, bool* val);

typedef CEHResult (*iCEURL_hasHttpFragment) (ICEURL* iURL, bool* val);

typedef CEHResult (*iCEURL_hasWellKnownScheme) (ICEURL* iURL, bool* val);

typedef CEHResult (*iCEURL_hasSupportedScheme) (ICEURL* iURL, bool* val);

typedef CEHResult (*iCEURL_hasUserDefinedScheme) (ICEURL* iURL, bool* val);

typedef CEHResult (*iCEURL_getScheme) (ICEURL* iURL, CEURIScheme* scheme);

typedef CEHResult (*iCEURL_getPropertyAsString2) (ICEURL* iURL, CEURIProperty2 name, void* value, CEURI_STRING_CONVERTER2 converter);

typedef CEHResult (*iCEURL_getFileInfo) (ICEURL* iURL, CEURL_FileInfo* fileInfo);

typedef CEHResult (*iCEURL_getHashcode) (ICEURL* iURL, UINT32 *const hashcodeOut);

typedef CEHResult (*iCEURL_setHttpProperty) (ICEURL* iURL, CEURIProperty2 name, struct ICEUString* value);

typedef CEHResult (*iCEURL_toImmutable) (ICEURL* iURL);

typedef CEHResult (*iCEURL_isImmutable) (ICEURL* iURL, bool* val);

typedef CEHResult (*iCEURL_dbgToString) (ICEURL* iURL, const UTF16CHAR* *const pStr16Out);

/*!
 * V-table of ICEURL
 */
struct ICEURL_vtbl
{
	iCEURL_queryInterface	_queryInterface;	//!< Query interface.
	iCEURL_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEURL_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEURL_init	_init;
	iCEURL_clone	_clone;
	iCEURL_parse	_parse;
	iCEURL_equals	_equals;
	iCEURL_hasHttpQuery	_hasHttpQuery;
	iCEURL_hasHttpFragment	_hasHttpFragment;
	iCEURL_hasWellKnownScheme	_hasWellKnownScheme;
	iCEURL_hasSupportedScheme	_hasSupportedScheme;
	iCEURL_hasUserDefinedScheme	_hasUserDefinedScheme;
	iCEURL_getScheme	_getScheme;
	iCEURL_getPropertyAsString2	_getPropertyAsString2;
	iCEURL_getFileInfo	_getFileInfo;
	iCEURL_getHashcode	_getHashcode;
	iCEURL_setHttpProperty	_setHttpProperty;
	iCEURL_toImmutable	_toImmutable;
	iCEURL_isImmutable	_isImmutable;
	iCEURL_dbgToString	_dbgToString;
};

/*! @}
 * end of ICEURL
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEURLRef CEComICEURLRef
 * @{
 */

class CEComICEURLRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEURLRef() : _iURL(0)
	{
#if defined(_DEBUG) || !defined(NDEBUG)
		dbgToString();
#endif //_DEBUG || !NDEBUG
	}
	CEComICEURLRef(ICEURL* iOther) : _iURL(0)
	{
		if (iOther)
		{
			_iURL = iOther;
			_iURL->_vtbl->_addRef(_iURL);
		}
	}

	CEComICEURLRef(const CEComICEURLRef& other) : _iURL(0)
	{
		if (other._iURL)
		{
			_iURL = other._iURL;
			_iURL->_vtbl->_addRef(_iURL);
		}
	}

	~CEComICEURLRef()
	{
		if (_iURL)
		{
			ICEURL* tmp = _iURL;
			_iURL = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEURL* iURL;
				void* _ptr;
			} uIntf;
			uIntf.iURL = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEURL, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iURL)
				{
					ICEURL* tmp = _iURL;
					_iURL = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iURL = uIntf.iURL;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEURL* object() const	{ return _iURL; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEURL* iOther)
	{
		if (_iURL)
		{
			ICEURL* tmp = _iURL;
			_iURL = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iURL = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEURL* detach()
	{
		ICEURL* iIntf = _iURL;
		_iURL = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEURL* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iURL;
		if (_iURL)
		{
			_iURL->_vtbl->_addRef(_iURL);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEURL*() const	{ return _iURL; }
	FORCEINLINE_WITHOUT_DEBUG ICEURL& operator*() const	{ return *_iURL; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iURL == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEURL* iOther) const	{ return (_iURL != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEURL* iOther) const	{ return (_iURL == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEURLRef& operator=(const CEComICEURLRef& other)	{ return operator=(other._iURL); }

	CEComICEURLRef& operator=(const ICEURL* iOther)
	{
		if (_iURL != iOther)
		{
			if (_iURL)
			{
				ICEURL* tmp = _iURL;
				_iURL = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iURL = const_cast<ICEURL*>(iOther);
			if (_iURL)
			{
				_iURL->_vtbl->_addRef(_iURL);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEURL** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEURLRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iURL && "has a com object reference. clear first.");
		return & _iURL; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEURLRef& other)
	{
		bool result = false;
		if (_iURL)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iURL);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iURL ? _iURL->_vtbl->_queryInterface(_iURL, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iURL) { _iURL->_vtbl->_addRef(_iURL); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iURL) { _iURL->_vtbl->_release(_iURL); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult init(struct ICEURL* base, const char* uri)	{ return _iURL ? _iURL->_vtbl->_init(_iURL, base, uri) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult clone(const struct ICEURL* other)	{ return _iURL ? _iURL->_vtbl->_clone(_iURL, other) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult parse(const struct ICEURL* base, struct ICEUString* url)	{ return _iURL ? _iURL->_vtbl->_parse(_iURL, base, url) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult equals(const struct ICEURL* other, CEURIProperty2 name, bool ignoreCase, bool* equals)	{ return _iURL ? _iURL->_vtbl->_equals(_iURL, other, name, ignoreCase, equals) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult hasHttpQuery(bool* val)	{ return _iURL ? _iURL->_vtbl->_hasHttpQuery(_iURL, val) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult hasHttpFragment(bool* val)	{ return _iURL ? _iURL->_vtbl->_hasHttpFragment(_iURL, val) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult hasWellKnownScheme(bool* val)	{ return _iURL ? _iURL->_vtbl->_hasWellKnownScheme(_iURL, val) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult hasSupportedScheme(bool* val)	{ return _iURL ? _iURL->_vtbl->_hasSupportedScheme(_iURL, val) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult hasUserDefinedScheme(bool* val)	{ return _iURL ? _iURL->_vtbl->_hasUserDefinedScheme(_iURL, val) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getScheme(CEURIScheme* scheme)	{ return _iURL ? _iURL->_vtbl->_getScheme(_iURL, scheme) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getPropertyAsString2(CEURIProperty2 name, void* value, CEURI_STRING_CONVERTER2 converter)	{ return _iURL ? _iURL->_vtbl->_getPropertyAsString2(_iURL, name, value, converter) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFileInfo(CEURL_FileInfo* fileInfo)	{ return _iURL ? _iURL->_vtbl->_getFileInfo(_iURL, fileInfo) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getHashcode(UINT32 *const hashcodeOut)	{ return _iURL ? _iURL->_vtbl->_getHashcode(_iURL, hashcodeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setHttpProperty(CEURIProperty2 name, struct ICEUString* value)	{ return _iURL ? _iURL->_vtbl->_setHttpProperty(_iURL, name, value) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult toImmutable()	{ return _iURL ? _iURL->_vtbl->_toImmutable(_iURL) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isImmutable(bool* val)	{ return _iURL ? _iURL->_vtbl->_isImmutable(_iURL, val) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult dbgToString(const UTF16CHAR* *const pStr16Out)	{ return _iURL ? _iURL->_vtbl->_dbgToString(_iURL, pStr16Out) : CE_SILK_ERR_UNINITIALIZED; }

#if defined(_DEBUG) || !defined(NDEBUG)
	const UTF16CHAR* dbgToString() { const UTF16CHAR* ret=0; if (_iURL) { dbgToString(&ret); } return ret; }
#endif // _DEBUG || !NDEBUG


private:
	ICEURL* _iURL;
};

/*! @}
 * end of CEComICEURLRef
 */

#endif // __cplusplus

#endif
