///////////////////////////////////////////////////////////////////////////////
// Copyright 2012 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICEVG_h
#define ICEVG_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"
#include "CEUITypes.h"
#include "CEDXMathTypes.h"
#include "CEPixelFormat.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------

struct ICEVGPathEnumerator;
struct ICEVGContext;
struct ICEVGPattern;
struct ICEPFGlyphLoader;
struct ICEDXContext2D;
struct CEPFGlyphBitmapInfo;
struct ICEUIFontGlyphBuffer;

//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

typedef enum
{
	eCEVGDrawMode_FILL,
	eCEVGDrawMode_STROKE
} eCEVGDrawMode;

typedef enum
{
	eCEVGAntialias_DEFAULT,
	eCEVGAntialias_NONE,
	eCEVGAntialias_GRAY,
	eCEVGAntialias_SUBPIXEL
} eCEVGAntialias;

typedef enum
{
	eCEVGOperator_CLEAR,

    eCEVGOperator_SOURCE,
    eCEVGOperator_OVER,
    eCEVGOperator_IN,
    eCEVGOperator_OUT,
    eCEVGOperator_ATOP,

    eCEVGOperator_DEST,
    eCEVGOperator_DEST_OVER,
    eCEVGOperator_DEST_IN,
    eCEVGOperator_DEST_OUT,
    eCEVGOperator_DEST_ATOP,

    eCEVGOperator_XOR,
    eCEVGOperator_ADD,
    eCEVGOperator_SATURATE
} eCEVGOperator;

typedef enum
{
	eCEVGFillRule_WINDING,
	eCEVGFillRule_EVENODD
} eCEVGFillRule;

typedef enum
{
	eCEVGPathDataType_MOVE_TO,
	eCEVGPathDataType_LINE_TO,
	eCEVGPathDataType_CURVE_TO,
	eCEVGPathDataType_CLOSE_PATH
} eCEVGPathDataType;

typedef enum
{
    eCEVGLineCap_BUTT,
	eCEVGLineCap_ROUND,
    eCEVGLineCap_SQUARE
} eCEVGLineCap;

typedef enum
{
	eCEVGLineJoin_MITER,
    eCEVGLineJoin_ROUND,
    eCEVGLineJoin_BEVEL
} eCEVGLineJoin;

typedef enum
{
	eCEVGFilter_FAST,
    eCEVGFilter_GOOD,
    eCEVGFilter_BEST,
    eCEVGFilter_NEAREST,
    eCEVGFilter_BILINEAR,
    eCEVGFilter_GAUSSIAN
} eCEVGFilter;

typedef enum
{
	eCEVGExtend_NONE,
    eCEVGExtend_REPEAT,
    eCEVGExtend_REFLECT,
    eCEVGExtend_PAD
} eCEVGExtend;

typedef enum
{
	eCEVGSurfaceLoadStatus_INIT,		// data is empty
	eCEVGSurfaceLoadStatus_LOADING,		// data is now loading
	eCEVGSurfaceLoadStatus_COMPLETED	// data ls completed
} eCEVGSurfaceLoadStatus;

typedef enum
{
	eCEVGSurfaceType_Image,
	eCEVGSurfaceType_OffScreen
} eCEVGSurfaceType;

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

typedef struct CEVGSurfaceInitParam
{
	CEPixelFormat	pixelFormat;	// pixel format of offscreen mode.
} CEVGSurfaceInitParam;

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEContext2D ICEContext2D
 * @{
 */

/*!
 * ID of ICEContext2D
 */
#define CEComIID_ICEContext2D 0xe4863b85


/*!
 * ICEContext2D
 * deprecated.
 */
typedef struct ICEContext2D
{
	const struct ICEContext2D_vtbl* _vtbl;
} ICEContext2D;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iContext2D	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEContext2D_queryInterface) (ICEContext2D* iContext2D, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEContext2D_addRef) (ICEContext2D* iContext2D);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEContext2D_release) (ICEContext2D* iContext2D);

typedef CEHResult (*iCEContext2D_getDXContext2D) (ICEContext2D* iContext2D, struct ICEDXContext2D* *const c2dOut);

/*!
 * V-table of ICEContext2D
 */
struct ICEContext2D_vtbl
{
	iCEContext2D_queryInterface	_queryInterface;	//!< Query interface.
	iCEContext2D_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEContext2D_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEContext2D_getDXContext2D	_getDXContext2D;
};

/*! @}
 * end of ICEContext2D
 */

/*! \defgroup ICEVG ICEVG
 * @{
 */

/*!
 * ID of ICEVG
 */
#define CEComIID_ICEVG 0x02d0cad9

/*!
 * ICEVG
 */
typedef struct ICEVG
{
	const struct ICEVG_vtbl* _vtbl;
} ICEVG;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iVG	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEVG_queryInterface) (ICEVG* iVG, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVG_addRef) (ICEVG* iVG);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVG_release) (ICEVG* iVG);

	//! Changes matrix to be the inverse of it's original value. Not all transformation 
	//! matrices have inverses; if the matrix collapses points together (it is degenerate), 
	//! then it has no inverse and this function will fail.

typedef CEHResult (*iCEVG_invertMatrix2DAffine) (ICEVG* iVG, CEDXMatrix2DAffineD* const matrix2DAffineRef);

typedef CEHResult (*iCEVG_multiplyMatrix2DAffine) (ICEVG* iVG, CEDXMatrix2DAffineD* const resultOut, const CEDXMatrix2DAffineD* m1, const CEDXMatrix2DAffineD* m2);

/*!
 * V-table of ICEVG
 */
struct ICEVG_vtbl
{
	iCEVG_queryInterface	_queryInterface;	//!< Query interface.
	iCEVG_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEVG_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEVG_invertMatrix2DAffine	_invertMatrix2DAffine;
	iCEVG_multiplyMatrix2DAffine	_multiplyMatrix2DAffine;
};

/*! @}
 * end of ICEVG
 */

/*! \defgroup ICEVGFactory ICEVGFactory
 * @{
 */

/*!
 * ID of ICEVGFactory
 */
#define CEComIID_ICEVGFactory 0x00bf5692

/*!
 * ICEVGFactory
 */
typedef struct ICEVGFactory
{
	const struct ICEVGFactory_vtbl* _vtbl;
} ICEVGFactory;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iVGFactory	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEVGFactory_queryInterface) (ICEVGFactory* iVGFactory, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGFactory_addRef) (ICEVGFactory* iVGFactory);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGFactory_release) (ICEVGFactory* iVGFactory);

	//! Changes matrix to be the inverse of it's original value. Not all transformation 
	//! matrices have inverses; if the matrix collapses points together (it is degenerate), 
	//! then it has no inverse and this function will fail.

typedef CEHResult (*iCEVGFactory_invertMatrix2DAffine) (ICEVGFactory* iVGFactory, CEDXMatrix2DAffineD* const matrix2DAffineRef);

typedef CEHResult (*iCEVGFactory_multiplyMatrix2DAffine) (ICEVGFactory* iVGFactory, CEDXMatrix2DAffineD* const resultOut, const CEDXMatrix2DAffineD* m1, const CEDXMatrix2DAffineD* m2);

	//! create ICEVGContext of path data.
	//! \param[out] piPathOut      pointer to the variable to receive the created ICEVGContext.

typedef CEHResult (*iCEVGFactory_createPath) (ICEVGFactory* iVGFactory, struct ICEVGContext* *const piPathOut);

	//! create ICEVGContext to render to the specified target
	//! \param[in] iRenderTarget  render target where the created ICEVGContext draws to.
	//! \param[out] piCtxOut      pointer to the variable to receive the created ICEVGContext.

typedef CEHResult (*iCEVGFactory_createContext) (ICEVGFactory* iVGFactory, struct ICEUnknown* iRenderTarget, struct ICEVGContext* *const piCtxOut);

	//! Create a ICEVGSurface instance.
	//! \param[in] isStereoImage     stereo image flag.
	//! \param[out] iSurfaceOut      pointer to the variable to receive the created ICEVGSurface

typedef CEHResult (*iCEVGFactory_createImageSurface) (ICEVGFactory* iVGFactory, bool isStereoImage, struct ICEVGSurface* *const iSurfaceOut);

	//! Create a ICEVGSurface instance. If the data is NULL pointer, the malloc function is called in this method. 
	//! In contrast, If the data have data,  this method creates an surface for the provided pixel data,
	//! The output buffer must be kept around until the ICEVGSurface is destoryed.
	//! \param[in] optData      surface array to set for surface
	//! \param[in] format    format of surface
	//! \param[in] width     width of surface
	//! \param[in] height    height of surface
	//! \param[out] iSurfaceOut      pointer to the variable to receive the created ICEVGSurface

typedef CEHResult (*iCEVGFactory_createOffScreenSurface) (ICEVGFactory* iVGFactory, UINT8* optData, CEPixelFormat format, UINT32 width, UINT32 height, struct ICEVGSurface* *const iSurfaceOut);

typedef CEHResult (*iCEVGFactory_createPattern) (ICEVGFactory* iVGFactory, struct ICEVGSurface* iSrcSurface, struct ICEVGPattern* *const iPatternOut);

typedef CEHResult (*iCEVGFactory_createRadialPatternF) (ICEVGFactory* iVGFactory, float cx0, float cy0, float radius0, float cx1, float cy1, float radius1, struct ICEVGPattern* *const radialPatternOut);

typedef CEHResult (*iCEVGFactory_createRGBAPatternF) (ICEVGFactory* iVGFactory, float red, float green, float blue, float alpha, struct ICEVGPattern* *const rgbaPatternOut);

typedef CEHResult (*iCEVGFactory_createLinearPatternF) (ICEVGFactory* iVGFactory, float x0, float y0, float x1, float y1, struct ICEVGPattern* *const linearPatternOut);

	//! create ICEVGContext to record to display list.
	//! \param[out] piCtxOut      pointer to the variable to receive the created ICEVGContext.

typedef CEHResult (*iCEVGFactory_createContextRec) (ICEVGFactory* iVGFactory, struct ICEVGContext* *const piCtxOut);

/*!
 * V-table of ICEVGFactory
 */
struct ICEVGFactory_vtbl
{
	iCEVGFactory_queryInterface	_queryInterface;	//!< Query interface.
	iCEVGFactory_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEVGFactory_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEVGFactory_invertMatrix2DAffine	_invertMatrix2DAffine;
	iCEVGFactory_multiplyMatrix2DAffine	_multiplyMatrix2DAffine;
	iCEVGFactory_createPath	_createPath;
	iCEVGFactory_createContext	_createContext;
	iCEVGFactory_createImageSurface	_createImageSurface;
	iCEVGFactory_createOffScreenSurface	_createOffScreenSurface;
	iCEVGFactory_createPattern	_createPattern;
	iCEVGFactory_createRadialPatternF	_createRadialPatternF;
	iCEVGFactory_createRGBAPatternF	_createRGBAPatternF;
	iCEVGFactory_createLinearPatternF	_createLinearPatternF;
	iCEVGFactory_createContextRec	_createContextRec;
};

/*! @}
 * end of ICEVGFactory
 */

/*! \defgroup ICEVGPathEnumerator ICEVGPathEnumerator
 * @{
 */

/*!
 * ID of ICEVGPathEnumerator
 */
#define CEComIID_ICEVGPathEnumerator 0x241d0e82

/*!
 * ICEVGPathEnumerator
 */
typedef struct ICEVGPathEnumerator
{
	const struct ICEVGPathEnumerator_vtbl* _vtbl;
} ICEVGPathEnumerator;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iVGPathEnumerator	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEVGPathEnumerator_queryInterface) (ICEVGPathEnumerator* iVGPathEnumerator, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGPathEnumerator_addRef) (ICEVGPathEnumerator* iVGPathEnumerator);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGPathEnumerator_release) (ICEVGPathEnumerator* iVGPathEnumerator);

typedef CEHResult (*iCEVGPathEnumerator_reset) (ICEVGPathEnumerator* iVGPathEnumerator);

typedef CEHResult (*iCEVGPathEnumerator_nextF) (ICEVGPathEnumerator* iVGPathEnumerator, eCEVGPathDataType *const typeOut, CEPointF* points3Out, INT_PTR *const nPointsRef);

/*!
 * V-table of ICEVGPathEnumerator
 */
struct ICEVGPathEnumerator_vtbl
{
	iCEVGPathEnumerator_queryInterface	_queryInterface;	//!< Query interface.
	iCEVGPathEnumerator_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEVGPathEnumerator_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEVGPathEnumerator_reset	_reset;
	iCEVGPathEnumerator_nextF	_nextF;
};

/*! @}
 * end of ICEVGPathEnumerator
 */

/*! \defgroup ICEVGPattern ICEVGPattern
 * @{
 */

/*!
 * ID of ICEVGPattern
 */
#define CEComIID_ICEVGPattern 0x993df760

/*!
 * ICEVGPattern
 */
typedef struct ICEVGPattern
{
	const struct ICEVGPattern_vtbl* _vtbl;
} ICEVGPattern;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iVGPattern	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEVGPattern_queryInterface) (ICEVGPattern* iVGPattern, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGPattern_addRef) (ICEVGPattern* iVGPattern);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGPattern_release) (ICEVGPattern* iVGPattern);

typedef CEHResult (*iCEVGPattern_setFilter) (ICEVGPattern* iVGPattern, eCEVGFilter filter);

typedef CEHResult (*iCEVGPattern_setMatrix2DAffine) (ICEVGPattern* iVGPattern, const CEDXMatrix2DAffineD* matrix2dAffine);

typedef CEHResult (*iCEVGPattern_setExtend) (ICEVGPattern* iVGPattern, eCEVGExtend extend);

typedef CEHResult (*iCEVGPattern_addColorStopRGBAF) (ICEVGPattern* iVGPattern, float offset, float  red, float green, float blue, float alpha);

/*!
 * V-table of ICEVGPattern
 */
struct ICEVGPattern_vtbl
{
	iCEVGPattern_queryInterface	_queryInterface;	//!< Query interface.
	iCEVGPattern_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEVGPattern_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEVGPattern_setFilter	_setFilter;
	iCEVGPattern_setMatrix2DAffine	_setMatrix2DAffine;
	iCEVGPattern_setExtend	_setExtend;
	iCEVGPattern_addColorStopRGBAF	_addColorStopRGBAF;
};

/*! @}
 * end of ICEVGPattern
 */

/*! \defgroup ICEVGSurfaceWriteFunction ICEVGSurfaceWriteFunction
 * @{
 */

/*!
 * ID of ICEVGSurfaceWriteFunction
 */
#define CEComIID_ICEVGSurfaceWriteFunction 0xb38f7124

/*!
 * ICEVGSurfaceWriteFunction
 */
typedef struct ICEVGSurfaceWriteFunction
{
	const struct ICEVGSurfaceWriteFunction_vtbl* _vtbl;
} ICEVGSurfaceWriteFunction;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iVGSurfaceWriteFunction	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEVGSurfaceWriteFunction_queryInterface) (ICEVGSurfaceWriteFunction* iVGSurfaceWriteFunction, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGSurfaceWriteFunction_addRef) (ICEVGSurfaceWriteFunction* iVGSurfaceWriteFunction);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGSurfaceWriteFunction_release) (ICEVGSurfaceWriteFunction* iVGSurfaceWriteFunction);

typedef CEHResult (*iCEVGSurfaceWriteFunction_write) (ICEVGSurfaceWriteFunction* iVGSurfaceWriteFunction, struct ICEUnknown* closure, const UCHAR8* data, UINT32 length);

/*!
 * V-table of ICEVGSurfaceWriteFunction
 */
struct ICEVGSurfaceWriteFunction_vtbl
{
	iCEVGSurfaceWriteFunction_queryInterface	_queryInterface;	//!< Query interface.
	iCEVGSurfaceWriteFunction_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEVGSurfaceWriteFunction_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEVGSurfaceWriteFunction_write	_write;
};

/*! @}
 * end of ICEVGSurfaceWriteFunction
 */

/*! \defgroup ICEVGSurface ICEVGSurface
 * @{
 */

/*!
 * ID of ICEVGSurface
 */
#define CEComIID_ICEVGSurface 0x7edb00af

/*!
 * ICEVGSurface
 */
typedef struct ICEVGSurface
{
	const struct ICEVGSurface_vtbl* _vtbl;
} ICEVGSurface;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iVGSurface	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEVGSurface_queryInterface) (ICEVGSurface* iVGSurface, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGSurface_addRef) (ICEVGSurface* iVGSurface);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGSurface_release) (ICEVGSurface* iVGSurface);

	//! Initialize VGSurface
	//! If create this interface, It is necessary to call this method.
	//! if offscreen mode, need pixel format, width and height.

typedef CEHResult (*iCEVGSurface_initialize) (ICEVGSurface* iVGSurface, const CEVGSurfaceInitParam* param);

	//! Get Image data.
	//! \param[out] strideOut  The stride of Image data
	//! \param[out] dataOut    The Image data.

typedef CEHResult (*iCEVGSurface_getImageDataRef) (ICEVGSurface* iVGSurface, INT_PTR *const strideOut, UCHAR8* *const dataOut);

typedef CEHResult (*iCEVGSurface_getDimension) (ICEVGSurface* iVGSurface, CEDim *const dimOut);

	//! creates clone of this surface.

typedef CEHResult (*iCEVGSurface_clone) (ICEVGSurface* iVGSurface, struct ICEVGSurface* *const newSurfaceOut);

typedef CEHResult (*iCEVGSurface_makeDirtyRectangleI) (ICEVGSurface* iVGSurface, int x, int y, int width, int height);

typedef CEHResult (*iCEVGSurface_clean) (ICEVGSurface* iVGSurface);

typedef CEHResult (*iCEVGSurface_getType) (ICEVGSurface* iVGSurface, eCEVGSurfaceType* typeOut);

/*!
 * V-table of ICEVGSurface
 */
struct ICEVGSurface_vtbl
{
	iCEVGSurface_queryInterface	_queryInterface;	//!< Query interface.
	iCEVGSurface_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEVGSurface_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEVGSurface_initialize	_initialize;
	iCEVGSurface_getImageDataRef	_getImageDataRef;
	iCEVGSurface_getDimension	_getDimension;
	iCEVGSurface_clone	_clone;
	iCEVGSurface_makeDirtyRectangleI	_makeDirtyRectangleI;
	iCEVGSurface_clean	_clean;
	iCEVGSurface_getType	_getType;
};

/*! @}
 * end of ICEVGSurface
 */

/*! \defgroup ICEVGImageSurfaceSource ICEVGImageSurfaceSource
 * @{
 */

/*!
 * ID of ICEVGImageSurfaceSource
 */
#define CEComIID_ICEVGImageSurfaceSource 0x58d9e292

/*!
 * ICEVGImageSurfaceSource
 */
typedef struct ICEVGImageSurfaceSource
{
	const struct ICEVGImageSurfaceSource_vtbl* _vtbl;
} ICEVGImageSurfaceSource;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iVGImageSurfaceSource	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEVGImageSurfaceSource_queryInterface) (ICEVGImageSurfaceSource* iVGImageSurfaceSource, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGImageSurfaceSource_addRef) (ICEVGImageSurfaceSource* iVGImageSurfaceSource);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGImageSurfaceSource_release) (ICEVGImageSurfaceSource* iVGImageSurfaceSource);

typedef CEHResult (*iCEVGImageSurfaceSource_appendData) (ICEVGImageSurfaceSource* iVGImageSurfaceSource, const UCHAR8* data, UINT32 length, bool allDataReceived);

typedef CEHResult (*iCEVGImageSurfaceSource_getStatus) (ICEVGImageSurfaceSource* iVGImageSurfaceSource, eCEVGSurfaceLoadStatus *const statusOut);

typedef CEHResult (*iCEVGImageSurfaceSource_getFrameCount) (ICEVGImageSurfaceSource* iVGImageSurfaceSource, INT32* frameCountOut);

typedef CEHResult (*iCEVGImageSurfaceSource_getLoopCount) (ICEVGImageSurfaceSource* iVGImageSurfaceSource, INT32* loopCountOut);

typedef CEHResult (*iCEVGImageSurfaceSource_getDelayTime) (ICEVGImageSurfaceSource* iVGImageSurfaceSource, INT32 index, INT32* delayTimeOut);

typedef CEHResult (*iCEVGImageSurfaceSource_setFrameIndex) (ICEVGImageSurfaceSource* iVGImageSurfaceSource, INT32 index);

/*!
 * V-table of ICEVGImageSurfaceSource
 */
struct ICEVGImageSurfaceSource_vtbl
{
	iCEVGImageSurfaceSource_queryInterface	_queryInterface;	//!< Query interface.
	iCEVGImageSurfaceSource_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEVGImageSurfaceSource_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEVGImageSurfaceSource_appendData	_appendData;
	iCEVGImageSurfaceSource_getStatus	_getStatus;
	iCEVGImageSurfaceSource_getFrameCount	_getFrameCount;
	iCEVGImageSurfaceSource_getLoopCount	_getLoopCount;
	iCEVGImageSurfaceSource_getDelayTime	_getDelayTime;
	iCEVGImageSurfaceSource_setFrameIndex	_setFrameIndex;
};

/*! @}
 * end of ICEVGImageSurfaceSource
 */

/*! \defgroup ICEVGContext ICEVGContext
 * @{
 */

/*!
 * ID of ICEVGContext
 */
#define CEComIID_ICEVGContext 0xef3fd1d2

/*!
 * ICEVGContext
 */
typedef struct ICEVGContext
{
	const struct ICEVGContext_vtbl* _vtbl;
} ICEVGContext;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iVGContext	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEVGContext_queryInterface) (ICEVGContext* iVGContext, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGContext_addRef) (ICEVGContext* iVGContext);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEVGContext_release) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_save) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_restore) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_appendPath) (ICEVGContext* iVGContext, struct ICEVGContext* iPath);

typedef CEHResult (*iCEVGContext_appendPathEn) (ICEVGContext* iVGContext, struct ICEVGPathEnumerator* iPathEn);

	//! @return CE_S_OK: path has current point. otherwise: has no current point and optXOut and optYOut will be set to 0.0.

typedef CEHResult (*iCEVGContext_getCurrentPointF) (ICEVGContext* iVGContext, float *const optXOut, float *const optYOut);

typedef CEHResult (*iCEVGContext_getPathExtentsF) (ICEVGContext* iVGContext, float *const xOut, float *const yOut, float *const widthOut, float *const heightOut);

typedef CEHResult (*iCEVGContext_getStrokeExtentsF) (ICEVGContext* iVGContext, float *const xOut, float *const yOut, float *const widthOut, float *const heightOut);

typedef CEHResult (*iCEVGContext_getFillExtentsF) (ICEVGContext* iVGContext, float *const xOut, float *const yOut, float *const widthOut, float *const heightOut);

typedef CEHResult (*iCEVGContext_getMatrix2DAffine) (ICEVGContext* iVGContext, CEDXMatrix2DAffineD* const matrix2DAffineOut);

typedef CEHResult (*iCEVGContext_transformMatrix2DAffine) (ICEVGContext* iVGContext, const CEDXMatrix2DAffineD* matrix2DAffine);

typedef CEHResult (*iCEVGContext_translateF) (ICEVGContext* iVGContext, float x, float y);

typedef CEHResult (*iCEVGContext_scaleF) (ICEVGContext* iVGContext, float x, float y);

typedef CEHResult (*iCEVGContext_rotateF) (ICEVGContext* iVGContext, float angle);

typedef CEHResult (*iCEVGContext_newPath) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_closePath) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_moveToF) (ICEVGContext* iVGContext, float x, float y);

typedef CEHResult (*iCEVGContext_lineToF) (ICEVGContext* iVGContext, float x, float y);

typedef CEHResult (*iCEVGContext_curveToF) (ICEVGContext* iVGContext, float x1, float y1, float x2, float y2, float x3, float y3);

typedef CEHResult (*iCEVGContext_addArcF) (ICEVGContext* iVGContext, float xc, float yc, float radius, float angle1, float angle2);

typedef CEHResult (*iCEVGContext_addArcNegativeF) (ICEVGContext* iVGContext, float xc, float yc, float radius, float angle1, float angle2);

typedef CEHResult (*iCEVGContext_addRectangleF) (ICEVGContext* iVGContext, float x, float y, float width, float height);

typedef CEHResult (*iCEVGContext_setFillRule) (ICEVGContext* iVGContext, eCEVGFillRule windRule);

typedef CEHResult (*iCEVGContext_getFillRule) (ICEVGContext* iVGContext, eCEVGFillRule *const windRuleOut);

typedef CEHResult (*iCEVGContext_isInFillF) (ICEVGContext* iVGContext, float x, float y, bool *const isInFillOut);

typedef CEHResult (*iCEVGContext_isInStroke) (ICEVGContext* iVGContext, float x, float y, bool *const isInStrokeOut);

typedef CEHResult (*iCEVGContext_copyPathData) (ICEVGContext* iVGContext, struct ICEVGPathEnumerator* *const iPathEnOut);

typedef CEHResult (*iCEVGContext_copyPropertiesTo) (ICEVGContext* iVGContext, struct ICEVGContext* vgcOut);

typedef CEHResult (*iCEVGContext_getTargetSurface) (ICEVGContext* iVGContext, struct ICEVGSurface* *const iSurfaceOut);

typedef CEHResult (*iCEVGContext_setSourceRGBA) (ICEVGContext* iVGContext, const CERGBAColor* rgbaColor);

typedef CEHResult (*iCEVGContext_setSourceRGBAF) (ICEVGContext* iVGContext, float r, float g, float b, float a);

typedef CEHResult (*iCEVGContext_setSourcePattern) (ICEVGContext* iVGContext, struct ICEVGPattern* iPattern);

typedef CEHResult (*iCEVGContext_setSourceSurfaceF) (ICEVGContext* iVGContext, struct ICEVGSurface* iSurface, float x, float y);

typedef CEHResult (*iCEVGContext_maskSurfaceF) (ICEVGContext* iVGContext, struct ICEVGSurface* iSurface, float surfaceX, float surfaceY);

typedef CEHResult (*iCEVGContext_fill) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_fillPreserve) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_stroke) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_strokePreserve) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_paint) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_pushGroup) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_popGroupToSource) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_paintWidthAlphaF) (ICEVGContext* iVGContext, float alpha);

typedef CEHResult (*iCEVGContext_userToDeviceF) (ICEVGContext* iVGContext, float *const xRef, float *const yRef);

typedef CEHResult (*iCEVGContext_deviceToUserF) (ICEVGContext* iVGContext, float *const xRef, float *const yRef);

typedef CEHResult (*iCEVGContext_userToDeviceDistanceF) (ICEVGContext* iVGContext, float *const xRef, float *const yRef);

typedef CEHResult (*iCEVGContext_deviceToUserDistanceF) (ICEVGContext* iVGContext, float *const xRef, float *const yRef);

typedef CEHResult (*iCEVGContext_setLineCap) (ICEVGContext* iVGContext, eCEVGLineCap lineCap);

typedef CEHResult (*iCEVGContext_setLineJoin) (ICEVGContext* iVGContext, eCEVGLineJoin lineJoin);

typedef CEHResult (*iCEVGContext_setLineWidthD) (ICEVGContext* iVGContext, double w);

typedef CEHResult (*iCEVGContext_getLineWidthD) (ICEVGContext* iVGContext, double *const wOut);

typedef CEHResult (*iCEVGContext_setMiterLimitF) (ICEVGContext* iVGContext, float miter);

typedef CEHResult (*iCEVGContext_setDashD) (ICEVGContext* iVGContext, const double* dashes, UINT32 nDashes, double offset);

typedef CEHResult (*iCEVGContext_setAntialias) (ICEVGContext* iVGContext, eCEVGAntialias v);

typedef CEHResult (*iCEVGContext_setOperator) (ICEVGContext* iVGContext, eCEVGOperator v);

typedef CEHResult (*iCEVGContext_setClip) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_setClipPreserve) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_setClipRectF) (ICEVGContext* iVGContext, const CERectF* rect, eCEVGFillRule windRule);

typedef CEHResult (*iCEVGContext_getClipExtents) (ICEVGContext* iVGContext, float *const xOut, float *const yOut, float *const widthOut, float *const heightOut);

typedef CEHResult (*iCEVGContext_resetClip) (ICEVGContext* iVGContext);

typedef CEHResult (*iCEVGContext_drawGlyphsI) (ICEVGContext* iVGContext, CEPointBase* ptToDrawAt, struct ICEUIFontGlyphBuffer* igb);

typedef CEHResult (*iCEVGContext_addGlyphPathsI) (ICEVGContext* iVGContext, CEPointBase* ptToDrawAt, struct ICEUIFontGlyphBuffer* igb);

typedef CEHResult (*iCEVGContext_drawGlyphI) (ICEVGContext* iVGContext, const CEPFGlyphBitmapInfo* gbmpInfo);

	//! deprecated.

typedef CEHResult (*iCEVGContext_drawRectangleI) (ICEVGContext* iVGContext, const CERect* rect, eCEVGDrawMode mode, eCEVGOperator fillOperator);

typedef CEHResult (*iCEVGContext_writeTargetSurfaceToPNGStream) (ICEVGContext* iVGContext, struct ICEVGSurfaceWriteFunction* iFunc, struct ICEUnknown* closure);

	//! temporary method.
	//! to do:
	//! We will erase below method, when we finish to connect Mithril and Silk.  

typedef CEHResult (*iCEVGContext_beginPaint) (ICEVGContext* iVGContext, struct ICEDXImmutableRegion* iUpdateRgn);

typedef CEHResult (*iCEVGContext_endPaint) (ICEVGContext* iVGContext);

/*!
 * V-table of ICEVGContext
 */
struct ICEVGContext_vtbl
{
	iCEVGContext_queryInterface	_queryInterface;	//!< Query interface.
	iCEVGContext_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEVGContext_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEVGContext_save	_save;
	iCEVGContext_restore	_restore;
	iCEVGContext_appendPath	_appendPath;
	iCEVGContext_appendPathEn	_appendPathEn;
	iCEVGContext_getCurrentPointF	_getCurrentPointF;
	iCEVGContext_getPathExtentsF	_getPathExtentsF;
	iCEVGContext_getStrokeExtentsF	_getStrokeExtentsF;
	iCEVGContext_getFillExtentsF	_getFillExtentsF;
	iCEVGContext_getMatrix2DAffine	_getMatrix2DAffine;
	iCEVGContext_transformMatrix2DAffine	_transformMatrix2DAffine;
	iCEVGContext_translateF	_translateF;
	iCEVGContext_scaleF	_scaleF;
	iCEVGContext_rotateF	_rotateF;
	iCEVGContext_newPath	_newPath;
	iCEVGContext_closePath	_closePath;
	iCEVGContext_moveToF	_moveToF;
	iCEVGContext_lineToF	_lineToF;
	iCEVGContext_curveToF	_curveToF;
	iCEVGContext_addArcF	_addArcF;
	iCEVGContext_addArcNegativeF	_addArcNegativeF;
	iCEVGContext_addRectangleF	_addRectangleF;
	iCEVGContext_setFillRule	_setFillRule;
	iCEVGContext_getFillRule	_getFillRule;
	iCEVGContext_isInFillF	_isInFillF;
	iCEVGContext_isInStroke	_isInStroke;
	iCEVGContext_copyPathData	_copyPathData;
	iCEVGContext_copyPropertiesTo	_copyPropertiesTo;
	iCEVGContext_getTargetSurface	_getTargetSurface;
	iCEVGContext_setSourceRGBA	_setSourceRGBA;
	iCEVGContext_setSourceRGBAF	_setSourceRGBAF;
	iCEVGContext_setSourcePattern	_setSourcePattern;
	iCEVGContext_setSourceSurfaceF	_setSourceSurfaceF;
	iCEVGContext_maskSurfaceF	_maskSurfaceF;
	iCEVGContext_fill	_fill;
	iCEVGContext_fillPreserve	_fillPreserve;
	iCEVGContext_stroke	_stroke;
	iCEVGContext_strokePreserve	_strokePreserve;
	iCEVGContext_paint	_paint;
	iCEVGContext_pushGroup	_pushGroup;
	iCEVGContext_popGroupToSource	_popGroupToSource;
	iCEVGContext_paintWidthAlphaF	_paintWidthAlphaF;
	iCEVGContext_userToDeviceF	_userToDeviceF;
	iCEVGContext_deviceToUserF	_deviceToUserF;
	iCEVGContext_userToDeviceDistanceF	_userToDeviceDistanceF;
	iCEVGContext_deviceToUserDistanceF	_deviceToUserDistanceF;
	iCEVGContext_setLineCap	_setLineCap;
	iCEVGContext_setLineJoin	_setLineJoin;
	iCEVGContext_setLineWidthD	_setLineWidthD;
	iCEVGContext_getLineWidthD	_getLineWidthD;
	iCEVGContext_setMiterLimitF	_setMiterLimitF;
	iCEVGContext_setDashD	_setDashD;
	iCEVGContext_setAntialias	_setAntialias;
	iCEVGContext_setOperator	_setOperator;
	iCEVGContext_setClip	_setClip;
	iCEVGContext_setClipPreserve	_setClipPreserve;
	iCEVGContext_setClipRectF	_setClipRectF;
	iCEVGContext_getClipExtents	_getClipExtents;
	iCEVGContext_resetClip	_resetClip;
	iCEVGContext_drawGlyphsI	_drawGlyphsI;
	iCEVGContext_addGlyphPathsI	_addGlyphPathsI;
	iCEVGContext_drawGlyphI	_drawGlyphI;
	iCEVGContext_drawRectangleI	_drawRectangleI;
	iCEVGContext_writeTargetSurfaceToPNGStream	_writeTargetSurfaceToPNGStream;
	iCEVGContext_beginPaint	_beginPaint;
	iCEVGContext_endPaint	_endPaint;
};

/*! @}
 * end of ICEVGContext
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEContext2DRef CEComICEContext2DRef
 * @{
 */

class CEComICEContext2DRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEContext2DRef() : _iContext2D(0) {}
	CEComICEContext2DRef(ICEContext2D* iOther) : _iContext2D(0)
	{
		if (iOther)
		{
			_iContext2D = iOther;
			_iContext2D->_vtbl->_addRef(_iContext2D);
		}
	}

	CEComICEContext2DRef(const CEComICEContext2DRef& other) : _iContext2D(0)
	{
		if (other._iContext2D)
		{
			_iContext2D = other._iContext2D;
			_iContext2D->_vtbl->_addRef(_iContext2D);
		}
	}

	~CEComICEContext2DRef()
	{
		if (_iContext2D)
		{
			ICEContext2D* tmp = _iContext2D;
			_iContext2D = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEContext2D* iContext2D;
				void* _ptr;
			} uIntf;
			uIntf.iContext2D = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEContext2D, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iContext2D)
				{
					ICEContext2D* tmp = _iContext2D;
					_iContext2D = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iContext2D = uIntf.iContext2D;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEContext2D* object() const	{ return _iContext2D; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEContext2D* iOther)
	{
		if (_iContext2D)
		{
			ICEContext2D* tmp = _iContext2D;
			_iContext2D = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iContext2D = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEContext2D* detach()
	{
		ICEContext2D* iIntf = _iContext2D;
		_iContext2D = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEContext2D* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iContext2D;
		if (_iContext2D)
		{
			_iContext2D->_vtbl->_addRef(_iContext2D);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEContext2D*() const	{ return _iContext2D; }
	FORCEINLINE_WITHOUT_DEBUG ICEContext2D& operator*() const	{ return *_iContext2D; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iContext2D == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEContext2D* iOther) const	{ return (_iContext2D != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEContext2D* iOther) const	{ return (_iContext2D == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEContext2DRef& operator=(const CEComICEContext2DRef& other)	{ return operator=(other._iContext2D); }

	CEComICEContext2DRef& operator=(const ICEContext2D* iOther)
	{
		if (_iContext2D != iOther)
		{
			if (_iContext2D)
			{
				ICEContext2D* tmp = _iContext2D;
				_iContext2D = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iContext2D = const_cast<ICEContext2D*>(iOther);
			if (_iContext2D)
			{
				_iContext2D->_vtbl->_addRef(_iContext2D);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEContext2D** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEContext2DRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iContext2D && "has a com object reference. clear first.");
		return & _iContext2D; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEContext2DRef& other)
	{
		bool result = false;
		if (_iContext2D)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iContext2D);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iContext2D ? _iContext2D->_vtbl->_queryInterface(_iContext2D, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iContext2D) { _iContext2D->_vtbl->_addRef(_iContext2D); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iContext2D) { _iContext2D->_vtbl->_release(_iContext2D); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDXContext2D(struct ICEDXContext2D* *const c2dOut)	{ return _iContext2D ? _iContext2D->_vtbl->_getDXContext2D(_iContext2D, c2dOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEContext2D* _iContext2D;
};

/*! @}
 * end of CEComICEContext2DRef
 */

/*! \defgroup CEComICEVGRef CEComICEVGRef
 * @{
 */

class CEComICEVGRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEVGRef() : _iVG(0) {}
	CEComICEVGRef(ICEVG* iOther) : _iVG(0)
	{
		if (iOther)
		{
			_iVG = iOther;
			_iVG->_vtbl->_addRef(_iVG);
		}
	}

	CEComICEVGRef(const CEComICEVGRef& other) : _iVG(0)
	{
		if (other._iVG)
		{
			_iVG = other._iVG;
			_iVG->_vtbl->_addRef(_iVG);
		}
	}

	~CEComICEVGRef()
	{
		if (_iVG)
		{
			ICEVG* tmp = _iVG;
			_iVG = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEVG* iVG;
				void* _ptr;
			} uIntf;
			uIntf.iVG = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEVG, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iVG)
				{
					ICEVG* tmp = _iVG;
					_iVG = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iVG = uIntf.iVG;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEVG* object() const	{ return _iVG; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEVG* iOther)
	{
		if (_iVG)
		{
			ICEVG* tmp = _iVG;
			_iVG = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iVG = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEVG* detach()
	{
		ICEVG* iIntf = _iVG;
		_iVG = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEVG* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iVG;
		if (_iVG)
		{
			_iVG->_vtbl->_addRef(_iVG);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEVG*() const	{ return _iVG; }
	FORCEINLINE_WITHOUT_DEBUG ICEVG& operator*() const	{ return *_iVG; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iVG == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEVG* iOther) const	{ return (_iVG != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEVG* iOther) const	{ return (_iVG == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEVGRef& operator=(const CEComICEVGRef& other)	{ return operator=(other._iVG); }

	CEComICEVGRef& operator=(const ICEVG* iOther)
	{
		if (_iVG != iOther)
		{
			if (_iVG)
			{
				ICEVG* tmp = _iVG;
				_iVG = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iVG = const_cast<ICEVG*>(iOther);
			if (_iVG)
			{
				_iVG->_vtbl->_addRef(_iVG);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEVG** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEVGRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iVG && "has a com object reference. clear first.");
		return & _iVG; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEVGRef& other)
	{
		bool result = false;
		if (_iVG)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iVG);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iVG ? _iVG->_vtbl->_queryInterface(_iVG, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iVG) { _iVG->_vtbl->_addRef(_iVG); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iVG) { _iVG->_vtbl->_release(_iVG); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult invertMatrix2DAffine(CEDXMatrix2DAffineD* const matrix2DAffineRef)	{ return _iVG ? _iVG->_vtbl->_invertMatrix2DAffine(_iVG, matrix2DAffineRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult multiplyMatrix2DAffine(CEDXMatrix2DAffineD* const resultOut, const CEDXMatrix2DAffineD* m1, const CEDXMatrix2DAffineD* m2)	{ return _iVG ? _iVG->_vtbl->_multiplyMatrix2DAffine(_iVG, resultOut, m1, m2) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEVG* _iVG;
};

/*! @}
 * end of CEComICEVGRef
 */

/*! \defgroup CEComICEVGFactoryRef CEComICEVGFactoryRef
 * @{
 */

class CEComICEVGFactoryRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEVGFactoryRef() : _iVGFactory(0) {}
	CEComICEVGFactoryRef(ICEVGFactory* iOther) : _iVGFactory(0)
	{
		if (iOther)
		{
			_iVGFactory = iOther;
			_iVGFactory->_vtbl->_addRef(_iVGFactory);
		}
	}

	CEComICEVGFactoryRef(const CEComICEVGFactoryRef& other) : _iVGFactory(0)
	{
		if (other._iVGFactory)
		{
			_iVGFactory = other._iVGFactory;
			_iVGFactory->_vtbl->_addRef(_iVGFactory);
		}
	}

	~CEComICEVGFactoryRef()
	{
		if (_iVGFactory)
		{
			ICEVGFactory* tmp = _iVGFactory;
			_iVGFactory = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEVGFactory* iVGFactory;
				void* _ptr;
			} uIntf;
			uIntf.iVGFactory = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEVGFactory, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iVGFactory)
				{
					ICEVGFactory* tmp = _iVGFactory;
					_iVGFactory = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iVGFactory = uIntf.iVGFactory;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEVGFactory* object() const	{ return _iVGFactory; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEVGFactory* iOther)
	{
		if (_iVGFactory)
		{
			ICEVGFactory* tmp = _iVGFactory;
			_iVGFactory = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iVGFactory = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEVGFactory* detach()
	{
		ICEVGFactory* iIntf = _iVGFactory;
		_iVGFactory = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEVGFactory* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iVGFactory;
		if (_iVGFactory)
		{
			_iVGFactory->_vtbl->_addRef(_iVGFactory);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEVGFactory*() const	{ return _iVGFactory; }
	FORCEINLINE_WITHOUT_DEBUG ICEVGFactory& operator*() const	{ return *_iVGFactory; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iVGFactory == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEVGFactory* iOther) const	{ return (_iVGFactory != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEVGFactory* iOther) const	{ return (_iVGFactory == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEVGFactoryRef& operator=(const CEComICEVGFactoryRef& other)	{ return operator=(other._iVGFactory); }

	CEComICEVGFactoryRef& operator=(const ICEVGFactory* iOther)
	{
		if (_iVGFactory != iOther)
		{
			if (_iVGFactory)
			{
				ICEVGFactory* tmp = _iVGFactory;
				_iVGFactory = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iVGFactory = const_cast<ICEVGFactory*>(iOther);
			if (_iVGFactory)
			{
				_iVGFactory->_vtbl->_addRef(_iVGFactory);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEVGFactory** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEVGFactoryRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iVGFactory && "has a com object reference. clear first.");
		return & _iVGFactory; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEVGFactoryRef& other)
	{
		bool result = false;
		if (_iVGFactory)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iVGFactory);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iVGFactory ? _iVGFactory->_vtbl->_queryInterface(_iVGFactory, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iVGFactory) { _iVGFactory->_vtbl->_addRef(_iVGFactory); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iVGFactory) { _iVGFactory->_vtbl->_release(_iVGFactory); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult invertMatrix2DAffine(CEDXMatrix2DAffineD* const matrix2DAffineRef)	{ return _iVGFactory ? _iVGFactory->_vtbl->_invertMatrix2DAffine(_iVGFactory, matrix2DAffineRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult multiplyMatrix2DAffine(CEDXMatrix2DAffineD* const resultOut, const CEDXMatrix2DAffineD* m1, const CEDXMatrix2DAffineD* m2)	{ return _iVGFactory ? _iVGFactory->_vtbl->_multiplyMatrix2DAffine(_iVGFactory, resultOut, m1, m2) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createPath(struct ICEVGContext* *const piPathOut)	{ return _iVGFactory ? _iVGFactory->_vtbl->_createPath(_iVGFactory, piPathOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createContext(struct ICEUnknown* iRenderTarget, struct ICEVGContext* *const piCtxOut)	{ return _iVGFactory ? _iVGFactory->_vtbl->_createContext(_iVGFactory, iRenderTarget, piCtxOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createImageSurface(bool isStereoImage, struct ICEVGSurface* *const iSurfaceOut)	{ return _iVGFactory ? _iVGFactory->_vtbl->_createImageSurface(_iVGFactory, isStereoImage, iSurfaceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createOffScreenSurface(UINT8* optData, CEPixelFormat format, UINT32 width, UINT32 height, struct ICEVGSurface* *const iSurfaceOut)	{ return _iVGFactory ? _iVGFactory->_vtbl->_createOffScreenSurface(_iVGFactory, optData, format, width, height, iSurfaceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createPattern(struct ICEVGSurface* iSrcSurface, struct ICEVGPattern* *const iPatternOut)	{ return _iVGFactory ? _iVGFactory->_vtbl->_createPattern(_iVGFactory, iSrcSurface, iPatternOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createRadialPatternF(float cx0, float cy0, float radius0, float cx1, float cy1, float radius1, struct ICEVGPattern* *const radialPatternOut)	{ return _iVGFactory ? _iVGFactory->_vtbl->_createRadialPatternF(_iVGFactory, cx0, cy0, radius0, cx1, cy1, radius1, radialPatternOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createRGBAPatternF(float red, float green, float blue, float alpha, struct ICEVGPattern* *const rgbaPatternOut)	{ return _iVGFactory ? _iVGFactory->_vtbl->_createRGBAPatternF(_iVGFactory, red, green, blue, alpha, rgbaPatternOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createLinearPatternF(float x0, float y0, float x1, float y1, struct ICEVGPattern* *const linearPatternOut)	{ return _iVGFactory ? _iVGFactory->_vtbl->_createLinearPatternF(_iVGFactory, x0, y0, x1, y1, linearPatternOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createContextRec(struct ICEVGContext* *const piCtxOut)	{ return _iVGFactory ? _iVGFactory->_vtbl->_createContextRec(_iVGFactory, piCtxOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEVGFactory* _iVGFactory;
};

/*! @}
 * end of CEComICEVGFactoryRef
 */

/*! \defgroup CEComICEVGPathEnumeratorRef CEComICEVGPathEnumeratorRef
 * @{
 */

class CEComICEVGPathEnumeratorRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEVGPathEnumeratorRef() : _iVGPathEnumerator(0) {}
	CEComICEVGPathEnumeratorRef(ICEVGPathEnumerator* iOther) : _iVGPathEnumerator(0)
	{
		if (iOther)
		{
			_iVGPathEnumerator = iOther;
			_iVGPathEnumerator->_vtbl->_addRef(_iVGPathEnumerator);
		}
	}

	CEComICEVGPathEnumeratorRef(const CEComICEVGPathEnumeratorRef& other) : _iVGPathEnumerator(0)
	{
		if (other._iVGPathEnumerator)
		{
			_iVGPathEnumerator = other._iVGPathEnumerator;
			_iVGPathEnumerator->_vtbl->_addRef(_iVGPathEnumerator);
		}
	}

	~CEComICEVGPathEnumeratorRef()
	{
		if (_iVGPathEnumerator)
		{
			ICEVGPathEnumerator* tmp = _iVGPathEnumerator;
			_iVGPathEnumerator = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEVGPathEnumerator* iVGPathEnumerator;
				void* _ptr;
			} uIntf;
			uIntf.iVGPathEnumerator = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEVGPathEnumerator, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iVGPathEnumerator)
				{
					ICEVGPathEnumerator* tmp = _iVGPathEnumerator;
					_iVGPathEnumerator = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iVGPathEnumerator = uIntf.iVGPathEnumerator;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEVGPathEnumerator* object() const	{ return _iVGPathEnumerator; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEVGPathEnumerator* iOther)
	{
		if (_iVGPathEnumerator)
		{
			ICEVGPathEnumerator* tmp = _iVGPathEnumerator;
			_iVGPathEnumerator = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iVGPathEnumerator = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEVGPathEnumerator* detach()
	{
		ICEVGPathEnumerator* iIntf = _iVGPathEnumerator;
		_iVGPathEnumerator = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEVGPathEnumerator* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iVGPathEnumerator;
		if (_iVGPathEnumerator)
		{
			_iVGPathEnumerator->_vtbl->_addRef(_iVGPathEnumerator);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEVGPathEnumerator*() const	{ return _iVGPathEnumerator; }
	FORCEINLINE_WITHOUT_DEBUG ICEVGPathEnumerator& operator*() const	{ return *_iVGPathEnumerator; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iVGPathEnumerator == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEVGPathEnumerator* iOther) const	{ return (_iVGPathEnumerator != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEVGPathEnumerator* iOther) const	{ return (_iVGPathEnumerator == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEVGPathEnumeratorRef& operator=(const CEComICEVGPathEnumeratorRef& other)	{ return operator=(other._iVGPathEnumerator); }

	CEComICEVGPathEnumeratorRef& operator=(const ICEVGPathEnumerator* iOther)
	{
		if (_iVGPathEnumerator != iOther)
		{
			if (_iVGPathEnumerator)
			{
				ICEVGPathEnumerator* tmp = _iVGPathEnumerator;
				_iVGPathEnumerator = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iVGPathEnumerator = const_cast<ICEVGPathEnumerator*>(iOther);
			if (_iVGPathEnumerator)
			{
				_iVGPathEnumerator->_vtbl->_addRef(_iVGPathEnumerator);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEVGPathEnumerator** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEVGPathEnumeratorRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iVGPathEnumerator && "has a com object reference. clear first.");
		return & _iVGPathEnumerator; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEVGPathEnumeratorRef& other)
	{
		bool result = false;
		if (_iVGPathEnumerator)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iVGPathEnumerator);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iVGPathEnumerator ? _iVGPathEnumerator->_vtbl->_queryInterface(_iVGPathEnumerator, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iVGPathEnumerator) { _iVGPathEnumerator->_vtbl->_addRef(_iVGPathEnumerator); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iVGPathEnumerator) { _iVGPathEnumerator->_vtbl->_release(_iVGPathEnumerator); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult reset()	{ return _iVGPathEnumerator ? _iVGPathEnumerator->_vtbl->_reset(_iVGPathEnumerator) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult nextF(eCEVGPathDataType *const typeOut, CEPointF* points3Out, INT_PTR *const nPointsRef)	{ return _iVGPathEnumerator ? _iVGPathEnumerator->_vtbl->_nextF(_iVGPathEnumerator, typeOut, points3Out, nPointsRef) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEVGPathEnumerator* _iVGPathEnumerator;
};

/*! @}
 * end of CEComICEVGPathEnumeratorRef
 */

/*! \defgroup CEComICEVGPatternRef CEComICEVGPatternRef
 * @{
 */

class CEComICEVGPatternRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEVGPatternRef() : _iVGPattern(0) {}
	CEComICEVGPatternRef(ICEVGPattern* iOther) : _iVGPattern(0)
	{
		if (iOther)
		{
			_iVGPattern = iOther;
			_iVGPattern->_vtbl->_addRef(_iVGPattern);
		}
	}

	CEComICEVGPatternRef(const CEComICEVGPatternRef& other) : _iVGPattern(0)
	{
		if (other._iVGPattern)
		{
			_iVGPattern = other._iVGPattern;
			_iVGPattern->_vtbl->_addRef(_iVGPattern);
		}
	}

	~CEComICEVGPatternRef()
	{
		if (_iVGPattern)
		{
			ICEVGPattern* tmp = _iVGPattern;
			_iVGPattern = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEVGPattern* iVGPattern;
				void* _ptr;
			} uIntf;
			uIntf.iVGPattern = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEVGPattern, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iVGPattern)
				{
					ICEVGPattern* tmp = _iVGPattern;
					_iVGPattern = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iVGPattern = uIntf.iVGPattern;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEVGPattern* object() const	{ return _iVGPattern; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEVGPattern* iOther)
	{
		if (_iVGPattern)
		{
			ICEVGPattern* tmp = _iVGPattern;
			_iVGPattern = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iVGPattern = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEVGPattern* detach()
	{
		ICEVGPattern* iIntf = _iVGPattern;
		_iVGPattern = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEVGPattern* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iVGPattern;
		if (_iVGPattern)
		{
			_iVGPattern->_vtbl->_addRef(_iVGPattern);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEVGPattern*() const	{ return _iVGPattern; }
	FORCEINLINE_WITHOUT_DEBUG ICEVGPattern& operator*() const	{ return *_iVGPattern; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iVGPattern == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEVGPattern* iOther) const	{ return (_iVGPattern != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEVGPattern* iOther) const	{ return (_iVGPattern == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEVGPatternRef& operator=(const CEComICEVGPatternRef& other)	{ return operator=(other._iVGPattern); }

	CEComICEVGPatternRef& operator=(const ICEVGPattern* iOther)
	{
		if (_iVGPattern != iOther)
		{
			if (_iVGPattern)
			{
				ICEVGPattern* tmp = _iVGPattern;
				_iVGPattern = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iVGPattern = const_cast<ICEVGPattern*>(iOther);
			if (_iVGPattern)
			{
				_iVGPattern->_vtbl->_addRef(_iVGPattern);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEVGPattern** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEVGPatternRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iVGPattern && "has a com object reference. clear first.");
		return & _iVGPattern; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEVGPatternRef& other)
	{
		bool result = false;
		if (_iVGPattern)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iVGPattern);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iVGPattern ? _iVGPattern->_vtbl->_queryInterface(_iVGPattern, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iVGPattern) { _iVGPattern->_vtbl->_addRef(_iVGPattern); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iVGPattern) { _iVGPattern->_vtbl->_release(_iVGPattern); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setFilter(eCEVGFilter filter)	{ return _iVGPattern ? _iVGPattern->_vtbl->_setFilter(_iVGPattern, filter) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setMatrix2DAffine(const CEDXMatrix2DAffineD* matrix2dAffine)	{ return _iVGPattern ? _iVGPattern->_vtbl->_setMatrix2DAffine(_iVGPattern, matrix2dAffine) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setExtend(eCEVGExtend extend)	{ return _iVGPattern ? _iVGPattern->_vtbl->_setExtend(_iVGPattern, extend) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult addColorStopRGBAF(float offset, float  red, float green, float blue, float alpha)	{ return _iVGPattern ? _iVGPattern->_vtbl->_addColorStopRGBAF(_iVGPattern, offset, red, green, blue, alpha) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEVGPattern* _iVGPattern;
};

/*! @}
 * end of CEComICEVGPatternRef
 */

/*! \defgroup CEComICEVGSurfaceWriteFunctionRef CEComICEVGSurfaceWriteFunctionRef
 * @{
 */

class CEComICEVGSurfaceWriteFunctionRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEVGSurfaceWriteFunctionRef() : _iVGSurfaceWriteFunction(0) {}
	CEComICEVGSurfaceWriteFunctionRef(ICEVGSurfaceWriteFunction* iOther) : _iVGSurfaceWriteFunction(0)
	{
		if (iOther)
		{
			_iVGSurfaceWriteFunction = iOther;
			_iVGSurfaceWriteFunction->_vtbl->_addRef(_iVGSurfaceWriteFunction);
		}
	}

	CEComICEVGSurfaceWriteFunctionRef(const CEComICEVGSurfaceWriteFunctionRef& other) : _iVGSurfaceWriteFunction(0)
	{
		if (other._iVGSurfaceWriteFunction)
		{
			_iVGSurfaceWriteFunction = other._iVGSurfaceWriteFunction;
			_iVGSurfaceWriteFunction->_vtbl->_addRef(_iVGSurfaceWriteFunction);
		}
	}

	~CEComICEVGSurfaceWriteFunctionRef()
	{
		if (_iVGSurfaceWriteFunction)
		{
			ICEVGSurfaceWriteFunction* tmp = _iVGSurfaceWriteFunction;
			_iVGSurfaceWriteFunction = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEVGSurfaceWriteFunction* iVGSurfaceWriteFunction;
				void* _ptr;
			} uIntf;
			uIntf.iVGSurfaceWriteFunction = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEVGSurfaceWriteFunction, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iVGSurfaceWriteFunction)
				{
					ICEVGSurfaceWriteFunction* tmp = _iVGSurfaceWriteFunction;
					_iVGSurfaceWriteFunction = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iVGSurfaceWriteFunction = uIntf.iVGSurfaceWriteFunction;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEVGSurfaceWriteFunction* object() const	{ return _iVGSurfaceWriteFunction; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEVGSurfaceWriteFunction* iOther)
	{
		if (_iVGSurfaceWriteFunction)
		{
			ICEVGSurfaceWriteFunction* tmp = _iVGSurfaceWriteFunction;
			_iVGSurfaceWriteFunction = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iVGSurfaceWriteFunction = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEVGSurfaceWriteFunction* detach()
	{
		ICEVGSurfaceWriteFunction* iIntf = _iVGSurfaceWriteFunction;
		_iVGSurfaceWriteFunction = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEVGSurfaceWriteFunction* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iVGSurfaceWriteFunction;
		if (_iVGSurfaceWriteFunction)
		{
			_iVGSurfaceWriteFunction->_vtbl->_addRef(_iVGSurfaceWriteFunction);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEVGSurfaceWriteFunction*() const	{ return _iVGSurfaceWriteFunction; }
	FORCEINLINE_WITHOUT_DEBUG ICEVGSurfaceWriteFunction& operator*() const	{ return *_iVGSurfaceWriteFunction; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iVGSurfaceWriteFunction == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEVGSurfaceWriteFunction* iOther) const	{ return (_iVGSurfaceWriteFunction != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEVGSurfaceWriteFunction* iOther) const	{ return (_iVGSurfaceWriteFunction == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEVGSurfaceWriteFunctionRef& operator=(const CEComICEVGSurfaceWriteFunctionRef& other)	{ return operator=(other._iVGSurfaceWriteFunction); }

	CEComICEVGSurfaceWriteFunctionRef& operator=(const ICEVGSurfaceWriteFunction* iOther)
	{
		if (_iVGSurfaceWriteFunction != iOther)
		{
			if (_iVGSurfaceWriteFunction)
			{
				ICEVGSurfaceWriteFunction* tmp = _iVGSurfaceWriteFunction;
				_iVGSurfaceWriteFunction = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iVGSurfaceWriteFunction = const_cast<ICEVGSurfaceWriteFunction*>(iOther);
			if (_iVGSurfaceWriteFunction)
			{
				_iVGSurfaceWriteFunction->_vtbl->_addRef(_iVGSurfaceWriteFunction);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEVGSurfaceWriteFunction** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEVGSurfaceWriteFunctionRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iVGSurfaceWriteFunction && "has a com object reference. clear first.");
		return & _iVGSurfaceWriteFunction; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEVGSurfaceWriteFunctionRef& other)
	{
		bool result = false;
		if (_iVGSurfaceWriteFunction)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iVGSurfaceWriteFunction);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iVGSurfaceWriteFunction ? _iVGSurfaceWriteFunction->_vtbl->_queryInterface(_iVGSurfaceWriteFunction, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iVGSurfaceWriteFunction) { _iVGSurfaceWriteFunction->_vtbl->_addRef(_iVGSurfaceWriteFunction); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iVGSurfaceWriteFunction) { _iVGSurfaceWriteFunction->_vtbl->_release(_iVGSurfaceWriteFunction); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult write(struct ICEUnknown* closure, const UCHAR8* data, UINT32 length)	{ return _iVGSurfaceWriteFunction ? _iVGSurfaceWriteFunction->_vtbl->_write(_iVGSurfaceWriteFunction, closure, data, length) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEVGSurfaceWriteFunction* _iVGSurfaceWriteFunction;
};

/*! @}
 * end of CEComICEVGSurfaceWriteFunctionRef
 */

/*! \defgroup CEComICEVGSurfaceRef CEComICEVGSurfaceRef
 * @{
 */

class CEComICEVGSurfaceRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEVGSurfaceRef() : _iVGSurface(0) {}
	CEComICEVGSurfaceRef(ICEVGSurface* iOther) : _iVGSurface(0)
	{
		if (iOther)
		{
			_iVGSurface = iOther;
			_iVGSurface->_vtbl->_addRef(_iVGSurface);
		}
	}

	CEComICEVGSurfaceRef(const CEComICEVGSurfaceRef& other) : _iVGSurface(0)
	{
		if (other._iVGSurface)
		{
			_iVGSurface = other._iVGSurface;
			_iVGSurface->_vtbl->_addRef(_iVGSurface);
		}
	}

	~CEComICEVGSurfaceRef()
	{
		if (_iVGSurface)
		{
			ICEVGSurface* tmp = _iVGSurface;
			_iVGSurface = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEVGSurface* iVGSurface;
				void* _ptr;
			} uIntf;
			uIntf.iVGSurface = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEVGSurface, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iVGSurface)
				{
					ICEVGSurface* tmp = _iVGSurface;
					_iVGSurface = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iVGSurface = uIntf.iVGSurface;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEVGSurface* object() const	{ return _iVGSurface; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEVGSurface* iOther)
	{
		if (_iVGSurface)
		{
			ICEVGSurface* tmp = _iVGSurface;
			_iVGSurface = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iVGSurface = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEVGSurface* detach()
	{
		ICEVGSurface* iIntf = _iVGSurface;
		_iVGSurface = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEVGSurface* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iVGSurface;
		if (_iVGSurface)
		{
			_iVGSurface->_vtbl->_addRef(_iVGSurface);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEVGSurface*() const	{ return _iVGSurface; }
	FORCEINLINE_WITHOUT_DEBUG ICEVGSurface& operator*() const	{ return *_iVGSurface; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iVGSurface == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEVGSurface* iOther) const	{ return (_iVGSurface != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEVGSurface* iOther) const	{ return (_iVGSurface == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEVGSurfaceRef& operator=(const CEComICEVGSurfaceRef& other)	{ return operator=(other._iVGSurface); }

	CEComICEVGSurfaceRef& operator=(const ICEVGSurface* iOther)
	{
		if (_iVGSurface != iOther)
		{
			if (_iVGSurface)
			{
				ICEVGSurface* tmp = _iVGSurface;
				_iVGSurface = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iVGSurface = const_cast<ICEVGSurface*>(iOther);
			if (_iVGSurface)
			{
				_iVGSurface->_vtbl->_addRef(_iVGSurface);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEVGSurface** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEVGSurfaceRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iVGSurface && "has a com object reference. clear first.");
		return & _iVGSurface; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEVGSurfaceRef& other)
	{
		bool result = false;
		if (_iVGSurface)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iVGSurface);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iVGSurface ? _iVGSurface->_vtbl->_queryInterface(_iVGSurface, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iVGSurface) { _iVGSurface->_vtbl->_addRef(_iVGSurface); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iVGSurface) { _iVGSurface->_vtbl->_release(_iVGSurface); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initialize(const CEVGSurfaceInitParam* param)	{ return _iVGSurface ? _iVGSurface->_vtbl->_initialize(_iVGSurface, param) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getImageDataRef(INT_PTR *const strideOut, UCHAR8* *const dataOut)	{ return _iVGSurface ? _iVGSurface->_vtbl->_getImageDataRef(_iVGSurface, strideOut, dataOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDimension(CEDim *const dimOut)	{ return _iVGSurface ? _iVGSurface->_vtbl->_getDimension(_iVGSurface, dimOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult clone(struct ICEVGSurface* *const newSurfaceOut)	{ return _iVGSurface ? _iVGSurface->_vtbl->_clone(_iVGSurface, newSurfaceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult makeDirtyRectangleI(int x, int y, int width, int height)	{ return _iVGSurface ? _iVGSurface->_vtbl->_makeDirtyRectangleI(_iVGSurface, x, y, width, height) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult clean()	{ return _iVGSurface ? _iVGSurface->_vtbl->_clean(_iVGSurface) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getType(eCEVGSurfaceType* typeOut)	{ return _iVGSurface ? _iVGSurface->_vtbl->_getType(_iVGSurface, typeOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEVGSurface* _iVGSurface;
};

/*! @}
 * end of CEComICEVGSurfaceRef
 */

/*! \defgroup CEComICEVGImageSurfaceSourceRef CEComICEVGImageSurfaceSourceRef
 * @{
 */

class CEComICEVGImageSurfaceSourceRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEVGImageSurfaceSourceRef() : _iVGImageSurfaceSource(0) {}
	CEComICEVGImageSurfaceSourceRef(ICEVGImageSurfaceSource* iOther) : _iVGImageSurfaceSource(0)
	{
		if (iOther)
		{
			_iVGImageSurfaceSource = iOther;
			_iVGImageSurfaceSource->_vtbl->_addRef(_iVGImageSurfaceSource);
		}
	}

	CEComICEVGImageSurfaceSourceRef(const CEComICEVGImageSurfaceSourceRef& other) : _iVGImageSurfaceSource(0)
	{
		if (other._iVGImageSurfaceSource)
		{
			_iVGImageSurfaceSource = other._iVGImageSurfaceSource;
			_iVGImageSurfaceSource->_vtbl->_addRef(_iVGImageSurfaceSource);
		}
	}

	~CEComICEVGImageSurfaceSourceRef()
	{
		if (_iVGImageSurfaceSource)
		{
			ICEVGImageSurfaceSource* tmp = _iVGImageSurfaceSource;
			_iVGImageSurfaceSource = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEVGImageSurfaceSource* iVGImageSurfaceSource;
				void* _ptr;
			} uIntf;
			uIntf.iVGImageSurfaceSource = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEVGImageSurfaceSource, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iVGImageSurfaceSource)
				{
					ICEVGImageSurfaceSource* tmp = _iVGImageSurfaceSource;
					_iVGImageSurfaceSource = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iVGImageSurfaceSource = uIntf.iVGImageSurfaceSource;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEVGImageSurfaceSource* object() const	{ return _iVGImageSurfaceSource; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEVGImageSurfaceSource* iOther)
	{
		if (_iVGImageSurfaceSource)
		{
			ICEVGImageSurfaceSource* tmp = _iVGImageSurfaceSource;
			_iVGImageSurfaceSource = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iVGImageSurfaceSource = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEVGImageSurfaceSource* detach()
	{
		ICEVGImageSurfaceSource* iIntf = _iVGImageSurfaceSource;
		_iVGImageSurfaceSource = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEVGImageSurfaceSource* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iVGImageSurfaceSource;
		if (_iVGImageSurfaceSource)
		{
			_iVGImageSurfaceSource->_vtbl->_addRef(_iVGImageSurfaceSource);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEVGImageSurfaceSource*() const	{ return _iVGImageSurfaceSource; }
	FORCEINLINE_WITHOUT_DEBUG ICEVGImageSurfaceSource& operator*() const	{ return *_iVGImageSurfaceSource; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iVGImageSurfaceSource == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEVGImageSurfaceSource* iOther) const	{ return (_iVGImageSurfaceSource != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEVGImageSurfaceSource* iOther) const	{ return (_iVGImageSurfaceSource == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEVGImageSurfaceSourceRef& operator=(const CEComICEVGImageSurfaceSourceRef& other)	{ return operator=(other._iVGImageSurfaceSource); }

	CEComICEVGImageSurfaceSourceRef& operator=(const ICEVGImageSurfaceSource* iOther)
	{
		if (_iVGImageSurfaceSource != iOther)
		{
			if (_iVGImageSurfaceSource)
			{
				ICEVGImageSurfaceSource* tmp = _iVGImageSurfaceSource;
				_iVGImageSurfaceSource = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iVGImageSurfaceSource = const_cast<ICEVGImageSurfaceSource*>(iOther);
			if (_iVGImageSurfaceSource)
			{
				_iVGImageSurfaceSource->_vtbl->_addRef(_iVGImageSurfaceSource);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEVGImageSurfaceSource** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEVGImageSurfaceSourceRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iVGImageSurfaceSource && "has a com object reference. clear first.");
		return & _iVGImageSurfaceSource; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEVGImageSurfaceSourceRef& other)
	{
		bool result = false;
		if (_iVGImageSurfaceSource)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iVGImageSurfaceSource);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iVGImageSurfaceSource ? _iVGImageSurfaceSource->_vtbl->_queryInterface(_iVGImageSurfaceSource, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iVGImageSurfaceSource) { _iVGImageSurfaceSource->_vtbl->_addRef(_iVGImageSurfaceSource); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iVGImageSurfaceSource) { _iVGImageSurfaceSource->_vtbl->_release(_iVGImageSurfaceSource); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult appendData(const UCHAR8* data, UINT32 length, bool allDataReceived)	{ return _iVGImageSurfaceSource ? _iVGImageSurfaceSource->_vtbl->_appendData(_iVGImageSurfaceSource, data, length, allDataReceived) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStatus(eCEVGSurfaceLoadStatus *const statusOut)	{ return _iVGImageSurfaceSource ? _iVGImageSurfaceSource->_vtbl->_getStatus(_iVGImageSurfaceSource, statusOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFrameCount(INT32* frameCountOut)	{ return _iVGImageSurfaceSource ? _iVGImageSurfaceSource->_vtbl->_getFrameCount(_iVGImageSurfaceSource, frameCountOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLoopCount(INT32* loopCountOut)	{ return _iVGImageSurfaceSource ? _iVGImageSurfaceSource->_vtbl->_getLoopCount(_iVGImageSurfaceSource, loopCountOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDelayTime(INT32 index, INT32* delayTimeOut)	{ return _iVGImageSurfaceSource ? _iVGImageSurfaceSource->_vtbl->_getDelayTime(_iVGImageSurfaceSource, index, delayTimeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setFrameIndex(INT32 index)	{ return _iVGImageSurfaceSource ? _iVGImageSurfaceSource->_vtbl->_setFrameIndex(_iVGImageSurfaceSource, index) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEVGImageSurfaceSource* _iVGImageSurfaceSource;
};

/*! @}
 * end of CEComICEVGImageSurfaceSourceRef
 */

/*! \defgroup CEComICEVGContextRef CEComICEVGContextRef
 * @{
 */

class CEComICEVGContextRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEVGContextRef() : _iVGContext(0) {}
	CEComICEVGContextRef(ICEVGContext* iOther) : _iVGContext(0)
	{
		if (iOther)
		{
			_iVGContext = iOther;
			_iVGContext->_vtbl->_addRef(_iVGContext);
		}
	}

	CEComICEVGContextRef(const CEComICEVGContextRef& other) : _iVGContext(0)
	{
		if (other._iVGContext)
		{
			_iVGContext = other._iVGContext;
			_iVGContext->_vtbl->_addRef(_iVGContext);
		}
	}

	~CEComICEVGContextRef()
	{
		if (_iVGContext)
		{
			ICEVGContext* tmp = _iVGContext;
			_iVGContext = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEVGContext* iVGContext;
				void* _ptr;
			} uIntf;
			uIntf.iVGContext = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEVGContext, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iVGContext)
				{
					ICEVGContext* tmp = _iVGContext;
					_iVGContext = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iVGContext = uIntf.iVGContext;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEVGContext* object() const	{ return _iVGContext; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEVGContext* iOther)
	{
		if (_iVGContext)
		{
			ICEVGContext* tmp = _iVGContext;
			_iVGContext = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iVGContext = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEVGContext* detach()
	{
		ICEVGContext* iIntf = _iVGContext;
		_iVGContext = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEVGContext* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iVGContext;
		if (_iVGContext)
		{
			_iVGContext->_vtbl->_addRef(_iVGContext);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEVGContext*() const	{ return _iVGContext; }
	FORCEINLINE_WITHOUT_DEBUG ICEVGContext& operator*() const	{ return *_iVGContext; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iVGContext == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEVGContext* iOther) const	{ return (_iVGContext != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEVGContext* iOther) const	{ return (_iVGContext == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEVGContextRef& operator=(const CEComICEVGContextRef& other)	{ return operator=(other._iVGContext); }

	CEComICEVGContextRef& operator=(const ICEVGContext* iOther)
	{
		if (_iVGContext != iOther)
		{
			if (_iVGContext)
			{
				ICEVGContext* tmp = _iVGContext;
				_iVGContext = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iVGContext = const_cast<ICEVGContext*>(iOther);
			if (_iVGContext)
			{
				_iVGContext->_vtbl->_addRef(_iVGContext);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEVGContext** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEVGContextRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iVGContext && "has a com object reference. clear first.");
		return & _iVGContext; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEVGContextRef& other)
	{
		bool result = false;
		if (_iVGContext)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iVGContext);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iVGContext ? _iVGContext->_vtbl->_queryInterface(_iVGContext, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iVGContext) { _iVGContext->_vtbl->_addRef(_iVGContext); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iVGContext) { _iVGContext->_vtbl->_release(_iVGContext); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult save()	{ return _iVGContext ? _iVGContext->_vtbl->_save(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult restore()	{ return _iVGContext ? _iVGContext->_vtbl->_restore(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult appendPath(struct ICEVGContext* iPath)	{ return _iVGContext ? _iVGContext->_vtbl->_appendPath(_iVGContext, iPath) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult appendPathEn(struct ICEVGPathEnumerator* iPathEn)	{ return _iVGContext ? _iVGContext->_vtbl->_appendPathEn(_iVGContext, iPathEn) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCurrentPointF(float *const optXOut, float *const optYOut)	{ return _iVGContext ? _iVGContext->_vtbl->_getCurrentPointF(_iVGContext, optXOut, optYOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getPathExtentsF(float *const xOut, float *const yOut, float *const widthOut, float *const heightOut)	{ return _iVGContext ? _iVGContext->_vtbl->_getPathExtentsF(_iVGContext, xOut, yOut, widthOut, heightOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStrokeExtentsF(float *const xOut, float *const yOut, float *const widthOut, float *const heightOut)	{ return _iVGContext ? _iVGContext->_vtbl->_getStrokeExtentsF(_iVGContext, xOut, yOut, widthOut, heightOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFillExtentsF(float *const xOut, float *const yOut, float *const widthOut, float *const heightOut)	{ return _iVGContext ? _iVGContext->_vtbl->_getFillExtentsF(_iVGContext, xOut, yOut, widthOut, heightOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getMatrix2DAffine(CEDXMatrix2DAffineD* const matrix2DAffineOut)	{ return _iVGContext ? _iVGContext->_vtbl->_getMatrix2DAffine(_iVGContext, matrix2DAffineOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult transformMatrix2DAffine(const CEDXMatrix2DAffineD* matrix2DAffine)	{ return _iVGContext ? _iVGContext->_vtbl->_transformMatrix2DAffine(_iVGContext, matrix2DAffine) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult translateF(float x, float y)	{ return _iVGContext ? _iVGContext->_vtbl->_translateF(_iVGContext, x, y) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult scaleF(float x, float y)	{ return _iVGContext ? _iVGContext->_vtbl->_scaleF(_iVGContext, x, y) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult rotateF(float angle)	{ return _iVGContext ? _iVGContext->_vtbl->_rotateF(_iVGContext, angle) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult newPath()	{ return _iVGContext ? _iVGContext->_vtbl->_newPath(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult closePath()	{ return _iVGContext ? _iVGContext->_vtbl->_closePath(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult moveToF(float x, float y)	{ return _iVGContext ? _iVGContext->_vtbl->_moveToF(_iVGContext, x, y) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult lineToF(float x, float y)	{ return _iVGContext ? _iVGContext->_vtbl->_lineToF(_iVGContext, x, y) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult curveToF(float x1, float y1, float x2, float y2, float x3, float y3)	{ return _iVGContext ? _iVGContext->_vtbl->_curveToF(_iVGContext, x1, y1, x2, y2, x3, y3) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult addArcF(float xc, float yc, float radius, float angle1, float angle2)	{ return _iVGContext ? _iVGContext->_vtbl->_addArcF(_iVGContext, xc, yc, radius, angle1, angle2) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult addArcNegativeF(float xc, float yc, float radius, float angle1, float angle2)	{ return _iVGContext ? _iVGContext->_vtbl->_addArcNegativeF(_iVGContext, xc, yc, radius, angle1, angle2) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult addRectangleF(float x, float y, float width, float height)	{ return _iVGContext ? _iVGContext->_vtbl->_addRectangleF(_iVGContext, x, y, width, height) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setFillRule(eCEVGFillRule windRule)	{ return _iVGContext ? _iVGContext->_vtbl->_setFillRule(_iVGContext, windRule) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFillRule(eCEVGFillRule *const windRuleOut)	{ return _iVGContext ? _iVGContext->_vtbl->_getFillRule(_iVGContext, windRuleOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isInFillF(float x, float y, bool *const isInFillOut)	{ return _iVGContext ? _iVGContext->_vtbl->_isInFillF(_iVGContext, x, y, isInFillOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isInStroke(float x, float y, bool *const isInStrokeOut)	{ return _iVGContext ? _iVGContext->_vtbl->_isInStroke(_iVGContext, x, y, isInStrokeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult copyPathData(struct ICEVGPathEnumerator* *const iPathEnOut)	{ return _iVGContext ? _iVGContext->_vtbl->_copyPathData(_iVGContext, iPathEnOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult copyPropertiesTo(struct ICEVGContext* vgcOut)	{ return _iVGContext ? _iVGContext->_vtbl->_copyPropertiesTo(_iVGContext, vgcOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getTargetSurface(struct ICEVGSurface* *const iSurfaceOut)	{ return _iVGContext ? _iVGContext->_vtbl->_getTargetSurface(_iVGContext, iSurfaceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setSourceRGBA(const CERGBAColor* rgbaColor)	{ return _iVGContext ? _iVGContext->_vtbl->_setSourceRGBA(_iVGContext, rgbaColor) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setSourceRGBAF(float r, float g, float b, float a)	{ return _iVGContext ? _iVGContext->_vtbl->_setSourceRGBAF(_iVGContext, r, g, b, a) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setSourcePattern(struct ICEVGPattern* iPattern)	{ return _iVGContext ? _iVGContext->_vtbl->_setSourcePattern(_iVGContext, iPattern) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setSourceSurfaceF(struct ICEVGSurface* iSurface, float x, float y)	{ return _iVGContext ? _iVGContext->_vtbl->_setSourceSurfaceF(_iVGContext, iSurface, x, y) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult maskSurfaceF(struct ICEVGSurface* iSurface, float surfaceX, float surfaceY)	{ return _iVGContext ? _iVGContext->_vtbl->_maskSurfaceF(_iVGContext, iSurface, surfaceX, surfaceY) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fill()	{ return _iVGContext ? _iVGContext->_vtbl->_fill(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult fillPreserve()	{ return _iVGContext ? _iVGContext->_vtbl->_fillPreserve(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult stroke()	{ return _iVGContext ? _iVGContext->_vtbl->_stroke(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult strokePreserve()	{ return _iVGContext ? _iVGContext->_vtbl->_strokePreserve(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult paint()	{ return _iVGContext ? _iVGContext->_vtbl->_paint(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult pushGroup()	{ return _iVGContext ? _iVGContext->_vtbl->_pushGroup(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult popGroupToSource()	{ return _iVGContext ? _iVGContext->_vtbl->_popGroupToSource(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult paintWidthAlphaF(float alpha)	{ return _iVGContext ? _iVGContext->_vtbl->_paintWidthAlphaF(_iVGContext, alpha) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult userToDeviceF(float *const xRef, float *const yRef)	{ return _iVGContext ? _iVGContext->_vtbl->_userToDeviceF(_iVGContext, xRef, yRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult deviceToUserF(float *const xRef, float *const yRef)	{ return _iVGContext ? _iVGContext->_vtbl->_deviceToUserF(_iVGContext, xRef, yRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult userToDeviceDistanceF(float *const xRef, float *const yRef)	{ return _iVGContext ? _iVGContext->_vtbl->_userToDeviceDistanceF(_iVGContext, xRef, yRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult deviceToUserDistanceF(float *const xRef, float *const yRef)	{ return _iVGContext ? _iVGContext->_vtbl->_deviceToUserDistanceF(_iVGContext, xRef, yRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setLineCap(eCEVGLineCap lineCap)	{ return _iVGContext ? _iVGContext->_vtbl->_setLineCap(_iVGContext, lineCap) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setLineJoin(eCEVGLineJoin lineJoin)	{ return _iVGContext ? _iVGContext->_vtbl->_setLineJoin(_iVGContext, lineJoin) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setLineWidthD(double w)	{ return _iVGContext ? _iVGContext->_vtbl->_setLineWidthD(_iVGContext, w) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLineWidthD(double *const wOut)	{ return _iVGContext ? _iVGContext->_vtbl->_getLineWidthD(_iVGContext, wOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setMiterLimitF(float miter)	{ return _iVGContext ? _iVGContext->_vtbl->_setMiterLimitF(_iVGContext, miter) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setDashD(const double* dashes, UINT32 nDashes, double offset)	{ return _iVGContext ? _iVGContext->_vtbl->_setDashD(_iVGContext, dashes, nDashes, offset) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setAntialias(eCEVGAntialias v)	{ return _iVGContext ? _iVGContext->_vtbl->_setAntialias(_iVGContext, v) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setOperator(eCEVGOperator v)	{ return _iVGContext ? _iVGContext->_vtbl->_setOperator(_iVGContext, v) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setClip()	{ return _iVGContext ? _iVGContext->_vtbl->_setClip(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setClipPreserve()	{ return _iVGContext ? _iVGContext->_vtbl->_setClipPreserve(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setClipRectF(const CERectF* rect, eCEVGFillRule windRule)	{ return _iVGContext ? _iVGContext->_vtbl->_setClipRectF(_iVGContext, rect, windRule) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getClipExtents(float *const xOut, float *const yOut, float *const widthOut, float *const heightOut)	{ return _iVGContext ? _iVGContext->_vtbl->_getClipExtents(_iVGContext, xOut, yOut, widthOut, heightOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult resetClip()	{ return _iVGContext ? _iVGContext->_vtbl->_resetClip(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawGlyphsI(CEPointBase* ptToDrawAt, struct ICEUIFontGlyphBuffer* igb)	{ return _iVGContext ? _iVGContext->_vtbl->_drawGlyphsI(_iVGContext, ptToDrawAt, igb) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult addGlyphPathsI(CEPointBase* ptToDrawAt, struct ICEUIFontGlyphBuffer* igb)	{ return _iVGContext ? _iVGContext->_vtbl->_addGlyphPathsI(_iVGContext, ptToDrawAt, igb) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawGlyphI(const CEPFGlyphBitmapInfo* gbmpInfo)	{ return _iVGContext ? _iVGContext->_vtbl->_drawGlyphI(_iVGContext, gbmpInfo) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawRectangleI(const CERect* rect, eCEVGDrawMode mode, eCEVGOperator fillOperator)	{ return _iVGContext ? _iVGContext->_vtbl->_drawRectangleI(_iVGContext, rect, mode, fillOperator) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult writeTargetSurfaceToPNGStream(struct ICEVGSurfaceWriteFunction* iFunc, struct ICEUnknown* closure)	{ return _iVGContext ? _iVGContext->_vtbl->_writeTargetSurfaceToPNGStream(_iVGContext, iFunc, closure) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult beginPaint(struct ICEDXImmutableRegion* iUpdateRgn)	{ return _iVGContext ? _iVGContext->_vtbl->_beginPaint(_iVGContext, iUpdateRgn) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult endPaint()	{ return _iVGContext ? _iVGContext->_vtbl->_endPaint(_iVGContext) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEVGContext* _iVGContext;
};

/*! @}
 * end of CEComICEVGContextRef
 */

#endif // __cplusplus

#endif
