///////////////////////////////////////////////////////////////////////////////
// Copyright 2004,2005,2006,2007,2008,2009 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICETextSupport_h
#define ICETextSupport_h

// TODO: generate like this factory methods from idl or ccd.

#include "CECom.h"
#include "ICEUString.h"
#include "ceustrmacros.h"

#if (!CE_OS(WIN32))
#include <wctype.h>
#endif

//! convert ascii utf16 string to null terminated c-string.
inline CEHResult CEUAsciiStr16ToCString(const UTF16CHAR* pC16s, UINT32 nC16s, UCHAR8* pDestC8s, UINT32 nC8s)
{
	CEU_VALIDATE_PTR(pC16s);
	CEU_VALIDATE_PTR(pDestC8s);
	CEU_ENSURE_TRUE((nC8s > 0), CE_SILK_ERR_BADARGS);
	CEU_ENSURE_TRUE((nC16s < nC8s), CE_SILK_ERR_BADARGS);

	CEHResult hr = CE_S_OK;
	UINT32 i=0;
	for (i=0; CESucceeded(hr) && (i < nC16s) && (i < (nC8s-1)); i++)
	{
		if (pC16s[i] & 0xff00)
			hr = CE_SILK_ERR_BADARGS;
		else
			pDestC8s[i] = (UCHAR8)(0xff & pC16s[i]);
	}
	pDestC8s[i] = 0;
	return hr;
}

inline UTF32CHAR CEUCharacter_toUpper(UTF32CHAR c32)
{
	UTF32CHAR ret = c32;
	if ( ! CEU_U32_IS_SURROGATE(c32) ) { ret = towupper((wchar_t)(0x0ffff & c32)); }
	return ret;
}

inline UTF32CHAR CEUCharacter_toLower(UTF32CHAR c32)
{
	UTF32CHAR ret = c32;
	if ( ! CEU_U32_IS_SURROGATE(c32) ) { ret = towlower((wchar_t)(0x0ffff & c32)); }
	return ret;
}


#if defined(_DEBUG) && defined(_WIN32)
#define CETEXTSUPPORT_VALIDATE_CSTR_LEN(_str, _len) {CEASSERT(_len == strlen(_str)); }
#else //#if defined(_DEBUG) && defined(_WIN32)
#define CETEXTSUPPORT_VALIDATE_CSTR_LEN(_str, _len) 
#endif //#if defined(_DEBUG) && defined(_WIN32)

inline bool ICEUString_equalsCStringLiteral(const UTF16CHAR* pUtf16s, UINT32 numOfC16s,  const char* lcCstr, UINT32 cstrLen, bool ignoreCase)
{
	bool ret = false;
	if ( pUtf16s && lcCstr )
	{
		if ( cstrLen == numOfC16s )
		{
			ret = true;
			for ( UINT32 i=0; lcCstr[i] && i<cstrLen; i++ )
			{
				UTF32CHAR c32a = pUtf16s[i];
				UTF32CHAR c32b = (UTF32CHAR)lcCstr[i];
				if ( ignoreCase )
				{
					c32a = CEUCharacter_toLower(c32a) ;
					c32b = CEUCharacter_toLower(c32b) ;
				}
				if ( c32a != c32b ) {
					ret = false;
					break;
				}
			}
		}
	}
	return ret;
}

inline CEHResult ICEUString_equalsCStringLiteral(ICEUString* iStr,  const char* lcCstr, UINT32 cstrLen, bool ignoreCase, bool& bSameOut)
{
	CEU_VALIDATE_PTR(iStr);

	CETEXTSUPPORT_VALIDATE_CSTR_LEN(lcCstr, cstrLen);

	const UTF16CHAR* pC16s = 0;
	UINT32 numOfC16s = 0;

	bSameOut = false;
	CEHResult hr = iStr->_vtbl->_getCharArray16(iStr, &pC16s, &numOfC16s);
	if (CESucceeded(hr))
	{
		bSameOut = ICEUString_equalsCStringLiteral(pC16s, numOfC16s, lcCstr, cstrLen, ignoreCase);
	}
	return hr;
}

inline CEHResult ICEUString_startsWithCStringLiteral(ICEUString* iStr,  const char* lcCstr, UINT32 cstrLen, bool ignoreCase, bool& bSameOut)
{
	CEU_VALIDATE_PTR(iStr);
	CETEXTSUPPORT_VALIDATE_CSTR_LEN(lcCstr, cstrLen);

	const UTF16CHAR* pC16s = 0;
	UINT32 numOfC16s = 0;

	bSameOut = false;
	CEHResult hr = iStr->_vtbl->_getCharArray16(iStr, &pC16s, &numOfC16s);
	if (CESucceeded(hr))
	{
		if ( numOfC16s >= cstrLen )
		{
			bSameOut = ICEUString_equalsCStringLiteral(pC16s, cstrLen, lcCstr, cstrLen, ignoreCase);
		}
	}
	return hr;
}

inline CEHResult ICEUString_endsWithCStringLiteral(ICEUString* iStr,  const char* lcCstr, UINT32 cstrLen, bool ignoreCase, bool& bSameOut)
{
	CEU_VALIDATE_PTR(iStr);
	CETEXTSUPPORT_VALIDATE_CSTR_LEN(lcCstr, cstrLen);

	const UTF16CHAR* pC16s = 0;
	UINT32 numOfC16s = 0;

	bSameOut = false;
	CEHResult hr = iStr->_vtbl->_getCharArray16(iStr, &pC16s, &numOfC16s);
	if (CESucceeded(hr))
	{
		if ( numOfC16s >= cstrLen )
		{
			UINT32 offset = numOfC16s - cstrLen;
			bSameOut = ICEUString_equalsCStringLiteral(pC16s+offset, cstrLen, lcCstr, cstrLen, ignoreCase);
		}
	}
	return hr;
}

inline bool ICEUString_isEmpty(ICEUString* istr)
{
	bool b = true;
	if ( istr )
	{
		if ( CEFailed( istr->_vtbl->_isEmpty(istr, &b) ) ) 
		{
			b = true;
		}
	}
	return b;
}

inline bool ICEUString_equals(ICEUString* istr1, ICEUString* istr2, bool bIgnoreCase)
{
	bool b = false;
	if ( istr1 )
	{
		if ( CEFailed( istr1->_vtbl->_equals(istr1, istr2, bIgnoreCase, &b) ) ) 
		{
			b = false;
		}
	}
	return b;
}

inline bool ICEUString_equalsCStringLiteral(ICEUString* istr, const char* str, UINT32 strLen, bool bIgnoreCase)
{
	bool b = false;
	ICEUString_equalsCStringLiteral(istr, str, strLen, bIgnoreCase, b);
	return b;
}

inline const UTF16CHAR* ICEUString_getCharArray16(ICEUString* istr, UINT32* numOfChars16Out=0)
{
	const UTF16CHAR* ret = 0;
	if ( istr )
	{
		UINT32 len=0;
		if ( !CEFailed(istr->_vtbl->_getCharArray16(istr, &ret, &len)) )
		{
			if ( numOfChars16Out ) 
			{
				*numOfChars16Out = len;
			}
		}
	}
	return ret;
}

inline CEHResult _ICEUStringCreate(UINT32 clsid, CEComICEUStringRef& strOut)
{
	strOut = 0;
	
	CEComICEClassFactoryRef cfRef = 0;

	CEHResult hr = CEComGetClassObject(clsid, CEComIID_ICEUString, (void**)&cfRef);
	if ( CESucceeded(hr) )
	{
		if ( cfRef )
		{
			strOut = 0;
			hr = cfRef.createInstance(0, CEComIID_ICEUString, (void**)&strOut);
		}
	}
	return hr;
}

//! create empty ICEUString. 
//! use the appropriate macro for the context.
#define ICEUStringCreateEmptyString		_ICEUStringCreate
#define ICEUStringCreate				_ICEUStringCreate

inline CEHResult ICEUStringCreateFromUTF16Array(UINT32 clsid, const UTF16CHAR* utf16Array, UINT32 sizeInBytes, CEComICEUStringRef& strOut)
{
	CEHResult hr = _ICEUStringCreate(clsid, strOut);
	if ( CESucceeded(hr) )
	{
		hr = strOut.initWithUTF16Array(utf16Array, sizeInBytes);
	}

	if ( CEFailed(hr) )
	{
		strOut = 0;
	}
	return hr;
}

inline CEHResult ICEUStringCreateFromUTF8Array(UINT32 clsid, const char* utf8Array, UINT32 sizeInBytes, CEComICEUStringRef& strOut)
{
	CEHResult hr = _ICEUStringCreate(clsid, strOut);
	if ( CESucceeded(hr) )
	{
		hr = strOut.initWithByteArray((const UCHAR8*)utf8Array, eICEI18nEncoding_utf_8, sizeInBytes);
	}
	if ( CEFailed(hr) )
	{
		strOut = 0;
	}
	return hr;
}

inline CEHResult ICEUStringCreateFromCStringLiteral(UINT32 clsid, const char* cstr, CEComICEUStringRef& strOut)
{
	CEHResult hr = _ICEUStringCreate(clsid, strOut);
	if ( CESucceeded(hr) )
	{
		hr = strOut.initWithCStringLiteral(cstr);
	}
	if ( CEFailed(hr) )
	{
		strOut = 0;
	}
	return hr;
}

inline CEHResult ICEUStringCreateFromICEUString(UINT32 clsid, /*const*/ ICEUString* fromIStr, CEComICEUStringRef& strOut)
{
	CEHResult hr = _ICEUStringCreate(clsid, strOut);
	if ( CESucceeded(hr) )
	{
		hr = strOut.initWithString(fromIStr);
	}
	if ( CEFailed(hr) )
	{
		strOut = 0;
	}
	return hr;
}

inline CEHResult ICEUStringReaderCreate(UINT32 clsid, CEComICEUStringReaderRef& strOut)
{
	strOut = 0;
	
	CEComICEClassFactoryRef cfRef;
	CEHResult hr = CEComGetClassObject(clsid, CEComIID_ICEUStringReader, (void**)&cfRef);
	if ( CESucceeded(hr) )
	{
		if ( cfRef )
		{
			strOut = 0;
			hr = cfRef.createInstance(0, CEComIID_ICEUStringReader, (void**)&strOut);
		}
	}
	return hr;
}

inline CEHResult ICEUStringWriterCreate(UINT32 clsid, CEComICEUStringWriterRef& strWriterOut)
{
	strWriterOut = 0;
	CEComICEClassFactoryRef cfRef;

	CEHResult hr = CEComGetClassObject(clsid, CEComIID_ICEUStringWriter, (void**)&cfRef);
	if ( CESucceeded(hr) )
	{
		if ( cfRef )
		{
			hr = cfRef.createInstance(0, CEComIID_ICEUStringWriter, (void**)&strWriterOut);
		}
	}
	return hr;
}	

inline CEHResult ICEUStringWriterCreateAndOpen(UINT32 clsid, CEComICEUStringWriterRef& strWriterOut)
{
	strWriterOut = 0;
	CEComICEClassFactoryRef cfRef;

	CEHResult hr = CEComGetClassObject(clsid, CEComIID_ICEUStringWriter, (void**)&cfRef);
	if ( CESucceeded(hr) )
	{
		if ( cfRef )
		{
			hr = cfRef.createInstance(0, CEComIID_ICEUStringWriter, (void**)&strWriterOut);
			if (CESucceeded(hr))
			{
				hr = strWriterOut.open();
				if ( CEFailed(hr) )
				{
					strWriterOut = 0;
				}
			}
		}
	}
	return hr;
}	


inline CEHResult ICEUTokenizerCreate(UINT32 clsid, CEComICEUTokenizerRef& strOut)
{
	strOut = 0;
	CEComICEClassFactoryRef cfRef;

	CEHResult hr = CEComGetClassObject(clsid, CEComIID_ICEUTokenizer, (void**)&cfRef);
	if ( CESucceeded(hr) )
	{
		if ( cfRef )
		{
			hr = cfRef.createInstance(0, CEComIID_ICEUTokenizer, (void**)&strOut);
		}
	}
	return hr;
}

inline CEHResult ICEUTokenizerCallbacksCreate(UINT32 clsid, CEComICEUTokenizerCallbacksRef& strOut)
{
	strOut = 0;
	CEComICEClassFactoryRef cfRef;

	CEHResult hr = CEComGetClassObject(clsid, CEComIID_ICEUTokenizerCallbacks, (void**)&cfRef);
	if ( CESucceeded(hr) )
	{
		if ( cfRef )
		{
			hr = cfRef.createInstance(0, CEComIID_ICEUTokenizerCallbacks, (void**)&strOut);
		}
	}
	return hr;
}


#define ICEUSTRING_DUP_WITH_CASECONV(_src, _iStrOut, _exp, _errOut) \
{ \
	CEU_VALIDATE_OUTPTR(iStrOut);\
	\
	if ( _src )\
	{\
		CEComICEUStringWriterRef out;\
		_errOut = ICEUStringWriterCreate(CEComStdClassID_CEUStringWriter, out);\
		if ( !_errOut )\
		{\
			_errOut = out.open();\
			if ( !_errOut )\
			{\
				CEComICEUStringReaderRef in;\
				_errOut = ICEUStringReaderCreate(CEComStdClassID_CEUStringReader, in);\
				if ( !_errOut )\
				{\
					_errOut = in.open(src);\
					if ( !_errOut )\
					{\
						UINT32 bytes=0;\
						UTF32CHAR c32 = INVALID_UCS4_CHAR;\
						while ( !_errOut && ((_errOut = in.read(&c32)) == CE_S_OK) )\
						{\
							_errOut = out.writeUTF32((_exp(c32)), &bytes);\
						}\
					}\
				}\
				in.close();\
			}\
			_errOut = out.close(_iStrOut);\
		}\
	}\
}


inline CEHResult ICEUString_toLower(ICEUString* src, ICEUString* *const iStrOut)
{
	CEHResult hr = CE_S_OK;
	ICEUSTRING_DUP_WITH_CASECONV(src, iStrOut, CEUCharacter_toLower, hr);
	return hr;
}

inline CEHResult ICEUString_toUpper(ICEUString* src, ICEUString* *const iStrOut)
{
	CEHResult hr = CE_S_OK;
	ICEUSTRING_DUP_WITH_CASECONV(src, iStrOut, CEUCharacter_toUpper, hr);
	return hr;
}


////////////////////////////////////////////////////////////////
// i18n string support.
////////////////////////////////////////////////////////////////
class CEUStrInfo {
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	static inline CEHResult char32AtLead16(const UTF16CHAR* c16Array, const UINT32 numOfChars16, const INT32 lead16, UTF32CHAR& c32Out)
	{
		CEU_VALIDATE_PTR(c16Array);
		c32Out = INVALID_UCS4_CHAR;
		CEHResult hr = CE_SILK_ERR_BADARRAYINDEX;

		CEASSERT((lead16 >= 0));
		CEU_ENSURE_TRUE((lead16 >= 0), hr);

		if ((UINT32)lead16 < numOfChars16) 
		{
			const UTF16CHAR* src = &c16Array[lead16];

			UTF16CHAR dest[2] = {0, 0};
			UINT32 destLen = 1;
			dest[0] = CEU_U32_UNKNOWN_CHAR;

			dest[0] = src[0];
			destLen = 1;

			if ( CEU_U16_IS_SURROGATE_LEAD(src[0]) ) 
			{
				if ((UINT32)(lead16+1) < numOfChars16)
				{
					if ( CEU_U16_IS_SURROGATE_TRAIL(src[1]) ) 
					{
						dest[1] = src[1];
						destLen = 2;
					}
				}
			}
			CEU_U16_TO_U32(src, destLen, c32Out);
			hr = CE_S_OK;
		}
		return hr;
	}

	static inline CEHResult offset16ToLead16(const UTF16CHAR* c16Array, const UINT32 numOfChars16, const INT32 offset16, INT32& lead16Out)
	{
		CEU_VALIDATE_PTR(c16Array);

		CEHResult hr = CE_SILK_ERR_BADARRAYINDEX;

		CEASSERT((offset16 >= 0));
		CEU_ENSURE_TRUE((offset16 >= 0), hr);

		if ((UINT32)offset16 < numOfChars16) 
		{
			hr = CE_S_OK;
			if (CEU_U16_IS_SURROGATE_LEAD(c16Array[offset16]))
			{
				lead16Out = offset16;
			}
			else if (CEU_U16_IS_SURROGATE_TRAIL(c16Array[offset16]))
			{
				hr = CE_SILK_ERR_BADARRAYINDEX;
				if (offset16 > 0)
				{
					hr = CE_S_OK;
					if (CEU_U16_IS_SURROGATE_TRAIL(c16Array[offset16-1]))
					{
						// broken sequence.
						lead16Out = offset16;
					}
					else
					{
						lead16Out = (offset16-1);
					}
				}
			}
			else
			{
				lead16Out = offset16;
			}
		}
		return hr;
	}

	static inline CEHResult getOffset16ToNextCharLead16At(const UTF16CHAR* c16Array, const UINT32 numOfChars16, INT32 pos16, INT32& offset16Out) 
	{
		CEHResult hr = CE_S_OK;
		CEASSERT((pos16 >= 0));
		CEU_ENSURE_TRUE((pos16 >= 0), CE_SILK_ERR_BADARRAYINDEX);
		if ( (UINT32)pos16 < numOfChars16 ) 
		{
			// at least 1 char is available.
			if ( CEU_U16_IS_SURROGATE_LEAD(c16Array[pos16]) )
			{
				if ( (UINT32)(pos16+1) < numOfChars16 ) 
				{
					if ( CEU_U16_IS_SURROGATE_TRAIL(c16Array[pos16+1]) ) 
					{
						offset16Out = 2;
					}
					else {
						offset16Out = 1;
					}
				}
				else {
					// no surrogate trail.
					offset16Out = 1;
				}
			}
			else if ( CEU_U16_IS_SURROGATE_TRAIL(c16Array[pos16]) ) 
			{
				// bad surrogate trail
				offset16Out = 1;
			}
			else {
				offset16Out = 1;
			}
		}
		else {
			hr = CE_SILK_ERR_BADARRAYINDEX;
		}
		return hr;
	}

	static inline CEHResult getOffset16ToPrevCharLead16At(const UTF16CHAR* c16Array, const UINT32 numOfChars16, INT32 pos16, INT32& offset16Out) 
	{
		CEHResult hr = CE_S_OK;
		CEASSERT(pos16 >= 0);
		CEU_ENSURE_TRUE((pos16 >= 0), CE_SILK_ERR_BADARRAYINDEX);
		CEU_ENSURE_TRUE(((UINT32)pos16 < numOfChars16), CE_SILK_ERR_BADARRAYINDEX);
		

		if (pos16 == 0)
		{
			// sos leached
			offset16Out = -1;
		}
		else
		{
			pos16--;	// move to prev char16.
			// at least 1 char is available.
			if ( CEU_U16_IS_SURROGATE_LEAD(c16Array[pos16]) )
			{
				// no surrogate trail.
				offset16Out = -1;
			}
			else if ( CEU_U16_IS_SURROGATE_TRAIL(c16Array[pos16]) ) 
			{
				if (pos16 > 0)
				{
					if (CEU_U16_IS_SURROGATE_LEAD(c16Array[pos16-1])) 
					{
						offset16Out = -2;
					}
					else
					{
						// bad surrogate.
						offset16Out = -1;
					}
				}
				else
				{
					// bad surrogate, sos leached.
					offset16Out = -1;
				}
			}
			else {
				offset16Out = -1;
			}
		}
		return hr;
	}
};

//////////////////////////////////////////////////////////////////
// unicode string utils
//////////////////////////////////////////////////////////////////

class CEUCharSeqEnumerator {
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	CEUCharSeqEnumerator()
		:_curCharSeqLen16(0),
		_moveOffset16(0),
		_inx(0),
		_numOfChars16(0),
		_c16Array(0),
		_bReverseMode(false)
	{}

	inline CEHResult init(ICEUString* src) 
	{
		CEU_VALIDATE_PTR(src);
		CEComICEUStringRef str = src;
		const UTF16CHAR* pC16s = 0;
		UINT32 numOfC16s = 0;

		CEHResult hr = str.getCharArray16(&pC16s, &numOfC16s);
		CEU_ENSURE_SUCCESS(hr, hr);

		return init(pC16s, numOfC16s);
	}

	inline CEHResult init(const UTF16CHAR* c16Array, const UINT32 numOfChars16) 
	{
		_c16Array = c16Array;
		_numOfChars16 = numOfChars16;
		reset(_bReverseMode);

		if ( !_c16Array ) { return CE_SILK_ERR_BADARGS; }
		return CE_S_OK;
	}

	inline CEHResult reset(bool bReverseMode) 
	{
		_bReverseMode = bReverseMode;
		if (_bReverseMode)
		{
			if (_numOfChars16 > 0)
			{
				_inx = _numOfChars16 - 1;
				CEUStrInfo::offset16ToLead16(_c16Array, _numOfChars16, _inx, _inx);
			}
		}
		else
		{
			_inx = 0;
		}
		_curCharSeqLen16 = 0;
		_moveOffset16 = 0;

		if ( !_c16Array ) { return CE_SILK_ERR_BADARGS; }
		return CE_S_OK;
	}

	inline CEHResult moveNext() 
	{
		CEHResult hr = CE_SILK_ERR_BADARRAYINDEX;
		_inx += _moveOffset16;

		if (_inx >= 0)
		{
			hr = CEUStrInfo::getOffset16ToNextCharLead16At(_c16Array, _numOfChars16, _inx , _curCharSeqLen16);
			if (CESucceeded(hr))
			{
				if (_bReverseMode)
				{
					hr = CE_SILK_ERR_BADARRAYINDEX;
					if (_inx >= 0)
					{
						hr = CEUStrInfo::getOffset16ToPrevCharLead16At(_c16Array, _numOfChars16, _inx , _moveOffset16);
					}
				}
				else
				{
					_moveOffset16 = _curCharSeqLen16;
				}
			}
		}
		return hr;
	}

	inline CEHResult current32(UTF32CHAR& c32Out) const
	{
		if ( 
			(UINT32)_inx >= _numOfChars16 ||
			_inx < 0 ||
			_curCharSeqLen16 == 0 ||
			!_c16Array
			) 
		{
			return CE_SILK_ERR_BADSTATE;
		}
		return CEUStrInfo::char32AtLead16(_c16Array, _numOfChars16, _inx, c32Out);
	}

	inline UINT32 nextCharSeqLeadIndex()const 
	{
		CEASSERT(_curCharSeqLen16==1||_curCharSeqLen16==2);
		return _inx + (curCharSeqLen16()); 
	}

	inline UINT32 charSeqLeadIndex()const { return _inx; }
	inline UINT32 curCharSeqLen16()const { return  _curCharSeqLen16; }

	inline CEHResult rest(CEUCharSeqEnumerator& restOut, bool widthoutCurrentChar=false)const
	{
		CEHResult hr = CE_SILK_ERR_BADARRAYINDEX;
		restOut.init(0,0);

		CEASSERT(_curCharSeqLen16==1||_curCharSeqLen16==2);

		if (_bReverseMode)
		{
			INT32 rest16 = widthoutCurrentChar ? charSeqLeadIndex() : nextCharSeqLeadIndex();
			if (rest16 >= 1)
			{
				restOut.init(_c16Array, rest16);
				hr = CE_S_OK;
			}
		}
		else
		{
			UINT32 fromIdx = widthoutCurrentChar ? nextCharSeqLeadIndex() : charSeqLeadIndex();
			INT32 rest16 = _numOfChars16 - fromIdx;

			if ( rest16 >= 1 )
			{
				restOut.init(&_c16Array[fromIdx], rest16);
				hr = CE_S_OK;
			}
		}
		return hr;
	}

	inline CEHResult restWithoutCurrentChar(CEUCharSeqEnumerator& restOut)const
	{
		return rest(restOut, true);
	}

	inline CEHResult charAt32(UINT32 offset16, UTF32CHAR& c32Out, INT32 *const optLead16Out=0)const
	{
		CEU_SET_OPTIONAL_OUTPTR(optLead16Out, 0);
		c32Out = INVALID_UCS4_CHAR;
		INT32 lead16 = 0;
		CEHResult hr = CEUStrInfo::offset16ToLead16(_c16Array, _numOfChars16, offset16, lead16);
		if (CESucceeded(hr))
		{
			CEU_SET_OPTIONAL_OUTPTR(optLead16Out, lead16);
			hr = CEUStrInfo::char32AtLead16(_c16Array, _numOfChars16, lead16, c32Out);
		}
		return hr;
	}

	inline CEHResult seek(INT32 numOfC16s)
	{
		CEHResult hr = CE_SILK_ERR_BADARRAYINDEX;
		INT32 newInx = _inx + numOfC16s;

		if (newInx >= 0 && (UINT32)newInx < _numOfChars16)
		{
			INT32 charSeqLen = 0;
			CEUStrInfo::offset16ToLead16(_c16Array, _numOfChars16, newInx, newInx);

			hr = CEUStrInfo::getOffset16ToNextCharLead16At(_c16Array, _numOfChars16, newInx, charSeqLen);
			if (CESucceeded(hr))
			{
				_inx = newInx;
				_curCharSeqLen16 = charSeqLen;
			}

			_moveOffset16 = charSeqLen;
			if (_bReverseMode)
			{
				CEUStrInfo::getOffset16ToPrevCharLead16At(_c16Array, _numOfChars16, newInx, _moveOffset16);
			}
		}
		else
		{
			_inx = _bReverseMode ? -1 : (INT32)_numOfChars16;
			_curCharSeqLen16 = 0;
			_moveOffset16 = 0;
		}
		return hr;
	}

	inline UINT32 currentLeadIndexInSrc(const UTF16CHAR* pSrc16s, UINT32 numOfSrc16s)const
	{
		const UTF16CHAR* pC16s = _c16Array;
		pC16s += _inx;

		CEASSERT(pC16s >= pSrc16s);

		UINT_PTR idx = (pC16s - pSrc16s);
		CEASSERT(idx < 0xffffffff);

		UINT32 ret = (UINT32)idx;
		CEASSERT(ret < numOfSrc16s);
		return ret;
	}

	inline const UTF16CHAR* charArray16()const { return _c16Array; }
	inline UINT32 numOfChars16()const { return _numOfChars16; }
	inline bool isReverseMode() const { return _bReverseMode; }

private:

	INT32		_curCharSeqLen16;
	INT32		_moveOffset16;
	INT32		_inx;
	UINT32		_numOfChars16;
	const UTF16CHAR* _c16Array;
	bool		_bReverseMode;
};

#endif //#ifndef ICETextSupport_h



