///////////////////////////////////////////////////////////////////////////////
// Copyright 2012 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICEPFRasterizer_h
#define ICEPFRasterizer_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"
#include "CEPFRasterizer.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------

struct ICEFSFile;
struct ICESymbol;

//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEPFFontResource ICEPFFontResource
 * @{
 */

/*!
 * ID of ICEPFFontResource
 */
#define CEComIID_ICEPFFontResource 0x93fc8e94


/*!
* ICEPFFontResource
*/
typedef struct ICEPFFontResource
{
	const struct ICEPFFontResource_vtbl* _vtbl;
} ICEPFFontResource;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iPFFontResource	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEPFFontResource_queryInterface) (ICEPFFontResource* iPFFontResource, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFFontResource_addRef) (ICEPFFontResource* iPFFontResource);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFFontResource_release) (ICEPFFontResource* iPFFontResource);


	/*!
	* Returns readable font resource stream.
	*
	* \param[in]	iPFFontResource	Specifies interface pointer
	* \param[out]	streamOut	Readable stream.
	*/
typedef CEHResult (*iCEPFFontResource_attachStream) (ICEPFFontResource* iPFFontResource, struct ICEFSFile* *const streamOut);

typedef CEHResult (*iCEPFFontResource_detachStream) (ICEPFFontResource* iPFFontResource, struct ICEFSFile* stream);


	/*!
	* Get font resource name string.
	*
	* \param[in]	iPFFontResource	Specifies interface pointer
	* \param[out]	nameOut	font resource name.
	*/
typedef CEHResult (*iCEPFFontResource_getName) (ICEPFFontResource* iPFFontResource, struct ICEFSPathname* *const nameOut);

typedef CEHResult (*iCEPFFontResource_getNameStr) (ICEPFFontResource* iPFFontResource, struct ICEUString* *const nameOut);


	/*!
	* Returns implementation class id.
	*
	* \param[in]	iPFFontResource	Specifies interface pointer
	* \param[out]	nameOut	font resource name.
	*/
typedef CEHResult (*iCEPFFontResource_getClassId) (ICEPFFontResource* iPFFontResource, UINT32 *const pClsidOut);


	/*!
	* Returns hashcode.
	*
	* \param[in]	iPFFontResource Specifies interface pointer
	* \param[out]	hashcodeOut hashcode.
	*/
typedef CEHResult (*iCEPFFontResource_getHashcode) (ICEPFFontResource* iPFFontResource, UINT32 *const pHashcodeOut);


	/*!
	* determine equality of this object and other.
	*
	* \param[in]	iPFFontResource Specifies interface pointer
	* \param[in]	other object to check.
	* \param[out]	equalsOut true if other is equal to this object; false otherwise.
	*/
typedef CEHResult (*iCEPFFontResource_equals) (ICEPFFontResource* iPFFontResource, struct ICEPFFontResource* pOther, bool *const pEqualsOut);


	/*!
	* returns byte data if supported.
	*
	* \param[in]	iPFFontResource Specifies interface pointer
	* \param[out]	bytesOut returns byte data.
	* \param[out]	nBytes number of bytes.
	*/
typedef CEHResult (*iCEPFFontResource_getBytes) (ICEPFFontResource* iPFFontResource, const UINT8* *const bytesOut, UINT32 *const nBytesOut);

/*!
 * V-table of ICEPFFontResource
 */
struct ICEPFFontResource_vtbl
{
	iCEPFFontResource_queryInterface	_queryInterface;	//!< Query interface.
	iCEPFFontResource_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEPFFontResource_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEPFFontResource_attachStream	_attachStream;
	iCEPFFontResource_detachStream	_detachStream;
	iCEPFFontResource_getName	_getName;
	iCEPFFontResource_getNameStr	_getNameStr;
	iCEPFFontResource_getClassId	_getClassId;
	iCEPFFontResource_getHashcode	_getHashcode;
	iCEPFFontResource_equals	_equals;
	iCEPFFontResource_getBytes	_getBytes;
};

/*! @}
 * end of ICEPFFontResource
 */

/*! \defgroup ICEPFGlyphLoader ICEPFGlyphLoader
 * @{
 */

/*!
 * ID of ICEPFGlyphLoader
 */
#define CEComIID_ICEPFGlyphLoader 0x8ac5d0b0


/*!
* ICEPFGlyphLoader
*/
typedef struct ICEPFGlyphLoader
{
	const struct ICEPFGlyphLoader_vtbl* _vtbl;
} ICEPFGlyphLoader;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iPFGlyphLoader	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEPFGlyphLoader_queryInterface) (ICEPFGlyphLoader* iPFGlyphLoader, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFGlyphLoader_addRef) (ICEPFGlyphLoader* iPFGlyphLoader);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFGlyphLoader_release) (ICEPFGlyphLoader* iPFGlyphLoader);


	/*!
	* Get the scaled font face information.
	*
	* \param[in]	iPFGlyphLoader	Specifies interface pointer
	* \param[out]	scaledFaceInfoOut	Address of an CEPFScaledFaceInfo, representing the scaled font face information.
	*/
typedef CEHResult (*iCEPFGlyphLoader_getScaledFaceInfo) (ICEPFGlyphLoader* iPFGlyphLoader, CEPFScaledFaceInfo* scaledFaceInfoOut);


	/*!
	* Get the ICEPFGlyphLoader settings.
	*
	* \param[in]	iPFGlyphLoader	Specifies interface pointer
	* \param[out]	glyphLoaderParamOut	Address of a pointer to an CEPFGlyphLoaderParam struct, represeting the glyph load setting parameters.
	* The returned pointer, glyphLoaderParamOut, is valid until ICEPFGlyphLoader is living, and is invalid after freed.
	*/
typedef CEHResult (*iCEPFGlyphLoader_getParam) (ICEPFGlyphLoader* iPFGlyphLoader, const CEPFGlyphLoaderParam* *const glyphLoaderParamOut);


	/*!
	* Get the glyph index from the code point
	*
	* \param[in]	iPFGlyphLoader	Specifies interface pointer
	* \param[in]	c32	Specifies the code point in UTF32
	* \param[in]	pCEPFCreateGlyphBitmapFuncCallData	Specifies the application depend data.
	* \param[out]	glyphIndexOut.
	* \param[out]	optXAdvanceOut.
    * \note
    *    CEHResult getGlyphIndex(const UTF32CHAR c32, UINT32 *const indexOut) is deprecated.
	*/
typedef CEHResult (*iCEPFGlyphLoader_getGlyphIndex2) (ICEPFGlyphLoader* iPFGlyphLoader, const UTF32CHAR c32, struct ICEUnknown* pCEPFCreateGlyphBitmapFuncCallData, UINT32 *const glyphIndexOut);

typedef CEHResult (*iCEPFGlyphLoader_getGlyphIndex3) (ICEPFGlyphLoader* iPFGlyphLoader, const UTF32CHAR c32, struct ICEUnknown* pCEPFCreateGlyphBitmapFuncCallData, UINT32 *const glyphIndexOut, CEUTextMeasurement *const optXAdvanceOut);

typedef CEHResult (*iCEPFGlyphLoader_getGlyphs16) (ICEPFGlyphLoader* iPFGlyphLoader, const UTF16CHAR* pC16s, UINT32 nC16s, bool bNoSurrogatePairs, UINT32 *const optGlyphsOut, UINT32 *const optNGlyphsRef);

typedef CEHResult (*iCEPFGlyphLoader_getGlyphPoint) (ICEPFGlyphLoader* iPFGlyphLoader, UINT32 glyphIndex, INT32 pointNumber, float *const xOut, float *const yOut);


	/*!
	* Load the glyph data from the font file
	*
	* \param[in]	iPFGlyphLoader	Specifies interface pointer
	* \param[in]	index	Specifies the glyph index
	* \param[in]	pCEPFCreateGlyphBitmapFuncCallData	Specifies the application depend data.
	* \note
	* The loaded glyph is valid until unloadGlyph() is called.
	*/
typedef CEHResult (*iCEPFGlyphLoader_loadGlyph) (ICEPFGlyphLoader* iPFGlyphLoader, const UINT32 glyphIndex, struct ICEUnknown* pCEPFCreateGlyphBitmapFuncCallData);

typedef CEHResult (*iCEPFGlyphLoader_loadGlyph2) (ICEPFGlyphLoader* iPFGlyphLoader, const UINT32 glyphIndex, struct ICEUnknown* pCEPFCreateGlyphBitmapFuncCallData, CEPFGlyphAdvance *const advanceOut);

typedef CEHResult (*iCEPFGlyphLoader_loadGlyph3) (ICEPFGlyphLoader* iPFGlyphLoader, const UINT32 glyphIndex, struct ICEUnknown* pCEPFCreateGlyphBitmapFuncCallData, const CEPFGlyphMetricsPt* *const metricsPtOut);


	/*!
	* Get the metric info of the loaded glyph
	*
	* \param[in]	iPFGlyphLoader	Specifies interface pointer
	* \param[out]	advanceOut	pointer to an CEPFGlyphAdvance struct, representing the loaded glyph advance. 
	* the advance depends on the eCEPFRasterizerOptions value of ICEPFRasterizer. if eCEPFRasterizerOptions_ALWAYS_AA_OUTLINE
	* bit is specified then the advance of outline glyph is returned. otherwise, the advance of embedded bitmap glyph(if available) or  
	* outline glyph is returned.
	*
	* \note
	* The method, loadGlyph(), should be called before this.
	*
	*/
typedef CEHResult (*iCEPFGlyphLoader_getGlyphAdvance) (ICEPFGlyphLoader* iPFGlyphLoader, const CEPFGlyphAdvance* *const advanceOut);

typedef CEHResult (*iCEPFGlyphLoader_getGlyphMetricsPt) (ICEPFGlyphLoader* iPFGlyphLoader, const CEPFGlyphMetricsPt* *const metricsPtOut);


	/*!
	* Get the metric info of the loaded glyph
	*
	* \param[in]	iPFGlyphLoader	Specifies interface pointer
	* \param[out]	bitmapInfoOut	Address of a pointer to an CEPFGlyphBitmapInfo struct, representing the loaded glyph bitmap data.
	*
	* \note
	* The method, loadGlyph(), should be called before this.
	*
	* The returned pointer, bitmapInfoOut, is valid until unloadGlyph() is called, and is invalid after that, 
	* because it points the cached glyph bitmap, so it may be relocated or released corresponding to the cached area usage.
	*/
typedef CEHResult (*iCEPFGlyphLoader_getGlyphBitmap) (ICEPFGlyphLoader* iPFGlyphLoader, const CEPFGlyphBitmapInfo* *const bitmapInfoOut, const CEPFGlyphAdvance* *const optGlyphBmpAdvanceOut);

typedef CEHResult (*iCEPFGlyphLoader_getGlyphOutline) (ICEPFGlyphLoader* iPFGlyphLoader, const CEPFGlyphOutlineInfo* *const outlineInfoOut, const CEPFGlyphAdvance* *const optGlyphOutlineAdvanceOut);


	/*!
	* Unload the glyph data from the font file
	*
	* \note The actual cached glyph metrics or advance or bitmaps may not be released immediately. It's up to the cache usage.
	*/

typedef CEHResult (*iCEPFGlyphLoader_unloadGlyph) (ICEPFGlyphLoader* iPFGlyphLoader);

/*!
 * V-table of ICEPFGlyphLoader
 */
struct ICEPFGlyphLoader_vtbl
{
	iCEPFGlyphLoader_queryInterface	_queryInterface;	//!< Query interface.
	iCEPFGlyphLoader_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEPFGlyphLoader_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEPFGlyphLoader_getScaledFaceInfo	_getScaledFaceInfo;
	iCEPFGlyphLoader_getParam	_getParam;
	iCEPFGlyphLoader_getGlyphIndex2	_getGlyphIndex2;
	iCEPFGlyphLoader_getGlyphIndex3	_getGlyphIndex3;
	iCEPFGlyphLoader_getGlyphs16	_getGlyphs16;
	iCEPFGlyphLoader_getGlyphPoint	_getGlyphPoint;
	iCEPFGlyphLoader_loadGlyph	_loadGlyph;
	iCEPFGlyphLoader_loadGlyph2	_loadGlyph2;
	iCEPFGlyphLoader_loadGlyph3	_loadGlyph3;
	iCEPFGlyphLoader_getGlyphAdvance	_getGlyphAdvance;
	iCEPFGlyphLoader_getGlyphMetricsPt	_getGlyphMetricsPt;
	iCEPFGlyphLoader_getGlyphBitmap	_getGlyphBitmap;
	iCEPFGlyphLoader_getGlyphOutline	_getGlyphOutline;
	iCEPFGlyphLoader_unloadGlyph	_unloadGlyph;
};

/*! @}
 * end of ICEPFGlyphLoader
 */

/*! \defgroup ICEPFGlyphLoaderCB ICEPFGlyphLoaderCB
 * @{
 */

/*!
 * ID of ICEPFGlyphLoaderCB
 */
#define CEComIID_ICEPFGlyphLoaderCB 0x40a27dab

/*!
 * ICEPFGlyphLoaderCB
 */
typedef struct ICEPFGlyphLoaderCB
{
	const struct ICEPFGlyphLoaderCB_vtbl* _vtbl;
} ICEPFGlyphLoaderCB;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iPFGlyphLoaderCB	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEPFGlyphLoaderCB_queryInterface) (ICEPFGlyphLoaderCB* iPFGlyphLoaderCB, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFGlyphLoaderCB_addRef) (ICEPFGlyphLoaderCB* iPFGlyphLoaderCB);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFGlyphLoaderCB_release) (ICEPFGlyphLoaderCB* iPFGlyphLoaderCB);

typedef CEHResult (*iCEPFGlyphLoaderCB_createGlyphBitmap) (ICEPFGlyphLoaderCB* iPFGlyphLoaderCB, const struct CEPFGlyphLoaderParam* pGlyphLoaderParam, const UINT32 glyphIndex, const CEPFGlyphBitmapInfo* pSrcGlyphBitmapInfo, const CEPFGlyphAdvance* pSrcGlyphAdvance, const CEComAllocatorRec* pMemFuncs, struct ICEUnknown* pCallData, UINT8** pRawBitmapBaseAddrOut, CEPFGlyphBitmapInfo* pGlyphBitmapInfoOut, CEPFGlyphAdvance* pGlyphAdvanceOut);

/*!
 * V-table of ICEPFGlyphLoaderCB
 */
struct ICEPFGlyphLoaderCB_vtbl
{
	iCEPFGlyphLoaderCB_queryInterface	_queryInterface;	//!< Query interface.
	iCEPFGlyphLoaderCB_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEPFGlyphLoaderCB_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEPFGlyphLoaderCB_createGlyphBitmap	_createGlyphBitmap;
};

/*! @}
 * end of ICEPFGlyphLoaderCB
 */

/*! \defgroup ICEPFFace ICEPFFace
 * @{
 */

/*!
 * ID of ICEPFFace
 */
#define CEComIID_ICEPFFace 0xb4fa3d81


/*!
* ICEPFFace
*
* \note
* The typical usage for measurement is:
*
*   CEComICEPFFaceRef iFace = ...;
*
*
*   iFace.createGlyphLoader(sizeInPoints, 0, ...);
*   TODO: fill this part...
*
*   UINT16 glyphIndex;
*   const CEPFGlyphAdvance* pAdvance;
*
*   iFace.getGlyphIndex(some_char_in_UTF32, &glyphIndex);
*   iFace.loadGlyph(size_in_points, glyphIndex);
*   iFace.getGlyphAdvance(glyphIndex, &pAdvance);
*
*   // use pAdvance for layouting
*
*   iFace.unloadGlyph();
*
*/
typedef struct ICEPFFace
{
	const struct ICEPFFace_vtbl* _vtbl;
} ICEPFFace;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iPFFace	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEPFFace_queryInterface) (ICEPFFace* iPFFace, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFFace_addRef) (ICEPFFace* iPFFace);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFFace_release) (ICEPFFace* iPFFace);


	/*!
	* Get the face information.
	*
	* \param[in]	iPFFace	Specifies interface pointer
	* \param[out]	faceInfoOut	Address of a pointer to an CEPFFaceInfo, representing the font face information.
	*/
typedef CEHResult (*iCEPFFace_getFaceInfo) (ICEPFFace* iPFFace, CEPFFaceInfo *const faceInfoOut);


	/*!
	* Creates a IPFGlyphLoader.
	*
	* \param[in]	iPFFace  Specifies interface pointer
	* \param[in]	sizeInPoints  size of glyph to render in points.
	* \param[in]	glyphStyleToRender	pointer to the CEPFFontStyle structure to render.
	* \param[out]	iPFGlyphLoaderOut	Address of a pointer to an ICEPFGlyphLoader, representing the glyph loader.
	*/
typedef CEHResult (*iCEPFFace_createGlyphLoader) (ICEPFFace* iPFFace, const CEFixed26d6 sizeInPoints, const CEPFFontStyle* glyphStyleToRender, struct ICEPFGlyphLoader* *const iPFGlyphLoaderOut);


	/*!
	 * Get specified font table entry. 
	 *
	 * \param[in] iPFFace Specifies interface pointer
	 * \param[in] fontTableTag font table tag id.
	 * \param[out] pPvTableEntryOut pointer to the font table entry.
	 *
     * \note Returns a pointer to the font table entry buffer that is 
	 * allocated by this method. Use deleteFontTableEntry() to delete the buffer.
	 */
typedef CEHResult (*iCEPFFace_getFontTableEntry) (ICEPFFace* iPFFace, UINT32 fontTabeleTag, void* *const pPvTableEntryOut);

typedef CEHResult (*iCEPFFace_lockFontTableEntries) (ICEPFFace* iPFFace);

typedef CEHResult (*iCEPFFace_unlockFontTableEntries) (ICEPFFace* iPFFace);


	/*!
	 * Deletes specified font table entry. see getFontTableEntry().
	 */

typedef CEHResult (*iCEPFFace_deleteFontTableEntry) (ICEPFFace* iPFFace, void* pTableEntry);

typedef CEHResult (*iCEPFFace_getUnitsPerEM) (ICEPFFace* iPFFace, INT32 *const unitsPerEmOut);

/*!
 * V-table of ICEPFFace
 */
struct ICEPFFace_vtbl
{
	iCEPFFace_queryInterface	_queryInterface;	//!< Query interface.
	iCEPFFace_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEPFFace_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEPFFace_getFaceInfo	_getFaceInfo;
	iCEPFFace_createGlyphLoader	_createGlyphLoader;
	iCEPFFace_getFontTableEntry	_getFontTableEntry;
	iCEPFFace_lockFontTableEntries	_lockFontTableEntries;
	iCEPFFace_unlockFontTableEntries	_unlockFontTableEntries;
	iCEPFFace_deleteFontTableEntry	_deleteFontTableEntry;
	iCEPFFace_getUnitsPerEM	_getUnitsPerEM;
};

/*! @}
 * end of ICEPFFace
 */

/*! \defgroup ICEPFFaceCollection ICEPFFaceCollection
 * @{
 */

/*!
 * ID of ICEPFFaceCollection
 */
#define CEComIID_ICEPFFaceCollection 0xdac20b27


/*!
* ICEPFFaceCollection
*/
typedef struct ICEPFFaceCollection
{
	const struct ICEPFFaceCollection_vtbl* _vtbl;
} ICEPFFaceCollection;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iPFFaceCollection	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEPFFaceCollection_queryInterface) (ICEPFFaceCollection* iPFFaceCollection, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFFaceCollection_addRef) (ICEPFFaceCollection* iPFFaceCollection);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFFaceCollection_release) (ICEPFFaceCollection* iPFFaceCollection);


	/*!
	* Get the number of faces contained in this face collection
	*
	* \param[in]	iPFFaceCollection	Specifies interface pointer
	* \param[out]	numOfFacesOut	Address of an variable to receive the number of faces
	*/
typedef CEHResult (*iCEPFFaceCollection_getNumOfFaces) (ICEPFFaceCollection* iPFFaceCollection, UINT32 *const numOfFacesOut);


	/*!
	* Open the specified font file
	*
	* \param[in]	iPFFaceCollection	Specifies interface pointer
	* \param[in]	faceIndex	Specifies the index of the font face in the font file
	* \param[out]	iPFFaceColOut	Address of a pointer to an ICEPRFace, representing the opened font face
	*/
typedef CEHResult (*iCEPFFaceCollection_openFace) (ICEPFFaceCollection* iPFFaceCollection, const UINT32 faceIndex, struct ICEPFFace* *const iPFFaceOut);

/*!
 * V-table of ICEPFFaceCollection
 */
struct ICEPFFaceCollection_vtbl
{
	iCEPFFaceCollection_queryInterface	_queryInterface;	//!< Query interface.
	iCEPFFaceCollection_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEPFFaceCollection_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEPFFaceCollection_getNumOfFaces	_getNumOfFaces;
	iCEPFFaceCollection_openFace	_openFace;
};

/*! @}
 * end of ICEPFFaceCollection
 */

/*! \defgroup ICEPFRasterizer ICEPFRasterizer
 * @{
 */

/*!
 * ID of ICEPFRasterizer
 */
#define CEComIID_ICEPFRasterizer 0x26659fda


/*!
* ICEPFRasterizer
*/
typedef struct ICEPFRasterizer
{
	const struct ICEPFRasterizer_vtbl* _vtbl;
} ICEPFRasterizer;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iPFRasterizer	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEPFRasterizer_queryInterface) (ICEPFRasterizer* iPFRasterizer, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFRasterizer_addRef) (ICEPFRasterizer* iPFRasterizer);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFRasterizer_release) (ICEPFRasterizer* iPFRasterizer);


	/*!
	* Open the specified font resource from stream.
	*
	* \param[in]	iPFRasterizer	Specifies interface pointer
	* \param[in]	fontResourceName	Specifies the pointer to the font resource name in ascii character array.
	* \param[in]	fontResourceNameSizeInBytes	Specifies the size of fontResourceName in bytes.
	* \param[in]	stream	readable A readable font data stream.
	* \param[out]	iPFFaceColOut	Address of a pointer to an ICEPRFaceCollection, representing the opened font face collection
	*/
typedef CEHResult (*iCEPFRasterizer_openFont) (ICEPFRasterizer* iPFRasterizer, struct ICEPFFontResource* fontResource, const eCEPFFontCharacterMap charmap, struct ICEPFFaceCollection* *const iPFFaceColOut);


	/*!
	* Activate/Deactivate glyph cache
	*
	* \param[in]	iPFRasterizer  Specifies interface pointer
	* \param[in]	isActive  Specifies the status of the glyph cache. 
	* \param[in]	pParam Specifies the activation parameters. If isActive argument is false, this value is ignored and can be a 0.
	*
	* \note
	* After deactivating the glyph cache, all glyph related requests can be failed.
	* In the implementation side, all glyph cache memory should be released on deactivated.
	*/
typedef CEHResult (*iCEPFRasterizer_activate) (ICEPFRasterizer* iPFRasterizer, const bool isActive, const CEPFRasterizerActivateParam* pParam);


	/*!
	* set rasterizer option.
	*
	* \param[in]	iPFRasterizer Specifies interface pointer
	* \param[in]	eCEPFRasterizerOptionsMask combination of eCEPFRasterizerOptions.
	*
	* \note
	* If a glyph is loaded (see ICEFace::loadGlyph()) then this method will be failed. So, unload first (see ICEFace::unloadGlyph()) and call this method.
	*
	* This method reset the glyph cache.
	*
	* see CEFRCreateFontRasterizer().
	*/
typedef CEHResult (*iCEPFRasterizer_setOption) (ICEPFRasterizer* iPFRasterizer, const UINT32 eCEPFRasterizerOptionsMask);


	/*!
	* get rasterizer option.
	*
	* \param[in]	iPFRasterizer Specifies interface pointer
	* \param[out]	eCEPFRasterizerOptionsMask combination of eCEPFRasterizerOptions.
	*
	* \note
	* see CEFRCreateFontRasterizer().
	*/
typedef CEHResult (*iCEPFRasterizer_getOption) (ICEPFRasterizer* iPFRasterizer, UINT32 *const eCEPFRasterizerOptionsMaskOut);


	/*!
	* specify rasterizer specific params.
	*
	* \param[in]	iPFRasterizerSpecifies interface pointer
	* \param[in]	params: pointer to the UINT32 array.
	* \param[in]	numOfParams: number of UINT32 item in params.
	*
	*/
typedef CEHResult (*iCEPFRasterizer_setRasterizerSpecificParams) (ICEPFRasterizer* iPFRasterizer, const UINT32* params, UINT32 numOfParams);


	/*!
	* returns rasterizer specific params.
	*
	* \param[in]	iPFRasterizerSpecifies interface pointer
	* \param[out]	params: pointer to the UINT32 array.
	* \param[out]	numOfParams: number of UINT32 item in params.
	*
	*/
typedef CEHResult (*iCEPFRasterizer_getRasterizerSpecificParams) (ICEPFRasterizer* iPFRasterizer, const UINT32* *const paramsOut, UINT32*const numOfParamsOut);


	/*!
	* delete iface.
	*/

typedef CEHResult (*iCEPFRasterizer_deleteMemoryFace) (ICEPFRasterizer* iPFRasterizer, struct ICEPFFace* iface);

/*!
 * V-table of ICEPFRasterizer
 */
struct ICEPFRasterizer_vtbl
{
	iCEPFRasterizer_queryInterface	_queryInterface;	//!< Query interface.
	iCEPFRasterizer_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEPFRasterizer_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEPFRasterizer_openFont	_openFont;
	iCEPFRasterizer_activate	_activate;
	iCEPFRasterizer_setOption	_setOption;
	iCEPFRasterizer_getOption	_getOption;
	iCEPFRasterizer_setRasterizerSpecificParams	_setRasterizerSpecificParams;
	iCEPFRasterizer_getRasterizerSpecificParams	_getRasterizerSpecificParams;
	iCEPFRasterizer_deleteMemoryFace	_deleteMemoryFace;
};

/*! @}
 * end of ICEPFRasterizer
 */

/*! \defgroup ICEPFRasterizerClass ICEPFRasterizerClass
 * @{
 */

/*!
 * ID of ICEPFRasterizerClass
 */
#define CEComIID_ICEPFRasterizerClass 0x2dc32f70


/*!
* ICEPFRasterizerClass
*/
typedef struct ICEPFRasterizerClass
{
	const struct ICEPFRasterizerClass_vtbl* _vtbl;
} ICEPFRasterizerClass;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iPFRasterizerClass	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEPFRasterizerClass_queryInterface) (ICEPFRasterizerClass* iPFRasterizerClass, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFRasterizerClass_addRef) (ICEPFRasterizerClass* iPFRasterizerClass);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFRasterizerClass_release) (ICEPFRasterizerClass* iPFRasterizerClass);


	/*!
	* create ICEPFRasterizer.
	*
	* \param[in]	iPFRasterizerClassSpecifies interface pointer
	* \param[in]	CEPFRasterizerInitParam.
	* \param[out]	iPFRasterizerOut	Address of a pointer to an ICEPRasterizer, representing the font rasterizer.
	*
	*/
typedef CEHResult (*iCEPFRasterizerClass_createInstance) (ICEPFRasterizerClass* iPFRasterizerClass, CEPFRasterizerInitParam* param, struct ICEPFRasterizer* *const iPFRasterizerOut);

/*!
 * V-table of ICEPFRasterizerClass
 */
struct ICEPFRasterizerClass_vtbl
{
	iCEPFRasterizerClass_queryInterface	_queryInterface;	//!< Query interface.
	iCEPFRasterizerClass_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEPFRasterizerClass_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEPFRasterizerClass_createInstance	_createInstance;
};

/*! @}
 * end of ICEPFRasterizerClass
 */

/*! \defgroup ICEPFResourceResolver ICEPFResourceResolver
 * @{
 */

/*!
 * ID of ICEPFResourceResolver
 */
#define CEComIID_ICEPFResourceResolver 0x98488a31


/*!
 * Font resource resolver
 * In most cases, this implementation is independent from that of font manager and font rasterizer.
 * Applications have to combine appropriate font resource resolver and font rasterizer on each
 * platform.
 */
typedef struct ICEPFResourceResolver
{
	const struct ICEPFResourceResolver_vtbl* _vtbl;
} ICEPFResourceResolver;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iPFResourceResolver	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEPFResourceResolver_queryInterface) (ICEPFResourceResolver* iPFResourceResolver, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFResourceResolver_addRef) (ICEPFResourceResolver* iPFResourceResolver);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFResourceResolver_release) (ICEPFResourceResolver* iPFResourceResolver);

	//! return the font resource for the specified font. 
	//!
	//! \param[in]  iFontFamilyName     font family name such as "HELVETICA".
	//!                                 contains only upper-case roman characters 
	//!                                 and ASCII symbols. 
	//!									you can check this normalized font family name
	//!									by asking FreeType for "tell me this font's ASCII name".
	//! \param[in]  style               pointer to the CEPFFontStyle structure.
	//! \param[in]  sizeInPoints        font size in points.
	//! \param[out] piFontResourceOut   return the font resource.
	//! \param[out] pIndexOut			when the returned font resource includes multiple font 
	//!									faces, this index specifies which face to use

typedef CEHResult (*iCEPFResourceResolver_getFontResource) (ICEPFResourceResolver* iPFResourceResolver, const struct ICESymbol* iFontFamilyName, const CEPFFontStyle* style, const CEFixed26d6 sizeInPoints, struct ICEPFFontResource* *const piFontResourceOut, UINT32 *const pIndexOut);

/*!
 * V-table of ICEPFResourceResolver
 */
struct ICEPFResourceResolver_vtbl
{
	iCEPFResourceResolver_queryInterface	_queryInterface;	//!< Query interface.
	iCEPFResourceResolver_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEPFResourceResolver_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEPFResourceResolver_getFontResource	_getFontResource;
};

/*! @}
 * end of ICEPFResourceResolver
 */

/*! \defgroup ICEPFManager ICEPFManager
 * @{
 */

/*!
 * ID of ICEPFManager
 */
#define CEComIID_ICEPFManager 0x29016805


/*!
 * Font manager
 * In most cases, applications retrieve this interface from the thread context, and get 
 * glyph loaders.
 */
typedef struct ICEPFManager
{
	const struct ICEPFManager_vtbl* _vtbl;
} ICEPFManager;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iPFManager	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEPFManager_queryInterface) (ICEPFManager* iPFManager, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFManager_addRef) (ICEPFManager* iPFManager);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEPFManager_release) (ICEPFManager* iPFManager);

	//! Set a font rasterizer to use.
	//! [Bug 5798] this method is temporary...

typedef CEHResult (*iCEPFManager_setRasterizer) (ICEPFManager* iPFManager, struct ICEPFRasterizer* iRasterizer);

	//! Set a font resource resolver to use.
	//! [Bug 5798] this method is temporary...

typedef CEHResult (*iCEPFManager_setResourceResolver) (ICEPFManager* iPFManager, struct ICEPFResourceResolver* iResolver);

	//! Return a glyph loader for the specified font. 
	//! Glyph loaders are cached in the manager.  
	//! \param[in]  pFontFamilyName     font family name such as "HELVETICA".
	//!                                 contains only upper-case roman characters 
	//!                                 and ASCII symbols. null-terminated.
	//!									you can check this normalized font family name
	//!									by asking FreeType for "tell me this font's ASCII name".
	//! \param[in]  style               combination of eCEPFFontStyle bits.
	//! \param[in]  sizeInPoints        font size in points.
	//! \param[out] piGlyphLoaderOut    return the glyph loader.

typedef CEHResult (*iCEPFManager_getGlyphLoader) (ICEPFManager* iPFManager, const char* pFontFamilyName, const CEPFFontStyle* style, const CEFixed26d6 sizeInPoints, struct ICEPFGlyphLoader* *const piGlyphLoaderOut);

	//! Purge cached resources such as glyph loaders.
	//! Application programs have to call this method before terminating the application programs.
	//! Otherwise ICEPFManager implementation may cause memory leak.

typedef CEHResult (*iCEPFManager_purgeCachedResources) (ICEPFManager* iPFManager);

/*!
 * V-table of ICEPFManager
 */
struct ICEPFManager_vtbl
{
	iCEPFManager_queryInterface	_queryInterface;	//!< Query interface.
	iCEPFManager_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEPFManager_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEPFManager_setRasterizer	_setRasterizer;
	iCEPFManager_setResourceResolver	_setResourceResolver;
	iCEPFManager_getGlyphLoader	_getGlyphLoader;
	iCEPFManager_purgeCachedResources	_purgeCachedResources;
};

/*! @}
 * end of ICEPFManager
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEPFFontResourceRef CEComICEPFFontResourceRef
 * @{
 */

class CEComICEPFFontResourceRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEPFFontResourceRef() : _iPFFontResource(0) {}
	CEComICEPFFontResourceRef(ICEPFFontResource* iOther) : _iPFFontResource(0)
	{
		if (iOther)
		{
			_iPFFontResource = iOther;
			_iPFFontResource->_vtbl->_addRef(_iPFFontResource);
		}
	}

	CEComICEPFFontResourceRef(const CEComICEPFFontResourceRef& other) : _iPFFontResource(0)
	{
		if (other._iPFFontResource)
		{
			_iPFFontResource = other._iPFFontResource;
			_iPFFontResource->_vtbl->_addRef(_iPFFontResource);
		}
	}

	~CEComICEPFFontResourceRef()
	{
		if (_iPFFontResource)
		{
			ICEPFFontResource* tmp = _iPFFontResource;
			_iPFFontResource = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEPFFontResource* iPFFontResource;
				void* _ptr;
			} uIntf;
			uIntf.iPFFontResource = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEPFFontResource, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iPFFontResource)
				{
					ICEPFFontResource* tmp = _iPFFontResource;
					_iPFFontResource = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iPFFontResource = uIntf.iPFFontResource;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEPFFontResource* object() const	{ return _iPFFontResource; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEPFFontResource* iOther)
	{
		if (_iPFFontResource)
		{
			ICEPFFontResource* tmp = _iPFFontResource;
			_iPFFontResource = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iPFFontResource = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEPFFontResource* detach()
	{
		ICEPFFontResource* iIntf = _iPFFontResource;
		_iPFFontResource = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEPFFontResource* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iPFFontResource;
		if (_iPFFontResource)
		{
			_iPFFontResource->_vtbl->_addRef(_iPFFontResource);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEPFFontResource*() const	{ return _iPFFontResource; }
	FORCEINLINE_WITHOUT_DEBUG ICEPFFontResource& operator*() const	{ return *_iPFFontResource; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iPFFontResource == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEPFFontResource* iOther) const	{ return (_iPFFontResource != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEPFFontResource* iOther) const	{ return (_iPFFontResource == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEPFFontResourceRef& operator=(const CEComICEPFFontResourceRef& other)	{ return operator=(other._iPFFontResource); }

	CEComICEPFFontResourceRef& operator=(const ICEPFFontResource* iOther)
	{
		if (_iPFFontResource != iOther)
		{
			if (_iPFFontResource)
			{
				ICEPFFontResource* tmp = _iPFFontResource;
				_iPFFontResource = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iPFFontResource = const_cast<ICEPFFontResource*>(iOther);
			if (_iPFFontResource)
			{
				_iPFFontResource->_vtbl->_addRef(_iPFFontResource);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEPFFontResource** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEPFFontResourceRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iPFFontResource && "has a com object reference. clear first.");
		return & _iPFFontResource; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEPFFontResourceRef& other)
	{
		bool result = false;
		if (_iPFFontResource)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iPFFontResource);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iPFFontResource ? _iPFFontResource->_vtbl->_queryInterface(_iPFFontResource, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iPFFontResource) { _iPFFontResource->_vtbl->_addRef(_iPFFontResource); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iPFFontResource) { _iPFFontResource->_vtbl->_release(_iPFFontResource); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult attachStream(struct ICEFSFile* *const streamOut)	{ return _iPFFontResource ? _iPFFontResource->_vtbl->_attachStream(_iPFFontResource, streamOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult detachStream(struct ICEFSFile* stream)	{ return _iPFFontResource ? _iPFFontResource->_vtbl->_detachStream(_iPFFontResource, stream) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getName(struct ICEFSPathname* *const nameOut)	{ return _iPFFontResource ? _iPFFontResource->_vtbl->_getName(_iPFFontResource, nameOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getNameStr(struct ICEUString* *const nameOut)	{ return _iPFFontResource ? _iPFFontResource->_vtbl->_getNameStr(_iPFFontResource, nameOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getClassId(UINT32 *const pClsidOut)	{ return _iPFFontResource ? _iPFFontResource->_vtbl->_getClassId(_iPFFontResource, pClsidOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getHashcode(UINT32 *const pHashcodeOut)	{ return _iPFFontResource ? _iPFFontResource->_vtbl->_getHashcode(_iPFFontResource, pHashcodeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult equals(struct ICEPFFontResource* pOther, bool *const pEqualsOut)	{ return _iPFFontResource ? _iPFFontResource->_vtbl->_equals(_iPFFontResource, pOther, pEqualsOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getBytes(const UINT8* *const bytesOut, UINT32 *const nBytesOut)	{ return _iPFFontResource ? _iPFFontResource->_vtbl->_getBytes(_iPFFontResource, bytesOut, nBytesOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEPFFontResource* _iPFFontResource;
};

/*! @}
 * end of CEComICEPFFontResourceRef
 */

/*! \defgroup CEComICEPFGlyphLoaderRef CEComICEPFGlyphLoaderRef
 * @{
 */

class CEComICEPFGlyphLoaderRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEPFGlyphLoaderRef() : _iPFGlyphLoader(0) {}
	CEComICEPFGlyphLoaderRef(ICEPFGlyphLoader* iOther) : _iPFGlyphLoader(0)
	{
		if (iOther)
		{
			_iPFGlyphLoader = iOther;
			_iPFGlyphLoader->_vtbl->_addRef(_iPFGlyphLoader);
		}
	}

	CEComICEPFGlyphLoaderRef(const CEComICEPFGlyphLoaderRef& other) : _iPFGlyphLoader(0)
	{
		if (other._iPFGlyphLoader)
		{
			_iPFGlyphLoader = other._iPFGlyphLoader;
			_iPFGlyphLoader->_vtbl->_addRef(_iPFGlyphLoader);
		}
	}

	~CEComICEPFGlyphLoaderRef()
	{
		if (_iPFGlyphLoader)
		{
			ICEPFGlyphLoader* tmp = _iPFGlyphLoader;
			_iPFGlyphLoader = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEPFGlyphLoader* iPFGlyphLoader;
				void* _ptr;
			} uIntf;
			uIntf.iPFGlyphLoader = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEPFGlyphLoader, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iPFGlyphLoader)
				{
					ICEPFGlyphLoader* tmp = _iPFGlyphLoader;
					_iPFGlyphLoader = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iPFGlyphLoader = uIntf.iPFGlyphLoader;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEPFGlyphLoader* object() const	{ return _iPFGlyphLoader; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEPFGlyphLoader* iOther)
	{
		if (_iPFGlyphLoader)
		{
			ICEPFGlyphLoader* tmp = _iPFGlyphLoader;
			_iPFGlyphLoader = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iPFGlyphLoader = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEPFGlyphLoader* detach()
	{
		ICEPFGlyphLoader* iIntf = _iPFGlyphLoader;
		_iPFGlyphLoader = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEPFGlyphLoader* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iPFGlyphLoader;
		if (_iPFGlyphLoader)
		{
			_iPFGlyphLoader->_vtbl->_addRef(_iPFGlyphLoader);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEPFGlyphLoader*() const	{ return _iPFGlyphLoader; }
	FORCEINLINE_WITHOUT_DEBUG ICEPFGlyphLoader& operator*() const	{ return *_iPFGlyphLoader; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iPFGlyphLoader == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEPFGlyphLoader* iOther) const	{ return (_iPFGlyphLoader != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEPFGlyphLoader* iOther) const	{ return (_iPFGlyphLoader == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEPFGlyphLoaderRef& operator=(const CEComICEPFGlyphLoaderRef& other)	{ return operator=(other._iPFGlyphLoader); }

	CEComICEPFGlyphLoaderRef& operator=(const ICEPFGlyphLoader* iOther)
	{
		if (_iPFGlyphLoader != iOther)
		{
			if (_iPFGlyphLoader)
			{
				ICEPFGlyphLoader* tmp = _iPFGlyphLoader;
				_iPFGlyphLoader = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iPFGlyphLoader = const_cast<ICEPFGlyphLoader*>(iOther);
			if (_iPFGlyphLoader)
			{
				_iPFGlyphLoader->_vtbl->_addRef(_iPFGlyphLoader);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEPFGlyphLoader** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEPFGlyphLoaderRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iPFGlyphLoader && "has a com object reference. clear first.");
		return & _iPFGlyphLoader; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEPFGlyphLoaderRef& other)
	{
		bool result = false;
		if (_iPFGlyphLoader)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iPFGlyphLoader);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_queryInterface(_iPFGlyphLoader, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iPFGlyphLoader) { _iPFGlyphLoader->_vtbl->_addRef(_iPFGlyphLoader); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iPFGlyphLoader) { _iPFGlyphLoader->_vtbl->_release(_iPFGlyphLoader); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getScaledFaceInfo(CEPFScaledFaceInfo* scaledFaceInfoOut)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_getScaledFaceInfo(_iPFGlyphLoader, scaledFaceInfoOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getParam(const CEPFGlyphLoaderParam* *const glyphLoaderParamOut)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_getParam(_iPFGlyphLoader, glyphLoaderParamOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphIndex2(const UTF32CHAR c32, struct ICEUnknown* pCEPFCreateGlyphBitmapFuncCallData, UINT32 *const glyphIndexOut)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_getGlyphIndex2(_iPFGlyphLoader, c32, pCEPFCreateGlyphBitmapFuncCallData, glyphIndexOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphIndex3(const UTF32CHAR c32, struct ICEUnknown* pCEPFCreateGlyphBitmapFuncCallData, UINT32 *const glyphIndexOut, CEUTextMeasurement *const optXAdvanceOut)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_getGlyphIndex3(_iPFGlyphLoader, c32, pCEPFCreateGlyphBitmapFuncCallData, glyphIndexOut, optXAdvanceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphs16(const UTF16CHAR* pC16s, UINT32 nC16s, bool bNoSurrogatePairs, UINT32 *const optGlyphsOut, UINT32 *const optNGlyphsRef)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_getGlyphs16(_iPFGlyphLoader, pC16s, nC16s, bNoSurrogatePairs, optGlyphsOut, optNGlyphsRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphPoint(UINT32 glyphIndex, INT32 pointNumber, float *const xOut, float *const yOut)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_getGlyphPoint(_iPFGlyphLoader, glyphIndex, pointNumber, xOut, yOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult loadGlyph(const UINT32 glyphIndex, struct ICEUnknown* pCEPFCreateGlyphBitmapFuncCallData)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_loadGlyph(_iPFGlyphLoader, glyphIndex, pCEPFCreateGlyphBitmapFuncCallData) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult loadGlyph2(const UINT32 glyphIndex, struct ICEUnknown* pCEPFCreateGlyphBitmapFuncCallData, CEPFGlyphAdvance *const advanceOut)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_loadGlyph2(_iPFGlyphLoader, glyphIndex, pCEPFCreateGlyphBitmapFuncCallData, advanceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult loadGlyph3(const UINT32 glyphIndex, struct ICEUnknown* pCEPFCreateGlyphBitmapFuncCallData, const CEPFGlyphMetricsPt* *const metricsPtOut)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_loadGlyph3(_iPFGlyphLoader, glyphIndex, pCEPFCreateGlyphBitmapFuncCallData, metricsPtOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphAdvance(const CEPFGlyphAdvance* *const advanceOut)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_getGlyphAdvance(_iPFGlyphLoader, advanceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphMetricsPt(const CEPFGlyphMetricsPt* *const metricsPtOut)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_getGlyphMetricsPt(_iPFGlyphLoader, metricsPtOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphBitmap(const CEPFGlyphBitmapInfo* *const bitmapInfoOut, const CEPFGlyphAdvance* *const optGlyphBmpAdvanceOut)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_getGlyphBitmap(_iPFGlyphLoader, bitmapInfoOut, optGlyphBmpAdvanceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphOutline(const CEPFGlyphOutlineInfo* *const outlineInfoOut, const CEPFGlyphAdvance* *const optGlyphOutlineAdvanceOut)	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_getGlyphOutline(_iPFGlyphLoader, outlineInfoOut, optGlyphOutlineAdvanceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unloadGlyph()	{ return _iPFGlyphLoader ? _iPFGlyphLoader->_vtbl->_unloadGlyph(_iPFGlyphLoader) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEPFGlyphLoader* _iPFGlyphLoader;
};

/*! @}
 * end of CEComICEPFGlyphLoaderRef
 */

/*! \defgroup CEComICEPFGlyphLoaderCBRef CEComICEPFGlyphLoaderCBRef
 * @{
 */

class CEComICEPFGlyphLoaderCBRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEPFGlyphLoaderCBRef() : _iPFGlyphLoaderCB(0) {}
	CEComICEPFGlyphLoaderCBRef(ICEPFGlyphLoaderCB* iOther) : _iPFGlyphLoaderCB(0)
	{
		if (iOther)
		{
			_iPFGlyphLoaderCB = iOther;
			_iPFGlyphLoaderCB->_vtbl->_addRef(_iPFGlyphLoaderCB);
		}
	}

	CEComICEPFGlyphLoaderCBRef(const CEComICEPFGlyphLoaderCBRef& other) : _iPFGlyphLoaderCB(0)
	{
		if (other._iPFGlyphLoaderCB)
		{
			_iPFGlyphLoaderCB = other._iPFGlyphLoaderCB;
			_iPFGlyphLoaderCB->_vtbl->_addRef(_iPFGlyphLoaderCB);
		}
	}

	~CEComICEPFGlyphLoaderCBRef()
	{
		if (_iPFGlyphLoaderCB)
		{
			ICEPFGlyphLoaderCB* tmp = _iPFGlyphLoaderCB;
			_iPFGlyphLoaderCB = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEPFGlyphLoaderCB* iPFGlyphLoaderCB;
				void* _ptr;
			} uIntf;
			uIntf.iPFGlyphLoaderCB = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEPFGlyphLoaderCB, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iPFGlyphLoaderCB)
				{
					ICEPFGlyphLoaderCB* tmp = _iPFGlyphLoaderCB;
					_iPFGlyphLoaderCB = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iPFGlyphLoaderCB = uIntf.iPFGlyphLoaderCB;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEPFGlyphLoaderCB* object() const	{ return _iPFGlyphLoaderCB; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEPFGlyphLoaderCB* iOther)
	{
		if (_iPFGlyphLoaderCB)
		{
			ICEPFGlyphLoaderCB* tmp = _iPFGlyphLoaderCB;
			_iPFGlyphLoaderCB = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iPFGlyphLoaderCB = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEPFGlyphLoaderCB* detach()
	{
		ICEPFGlyphLoaderCB* iIntf = _iPFGlyphLoaderCB;
		_iPFGlyphLoaderCB = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEPFGlyphLoaderCB* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iPFGlyphLoaderCB;
		if (_iPFGlyphLoaderCB)
		{
			_iPFGlyphLoaderCB->_vtbl->_addRef(_iPFGlyphLoaderCB);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEPFGlyphLoaderCB*() const	{ return _iPFGlyphLoaderCB; }
	FORCEINLINE_WITHOUT_DEBUG ICEPFGlyphLoaderCB& operator*() const	{ return *_iPFGlyphLoaderCB; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iPFGlyphLoaderCB == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEPFGlyphLoaderCB* iOther) const	{ return (_iPFGlyphLoaderCB != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEPFGlyphLoaderCB* iOther) const	{ return (_iPFGlyphLoaderCB == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEPFGlyphLoaderCBRef& operator=(const CEComICEPFGlyphLoaderCBRef& other)	{ return operator=(other._iPFGlyphLoaderCB); }

	CEComICEPFGlyphLoaderCBRef& operator=(const ICEPFGlyphLoaderCB* iOther)
	{
		if (_iPFGlyphLoaderCB != iOther)
		{
			if (_iPFGlyphLoaderCB)
			{
				ICEPFGlyphLoaderCB* tmp = _iPFGlyphLoaderCB;
				_iPFGlyphLoaderCB = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iPFGlyphLoaderCB = const_cast<ICEPFGlyphLoaderCB*>(iOther);
			if (_iPFGlyphLoaderCB)
			{
				_iPFGlyphLoaderCB->_vtbl->_addRef(_iPFGlyphLoaderCB);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEPFGlyphLoaderCB** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEPFGlyphLoaderCBRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iPFGlyphLoaderCB && "has a com object reference. clear first.");
		return & _iPFGlyphLoaderCB; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEPFGlyphLoaderCBRef& other)
	{
		bool result = false;
		if (_iPFGlyphLoaderCB)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iPFGlyphLoaderCB);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iPFGlyphLoaderCB ? _iPFGlyphLoaderCB->_vtbl->_queryInterface(_iPFGlyphLoaderCB, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iPFGlyphLoaderCB) { _iPFGlyphLoaderCB->_vtbl->_addRef(_iPFGlyphLoaderCB); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iPFGlyphLoaderCB) { _iPFGlyphLoaderCB->_vtbl->_release(_iPFGlyphLoaderCB); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createGlyphBitmap(const struct CEPFGlyphLoaderParam* pGlyphLoaderParam, const UINT32 glyphIndex, const CEPFGlyphBitmapInfo* pSrcGlyphBitmapInfo, const CEPFGlyphAdvance* pSrcGlyphAdvance, const CEComAllocatorRec* pMemFuncs, struct ICEUnknown* pCallData, UINT8** pRawBitmapBaseAddrOut, CEPFGlyphBitmapInfo* pGlyphBitmapInfoOut, CEPFGlyphAdvance* pGlyphAdvanceOut)	{ return _iPFGlyphLoaderCB ? _iPFGlyphLoaderCB->_vtbl->_createGlyphBitmap(_iPFGlyphLoaderCB, pGlyphLoaderParam, glyphIndex, pSrcGlyphBitmapInfo, pSrcGlyphAdvance, pMemFuncs, pCallData, pRawBitmapBaseAddrOut, pGlyphBitmapInfoOut, pGlyphAdvanceOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEPFGlyphLoaderCB* _iPFGlyphLoaderCB;
};

/*! @}
 * end of CEComICEPFGlyphLoaderCBRef
 */

/*! \defgroup CEComICEPFFaceRef CEComICEPFFaceRef
 * @{
 */

class CEComICEPFFaceRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEPFFaceRef() : _iPFFace(0) {}
	CEComICEPFFaceRef(ICEPFFace* iOther) : _iPFFace(0)
	{
		if (iOther)
		{
			_iPFFace = iOther;
			_iPFFace->_vtbl->_addRef(_iPFFace);
		}
	}

	CEComICEPFFaceRef(const CEComICEPFFaceRef& other) : _iPFFace(0)
	{
		if (other._iPFFace)
		{
			_iPFFace = other._iPFFace;
			_iPFFace->_vtbl->_addRef(_iPFFace);
		}
	}

	~CEComICEPFFaceRef()
	{
		if (_iPFFace)
		{
			ICEPFFace* tmp = _iPFFace;
			_iPFFace = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEPFFace* iPFFace;
				void* _ptr;
			} uIntf;
			uIntf.iPFFace = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEPFFace, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iPFFace)
				{
					ICEPFFace* tmp = _iPFFace;
					_iPFFace = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iPFFace = uIntf.iPFFace;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEPFFace* object() const	{ return _iPFFace; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEPFFace* iOther)
	{
		if (_iPFFace)
		{
			ICEPFFace* tmp = _iPFFace;
			_iPFFace = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iPFFace = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEPFFace* detach()
	{
		ICEPFFace* iIntf = _iPFFace;
		_iPFFace = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEPFFace* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iPFFace;
		if (_iPFFace)
		{
			_iPFFace->_vtbl->_addRef(_iPFFace);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEPFFace*() const	{ return _iPFFace; }
	FORCEINLINE_WITHOUT_DEBUG ICEPFFace& operator*() const	{ return *_iPFFace; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iPFFace == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEPFFace* iOther) const	{ return (_iPFFace != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEPFFace* iOther) const	{ return (_iPFFace == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEPFFaceRef& operator=(const CEComICEPFFaceRef& other)	{ return operator=(other._iPFFace); }

	CEComICEPFFaceRef& operator=(const ICEPFFace* iOther)
	{
		if (_iPFFace != iOther)
		{
			if (_iPFFace)
			{
				ICEPFFace* tmp = _iPFFace;
				_iPFFace = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iPFFace = const_cast<ICEPFFace*>(iOther);
			if (_iPFFace)
			{
				_iPFFace->_vtbl->_addRef(_iPFFace);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEPFFace** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEPFFaceRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iPFFace && "has a com object reference. clear first.");
		return & _iPFFace; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEPFFaceRef& other)
	{
		bool result = false;
		if (_iPFFace)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iPFFace);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iPFFace ? _iPFFace->_vtbl->_queryInterface(_iPFFace, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iPFFace) { _iPFFace->_vtbl->_addRef(_iPFFace); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iPFFace) { _iPFFace->_vtbl->_release(_iPFFace); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFaceInfo(CEPFFaceInfo *const faceInfoOut)	{ return _iPFFace ? _iPFFace->_vtbl->_getFaceInfo(_iPFFace, faceInfoOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createGlyphLoader(const CEFixed26d6 sizeInPoints, const CEPFFontStyle* glyphStyleToRender, struct ICEPFGlyphLoader* *const iPFGlyphLoaderOut)	{ return _iPFFace ? _iPFFace->_vtbl->_createGlyphLoader(_iPFFace, sizeInPoints, glyphStyleToRender, iPFGlyphLoaderOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFontTableEntry(UINT32 fontTabeleTag, void* *const pPvTableEntryOut)	{ return _iPFFace ? _iPFFace->_vtbl->_getFontTableEntry(_iPFFace, fontTabeleTag, pPvTableEntryOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult lockFontTableEntries()	{ return _iPFFace ? _iPFFace->_vtbl->_lockFontTableEntries(_iPFFace) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unlockFontTableEntries()	{ return _iPFFace ? _iPFFace->_vtbl->_unlockFontTableEntries(_iPFFace) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult deleteFontTableEntry(void* pTableEntry)	{ return _iPFFace ? _iPFFace->_vtbl->_deleteFontTableEntry(_iPFFace, pTableEntry) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getUnitsPerEM(INT32 *const unitsPerEmOut)	{ return _iPFFace ? _iPFFace->_vtbl->_getUnitsPerEM(_iPFFace, unitsPerEmOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEPFFace* _iPFFace;
};

/*! @}
 * end of CEComICEPFFaceRef
 */

/*! \defgroup CEComICEPFFaceCollectionRef CEComICEPFFaceCollectionRef
 * @{
 */

class CEComICEPFFaceCollectionRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEPFFaceCollectionRef() : _iPFFaceCollection(0) {}
	CEComICEPFFaceCollectionRef(ICEPFFaceCollection* iOther) : _iPFFaceCollection(0)
	{
		if (iOther)
		{
			_iPFFaceCollection = iOther;
			_iPFFaceCollection->_vtbl->_addRef(_iPFFaceCollection);
		}
	}

	CEComICEPFFaceCollectionRef(const CEComICEPFFaceCollectionRef& other) : _iPFFaceCollection(0)
	{
		if (other._iPFFaceCollection)
		{
			_iPFFaceCollection = other._iPFFaceCollection;
			_iPFFaceCollection->_vtbl->_addRef(_iPFFaceCollection);
		}
	}

	~CEComICEPFFaceCollectionRef()
	{
		if (_iPFFaceCollection)
		{
			ICEPFFaceCollection* tmp = _iPFFaceCollection;
			_iPFFaceCollection = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEPFFaceCollection* iPFFaceCollection;
				void* _ptr;
			} uIntf;
			uIntf.iPFFaceCollection = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEPFFaceCollection, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iPFFaceCollection)
				{
					ICEPFFaceCollection* tmp = _iPFFaceCollection;
					_iPFFaceCollection = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iPFFaceCollection = uIntf.iPFFaceCollection;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEPFFaceCollection* object() const	{ return _iPFFaceCollection; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEPFFaceCollection* iOther)
	{
		if (_iPFFaceCollection)
		{
			ICEPFFaceCollection* tmp = _iPFFaceCollection;
			_iPFFaceCollection = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iPFFaceCollection = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEPFFaceCollection* detach()
	{
		ICEPFFaceCollection* iIntf = _iPFFaceCollection;
		_iPFFaceCollection = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEPFFaceCollection* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iPFFaceCollection;
		if (_iPFFaceCollection)
		{
			_iPFFaceCollection->_vtbl->_addRef(_iPFFaceCollection);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEPFFaceCollection*() const	{ return _iPFFaceCollection; }
	FORCEINLINE_WITHOUT_DEBUG ICEPFFaceCollection& operator*() const	{ return *_iPFFaceCollection; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iPFFaceCollection == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEPFFaceCollection* iOther) const	{ return (_iPFFaceCollection != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEPFFaceCollection* iOther) const	{ return (_iPFFaceCollection == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEPFFaceCollectionRef& operator=(const CEComICEPFFaceCollectionRef& other)	{ return operator=(other._iPFFaceCollection); }

	CEComICEPFFaceCollectionRef& operator=(const ICEPFFaceCollection* iOther)
	{
		if (_iPFFaceCollection != iOther)
		{
			if (_iPFFaceCollection)
			{
				ICEPFFaceCollection* tmp = _iPFFaceCollection;
				_iPFFaceCollection = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iPFFaceCollection = const_cast<ICEPFFaceCollection*>(iOther);
			if (_iPFFaceCollection)
			{
				_iPFFaceCollection->_vtbl->_addRef(_iPFFaceCollection);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEPFFaceCollection** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEPFFaceCollectionRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iPFFaceCollection && "has a com object reference. clear first.");
		return & _iPFFaceCollection; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEPFFaceCollectionRef& other)
	{
		bool result = false;
		if (_iPFFaceCollection)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iPFFaceCollection);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iPFFaceCollection ? _iPFFaceCollection->_vtbl->_queryInterface(_iPFFaceCollection, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iPFFaceCollection) { _iPFFaceCollection->_vtbl->_addRef(_iPFFaceCollection); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iPFFaceCollection) { _iPFFaceCollection->_vtbl->_release(_iPFFaceCollection); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getNumOfFaces(UINT32 *const numOfFacesOut)	{ return _iPFFaceCollection ? _iPFFaceCollection->_vtbl->_getNumOfFaces(_iPFFaceCollection, numOfFacesOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult openFace(const UINT32 faceIndex, struct ICEPFFace* *const iPFFaceOut)	{ return _iPFFaceCollection ? _iPFFaceCollection->_vtbl->_openFace(_iPFFaceCollection, faceIndex, iPFFaceOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEPFFaceCollection* _iPFFaceCollection;
};

/*! @}
 * end of CEComICEPFFaceCollectionRef
 */

/*! \defgroup CEComICEPFRasterizerRef CEComICEPFRasterizerRef
 * @{
 */

class CEComICEPFRasterizerRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEPFRasterizerRef() : _iPFRasterizer(0) {}
	CEComICEPFRasterizerRef(ICEPFRasterizer* iOther) : _iPFRasterizer(0)
	{
		if (iOther)
		{
			_iPFRasterizer = iOther;
			_iPFRasterizer->_vtbl->_addRef(_iPFRasterizer);
		}
	}

	CEComICEPFRasterizerRef(const CEComICEPFRasterizerRef& other) : _iPFRasterizer(0)
	{
		if (other._iPFRasterizer)
		{
			_iPFRasterizer = other._iPFRasterizer;
			_iPFRasterizer->_vtbl->_addRef(_iPFRasterizer);
		}
	}

	~CEComICEPFRasterizerRef()
	{
		if (_iPFRasterizer)
		{
			ICEPFRasterizer* tmp = _iPFRasterizer;
			_iPFRasterizer = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEPFRasterizer* iPFRasterizer;
				void* _ptr;
			} uIntf;
			uIntf.iPFRasterizer = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEPFRasterizer, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iPFRasterizer)
				{
					ICEPFRasterizer* tmp = _iPFRasterizer;
					_iPFRasterizer = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iPFRasterizer = uIntf.iPFRasterizer;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEPFRasterizer* object() const	{ return _iPFRasterizer; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEPFRasterizer* iOther)
	{
		if (_iPFRasterizer)
		{
			ICEPFRasterizer* tmp = _iPFRasterizer;
			_iPFRasterizer = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iPFRasterizer = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEPFRasterizer* detach()
	{
		ICEPFRasterizer* iIntf = _iPFRasterizer;
		_iPFRasterizer = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEPFRasterizer* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iPFRasterizer;
		if (_iPFRasterizer)
		{
			_iPFRasterizer->_vtbl->_addRef(_iPFRasterizer);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEPFRasterizer*() const	{ return _iPFRasterizer; }
	FORCEINLINE_WITHOUT_DEBUG ICEPFRasterizer& operator*() const	{ return *_iPFRasterizer; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iPFRasterizer == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEPFRasterizer* iOther) const	{ return (_iPFRasterizer != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEPFRasterizer* iOther) const	{ return (_iPFRasterizer == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEPFRasterizerRef& operator=(const CEComICEPFRasterizerRef& other)	{ return operator=(other._iPFRasterizer); }

	CEComICEPFRasterizerRef& operator=(const ICEPFRasterizer* iOther)
	{
		if (_iPFRasterizer != iOther)
		{
			if (_iPFRasterizer)
			{
				ICEPFRasterizer* tmp = _iPFRasterizer;
				_iPFRasterizer = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iPFRasterizer = const_cast<ICEPFRasterizer*>(iOther);
			if (_iPFRasterizer)
			{
				_iPFRasterizer->_vtbl->_addRef(_iPFRasterizer);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEPFRasterizer** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEPFRasterizerRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iPFRasterizer && "has a com object reference. clear first.");
		return & _iPFRasterizer; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEPFRasterizerRef& other)
	{
		bool result = false;
		if (_iPFRasterizer)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iPFRasterizer);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iPFRasterizer ? _iPFRasterizer->_vtbl->_queryInterface(_iPFRasterizer, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iPFRasterizer) { _iPFRasterizer->_vtbl->_addRef(_iPFRasterizer); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iPFRasterizer) { _iPFRasterizer->_vtbl->_release(_iPFRasterizer); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult openFont(struct ICEPFFontResource* fontResource, const eCEPFFontCharacterMap charmap, struct ICEPFFaceCollection* *const iPFFaceColOut)	{ return _iPFRasterizer ? _iPFRasterizer->_vtbl->_openFont(_iPFRasterizer, fontResource, charmap, iPFFaceColOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult activate(const bool isActive, const CEPFRasterizerActivateParam* pParam)	{ return _iPFRasterizer ? _iPFRasterizer->_vtbl->_activate(_iPFRasterizer, isActive, pParam) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setOption(const UINT32 eCEPFRasterizerOptionsMask)	{ return _iPFRasterizer ? _iPFRasterizer->_vtbl->_setOption(_iPFRasterizer, eCEPFRasterizerOptionsMask) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getOption(UINT32 *const eCEPFRasterizerOptionsMaskOut)	{ return _iPFRasterizer ? _iPFRasterizer->_vtbl->_getOption(_iPFRasterizer, eCEPFRasterizerOptionsMaskOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setRasterizerSpecificParams(const UINT32* params, UINT32 numOfParams)	{ return _iPFRasterizer ? _iPFRasterizer->_vtbl->_setRasterizerSpecificParams(_iPFRasterizer, params, numOfParams) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getRasterizerSpecificParams(const UINT32* *const paramsOut, UINT32*const numOfParamsOut)	{ return _iPFRasterizer ? _iPFRasterizer->_vtbl->_getRasterizerSpecificParams(_iPFRasterizer, paramsOut, numOfParamsOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult deleteMemoryFace(struct ICEPFFace* iface)	{ return _iPFRasterizer ? _iPFRasterizer->_vtbl->_deleteMemoryFace(_iPFRasterizer, iface) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEPFRasterizer* _iPFRasterizer;
};

/*! @}
 * end of CEComICEPFRasterizerRef
 */

/*! \defgroup CEComICEPFRasterizerClassRef CEComICEPFRasterizerClassRef
 * @{
 */

class CEComICEPFRasterizerClassRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEPFRasterizerClassRef() : _iPFRasterizerClass(0) {}
	CEComICEPFRasterizerClassRef(ICEPFRasterizerClass* iOther) : _iPFRasterizerClass(0)
	{
		if (iOther)
		{
			_iPFRasterizerClass = iOther;
			_iPFRasterizerClass->_vtbl->_addRef(_iPFRasterizerClass);
		}
	}

	CEComICEPFRasterizerClassRef(const CEComICEPFRasterizerClassRef& other) : _iPFRasterizerClass(0)
	{
		if (other._iPFRasterizerClass)
		{
			_iPFRasterizerClass = other._iPFRasterizerClass;
			_iPFRasterizerClass->_vtbl->_addRef(_iPFRasterizerClass);
		}
	}

	~CEComICEPFRasterizerClassRef()
	{
		if (_iPFRasterizerClass)
		{
			ICEPFRasterizerClass* tmp = _iPFRasterizerClass;
			_iPFRasterizerClass = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEPFRasterizerClass* iPFRasterizerClass;
				void* _ptr;
			} uIntf;
			uIntf.iPFRasterizerClass = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEPFRasterizerClass, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iPFRasterizerClass)
				{
					ICEPFRasterizerClass* tmp = _iPFRasterizerClass;
					_iPFRasterizerClass = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iPFRasterizerClass = uIntf.iPFRasterizerClass;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEPFRasterizerClass* object() const	{ return _iPFRasterizerClass; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEPFRasterizerClass* iOther)
	{
		if (_iPFRasterizerClass)
		{
			ICEPFRasterizerClass* tmp = _iPFRasterizerClass;
			_iPFRasterizerClass = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iPFRasterizerClass = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEPFRasterizerClass* detach()
	{
		ICEPFRasterizerClass* iIntf = _iPFRasterizerClass;
		_iPFRasterizerClass = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEPFRasterizerClass* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iPFRasterizerClass;
		if (_iPFRasterizerClass)
		{
			_iPFRasterizerClass->_vtbl->_addRef(_iPFRasterizerClass);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEPFRasterizerClass*() const	{ return _iPFRasterizerClass; }
	FORCEINLINE_WITHOUT_DEBUG ICEPFRasterizerClass& operator*() const	{ return *_iPFRasterizerClass; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iPFRasterizerClass == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEPFRasterizerClass* iOther) const	{ return (_iPFRasterizerClass != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEPFRasterizerClass* iOther) const	{ return (_iPFRasterizerClass == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEPFRasterizerClassRef& operator=(const CEComICEPFRasterizerClassRef& other)	{ return operator=(other._iPFRasterizerClass); }

	CEComICEPFRasterizerClassRef& operator=(const ICEPFRasterizerClass* iOther)
	{
		if (_iPFRasterizerClass != iOther)
		{
			if (_iPFRasterizerClass)
			{
				ICEPFRasterizerClass* tmp = _iPFRasterizerClass;
				_iPFRasterizerClass = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iPFRasterizerClass = const_cast<ICEPFRasterizerClass*>(iOther);
			if (_iPFRasterizerClass)
			{
				_iPFRasterizerClass->_vtbl->_addRef(_iPFRasterizerClass);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEPFRasterizerClass** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEPFRasterizerClassRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iPFRasterizerClass && "has a com object reference. clear first.");
		return & _iPFRasterizerClass; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEPFRasterizerClassRef& other)
	{
		bool result = false;
		if (_iPFRasterizerClass)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iPFRasterizerClass);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iPFRasterizerClass ? _iPFRasterizerClass->_vtbl->_queryInterface(_iPFRasterizerClass, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iPFRasterizerClass) { _iPFRasterizerClass->_vtbl->_addRef(_iPFRasterizerClass); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iPFRasterizerClass) { _iPFRasterizerClass->_vtbl->_release(_iPFRasterizerClass); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createInstance(CEPFRasterizerInitParam* param, struct ICEPFRasterizer* *const iPFRasterizerOut)	{ return _iPFRasterizerClass ? _iPFRasterizerClass->_vtbl->_createInstance(_iPFRasterizerClass, param, iPFRasterizerOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEPFRasterizerClass* _iPFRasterizerClass;
};

/*! @}
 * end of CEComICEPFRasterizerClassRef
 */

/*! \defgroup CEComICEPFResourceResolverRef CEComICEPFResourceResolverRef
 * @{
 */

class CEComICEPFResourceResolverRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEPFResourceResolverRef() : _iPFResourceResolver(0) {}
	CEComICEPFResourceResolverRef(ICEPFResourceResolver* iOther) : _iPFResourceResolver(0)
	{
		if (iOther)
		{
			_iPFResourceResolver = iOther;
			_iPFResourceResolver->_vtbl->_addRef(_iPFResourceResolver);
		}
	}

	CEComICEPFResourceResolverRef(const CEComICEPFResourceResolverRef& other) : _iPFResourceResolver(0)
	{
		if (other._iPFResourceResolver)
		{
			_iPFResourceResolver = other._iPFResourceResolver;
			_iPFResourceResolver->_vtbl->_addRef(_iPFResourceResolver);
		}
	}

	~CEComICEPFResourceResolverRef()
	{
		if (_iPFResourceResolver)
		{
			ICEPFResourceResolver* tmp = _iPFResourceResolver;
			_iPFResourceResolver = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEPFResourceResolver* iPFResourceResolver;
				void* _ptr;
			} uIntf;
			uIntf.iPFResourceResolver = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEPFResourceResolver, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iPFResourceResolver)
				{
					ICEPFResourceResolver* tmp = _iPFResourceResolver;
					_iPFResourceResolver = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iPFResourceResolver = uIntf.iPFResourceResolver;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEPFResourceResolver* object() const	{ return _iPFResourceResolver; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEPFResourceResolver* iOther)
	{
		if (_iPFResourceResolver)
		{
			ICEPFResourceResolver* tmp = _iPFResourceResolver;
			_iPFResourceResolver = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iPFResourceResolver = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEPFResourceResolver* detach()
	{
		ICEPFResourceResolver* iIntf = _iPFResourceResolver;
		_iPFResourceResolver = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEPFResourceResolver* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iPFResourceResolver;
		if (_iPFResourceResolver)
		{
			_iPFResourceResolver->_vtbl->_addRef(_iPFResourceResolver);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEPFResourceResolver*() const	{ return _iPFResourceResolver; }
	FORCEINLINE_WITHOUT_DEBUG ICEPFResourceResolver& operator*() const	{ return *_iPFResourceResolver; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iPFResourceResolver == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEPFResourceResolver* iOther) const	{ return (_iPFResourceResolver != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEPFResourceResolver* iOther) const	{ return (_iPFResourceResolver == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEPFResourceResolverRef& operator=(const CEComICEPFResourceResolverRef& other)	{ return operator=(other._iPFResourceResolver); }

	CEComICEPFResourceResolverRef& operator=(const ICEPFResourceResolver* iOther)
	{
		if (_iPFResourceResolver != iOther)
		{
			if (_iPFResourceResolver)
			{
				ICEPFResourceResolver* tmp = _iPFResourceResolver;
				_iPFResourceResolver = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iPFResourceResolver = const_cast<ICEPFResourceResolver*>(iOther);
			if (_iPFResourceResolver)
			{
				_iPFResourceResolver->_vtbl->_addRef(_iPFResourceResolver);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEPFResourceResolver** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEPFResourceResolverRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iPFResourceResolver && "has a com object reference. clear first.");
		return & _iPFResourceResolver; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEPFResourceResolverRef& other)
	{
		bool result = false;
		if (_iPFResourceResolver)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iPFResourceResolver);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iPFResourceResolver ? _iPFResourceResolver->_vtbl->_queryInterface(_iPFResourceResolver, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iPFResourceResolver) { _iPFResourceResolver->_vtbl->_addRef(_iPFResourceResolver); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iPFResourceResolver) { _iPFResourceResolver->_vtbl->_release(_iPFResourceResolver); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getFontResource(const struct ICESymbol* iFontFamilyName, const CEPFFontStyle* style, const CEFixed26d6 sizeInPoints, struct ICEPFFontResource* *const piFontResourceOut, UINT32 *const pIndexOut)	{ return _iPFResourceResolver ? _iPFResourceResolver->_vtbl->_getFontResource(_iPFResourceResolver, iFontFamilyName, style, sizeInPoints, piFontResourceOut, pIndexOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEPFResourceResolver* _iPFResourceResolver;
};

/*! @}
 * end of CEComICEPFResourceResolverRef
 */

/*! \defgroup CEComICEPFManagerRef CEComICEPFManagerRef
 * @{
 */

class CEComICEPFManagerRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEPFManagerRef() : _iPFManager(0) {}
	CEComICEPFManagerRef(ICEPFManager* iOther) : _iPFManager(0)
	{
		if (iOther)
		{
			_iPFManager = iOther;
			_iPFManager->_vtbl->_addRef(_iPFManager);
		}
	}

	CEComICEPFManagerRef(const CEComICEPFManagerRef& other) : _iPFManager(0)
	{
		if (other._iPFManager)
		{
			_iPFManager = other._iPFManager;
			_iPFManager->_vtbl->_addRef(_iPFManager);
		}
	}

	~CEComICEPFManagerRef()
	{
		if (_iPFManager)
		{
			ICEPFManager* tmp = _iPFManager;
			_iPFManager = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEPFManager* iPFManager;
				void* _ptr;
			} uIntf;
			uIntf.iPFManager = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEPFManager, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iPFManager)
				{
					ICEPFManager* tmp = _iPFManager;
					_iPFManager = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iPFManager = uIntf.iPFManager;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEPFManager* object() const	{ return _iPFManager; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEPFManager* iOther)
	{
		if (_iPFManager)
		{
			ICEPFManager* tmp = _iPFManager;
			_iPFManager = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iPFManager = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEPFManager* detach()
	{
		ICEPFManager* iIntf = _iPFManager;
		_iPFManager = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEPFManager* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iPFManager;
		if (_iPFManager)
		{
			_iPFManager->_vtbl->_addRef(_iPFManager);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEPFManager*() const	{ return _iPFManager; }
	FORCEINLINE_WITHOUT_DEBUG ICEPFManager& operator*() const	{ return *_iPFManager; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iPFManager == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEPFManager* iOther) const	{ return (_iPFManager != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEPFManager* iOther) const	{ return (_iPFManager == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEPFManagerRef& operator=(const CEComICEPFManagerRef& other)	{ return operator=(other._iPFManager); }

	CEComICEPFManagerRef& operator=(const ICEPFManager* iOther)
	{
		if (_iPFManager != iOther)
		{
			if (_iPFManager)
			{
				ICEPFManager* tmp = _iPFManager;
				_iPFManager = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iPFManager = const_cast<ICEPFManager*>(iOther);
			if (_iPFManager)
			{
				_iPFManager->_vtbl->_addRef(_iPFManager);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEPFManager** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEPFManagerRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iPFManager && "has a com object reference. clear first.");
		return & _iPFManager; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEPFManagerRef& other)
	{
		bool result = false;
		if (_iPFManager)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iPFManager);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iPFManager ? _iPFManager->_vtbl->_queryInterface(_iPFManager, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iPFManager) { _iPFManager->_vtbl->_addRef(_iPFManager); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iPFManager) { _iPFManager->_vtbl->_release(_iPFManager); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setRasterizer(struct ICEPFRasterizer* iRasterizer)	{ return _iPFManager ? _iPFManager->_vtbl->_setRasterizer(_iPFManager, iRasterizer) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setResourceResolver(struct ICEPFResourceResolver* iResolver)	{ return _iPFManager ? _iPFManager->_vtbl->_setResourceResolver(_iPFManager, iResolver) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getGlyphLoader(const char* pFontFamilyName, const CEPFFontStyle* style, const CEFixed26d6 sizeInPoints, struct ICEPFGlyphLoader* *const piGlyphLoaderOut)	{ return _iPFManager ? _iPFManager->_vtbl->_getGlyphLoader(_iPFManager, pFontFamilyName, style, sizeInPoints, piGlyphLoaderOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult purgeCachedResources()	{ return _iPFManager ? _iPFManager->_vtbl->_purgeCachedResources(_iPFManager) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEPFManager* _iPFManager;
};

/*! @}
 * end of CEComICEPFManagerRef
 */

#endif // __cplusplus

#endif
