///////////////////////////////////////////////////////////////////////////////
// Copyright 2012 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICEJSContext_h
#define ICEJSContext_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------


//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEJSContext ICEJSContext
 * @{
 */

/*!
 * ID of ICEJSContext
 */
#define CEComIID_ICEJSContext 0xc033d0ec


/*!
 * ICEJSContext
 */
typedef struct ICEJSContext
{
	const struct ICEJSContext_vtbl* _vtbl;
} ICEJSContext;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iJSContext	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEJSContext_queryInterface) (ICEJSContext* iJSContext, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEJSContext_addRef) (ICEJSContext* iJSContext);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEJSContext_release) (ICEJSContext* iJSContext);

typedef CEHResult (*iCEJSContext_evalScript) (ICEJSContext* iJSContext, const UTF16CHAR* scriptText, UINT32 scriptTextLen);

/*!
 * V-table of ICEJSContext
 */
struct ICEJSContext_vtbl
{
	iCEJSContext_queryInterface	_queryInterface;	//!< Query interface.
	iCEJSContext_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEJSContext_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEJSContext_evalScript	_evalScript;
};

/*! @}
 * end of ICEJSContext
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEJSContextRef CEComICEJSContextRef
 * @{
 */

class CEComICEJSContextRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEJSContextRef() : _iJSContext(0) {}
	CEComICEJSContextRef(ICEJSContext* iOther) : _iJSContext(0)
	{
		if (iOther)
		{
			_iJSContext = iOther;
			_iJSContext->_vtbl->_addRef(_iJSContext);
		}
	}

	CEComICEJSContextRef(const CEComICEJSContextRef& other) : _iJSContext(0)
	{
		if (other._iJSContext)
		{
			_iJSContext = other._iJSContext;
			_iJSContext->_vtbl->_addRef(_iJSContext);
		}
	}

	~CEComICEJSContextRef()
	{
		if (_iJSContext)
		{
			ICEJSContext* tmp = _iJSContext;
			_iJSContext = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEJSContext* iJSContext;
				void* _ptr;
			} uIntf;
			uIntf.iJSContext = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEJSContext, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iJSContext)
				{
					ICEJSContext* tmp = _iJSContext;
					_iJSContext = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iJSContext = uIntf.iJSContext;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEJSContext* object() const	{ return _iJSContext; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEJSContext* iOther)
	{
		if (_iJSContext)
		{
			ICEJSContext* tmp = _iJSContext;
			_iJSContext = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iJSContext = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEJSContext* detach()
	{
		ICEJSContext* iIntf = _iJSContext;
		_iJSContext = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEJSContext* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iJSContext;
		if (_iJSContext)
		{
			_iJSContext->_vtbl->_addRef(_iJSContext);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEJSContext*() const	{ return _iJSContext; }
	FORCEINLINE_WITHOUT_DEBUG ICEJSContext& operator*() const	{ return *_iJSContext; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iJSContext == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEJSContext* iOther) const	{ return (_iJSContext != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEJSContext* iOther) const	{ return (_iJSContext == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEJSContextRef& operator=(const CEComICEJSContextRef& other)	{ return operator=(other._iJSContext); }

	CEComICEJSContextRef& operator=(const ICEJSContext* iOther)
	{
		if (_iJSContext != iOther)
		{
			if (_iJSContext)
			{
				ICEJSContext* tmp = _iJSContext;
				_iJSContext = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iJSContext = const_cast<ICEJSContext*>(iOther);
			if (_iJSContext)
			{
				_iJSContext->_vtbl->_addRef(_iJSContext);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEJSContext** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEJSContextRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iJSContext && "has a com object reference. clear first.");
		return & _iJSContext; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEJSContextRef& other)
	{
		bool result = false;
		if (_iJSContext)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iJSContext);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iJSContext ? _iJSContext->_vtbl->_queryInterface(_iJSContext, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iJSContext) { _iJSContext->_vtbl->_addRef(_iJSContext); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iJSContext) { _iJSContext->_vtbl->_release(_iJSContext); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult evalScript(const UTF16CHAR* scriptText, UINT32 scriptTextLen)	{ return _iJSContext ? _iJSContext->_vtbl->_evalScript(_iJSContext, scriptText, scriptTextLen) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEJSContext* _iJSContext;
};

/*! @}
 * end of CEComICEJSContextRef
 */

#endif // __cplusplus

#endif
