///////////////////////////////////////////////////////////////////////////////
// Copyright 2012 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef CEApiFSPathname_h
#define CEApiFSPathname_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"
#include "ICEUString.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------


//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

typedef enum
{
	eCEUPathnameType_Unknown = 0,	//!< Unknown pathname type
	eCEUPathnameType_Posix,			//!< Posix-style pathname (e.g. "/usr/include")
	eCEUPathnameType_Win32Local,	//!< Win32 local pathname (e.g. "C:\Windows\System32")
	eCEUPathnameType_Win32UNC,		//!< Win32 UNC (e.g. \\smbserver\share\localdirname)
	eCEUPathnameType_Hydra			//!< Hydra style pathname (e.g. "app0:data/include")
} eCEUPathnameType;

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEFSPathname ICEFSPathname
 * @{
 */

/*!
 * ID of ICEFSPathname
 */
#define CEComIID_ICEFSPathname 0x4beaffec


/*!
 * ICEFSPathname
 */
typedef struct ICEFSPathname
{
	const struct ICEFSPathname_vtbl* _vtbl;
} ICEFSPathname;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iFSPathname	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSPathname_queryInterface) (ICEFSPathname* iFSPathname, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSPathname_addRef) (ICEFSPathname* iFSPathname);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSPathname_release) (ICEFSPathname* iFSPathname);


/*!
 * <b>Summary:</b>
 * Initialize pathname object.<br>
 *
 * \param[in] iFSPathname Specifies interface pointer
 * \param[out] strPath Specifies the pathname string
 *
 * \return Error status
 *
 * <b>Description:</b>
 * compose the string object from string.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSPathname_init) (ICEFSPathname* iFSPathname, struct ICEUString* strPath);


/*!
 * <b>Summary:</b>
 * compose the string representation from this pathname object.<br>
 *
 * \param[in] iFSPathname Specifies interface pointer
 * \param[out] strPathOut Specifies the pointer to the variable to get the converted pathname string
 *
 * \return Error status
 *
 * <b>Description:</b>
 * compose the string representation from this pathname object.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSPathname_toString) (ICEFSPathname* iFSPathname, struct ICEUString** strOut);


/*!
 * <b>Summary:</b>
 * returns whether this pathname object is an absolute path or a relative path.<br>
 *
 * <b>Description:</b>
 * returns whether this pathname object is an absolute path or a relative path.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */

typedef CEHResult (*iCEFSPathname_isAbsolute) (ICEFSPathname* iFSPathname, bool* flagOut);


/*!
 * <b>Summary:</b>
 * get the length of the relpath part.<br>
 *
 * <b>Description:</b>
 * get the length of the relpath part.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * "aaa/bbb/ccc" ---> 3 <br>
 * "/aaa/bbb/ccc" --> 3 <br>
 * "/" ---> 0
 *
 * <b>See also:</b>
 * None
 *
 */

typedef CEHResult (*iCEFSPathname_getRelativePathLength) (ICEFSPathname* iFSPathname, UINT32* lengthOut);


/*!
 * <b>Summary:</b>
 * create the pathname object with one-level shorter relpath part.<br>
 *
 * \param[in] iFSPathname Specifies interface pointer
 * \param[out] subPathOut Specifies the reference to the variable to get the one-level shortened pathname
 *
 * \return JGSS_SUCCESS : no error. <br>
 *         JGSS_ERR_OPERATION_FAILED : the relpath part of this pathname is the root directory (getRelativePathLength() == 0).
 *
 * <b>Description:</b>
 * create the pathname object with one-level shorter relpath part.<br>
 * (e.g. "/aaa/bbb/ccc" --> "/aaa/bbb")
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSPathname_getSubPath) (ICEFSPathname* iFSPathname, struct ICEFSPathname** subPathOut);


/*!
 * <b>Summary:</b>
 * create the string object of the last directroy entry name of the relpath.<br>
 *
 * \param[in] iFSPathname Specifies interface pointer
 * \param[out] dirEntryNameOut Specifies the reference to the variable to get the last directroy entry name of the relpath
 *
 * \return an error code
 *
 * <b>Description:</b>
 * create the string object of the last directroy entry name of the relpath.<br>
 * (e.g. "/aaa/bbb/ccc" --> "ccc")
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * \todo when this path is the root directory ("/"), what should be returned?
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSPathname_getLastDirEntryName) (ICEFSPathname* iFSPathname, struct ICEUString** dirEntryNameOut);


/*!
 * <b>Summary:</b>
 * create the string object before the extension part of the last directroy entry name of the relpath.<br>
 *
 * \param[in] iFSPathname Specifies interface pointer
 * \param[out] basenameOut Specifies the reference to the variable to get the basename 
 *
 * \return error code
 *
 * <b>Description:</b>
 * create the string object of the extension part of the last directroy entry name of the relpath.<br>
 * (e.g. "/aaa/bbb/ccc.ext" --> "ccc")
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * \todo when there is no exntension part, what should be returned?
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSPathname_getBasename) (ICEFSPathname* iFSPathname, struct ICEUString** basenameOut);


/*!
 * <b>Summary:</b>
 * create the string object of the extension part of the last directroy entry name of the relpath.<br>
 *
 * \param[in] iFSPathname Specifies interface pointer
 * \param[out] extOut Specifies the reference to the variable to get the extension 
 *
 * \return error code
 *
 * <b>Description:</b>
 * create the string object of the extension part of the last directroy entry name of the relpath.<br>
 * (e.g. "/aaa/bbb/ccc.ext" --> "ext")
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * \todo when there is no exntension part, what should be returned?
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSPathname_getExtension) (ICEFSPathname* iFSPathname, struct ICEUString** extOut);


/*!
 * <b>Summary:</b>
 * get the path to the common ancestor of given 2 absolute paths.<br>
 *
 * <b>Description:</b>
 * get the path to the common ancestor of given 2 absolute paths.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */

typedef CEHResult (*iCEFSPathname_getCommonAncestor) (ICEFSPathname* iFSPathname, struct ICEFSPathname* absPath1, struct ICEFSPathname* absPath2, struct ICEFSPathname* commonPathOut);


/*!
 * <b>Summary:</b>
 * append the relative path to the end of the absolute path.<br>
 *
 * <b>Description:</b>
 * concat the relative path to the end of the absolute path.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */	

typedef CEHResult (*iCEFSPathname_append) (ICEFSPathname* iFSPathname, struct ICEFSPathname* relpath, struct ICEFSPathname** pathOut);


/*!
 * <b>Summary:</b>
 * TODO<br>
 *
 * \param[in]	iFSPathname	Specifies interface pointer
 * \param	token	TODO
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSPathname_appendDirEntry) (ICEFSPathname* iFSPathname, struct ICEUString* token);


/*!
 * <b>Summary:</b>
 * TODO<br>
 *
 * \param[in]	iFSPathname	Specifies interface pointer
 * \param	token	TODO
 *
 * \return Error status
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSPathname_setDriveLetter) (ICEFSPathname* iFSPathname, struct ICEUString* token);


/*!
 * <b>Summary:</b>
 * TODO<br>
 *
 * \param[in]	iFSPathname	Specifies interface pointer
 * \param	token	TODO
 *
 * \return Error type
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSPathname_setType) (ICEFSPathname* iFSPathname, eCEUPathnameType type);


/*!
 * <b>Summary:</b>
 * TODO<br>
 *
 * \param[in]	iFSPathname	Specifies interface pointer
 * \param	token	TODO
 *
 * \return Error isAbsolute
 *
 * <b>Description:</b>
 * None
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 *
 */
typedef CEHResult (*iCEFSPathname_setIsAbsolute) (ICEFSPathname* iFSPathname, bool isAbsolute);

/*!
 * V-table of ICEFSPathname
 */
struct ICEFSPathname_vtbl
{
	iCEFSPathname_queryInterface	_queryInterface;	//!< Query interface.
	iCEFSPathname_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEFSPathname_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEFSPathname_init	_init;	//!< Initialize pathname object.
	iCEFSPathname_toString	_toString;	//!< compose the string representation from this pathname object.
	iCEFSPathname_isAbsolute	_isAbsolute;	//!< returns whether this pathname object is an absolute path or a relative path.
	iCEFSPathname_getRelativePathLength	_getRelativePathLength;	//!< get the length of the relpath part.
	iCEFSPathname_getSubPath	_getSubPath;	//!< create the pathname object with one-level shorter relpath part.
	iCEFSPathname_getLastDirEntryName	_getLastDirEntryName;	//!< create the string object of the last directroy entry name of the relpath.
	iCEFSPathname_getBasename	_getBasename;	//!< create the string object before the extension part of the last directroy entry name of the relpath.
	iCEFSPathname_getExtension	_getExtension;	//!< create the string object of the extension part of the last directroy entry name of the relpath.
	iCEFSPathname_getCommonAncestor	_getCommonAncestor;	//!< get the path to the common ancestor of given 2 absolute paths.
	iCEFSPathname_append	_append;	//!< append the relative path to the end of the absolute path.
	iCEFSPathname_appendDirEntry	_appendDirEntry;	//!< TODO
	iCEFSPathname_setDriveLetter	_setDriveLetter;	//!< TODO
	iCEFSPathname_setType	_setType;	//!< TODO
	iCEFSPathname_setIsAbsolute	_setIsAbsolute;	//!< TODO
};

/*! @}
 * end of ICEFSPathname
 */

/*! \defgroup ICEFSPathnameFactory ICEFSPathnameFactory
 * @{
 */

/*!
 * ID of ICEFSPathnameFactory
 */
#define CEComIID_ICEFSPathnameFactory 0xd43f5b5d


/*!
 * ICEFSPathnameFactory
 */
typedef struct ICEFSPathnameFactory
{
	const struct ICEFSPathnameFactory_vtbl* _vtbl;
} ICEFSPathnameFactory;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iFSPathnameFactory	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEFSPathnameFactory_queryInterface) (ICEFSPathnameFactory* iFSPathnameFactory, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSPathnameFactory_addRef) (ICEFSPathnameFactory* iFSPathnameFactory);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEFSPathnameFactory_release) (ICEFSPathnameFactory* iFSPathnameFactory);

/*!
 * V-table of ICEFSPathnameFactory
 */
struct ICEFSPathnameFactory_vtbl
{
	iCEFSPathnameFactory_queryInterface	_queryInterface;	//!< Query interface.
	iCEFSPathnameFactory_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEFSPathnameFactory_release	_release;	//!< Decrease the reference count of the specified interface pointer.
};

/*! @}
 * end of ICEFSPathnameFactory
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEFSPathnameRef CEComICEFSPathnameRef
 * @{
 */

class CEComICEFSPathnameRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEFSPathnameRef() : _iFSPathname(0) {}
	CEComICEFSPathnameRef(ICEFSPathname* iOther) : _iFSPathname(0)
	{
		if (iOther)
		{
			_iFSPathname = iOther;
			_iFSPathname->_vtbl->_addRef(_iFSPathname);
		}
	}

	CEComICEFSPathnameRef(const CEComICEFSPathnameRef& other) : _iFSPathname(0)
	{
		if (other._iFSPathname)
		{
			_iFSPathname = other._iFSPathname;
			_iFSPathname->_vtbl->_addRef(_iFSPathname);
		}
	}

	~CEComICEFSPathnameRef()
	{
		if (_iFSPathname)
		{
			ICEFSPathname* tmp = _iFSPathname;
			_iFSPathname = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEFSPathname* iFSPathname;
				void* _ptr;
			} uIntf;
			uIntf.iFSPathname = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEFSPathname, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iFSPathname)
				{
					ICEFSPathname* tmp = _iFSPathname;
					_iFSPathname = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iFSPathname = uIntf.iFSPathname;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEFSPathname* object() const	{ return _iFSPathname; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEFSPathname* iOther)
	{
		if (_iFSPathname)
		{
			ICEFSPathname* tmp = _iFSPathname;
			_iFSPathname = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iFSPathname = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEFSPathname* detach()
	{
		ICEFSPathname* iIntf = _iFSPathname;
		_iFSPathname = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEFSPathname* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iFSPathname;
		if (_iFSPathname)
		{
			_iFSPathname->_vtbl->_addRef(_iFSPathname);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEFSPathname*() const	{ return _iFSPathname; }
	FORCEINLINE_WITHOUT_DEBUG ICEFSPathname& operator*() const	{ return *_iFSPathname; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iFSPathname == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEFSPathname* iOther) const	{ return (_iFSPathname != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEFSPathname* iOther) const	{ return (_iFSPathname == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEFSPathnameRef& operator=(const CEComICEFSPathnameRef& other)	{ return operator=(other._iFSPathname); }

	CEComICEFSPathnameRef& operator=(const ICEFSPathname* iOther)
	{
		if (_iFSPathname != iOther)
		{
			if (_iFSPathname)
			{
				ICEFSPathname* tmp = _iFSPathname;
				_iFSPathname = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iFSPathname = const_cast<ICEFSPathname*>(iOther);
			if (_iFSPathname)
			{
				_iFSPathname->_vtbl->_addRef(_iFSPathname);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEFSPathname** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEFSPathnameRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iFSPathname && "has a com object reference. clear first.");
		return & _iFSPathname; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEFSPathnameRef& other)
	{
		bool result = false;
		if (_iFSPathname)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iFSPathname);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iFSPathname ? _iFSPathname->_vtbl->_queryInterface(_iFSPathname, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iFSPathname) { _iFSPathname->_vtbl->_addRef(_iFSPathname); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iFSPathname) { _iFSPathname->_vtbl->_release(_iFSPathname); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult init(struct ICEUString* strPath)	{ return _iFSPathname ? _iFSPathname->_vtbl->_init(_iFSPathname, strPath) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult toString(struct ICEUString** strOut)	{ return _iFSPathname ? _iFSPathname->_vtbl->_toString(_iFSPathname, strOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isAbsolute(bool* flagOut)	{ return _iFSPathname ? _iFSPathname->_vtbl->_isAbsolute(_iFSPathname, flagOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getRelativePathLength(UINT32* lengthOut)	{ return _iFSPathname ? _iFSPathname->_vtbl->_getRelativePathLength(_iFSPathname, lengthOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getSubPath(struct ICEFSPathname** subPathOut)	{ return _iFSPathname ? _iFSPathname->_vtbl->_getSubPath(_iFSPathname, subPathOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLastDirEntryName(struct ICEUString** dirEntryNameOut)	{ return _iFSPathname ? _iFSPathname->_vtbl->_getLastDirEntryName(_iFSPathname, dirEntryNameOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getBasename(struct ICEUString** basenameOut)	{ return _iFSPathname ? _iFSPathname->_vtbl->_getBasename(_iFSPathname, basenameOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getExtension(struct ICEUString** extOut)	{ return _iFSPathname ? _iFSPathname->_vtbl->_getExtension(_iFSPathname, extOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCommonAncestor(struct ICEFSPathname* absPath1, struct ICEFSPathname* absPath2, struct ICEFSPathname* commonPathOut)	{ return _iFSPathname ? _iFSPathname->_vtbl->_getCommonAncestor(_iFSPathname, absPath1, absPath2, commonPathOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult append(struct ICEFSPathname* relpath, struct ICEFSPathname** pathOut)	{ return _iFSPathname ? _iFSPathname->_vtbl->_append(_iFSPathname, relpath, pathOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult appendDirEntry(struct ICEUString* token)	{ return _iFSPathname ? _iFSPathname->_vtbl->_appendDirEntry(_iFSPathname, token) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setDriveLetter(struct ICEUString* token)	{ return _iFSPathname ? _iFSPathname->_vtbl->_setDriveLetter(_iFSPathname, token) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setType(eCEUPathnameType type)	{ return _iFSPathname ? _iFSPathname->_vtbl->_setType(_iFSPathname, type) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setIsAbsolute(bool isAbsolute)	{ return _iFSPathname ? _iFSPathname->_vtbl->_setIsAbsolute(_iFSPathname, isAbsolute) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEFSPathname* _iFSPathname;
};

/*! @}
 * end of CEComICEFSPathnameRef
 */

/*! \defgroup CEComICEFSPathnameFactoryRef CEComICEFSPathnameFactoryRef
 * @{
 */

class CEComICEFSPathnameFactoryRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEFSPathnameFactoryRef() : _iFSPathnameFactory(0) {}
	CEComICEFSPathnameFactoryRef(ICEFSPathnameFactory* iOther) : _iFSPathnameFactory(0)
	{
		if (iOther)
		{
			_iFSPathnameFactory = iOther;
			_iFSPathnameFactory->_vtbl->_addRef(_iFSPathnameFactory);
		}
	}

	CEComICEFSPathnameFactoryRef(const CEComICEFSPathnameFactoryRef& other) : _iFSPathnameFactory(0)
	{
		if (other._iFSPathnameFactory)
		{
			_iFSPathnameFactory = other._iFSPathnameFactory;
			_iFSPathnameFactory->_vtbl->_addRef(_iFSPathnameFactory);
		}
	}

	~CEComICEFSPathnameFactoryRef()
	{
		if (_iFSPathnameFactory)
		{
			ICEFSPathnameFactory* tmp = _iFSPathnameFactory;
			_iFSPathnameFactory = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEFSPathnameFactory* iFSPathnameFactory;
				void* _ptr;
			} uIntf;
			uIntf.iFSPathnameFactory = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEFSPathnameFactory, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iFSPathnameFactory)
				{
					ICEFSPathnameFactory* tmp = _iFSPathnameFactory;
					_iFSPathnameFactory = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iFSPathnameFactory = uIntf.iFSPathnameFactory;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEFSPathnameFactory* object() const	{ return _iFSPathnameFactory; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEFSPathnameFactory* iOther)
	{
		if (_iFSPathnameFactory)
		{
			ICEFSPathnameFactory* tmp = _iFSPathnameFactory;
			_iFSPathnameFactory = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iFSPathnameFactory = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEFSPathnameFactory* detach()
	{
		ICEFSPathnameFactory* iIntf = _iFSPathnameFactory;
		_iFSPathnameFactory = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEFSPathnameFactory* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iFSPathnameFactory;
		if (_iFSPathnameFactory)
		{
			_iFSPathnameFactory->_vtbl->_addRef(_iFSPathnameFactory);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEFSPathnameFactory*() const	{ return _iFSPathnameFactory; }
	FORCEINLINE_WITHOUT_DEBUG ICEFSPathnameFactory& operator*() const	{ return *_iFSPathnameFactory; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iFSPathnameFactory == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEFSPathnameFactory* iOther) const	{ return (_iFSPathnameFactory != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEFSPathnameFactory* iOther) const	{ return (_iFSPathnameFactory == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEFSPathnameFactoryRef& operator=(const CEComICEFSPathnameFactoryRef& other)	{ return operator=(other._iFSPathnameFactory); }

	CEComICEFSPathnameFactoryRef& operator=(const ICEFSPathnameFactory* iOther)
	{
		if (_iFSPathnameFactory != iOther)
		{
			if (_iFSPathnameFactory)
			{
				ICEFSPathnameFactory* tmp = _iFSPathnameFactory;
				_iFSPathnameFactory = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iFSPathnameFactory = const_cast<ICEFSPathnameFactory*>(iOther);
			if (_iFSPathnameFactory)
			{
				_iFSPathnameFactory->_vtbl->_addRef(_iFSPathnameFactory);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEFSPathnameFactory** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEFSPathnameFactoryRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iFSPathnameFactory && "has a com object reference. clear first.");
		return & _iFSPathnameFactory; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEFSPathnameFactoryRef& other)
	{
		bool result = false;
		if (_iFSPathnameFactory)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iFSPathnameFactory);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iFSPathnameFactory ? _iFSPathnameFactory->_vtbl->_queryInterface(_iFSPathnameFactory, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iFSPathnameFactory) { _iFSPathnameFactory->_vtbl->_addRef(_iFSPathnameFactory); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iFSPathnameFactory) { _iFSPathnameFactory->_vtbl->_release(_iFSPathnameFactory); } }


private:
	ICEFSPathnameFactory* _iFSPathnameFactory;
};

/*! @}
 * end of CEComICEFSPathnameFactoryRef
 */

#endif // __cplusplus

#endif
