///////////////////////////////////////////////////////////////////////////////
// Copyright 2012 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICEScriptable_h
#define ICEScriptable_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------

struct ICEUString;
struct ICESymbol;
struct CEVariant;
struct CEVariantArgs;
struct ICEHtmlBrowserManager;

//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEScriptable ICEScriptable
 * @{
 */

/*!
 * ID of ICEScriptable
 */
#define CEComIID_ICEScriptable 0xf733331e


/*!
 * ICEScriptable
 */
typedef struct ICEScriptable
{
	const struct ICEScriptable_vtbl* _vtbl;
} ICEScriptable;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iScriptable	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEScriptable_queryInterface) (ICEScriptable* iScriptable, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEScriptable_addRef) (ICEScriptable* iScriptable);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEScriptable_release) (ICEScriptable* iScriptable);


/*! 
 * <b>Summary:</b>
 * get the specified property value.<br>
 * 
 * \param[in]	iScriptable	Specifies interface pointer
 * \param[in]	iPropName	Specifies the property identifier
 * \param[out]	propValueOut	Pointer to the variable to receive the property value
 *
 * \return Error status
 */
typedef CEHResult (*iCEScriptable_getProperty) (ICEScriptable* iScriptable, struct ICESymbol* iPropName, CEVariant *const propValueOut);


/*! 
 * <b>Summary:</b>
 * set the value to the specified property.<br>
 * 
 * \param[in]	iScriptable	Specifies interface pointer
 * \param[in]	iPropName	Specifies the property identifier
 * \param[out]	propValueOut	Pointer to the variable to hold the property value to be set.
 *
 * \return Error status
 */
typedef CEHResult (*iCEScriptable_setProperty) (ICEScriptable* iScriptable, struct ICESymbol* iPropName, CEVariant* propValue);


/*! 
 * <b>Summary:</b>
 * remove the specified property.<br>
 * 
 * \param[in]	iScriptable	Specifies interface pointer
 * \param[in]	iPropName	Specifies the property identifier
 *
 * \return Error status
 */
typedef CEHResult (*iCEScriptable_removeProperty) (ICEScriptable* iScriptable, struct ICESymbol* iPropName);


/*! 
 * <b>Summary:</b>
 * check if a given property exists.<br>
 * 
 * \param[in]	iScriptable	Specifies interface pointer
 * \param[in]	iPropName	Specifies the property identifier
 * \param[out]	bHasProperty	true if the property exists, else false.
 *
 * \return Error status
 */
typedef CEHResult (*iCEScriptable_hasProperty) (ICEScriptable* iScriptable, struct ICESymbol* iPropName, bool& bHasProperty);


/*! 
 * <b>Summary:</b>
 * check if a given method exists.<br>
 * 
 * \param[in]	iScriptable	Specifies interface pointer
 * \param[in]	iMethodName	Specifies the method identifier
 * \param[out]	bHasMethod	true if the method exists, else false.
 *
 * \return Error status
 */
typedef CEHResult (*iCEScriptable_hasMethod) (ICEScriptable* iScriptable, struct ICESymbol* iMethodName, bool& bHasMethod);


/*! 
 * <b>Summary:</b>
 * invoke the specified method.<br>
 * 
 * \param[in]	iScriptable	Specifies interface pointer
 * \param[in]	iMethodName	Specifies the method identifier
 * \param[in/out]	args	Pointer to the variable to hold the args value.
 *
 * \return Error status
 */
typedef CEHResult (*iCEScriptable_invokeMethod) (ICEScriptable* iScriptable, struct ICESymbol* iMethodName, CEVariantArgs* args);


/*! 
 * <b>Summary:</b>
 * apply the own method.<br>
 * 
 * \param[int/out]	args	Pointer to the variable to hold the args value.
 *
 * \param[in]	iScriptable	Specifies interface pointer
 *
 * \return Error status
 */
typedef CEHResult (*iCEScriptable_applyMethod) (ICEScriptable* iScriptable, CEVariantArgs* args);

/*!
 * V-table of ICEScriptable
 */
struct ICEScriptable_vtbl
{
	iCEScriptable_queryInterface	_queryInterface;	//!< Query interface.
	iCEScriptable_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEScriptable_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEScriptable_getProperty	_getProperty;	//!< get the specified property value.
	iCEScriptable_setProperty	_setProperty;	//!< set the value to the specified property.
	iCEScriptable_removeProperty	_removeProperty;	//!< remove the specified property.
	iCEScriptable_hasProperty	_hasProperty;	//!< check if a given property exists.
	iCEScriptable_hasMethod	_hasMethod;	//!< check if a given method exists.
	iCEScriptable_invokeMethod	_invokeMethod;	//!< invoke the specified method.
	iCEScriptable_applyMethod	_applyMethod;	//!< apply the own method.
};

/*! @}
 * end of ICEScriptable
 */

/*! \defgroup ICEScriptContext ICEScriptContext
 * @{
 */

/*!
 * ID of ICEScriptContext
 */
#define CEComIID_ICEScriptContext 0x65a0cac6


/*!
 * ICEScriptingContext
 */
typedef struct ICEScriptContext
{
	const struct ICEScriptContext_vtbl* _vtbl;
} ICEScriptContext;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iScriptContext	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEScriptContext_queryInterface) (ICEScriptContext* iScriptContext, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEScriptContext_addRef) (ICEScriptContext* iScriptContext);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEScriptContext_release) (ICEScriptContext* iScriptContext);


	/*!
	 * find the unique symbol object in the caller's name scope
	 *
	 * \param[in] iScriptContext  Specifies interface pointer
	 * \param[in] iName  specifies the name of the symbol to find out in the caller's name scope
	 * \param[out] iSymbolOut  the pointer to ICEXmlSymbol interface pointer to receive the found result
	 */
typedef CEHResult (*iCEScriptContext_findSymbol) (ICEScriptContext* iScriptContext, struct ICEUString* iName, struct ICESymbol* *const iSymbolOut);


	/*!
	 * add the symbol-value pair into the caller's name scope
	 *
	 * \param[in] iScriptContext  Specifies interface pointer
	 * \param[in] iName  specifies the name of the symbol to find out in the caller's name scope
	 * \param[out] iSymbolOut  the pointer to ICEXmlSymbol interface pointer to receive the found result
	 */
typedef CEHResult (*iCEScriptContext_intern) (ICEScriptContext* iScriptContext, struct ICEUString* iName, struct ICESymbol* *const iSymbolOut);

typedef CEHResult (*iCEScriptContext_internByUTF16CharArray) (ICEScriptContext* iScriptContext, const UTF16CHAR* pCharArray16, UINT32 numOfChars16, struct ICESymbol* *const iSymbolOut);

typedef CEHResult (*iCEScriptContext_getProperty) (ICEScriptContext* iScriptContext, struct ICESymbol* iPropName, CEVariant *const propValueOut);

typedef CEHResult (*iCEScriptContext_setProperty) (ICEScriptContext* iScriptContext, struct ICESymbol* iPropName, CEVariant* propValue);

/*!
 * V-table of ICEScriptContext
 */
struct ICEScriptContext_vtbl
{
	iCEScriptContext_queryInterface	_queryInterface;	//!< Query interface.
	iCEScriptContext_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEScriptContext_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEScriptContext_findSymbol	_findSymbol;
	iCEScriptContext_intern	_intern;
	iCEScriptContext_internByUTF16CharArray	_internByUTF16CharArray;
	iCEScriptContext_getProperty	_getProperty;
	iCEScriptContext_setProperty	_setProperty;
};

/*! @}
 * end of ICEScriptContext
 */

/*! \defgroup ICEScriptableMessage ICEScriptableMessage
 * @{
 */

/*!
 * ID of ICEScriptableMessage
 */
#define CEComIID_ICEScriptableMessage 0xfb370901


/*!
 * ICEScriptableMessage
 * this interface used for ICEHtmlBrowserManager.postMessage().
 */
typedef struct ICEScriptableMessage
{
	const struct ICEScriptableMessage_vtbl* _vtbl;
} ICEScriptableMessage;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iScriptableMessage	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEScriptableMessage_queryInterface) (ICEScriptableMessage* iScriptableMessage, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEScriptableMessage_addRef) (ICEScriptableMessage* iScriptableMessage);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEScriptableMessage_release) (ICEScriptableMessage* iScriptableMessage);

typedef CEHResult (*iCEScriptableMessage_processMessage) (ICEScriptableMessage* iScriptableMessage);

/*!
 * V-table of ICEScriptableMessage
 */
struct ICEScriptableMessage_vtbl
{
	iCEScriptableMessage_queryInterface	_queryInterface;	//!< Query interface.
	iCEScriptableMessage_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEScriptableMessage_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEScriptableMessage_processMessage	_processMessage;
};

/*! @}
 * end of ICEScriptableMessage
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEScriptableRef CEComICEScriptableRef
 * @{
 */

class CEComICEScriptableRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEScriptableRef() : _iScriptable(0) {}
	CEComICEScriptableRef(ICEScriptable* iOther) : _iScriptable(0)
	{
		if (iOther)
		{
			_iScriptable = iOther;
			_iScriptable->_vtbl->_addRef(_iScriptable);
		}
	}

	CEComICEScriptableRef(const CEComICEScriptableRef& other) : _iScriptable(0)
	{
		if (other._iScriptable)
		{
			_iScriptable = other._iScriptable;
			_iScriptable->_vtbl->_addRef(_iScriptable);
		}
	}

	~CEComICEScriptableRef()
	{
		if (_iScriptable)
		{
			ICEScriptable* tmp = _iScriptable;
			_iScriptable = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEScriptable* iScriptable;
				void* _ptr;
			} uIntf;
			uIntf.iScriptable = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEScriptable, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iScriptable)
				{
					ICEScriptable* tmp = _iScriptable;
					_iScriptable = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iScriptable = uIntf.iScriptable;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEScriptable* object() const	{ return _iScriptable; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEScriptable* iOther)
	{
		if (_iScriptable)
		{
			ICEScriptable* tmp = _iScriptable;
			_iScriptable = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iScriptable = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEScriptable* detach()
	{
		ICEScriptable* iIntf = _iScriptable;
		_iScriptable = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEScriptable* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iScriptable;
		if (_iScriptable)
		{
			_iScriptable->_vtbl->_addRef(_iScriptable);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEScriptable*() const	{ return _iScriptable; }
	FORCEINLINE_WITHOUT_DEBUG ICEScriptable& operator*() const	{ return *_iScriptable; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iScriptable == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEScriptable* iOther) const	{ return (_iScriptable != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEScriptable* iOther) const	{ return (_iScriptable == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEScriptableRef& operator=(const CEComICEScriptableRef& other)	{ return operator=(other._iScriptable); }

	CEComICEScriptableRef& operator=(const ICEScriptable* iOther)
	{
		if (_iScriptable != iOther)
		{
			if (_iScriptable)
			{
				ICEScriptable* tmp = _iScriptable;
				_iScriptable = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iScriptable = const_cast<ICEScriptable*>(iOther);
			if (_iScriptable)
			{
				_iScriptable->_vtbl->_addRef(_iScriptable);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEScriptable** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEScriptableRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iScriptable && "has a com object reference. clear first.");
		return & _iScriptable; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEScriptableRef& other)
	{
		bool result = false;
		if (_iScriptable)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iScriptable);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iScriptable ? _iScriptable->_vtbl->_queryInterface(_iScriptable, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iScriptable) { _iScriptable->_vtbl->_addRef(_iScriptable); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iScriptable) { _iScriptable->_vtbl->_release(_iScriptable); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getProperty(struct ICESymbol* iPropName, CEVariant *const propValueOut)	{ return _iScriptable ? _iScriptable->_vtbl->_getProperty(_iScriptable, iPropName, propValueOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setProperty(struct ICESymbol* iPropName, CEVariant* propValue)	{ return _iScriptable ? _iScriptable->_vtbl->_setProperty(_iScriptable, iPropName, propValue) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult removeProperty(struct ICESymbol* iPropName)	{ return _iScriptable ? _iScriptable->_vtbl->_removeProperty(_iScriptable, iPropName) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult hasProperty(struct ICESymbol* iPropName, bool& bHasProperty)	{ return _iScriptable ? _iScriptable->_vtbl->_hasProperty(_iScriptable, iPropName, bHasProperty) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult hasMethod(struct ICESymbol* iMethodName, bool& bHasMethod)	{ return _iScriptable ? _iScriptable->_vtbl->_hasMethod(_iScriptable, iMethodName, bHasMethod) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult invokeMethod(struct ICESymbol* iMethodName, CEVariantArgs* args)	{ return _iScriptable ? _iScriptable->_vtbl->_invokeMethod(_iScriptable, iMethodName, args) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult applyMethod(CEVariantArgs* args)	{ return _iScriptable ? _iScriptable->_vtbl->_applyMethod(_iScriptable, args) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEScriptable* _iScriptable;
};

/*! @}
 * end of CEComICEScriptableRef
 */

/*! \defgroup CEComICEScriptContextRef CEComICEScriptContextRef
 * @{
 */

class CEComICEScriptContextRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEScriptContextRef() : _iScriptContext(0) {}
	CEComICEScriptContextRef(ICEScriptContext* iOther) : _iScriptContext(0)
	{
		if (iOther)
		{
			_iScriptContext = iOther;
			_iScriptContext->_vtbl->_addRef(_iScriptContext);
		}
	}

	CEComICEScriptContextRef(const CEComICEScriptContextRef& other) : _iScriptContext(0)
	{
		if (other._iScriptContext)
		{
			_iScriptContext = other._iScriptContext;
			_iScriptContext->_vtbl->_addRef(_iScriptContext);
		}
	}

	~CEComICEScriptContextRef()
	{
		if (_iScriptContext)
		{
			ICEScriptContext* tmp = _iScriptContext;
			_iScriptContext = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEScriptContext* iScriptContext;
				void* _ptr;
			} uIntf;
			uIntf.iScriptContext = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEScriptContext, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iScriptContext)
				{
					ICEScriptContext* tmp = _iScriptContext;
					_iScriptContext = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iScriptContext = uIntf.iScriptContext;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEScriptContext* object() const	{ return _iScriptContext; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEScriptContext* iOther)
	{
		if (_iScriptContext)
		{
			ICEScriptContext* tmp = _iScriptContext;
			_iScriptContext = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iScriptContext = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEScriptContext* detach()
	{
		ICEScriptContext* iIntf = _iScriptContext;
		_iScriptContext = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEScriptContext* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iScriptContext;
		if (_iScriptContext)
		{
			_iScriptContext->_vtbl->_addRef(_iScriptContext);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEScriptContext*() const	{ return _iScriptContext; }
	FORCEINLINE_WITHOUT_DEBUG ICEScriptContext& operator*() const	{ return *_iScriptContext; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iScriptContext == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEScriptContext* iOther) const	{ return (_iScriptContext != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEScriptContext* iOther) const	{ return (_iScriptContext == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEScriptContextRef& operator=(const CEComICEScriptContextRef& other)	{ return operator=(other._iScriptContext); }

	CEComICEScriptContextRef& operator=(const ICEScriptContext* iOther)
	{
		if (_iScriptContext != iOther)
		{
			if (_iScriptContext)
			{
				ICEScriptContext* tmp = _iScriptContext;
				_iScriptContext = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iScriptContext = const_cast<ICEScriptContext*>(iOther);
			if (_iScriptContext)
			{
				_iScriptContext->_vtbl->_addRef(_iScriptContext);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEScriptContext** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEScriptContextRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iScriptContext && "has a com object reference. clear first.");
		return & _iScriptContext; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEScriptContextRef& other)
	{
		bool result = false;
		if (_iScriptContext)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iScriptContext);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iScriptContext ? _iScriptContext->_vtbl->_queryInterface(_iScriptContext, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iScriptContext) { _iScriptContext->_vtbl->_addRef(_iScriptContext); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iScriptContext) { _iScriptContext->_vtbl->_release(_iScriptContext); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult findSymbol(struct ICEUString* iName, struct ICESymbol* *const iSymbolOut)	{ return _iScriptContext ? _iScriptContext->_vtbl->_findSymbol(_iScriptContext, iName, iSymbolOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult intern(struct ICEUString* iName, struct ICESymbol* *const iSymbolOut)	{ return _iScriptContext ? _iScriptContext->_vtbl->_intern(_iScriptContext, iName, iSymbolOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult internByUTF16CharArray(const UTF16CHAR* pCharArray16, UINT32 numOfChars16, struct ICESymbol* *const iSymbolOut)	{ return _iScriptContext ? _iScriptContext->_vtbl->_internByUTF16CharArray(_iScriptContext, pCharArray16, numOfChars16, iSymbolOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getProperty(struct ICESymbol* iPropName, CEVariant *const propValueOut)	{ return _iScriptContext ? _iScriptContext->_vtbl->_getProperty(_iScriptContext, iPropName, propValueOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setProperty(struct ICESymbol* iPropName, CEVariant* propValue)	{ return _iScriptContext ? _iScriptContext->_vtbl->_setProperty(_iScriptContext, iPropName, propValue) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEScriptContext* _iScriptContext;
};

/*! @}
 * end of CEComICEScriptContextRef
 */

/*! \defgroup CEComICEScriptableMessageRef CEComICEScriptableMessageRef
 * @{
 */

class CEComICEScriptableMessageRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEScriptableMessageRef() : _iScriptableMessage(0) {}
	CEComICEScriptableMessageRef(ICEScriptableMessage* iOther) : _iScriptableMessage(0)
	{
		if (iOther)
		{
			_iScriptableMessage = iOther;
			_iScriptableMessage->_vtbl->_addRef(_iScriptableMessage);
		}
	}

	CEComICEScriptableMessageRef(const CEComICEScriptableMessageRef& other) : _iScriptableMessage(0)
	{
		if (other._iScriptableMessage)
		{
			_iScriptableMessage = other._iScriptableMessage;
			_iScriptableMessage->_vtbl->_addRef(_iScriptableMessage);
		}
	}

	~CEComICEScriptableMessageRef()
	{
		if (_iScriptableMessage)
		{
			ICEScriptableMessage* tmp = _iScriptableMessage;
			_iScriptableMessage = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEScriptableMessage* iScriptableMessage;
				void* _ptr;
			} uIntf;
			uIntf.iScriptableMessage = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEScriptableMessage, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iScriptableMessage)
				{
					ICEScriptableMessage* tmp = _iScriptableMessage;
					_iScriptableMessage = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iScriptableMessage = uIntf.iScriptableMessage;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEScriptableMessage* object() const	{ return _iScriptableMessage; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEScriptableMessage* iOther)
	{
		if (_iScriptableMessage)
		{
			ICEScriptableMessage* tmp = _iScriptableMessage;
			_iScriptableMessage = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iScriptableMessage = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEScriptableMessage* detach()
	{
		ICEScriptableMessage* iIntf = _iScriptableMessage;
		_iScriptableMessage = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEScriptableMessage* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iScriptableMessage;
		if (_iScriptableMessage)
		{
			_iScriptableMessage->_vtbl->_addRef(_iScriptableMessage);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEScriptableMessage*() const	{ return _iScriptableMessage; }
	FORCEINLINE_WITHOUT_DEBUG ICEScriptableMessage& operator*() const	{ return *_iScriptableMessage; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iScriptableMessage == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEScriptableMessage* iOther) const	{ return (_iScriptableMessage != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEScriptableMessage* iOther) const	{ return (_iScriptableMessage == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEScriptableMessageRef& operator=(const CEComICEScriptableMessageRef& other)	{ return operator=(other._iScriptableMessage); }

	CEComICEScriptableMessageRef& operator=(const ICEScriptableMessage* iOther)
	{
		if (_iScriptableMessage != iOther)
		{
			if (_iScriptableMessage)
			{
				ICEScriptableMessage* tmp = _iScriptableMessage;
				_iScriptableMessage = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iScriptableMessage = const_cast<ICEScriptableMessage*>(iOther);
			if (_iScriptableMessage)
			{
				_iScriptableMessage->_vtbl->_addRef(_iScriptableMessage);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEScriptableMessage** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEScriptableMessageRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iScriptableMessage && "has a com object reference. clear first.");
		return & _iScriptableMessage; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEScriptableMessageRef& other)
	{
		bool result = false;
		if (_iScriptableMessage)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iScriptableMessage);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iScriptableMessage ? _iScriptableMessage->_vtbl->_queryInterface(_iScriptableMessage, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iScriptableMessage) { _iScriptableMessage->_vtbl->_addRef(_iScriptableMessage); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iScriptableMessage) { _iScriptableMessage->_vtbl->_release(_iScriptableMessage); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult processMessage()	{ return _iScriptableMessage ? _iScriptableMessage->_vtbl->_processMessage(_iScriptableMessage) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEScriptableMessage* _iScriptableMessage;
};

/*! @}
 * end of CEComICEScriptableMessageRef
 */

#endif // __cplusplus

#endif
