///////////////////////////////////////////////////////////////////////////////
// Copyright 2012 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICEI18n_h
#define ICEI18n_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CECom.h"
#include "CEApiUnknown.h"
#include "ceutypes.h"
#include "ICEI18nTypes.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------

struct ICEFSPathname;

//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

typedef enum
{
	eCEI18nCharsetDetectStatus_NoAnswerYet,
	eCEI18nCharsetDetectStatus_ProbablyAnswer,
	eCEI18nCharsetDetectStatus_SureAnswer,
	eCEI18nCharsetDetectStatus_NoAnswerMatch
} eCEI18nCharsetDetectStatus;


/*!
* i18n specific error code.
*
* If returned CEHResult is JGSS_SUCCESS, returned eCEI18nErr indicates the i18n specific error code.
* otherwise (CEHResult != JGSS_SUCCESS), the value of eCEI18nErr is unspecified.
*
*/
typedef enum
{
  eCEI18nErr_None = 0,
  // i18nErr_Failure
  //    The mask to indicate the severiality of the error
  eCEI18nErr_Failure         = 0x80000000,

  eCEI18nErr_ErrorGeneric    = eCEI18nErr_Failure  | 1,
  eCEI18nErr_MemErr          = eCEI18nErr_Failure  | 2,

  eCEI18nErr_NullPointer     = eCEI18nErr_Failure  | 3,
  eCEI18nErr_IllegalValue    = eCEI18nErr_Failure  | 4,
  eCEI18nErr_Unexpected      = eCEI18nErr_Failure  | 5,

  // i18nErr_NoMapping:
  //   The converter hit an undefined character in this charset
  eCEI18nErr_NoMapping       = eCEI18nErr_Failure  | 6,

  // i18nErr_IllegalInput:
  //   The input byte sequence does not match with the charset's 
  //   encoding structure
  eCEI18nErr_IllegalInput    = eCEI18nErr_Failure  | 7,

  // i18nErr_MoreOutput:
  //   The converter consume some data but need more output
  //   buffer to complete convert the whole input buffer
  eCEI18nErr_MoreOutput    = 8, 
  // i18nErr_MoreInput:
  //   The converter consume some data but need more input
  //   buffer to complete convert the last multibyte characters.
  //   this error will be generated if the last byte pass to the converter 
  //   is not the last byte of a multibyte characters.
  eCEI18nErr_MoreInput     = 9

} eCEI18nErr;

typedef enum
{
	eCEI18nNormalizeType_NFC, 
	eCEI18nNormalizeType_NFD,
	eCEI18nNormalizeType_NFKD,
	eCEI18nNormalizeType_NFKC
} eCEI18nNormalizeType;

typedef enum
{
	eCEI18nCollationStrength_Primary,
	eCEI18nCollationStrength_Secondary,
	eCEI18nCollationStrength_Tertiary,
	eCEI18nCollationStrength_Identical
} eCEI18nCollationStrength;

typedef enum
{
	eCEI18nCollationDecomposition_None,
	eCEI18nCollationDecomposition_Canonical,
	eCEI18nCollationDecomposition_Full
} eCEI18nCollationDecomposition;

typedef enum
{
	eCEI18nGIFlags_None				= 0,
	eCEI18nGIFlags_GlyphIndexies	= 1 << 0,
	eCEI18nGIFlags_BasedIndexies	= 1 << 1,
	eCEI18nGIFlags_Positions		= 1 << 2,
	eCEI18nGIFlags_All				= (eCEI18nGIFlags_GlyphIndexies|eCEI18nGIFlags_BasedIndexies|eCEI18nGIFlags_Positions)
} eCEI18nGIFlags;

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

typedef struct CEI18nConverterInitArg
{
	ICEFSPathname* dataDir;
} CEI18nConverterInitArg;

typedef struct CEI18nCTLLayoutEngine
{
} CEI18nCTLLayoutEngine;

typedef struct CEI18nGlyphInfo
{
	UINT32	eCEI18nGIFlagsIn;		//![in]
	CEUTextMeasurement	width;		//![out]
	UINT32	numOfGlyphs;			//![out]
	UINT32* pGlyphs;				//![out]
	INT32*	pCharIndices;			//![out]
	float*  pPositions;				//! [out] (X1, Y1,..., XnumOfGlyphs, YnumOfGlyphs, width, height)
	CEI18nCTLLayoutEngine*	pLe;	// internal use. 
} CEI18nGlyphInfo;

typedef struct CEI18nBiDiCCodeStack
{
} CEI18nBiDiCCodeStack;

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEI18nLocale ICEI18nLocale
 * @{
 */

/*!
 * ID of ICEI18nLocale
 */
#define CEComIID_ICEI18nLocale 0x83889cc8

/*!
 * ICEI18nLocale
 */
typedef struct ICEI18nLocale
{
	const struct ICEI18nLocale_vtbl* _vtbl;
} ICEI18nLocale;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nLocale	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nLocale_queryInterface) (ICEI18nLocale* iI18nLocale, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nLocale_addRef) (ICEI18nLocale* iI18nLocale);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nLocale_release) (ICEI18nLocale* iI18nLocale);


	/*!
	*  returns ISO 639 two letter language code.
	* http://www.w3.org/International/articles/language-tags/
	* http://www.ics.uci.edu/pub/ietf/http/related/iso639.txt
	*/

typedef CEHResult (*iCEI18nLocale_getISO2LLanguageCode) (ICEI18nLocale* iI18nLocale, struct ICEUString* *const optIStrOut, struct ICESymbol* *const optISymbolOut);


	/*!
	 *  returns ISO 3166 two letter country code.
	 *  http://www.w3.org/International/articles/language-tags/
	 *	 http://www.chemie.fu-berlin.de/diverse/doc/ISO_3166.html
	 */

typedef CEHResult (*iCEI18nLocale_getISO2LCountryCode) (ICEI18nLocale* iI18nLocale, struct ICEUString* *const optIStrOut, struct ICESymbol* *const optISymbolOut);


	/*!
	* returns HTML/xhtml language tag (RFC 3066). see http://www.ietf.org/rfc/rfc3066.txt
	* http://www.w3.org/International/articles/language-tags/
	*/

typedef CEHResult (*iCEI18nLocale_getLanguageTag) (ICEI18nLocale* iI18nLocale, struct ICEUString* *const optIStrOut, struct ICESymbol* *const optISymbolOut);


	/*!
	*  returns ISO 639 three letter language code.
	* http://www.ics.uci.edu/pub/ietf/http/related/iso639.txt
	* http://www.w3.org/International/articles/language-tags/
	*/

typedef CEHResult (*iCEI18nLocale_getISO3LLanguageCode) (ICEI18nLocale* iI18nLocale, struct ICEUString* *const optIStrOut, struct ICESymbol* *const optISymbolOut);


	/*!
	 *  returns ISO 3166 three letter country code.
	 *	 http://www.chemie.fu-berlin.de/diverse/doc/ISO_3166.html
	 *  http://www.w3.org/International/articles/language-tags/
	 */

typedef CEHResult (*iCEI18nLocale_getISO3LCountryCode) (ICEI18nLocale* iI18nLocale, struct ICEUString* *const optIStrOut, struct ICESymbol* *const optISymbolOut);

typedef CEHResult (*iCEI18nLocale_getDisplayLanguage) (ICEI18nLocale* iI18nLocale, const struct ICEI18nLocale* displayLocale, struct ICEUString* *const strOut);

typedef CEHResult (*iCEI18nLocale_getDisplayCountry) (ICEI18nLocale* iI18nLocale, const struct ICEI18nLocale* displayLocale, struct ICEUString* *const strOut);

typedef CEHResult (*iCEI18nLocale_equalsCStringLiteral) (ICEI18nLocale* iI18nLocale, const char* lang, const char* country, bool *const bLangMatchedOut, bool *const bCountryMatchedOut);

typedef CEHResult (*iCEI18nLocale_equals) (ICEI18nLocale* iI18nLocale, const struct ICEI18nLocale* other, bool *const bSameLangOut, bool *const bSameCountryOut);


	/*!
	* returns parent locale of this object.
	*
	* For instance, the locale "ja" is a parent of the locale "ja-JP".
	*
	*/

typedef CEHResult (*iCEI18nLocale_getParentLocale) (ICEI18nLocale* iI18nLocale, struct ICEI18nLocale* *const parentOut);

typedef CEHResult (*iCEI18nLocale_dbgToString) (ICEI18nLocale* iI18nLocale, const UTF16CHAR* *const pStr16Out);

/*!
 * V-table of ICEI18nLocale
 */
struct ICEI18nLocale_vtbl
{
	iCEI18nLocale_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nLocale_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nLocale_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nLocale_getISO2LLanguageCode	_getISO2LLanguageCode;
	iCEI18nLocale_getISO2LCountryCode	_getISO2LCountryCode;
	iCEI18nLocale_getLanguageTag	_getLanguageTag;
	iCEI18nLocale_getISO3LLanguageCode	_getISO3LLanguageCode;
	iCEI18nLocale_getISO3LCountryCode	_getISO3LCountryCode;
	iCEI18nLocale_getDisplayLanguage	_getDisplayLanguage;
	iCEI18nLocale_getDisplayCountry	_getDisplayCountry;
	iCEI18nLocale_equalsCStringLiteral	_equalsCStringLiteral;
	iCEI18nLocale_equals	_equals;
	iCEI18nLocale_getParentLocale	_getParentLocale;
	iCEI18nLocale_dbgToString	_dbgToString;
};

/*! @}
 * end of ICEI18nLocale
 */

/*! \defgroup ICEI18nCharsetEncoding ICEI18nCharsetEncoding
 * @{
 */

/*!
 * ID of ICEI18nCharsetEncoding
 */
#define CEComIID_ICEI18nCharsetEncoding 0xf34d01cc

/*!
 * ICEI18nCharsetEncoding
 */
typedef struct ICEI18nCharsetEncoding
{
	const struct ICEI18nCharsetEncoding_vtbl* _vtbl;
} ICEI18nCharsetEncoding;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nCharsetEncoding	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nCharsetEncoding_queryInterface) (ICEI18nCharsetEncoding* iI18nCharsetEncoding, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nCharsetEncoding_addRef) (ICEI18nCharsetEncoding* iI18nCharsetEncoding);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nCharsetEncoding_release) (ICEI18nCharsetEncoding* iI18nCharsetEncoding);

	//! http://www.iana.org/assignments/character-sets

typedef CEHResult (*iCEI18nCharsetEncoding_getIanaName) (ICEI18nCharsetEncoding* iI18nCharsetEncoding, struct ICEUString* *const ianaNameOut);


	/*! returns MIB enum (http://www.iana.org/assignments/character-sets) and implementation depend uniq id of this charset encoding.
	*
	* ex..
	* UINT32 id = 0;
	* eCEI18nIanaMIBEnum mibEnum = 0;
	* enc.getId(&mibEnum, &id);
	* if ( mibEnum == eCEI18nIanaMIBEnum_other )
	* {
	*		// this charset encoding is not registerd in RFC 3808.
	* }
	*/

typedef CEHResult (*iCEI18nCharsetEncoding_getId) (ICEI18nCharsetEncoding* iI18nCharsetEncoding, eCEI18nIanaMIBEnum *const ianaMIBEnumOut, UINT32 *const idOut);

	//! returns string representation of this instance for display. ex.. "Unicode (utf-8)".

typedef CEHResult (*iCEI18nCharsetEncoding_getDisplayName) (ICEI18nCharsetEncoding* iI18nCharsetEncoding, const struct ICEI18nLocale* displayLocale, struct ICEUString* *const displayNameOut);

typedef CEHResult (*iCEI18nCharsetEncoding_getUnicodeEncoder) (ICEI18nCharsetEncoding* iI18nCharsetEncoding, struct ICEI18nUnicodeEncoder* *const unicodeEncoderOut);

typedef CEHResult (*iCEI18nCharsetEncoding_getUnicodeDecoder) (ICEI18nCharsetEncoding* iI18nCharsetEncoding, struct ICEI18nUnicodeDecoder* *const unicodeDecoderOut);

typedef CEHResult (*iCEI18nCharsetEncoding_getLocale) (ICEI18nCharsetEncoding* iI18nCharsetEncoding, struct ICEI18nLocale* *const localeOut);

typedef CEHResult (*iCEI18nCharsetEncoding_equals) (ICEI18nCharsetEncoding* iI18nCharsetEncoding, const struct ICEI18nCharsetEncoding* other, bool *const bEqualsOut);

typedef CEHResult (*iCEI18nCharsetEncoding_dbgToString) (ICEI18nCharsetEncoding* iI18nCharsetEncoding, const UTF16CHAR* *const pStr16Out);

/*!
 * V-table of ICEI18nCharsetEncoding
 */
struct ICEI18nCharsetEncoding_vtbl
{
	iCEI18nCharsetEncoding_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nCharsetEncoding_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nCharsetEncoding_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nCharsetEncoding_getIanaName	_getIanaName;
	iCEI18nCharsetEncoding_getId	_getId;
	iCEI18nCharsetEncoding_getDisplayName	_getDisplayName;
	iCEI18nCharsetEncoding_getUnicodeEncoder	_getUnicodeEncoder;
	iCEI18nCharsetEncoding_getUnicodeDecoder	_getUnicodeDecoder;
	iCEI18nCharsetEncoding_getLocale	_getLocale;
	iCEI18nCharsetEncoding_equals	_equals;
	iCEI18nCharsetEncoding_dbgToString	_dbgToString;
};

/*! @}
 * end of ICEI18nCharsetEncoding
 */

/*! \defgroup ICEI18nClassFactory ICEI18nClassFactory
 * @{
 */

/*!
 * ID of ICEI18nClassFactory
 */
#define CEComIID_ICEI18nClassFactory 0xdcf45c7f

/*!
 * ICEI18nClassFactory
 */
typedef struct ICEI18nClassFactory
{
	const struct ICEI18nClassFactory_vtbl* _vtbl;
} ICEI18nClassFactory;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nClassFactory	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nClassFactory_queryInterface) (ICEI18nClassFactory* iI18nClassFactory, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nClassFactory_addRef) (ICEI18nClassFactory* iI18nClassFactory);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nClassFactory_release) (ICEI18nClassFactory* iI18nClassFactory);


	/*! 
	* specifiy 0 to encodingsOut to get the number of supported encodings.
	* otherwise, encodingsOut must be allocated by caller before call this method. And specify 
	* the number of allocated items to numOfEncodingsRef. If encodingsOut is not 0, 
	* set the number of items of encodingsOut to numOfEncodingsRef.
	*
	* bIanaRegisteredOnly: true: returns IANA registered charset encodings only. false: returns all charset encodings.
	*
	* return: JGSS_SUCCESS: supported ICEI18nCharsetEncoding interface pointers are returned in encodingOut.
	*						And, the number of returned items is setted to the numOfEncodingsRef.
	*						Since, encodingsOut items are addRef'ed by this call, caller must 
	*						call release() for each items when those items are no-longer necessary.
	*/

typedef CEHResult (*iCEI18nClassFactory_getCharsetEncodings) (ICEI18nClassFactory* iI18nClassFactory, bool bIanaRegisteredOnly, struct ICEI18nCharsetEncoding* *const encodingsOut, UINT32 *const numOfEncodingsRef);

typedef CEHResult (*iCEI18nClassFactory_getCharsetEncodingFromIanaName) (ICEI18nClassFactory* iI18nClassFactory, const UTF16CHAR* str16, const UINT32 numOfChar16, struct ICEI18nCharsetEncoding* *const charsetEncodingOut);

typedef CEHResult (*iCEI18nClassFactory_getCharsetEncodingFromIanaCStringLiteral) (ICEI18nClassFactory* iI18nClassFactory, const char* str8, struct ICEI18nCharsetEncoding* *const charsetEncodingOut);


	/*! 
	* specifiy 0 to encodingsOut to get the number of supported ICEI18nCharsetDetector implementations.
	* otherwise, charsetDetectorInfosOut must be allocated by caller before call this method. And specify 
	* the number of allocated items to numOfCharsetDetectorInfosRef. If charsetDetectorInfosOut is not 0, 
	* set the number of items of charsetDetectorInfosOut to numOfCharsetDetectorInfosRef.
	*
	* return: JGSS_SUCCESS: supported ICEI18nCharsetEncoding interface pointers are returned in encodingOut.
	*						And, the number of returned items is setted to the numOfCharsetDetectorInfosRef.
	*						Since, encodingsOut items are addRef'ed by this call, caller must 
	*						call release() for each items when those items are no-longer necessary.
	*/

typedef CEHResult (*iCEI18nClassFactory_getCharsetDetectorInfos) (ICEI18nClassFactory* iI18nClassFactory, struct ICEI18nCharsetDetectorInfo* *const charsetDetectorInfosOut, UINT32 *const numOfCharsetDetectorInfosRef);

typedef CEHResult (*iCEI18nClassFactory_getCharsetDetectorInfo) (ICEI18nClassFactory* iI18nClassFactory, UINT32 charsetDetectorCLSID, struct ICEI18nCharsetDetectorInfo* *const charsetDetectorInfoOut);


	/*!
	* returns available ICEI18nLocale(s). These ICEI18nLocale(s) can be specified to get the display-name of I18n cecom interface pointers.
	* specify 0 to availableLocalesOut to get the number of available locales. If availableLocalesOut is not 0, 
	* set the number of items of availableLocalesOut to numOfAvailableLocalesRef.
	*
	* return: JGSS_SUCCESS: available ICEI18nLocale interface pointers are returned in availableLocalesOut.
	*						And, the number of returned items is setted to the numOfAvailableLocalesRef.
	*						Since, availableLocalesOut items are addRef'ed by this call, caller must 
	*						call release() for each items when those items are no-longer necessary.
	*/

typedef CEHResult (*iCEI18nClassFactory_getAvailableLocales) (ICEI18nClassFactory* iI18nClassFactory, struct ICEI18nLocale* *const availableLocalesOut, UINT32 *const numOfAvailableLocalesRef);

	//! create a new locale with default system setting.

typedef CEHResult (*iCEI18nClassFactory_getLocaleFromCStringLiteral) (ICEI18nClassFactory* iI18nClassFactory, const char* isoLangCStrLiteral, const char* isoCountryCStrLiteral, struct ICEI18nLocale* *const localeOut);

typedef CEHResult (*iCEI18nClassFactory_getLocaleFromUtf16Array) (ICEI18nClassFactory* iI18nClassFactory, UTF16CHAR* isoLang16, UINT32 numOfIsoLangChars16, UTF16CHAR* isoCountry16, UINT32 numOfisoCountryChars16, struct ICEI18nLocale* *const localeOut);

typedef CEHResult (*iCEI18nClassFactory_getLocaleFromLanguageTagCStringLiteral) (ICEI18nClassFactory* iI18nClassFactory, const char* langTagCStringLiteral, struct ICEI18nLocale* *const localeOut);

typedef CEHResult (*iCEI18nClassFactory_getLocaleFromLanguageTagUtf16Array) (ICEI18nClassFactory* iI18nClassFactory, const UTF16CHAR* langTag16, UINT32 numOfLangTagChars16, struct ICEI18nLocale* *const localeOut);

typedef CEHResult (*iCEI18nClassFactory_getCharacterClassForLocale) (ICEI18nClassFactory* iI18nClassFactory, const struct ICEI18nLocale* i18nLocale, struct ICEI18nCharacterClass* *const charClassOut);

typedef CEHResult (*iCEI18nClassFactory_initConverter) (ICEI18nClassFactory* iI18nClassFactory, CEI18nConverterInitArg* arg);

typedef CEHResult (*iCEI18nClassFactory_shutdownConverter) (ICEI18nClassFactory* iI18nClassFactory);

typedef CEHResult (*iCEI18nClassFactory_initResource) (ICEI18nClassFactory* iI18nClassFactory, struct ICEI18nClassFactory* parentFactory);

typedef CEHResult (*iCEI18nClassFactory_shutdownResource) (ICEI18nClassFactory* iI18nClassFactory, struct ICEI18nClassFactory* parentFactory);

	//! returns resource property map of this module.

typedef CEHResult (*iCEI18nClassFactory_getResourcePropertyMap) (ICEI18nClassFactory* iI18nClassFactory, struct ICEI18nResourcePropertyMap* *const propMapOut);

	//! returns ICEI18nCTLEngine interface pointer.

typedef CEHResult (*iCEI18nClassFactory_getCTLEngine) (ICEI18nClassFactory* iI18nClassFactory, struct ICEI18nCTLEngine* *const ctlEngineOut);

	//! returns ICEI18nBreakIterator interface pointer.

typedef CEHResult (*iCEI18nClassFactory_getBreakIterator) (ICEI18nClassFactory* iI18nClassFactory, struct ICEI18nLocale* locale, eCEI18nBreakIteratorType type, struct ICEI18nBreakIterator* *const breakIteratorOut);

	//! returns ICEI18nBiDiUtilClass interface pointer. 

typedef CEHResult (*iCEI18nClassFactory_getBiDiUtilClass) (ICEI18nClassFactory* iI18nClassFactory, struct ICEI18nBiDiUtilClass* *const bidiUtilClassOut);

/*!
 * V-table of ICEI18nClassFactory
 */
struct ICEI18nClassFactory_vtbl
{
	iCEI18nClassFactory_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nClassFactory_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nClassFactory_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nClassFactory_getCharsetEncodings	_getCharsetEncodings;
	iCEI18nClassFactory_getCharsetEncodingFromIanaName	_getCharsetEncodingFromIanaName;
	iCEI18nClassFactory_getCharsetEncodingFromIanaCStringLiteral	_getCharsetEncodingFromIanaCStringLiteral;
	iCEI18nClassFactory_getCharsetDetectorInfos	_getCharsetDetectorInfos;
	iCEI18nClassFactory_getCharsetDetectorInfo	_getCharsetDetectorInfo;
	iCEI18nClassFactory_getAvailableLocales	_getAvailableLocales;
	iCEI18nClassFactory_getLocaleFromCStringLiteral	_getLocaleFromCStringLiteral;
	iCEI18nClassFactory_getLocaleFromUtf16Array	_getLocaleFromUtf16Array;
	iCEI18nClassFactory_getLocaleFromLanguageTagCStringLiteral	_getLocaleFromLanguageTagCStringLiteral;
	iCEI18nClassFactory_getLocaleFromLanguageTagUtf16Array	_getLocaleFromLanguageTagUtf16Array;
	iCEI18nClassFactory_getCharacterClassForLocale	_getCharacterClassForLocale;
	iCEI18nClassFactory_initConverter	_initConverter;
	iCEI18nClassFactory_shutdownConverter	_shutdownConverter;
	iCEI18nClassFactory_initResource	_initResource;
	iCEI18nClassFactory_shutdownResource	_shutdownResource;
	iCEI18nClassFactory_getResourcePropertyMap	_getResourcePropertyMap;
	iCEI18nClassFactory_getCTLEngine	_getCTLEngine;
	iCEI18nClassFactory_getBreakIterator	_getBreakIterator;
	iCEI18nClassFactory_getBiDiUtilClass	_getBiDiUtilClass;
};

/*! @}
 * end of ICEI18nClassFactory
 */

/*! \defgroup ICEI18nCharacterClass ICEI18nCharacterClass
 * @{
 */

/*!
 * ID of ICEI18nCharacterClass
 */
#define CEComIID_ICEI18nCharacterClass 0xf00a0378


/*!
* ICEI18nCharacterClass
*/
typedef struct ICEI18nCharacterClass
{
	const struct ICEI18nCharacterClass_vtbl* _vtbl;
} ICEI18nCharacterClass;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nCharacterClass	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nCharacterClass_queryInterface) (ICEI18nCharacterClass* iI18nCharacterClass, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nCharacterClass_addRef) (ICEI18nCharacterClass* iI18nCharacterClass);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nCharacterClass_release) (ICEI18nCharacterClass* iI18nCharacterClass);

typedef CEHResult (*iCEI18nCharacterClass_isBreakableBetween) (ICEI18nCharacterClass* iI18nCharacterClass, UTF32CHAR c1, UTF32CHAR c2, bool *const bBreakableOut);

typedef CEHResult (*iCEI18nCharacterClass_isBreakableChar) (ICEI18nCharacterClass* iI18nCharacterClass, UTF32CHAR c, bool *const bBreakableOut);

typedef CEHResult (*iCEI18nCharacterClass_isBreakableBetweenGCB) (ICEI18nCharacterClass* iI18nCharacterClass, UTF32CHAR c1, UTF32CHAR c2, bool *const bBreakableOut);

typedef CEHResult (*iCEI18nCharacterClass_isBreakableBetweenWord) (ICEI18nCharacterClass* iI18nCharacterClass, UTF32CHAR c1, UTF32CHAR c2, bool *const bBreakableOut);

typedef CEHResult (*iCEI18nCharacterClass_toUpper) (ICEI18nCharacterClass* iI18nCharacterClass, UTF32CHAR c32, UTF32CHAR *const c32Out);

typedef CEHResult (*iCEI18nCharacterClass_toLower) (ICEI18nCharacterClass* iI18nCharacterClass, UTF32CHAR c32, UTF32CHAR *const c32Out);

typedef CEHResult (*iCEI18nCharacterClass_toTitle) (ICEI18nCharacterClass* iI18nCharacterClass, UTF32CHAR c32, UTF32CHAR *const c32Out);

typedef CEHResult (*iCEI18nCharacterClass_getScriptCode) (ICEI18nCharacterClass* iI18nCharacterClass, UTF32CHAR c32, eCEI18nScriptCode *const scOut);

typedef CEHResult (*iCEI18nCharacterClass_getURangeCode) (ICEI18nCharacterClass* iI18nCharacterClass, UTF32CHAR c32, eCEI18nURangeCode *const rangeOut);

	//! returns rough bidi direction. if c32's bidi direction is neutral then returns eCEI18nBiDiDirection_Unknown.

typedef CEHResult (*iCEI18nCharacterClass_getBiDIDirection) (ICEI18nCharacterClass* iI18nCharacterClass, UTF32CHAR c32, eCEI18nBiDiDirection *const bidiDirectionOut);

/*!
 * V-table of ICEI18nCharacterClass
 */
struct ICEI18nCharacterClass_vtbl
{
	iCEI18nCharacterClass_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nCharacterClass_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nCharacterClass_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nCharacterClass_isBreakableBetween	_isBreakableBetween;
	iCEI18nCharacterClass_isBreakableChar	_isBreakableChar;
	iCEI18nCharacterClass_isBreakableBetweenGCB	_isBreakableBetweenGCB;
	iCEI18nCharacterClass_isBreakableBetweenWord	_isBreakableBetweenWord;
	iCEI18nCharacterClass_toUpper	_toUpper;
	iCEI18nCharacterClass_toLower	_toLower;
	iCEI18nCharacterClass_toTitle	_toTitle;
	iCEI18nCharacterClass_getScriptCode	_getScriptCode;
	iCEI18nCharacterClass_getURangeCode	_getURangeCode;
	iCEI18nCharacterClass_getBiDIDirection	_getBiDIDirection;
};

/*! @}
 * end of ICEI18nCharacterClass
 */

/*! \defgroup ICEI18nCharsetDetectorInfo ICEI18nCharsetDetectorInfo
 * @{
 */

/*!
 * ID of ICEI18nCharsetDetectorInfo
 */
#define CEComIID_ICEI18nCharsetDetectorInfo 0x65d00fe8


/*!
 * ICEI18nCharsetDetectorInfo : 
 */
typedef struct ICEI18nCharsetDetectorInfo
{
	const struct ICEI18nCharsetDetectorInfo_vtbl* _vtbl;
} ICEI18nCharsetDetectorInfo;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nCharsetDetectorInfo	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nCharsetDetectorInfo_queryInterface) (ICEI18nCharsetDetectorInfo* iI18nCharsetDetectorInfo, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nCharsetDetectorInfo_addRef) (ICEI18nCharsetDetectorInfo* iI18nCharsetDetectorInfo);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nCharsetDetectorInfo_release) (ICEI18nCharsetDetectorInfo* iI18nCharsetDetectorInfo);

typedef CEHResult (*iCEI18nCharsetDetectorInfo_getCLSID) (ICEI18nCharsetDetectorInfo* iI18nCharsetDetectorInfo, UINT32 *const clsidOut);

typedef CEHResult (*iCEI18nCharsetDetectorInfo_getMinimumInputSizeInBytes) (ICEI18nCharsetDetectorInfo* iI18nCharsetDetectorInfo, UINT32 *const minInputSizeInBytesOut);

typedef CEHResult (*iCEI18nCharsetDetectorInfo_getDisplayName) (ICEI18nCharsetDetectorInfo* iI18nCharsetDetectorInfo, const struct ICEI18nLocale* displayLocale, struct ICEUString* *const displayNameOut);

typedef CEHResult (*iCEI18nCharsetDetectorInfo_createCharsetDetector) (ICEI18nCharsetDetectorInfo* iI18nCharsetDetectorInfo, struct ICEI18nCharsetDetector* *const detectorOut);

/*!
 * V-table of ICEI18nCharsetDetectorInfo
 */
struct ICEI18nCharsetDetectorInfo_vtbl
{
	iCEI18nCharsetDetectorInfo_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nCharsetDetectorInfo_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nCharsetDetectorInfo_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nCharsetDetectorInfo_getCLSID	_getCLSID;
	iCEI18nCharsetDetectorInfo_getMinimumInputSizeInBytes	_getMinimumInputSizeInBytes;
	iCEI18nCharsetDetectorInfo_getDisplayName	_getDisplayName;
	iCEI18nCharsetDetectorInfo_createCharsetDetector	_createCharsetDetector;
};

/*! @}
 * end of ICEI18nCharsetDetectorInfo
 */

/*! \defgroup ICEI18nCharsetDetector ICEI18nCharsetDetector
 * @{
 */

/*!
 * ID of ICEI18nCharsetDetector
 */
#define CEComIID_ICEI18nCharsetDetector 0x91c54498


/*!
 * ICEI18nCharsetDetector : 
 */
typedef struct ICEI18nCharsetDetector
{
	const struct ICEI18nCharsetDetector_vtbl* _vtbl;
} ICEI18nCharsetDetector;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nCharsetDetector	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nCharsetDetector_queryInterface) (ICEI18nCharsetDetector* iI18nCharsetDetector, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nCharsetDetector_addRef) (ICEI18nCharsetDetector* iI18nCharsetDetector);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nCharsetDetector_release) (ICEI18nCharsetDetector* iI18nCharsetDetector);


	/*!
	 * <b>Summary:</b>
	 * Detects character set.
	 *
	 * \param[in]	iI18nCharsetDetector Specifies interface pointer
	 * \param[in]	inbytes input bytes to detect.
	 * \param[in]	inLen inbytes length.
	 * \param[in]   bSimple specify true to simple detection. 
	 * \param[out]	detectStatusOut status of charset detection.
	 * \param[out]	charsetEncodingOut result of charset detection.
	 *
	 * \return error status
	 *
	 * <b>Description:</b>
	 * None
	 *
	 * <b>Precondition:</b>
	 * None
	 *
	 * <b>Limitation:</b>
	 * None.
	 *
	 * <b>Example:</b>
	 * None
	 *
	 * <b>See also:</b>
	 * None
	 */
typedef CEHResult (*iCEI18nCharsetDetector_doIt) (ICEI18nCharsetDetector* iI18nCharsetDetector, const UCHAR8* inbytes, UINT32 inLen, eCEI18nCharsetDetectStatus *const detectStatusOut, struct ICEI18nCharsetEncoding* *const charsetEncodingOut);


	/*!
	 * <b>Summary:</b>
	 * Finishes the charset detectection.
	 *
	 * \param[in]	iI18nCharsetDetector Specifies interface pointer
	 * \param[out]	detectStatusOut status of charset detection.
	 * \param[out]	charsetEncodingOut result of charset detection.
	 *
	 * \return error status
	 *
	 * <b>Description:</b>
	 * None
	 *
	 * <b>Precondition:</b>
	 * None
	 *
	 * <b>Limitation:</b>
	 * None.
	 *
	 * <b>Example:</b>
	 * None
	 *
	 * <b>See also:</b>
	 * None
	 */
typedef CEHResult (*iCEI18nCharsetDetector_finish) (ICEI18nCharsetDetector* iI18nCharsetDetector, eCEI18nCharsetDetectStatus *const detectStatusOut, struct ICEI18nCharsetEncoding* *const charsetEncodingOut);


	/*!
	 * <b>Summary:</b>
	 * Resets the Character detector.
	 *
	 * \param[in]	iI18nCharsetDetector  Specifies interface pointer
	 * \param[in]	bSimple  specify true to simple charset detection.
	 * \param[in]   iCharsetDetectorHint specify charset detection hint or 0. see ICEI18nClassFactory::getCharsetDetectorHints().
	 *
	 * \return error status
	 *
	 * <b>Description:</b>
	 * None
	 *
	 * <b>Precondition:</b>
	 * None
	 *
	 * <b>Limitation:</b>
	 * None.
	 *
	 * <b>Example:</b>
	 * None
	 *
	 * <b>See also:</b>
	 * None
	 */
typedef CEHResult (*iCEI18nCharsetDetector_reset) (ICEI18nCharsetDetector* iI18nCharsetDetector);

typedef CEHResult (*iCEI18nCharsetDetector_dbgToString) (ICEI18nCharsetDetector* iI18nCharsetDetector, const UTF16CHAR* *const pStr16Out);

/*!
 * V-table of ICEI18nCharsetDetector
 */
struct ICEI18nCharsetDetector_vtbl
{
	iCEI18nCharsetDetector_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nCharsetDetector_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nCharsetDetector_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nCharsetDetector_doIt	_doIt;
	iCEI18nCharsetDetector_finish	_finish;
	iCEI18nCharsetDetector_reset	_reset;
	iCEI18nCharsetDetector_dbgToString	_dbgToString;
};

/*! @}
 * end of ICEI18nCharsetDetector
 */

/*! \defgroup ICEI18nUnicodeEncoder ICEI18nUnicodeEncoder
 * @{
 */

/*!
 * ID of ICEI18nUnicodeEncoder
 */
#define CEComIID_ICEI18nUnicodeEncoder 0x3e87b62a


/*!
 * ICEI18nUnicodeEncoder : encode unicode data bytes to specific encoding.
 */
typedef struct ICEI18nUnicodeEncoder
{
	const struct ICEI18nUnicodeEncoder_vtbl* _vtbl;
} ICEI18nUnicodeEncoder;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nUnicodeEncoder	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nUnicodeEncoder_queryInterface) (ICEI18nUnicodeEncoder* iI18nUnicodeEncoder, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nUnicodeEncoder_addRef) (ICEI18nUnicodeEncoder* iI18nUnicodeEncoder);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nUnicodeEncoder_release) (ICEI18nUnicodeEncoder* iI18nUnicodeEncoder);


	/*!
	 * <b>Summary:</b>
	 * * Estimate max length (of worst case) of converted encoding specific buffer.
	 *
	 * \param[in]	iI18nUnicodeEncoder Specifies interface pointer
	 * \param[in]	utf16Array specify source UTF16 array.
	 * \param[in]	numOfUtf16 specify number of UTF16CHAR of utf16Bytes. 
	 * \param[out]	numOfBytesOut required number of bytes.
	 *
	 * \return error status
	 *
	 */
typedef CEHResult (*iCEI18nUnicodeEncoder_requiredSize) (ICEI18nUnicodeEncoder* iI18nUnicodeEncoder, const UTF16CHAR* utf16Array, UINT32 numOfUtf16, UINT32 *const numOfBytesOut, eCEI18nErr *const i18nErrOut);


	/*!
	 * <b>Summary:</b>
	 * Encodes specified unicode data.
	 *
	 * \param[in]	iI18nUnicodeEncoder Specifies interface pointer
	 * \param[in]	srcUtf16Chars specify source UTF16 data.
	 * \param[in, out]	numOfSrcUtf16CharsRef specify number of utf16Bytes. processed number of UTF16CHARs  returned.
	 * \param[out]	destBytesOut encoded data.
	 * \param[in, out]	numOfDestBytesRef specify number of bytes. processed number of bytes returned.
	 *
	 * \return error status
	 *
	 * <b>Description:</b>
	 * None
	 *
	 * <b>Precondition:</b>
	 * None
	 *
	 * <b>Limitation:</b>
	 * None.
	 *
	 * <b>Example:</b>
	 * None
	 *
	 * <b>See also:</b>
	 * None
	 *
	 *
	 */
typedef CEHResult (*iCEI18nUnicodeEncoder_convert) (ICEI18nUnicodeEncoder* iI18nUnicodeEncoder, const UTF16CHAR* srcUtf16Chars, UINT32 *const numOfSrcUtf16CharsRef, void *const destBytesOut, UINT32 *const numOfDestBytesRef, eCEI18nErr *const i18nErrOut);

typedef CEHResult (*iCEI18nUnicodeEncoder_convertEx) (ICEI18nUnicodeEncoder* iI18nUnicodeEncoder, const UTF16CHAR* srcUtf16Chars, UINT32 *const numOfSrcUtf16CharsRef, void *const destBytesOut, UINT32 *const numOfDestBytesRef, UCHAR8 altCharForUnknownChar8, eCEI18nErr *const i18nErrOut);

typedef CEHResult (*iCEI18nUnicodeEncoder_getCharsetEncoding) (ICEI18nUnicodeEncoder* iI18nUnicodeEncoder, struct ICEI18nCharsetEncoding* *const encodingOut);

typedef CEHResult (*iCEI18nUnicodeEncoder_dbgToString) (ICEI18nUnicodeEncoder* iI18nUnicodeEncoder, const UTF16CHAR* *const pStr16Out);

/*!
 * V-table of ICEI18nUnicodeEncoder
 */
struct ICEI18nUnicodeEncoder_vtbl
{
	iCEI18nUnicodeEncoder_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nUnicodeEncoder_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nUnicodeEncoder_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nUnicodeEncoder_requiredSize	_requiredSize;
	iCEI18nUnicodeEncoder_convert	_convert;
	iCEI18nUnicodeEncoder_convertEx	_convertEx;
	iCEI18nUnicodeEncoder_getCharsetEncoding	_getCharsetEncoding;
	iCEI18nUnicodeEncoder_dbgToString	_dbgToString;
};

/*! @}
 * end of ICEI18nUnicodeEncoder
 */

/*! \defgroup ICEI18nUnicodeDecoder ICEI18nUnicodeDecoder
 * @{
 */

/*!
 * ID of ICEI18nUnicodeDecoder
 */
#define CEComIID_ICEI18nUnicodeDecoder 0xfa09de02


/*!
 * ICEI18nUnicodeDecoder : decode specific encoding data bytes to unicode.
 */
typedef struct ICEI18nUnicodeDecoder
{
	const struct ICEI18nUnicodeDecoder_vtbl* _vtbl;
} ICEI18nUnicodeDecoder;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nUnicodeDecoder	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nUnicodeDecoder_queryInterface) (ICEI18nUnicodeDecoder* iI18nUnicodeDecoder, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nUnicodeDecoder_addRef) (ICEI18nUnicodeDecoder* iI18nUnicodeDecoder);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nUnicodeDecoder_release) (ICEI18nUnicodeDecoder* iI18nUnicodeDecoder);


	/*!
	 * <b>Summary:</b>
	 * decodes specified bytes to unicode data.
	 *
	 * \param[in]	iI18nUnicodeDecoder Specifies interface pointer
	 * \param[in]	bytes encoded source data bytes.
	 * \param[in]	numOfBytes specify number of bytes.
	 * \param[out]  numOfUtf16Out required number of UTF16CHAR.
	 * \param[out]  i18nErrOut i18n specific error code.
	 *
	 * \return error status
	 *
	 */
typedef CEHResult (*iCEI18nUnicodeDecoder_requiredSize) (ICEI18nUnicodeDecoder* iI18nUnicodeDecoder, const void* bytes, UINT32 numOfBytes, UINT32 *const numOfUtf16Out, eCEI18nErr *const i18nErrOut);


	/*!
	 * <b>Summary:</b>
	 * decodes specified bytes to unicode data.
	 *
	 * \param[in]	iI18nUnicodeDecoder Specifies interface pointer
	 * \param[in]	srcBytes source data bytes.
	 * \param[in, out]	numOfSrcBytesRef specify number of bytes. processed bytes length returned.
	 * \param[out]  destUtf16CharsOut decoded unicode data.
	 * \param[in, out]  destNumOfUtf16CharsRef specify number of UTF16CHARs. decoded number of UTF16CHARs returned.
	 *
	 * \return error status
	 *
	 * <b>Description:</b>
	 * None
	 *
	 * <b>Precondition:</b>
	 * None
	 *
	 * <b>Limitation:</b>
	 * None.
	 *
	 * <b>Example:</b>
	 * None
	 *
	 * <b>See also:</b>
	 * None
	 */
typedef CEHResult (*iCEI18nUnicodeDecoder_convert) (ICEI18nUnicodeDecoder* iI18nUnicodeDecoder, const void* srcBytes, UINT32 *const numOfSrcBytesRef, UTF16CHAR *const destUtf16CharsOut, UINT32 *const destNumOfUtf16CharsRef, eCEI18nErr *const i18nErrOut);

typedef CEHResult (*iCEI18nUnicodeDecoder_convertEx) (ICEI18nUnicodeDecoder* iI18nUnicodeDecoder, const void* srcBytes, UINT32 *const numOfSrcBytesRef, UTF16CHAR *const destUtf16CharsOut, UINT32 *const destNumOfUtf16CharsRef, UCHAR8 altCharForUnknownChar8, eCEI18nErr *const i18nErrOut);

typedef CEHResult (*iCEI18nUnicodeDecoder_getCharsetEncoding) (ICEI18nUnicodeDecoder* iI18nUnicodeDecoder, struct ICEI18nCharsetEncoding* *const encodingOut);

typedef CEHResult (*iCEI18nUnicodeDecoder_dbgToString) (ICEI18nUnicodeDecoder* iI18nUnicodeDecoder, const UTF16CHAR* *const pStr16Out);

/*!
 * V-table of ICEI18nUnicodeDecoder
 */
struct ICEI18nUnicodeDecoder_vtbl
{
	iCEI18nUnicodeDecoder_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nUnicodeDecoder_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nUnicodeDecoder_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nUnicodeDecoder_requiredSize	_requiredSize;
	iCEI18nUnicodeDecoder_convert	_convert;
	iCEI18nUnicodeDecoder_convertEx	_convertEx;
	iCEI18nUnicodeDecoder_getCharsetEncoding	_getCharsetEncoding;
	iCEI18nUnicodeDecoder_dbgToString	_dbgToString;
};

/*! @}
 * end of ICEI18nUnicodeDecoder
 */

/*! \defgroup ICEI18nResourcePropertyMap ICEI18nResourcePropertyMap
 * @{
 */

/*!
 * ID of ICEI18nResourcePropertyMap
 */
#define CEComIID_ICEI18nResourcePropertyMap 0x00268dcd

/*!
 * ICEI18nResourcePropertyMap
 */
typedef struct ICEI18nResourcePropertyMap
{
	const struct ICEI18nResourcePropertyMap_vtbl* _vtbl;
} ICEI18nResourcePropertyMap;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nResourcePropertyMap	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nResourcePropertyMap_queryInterface) (ICEI18nResourcePropertyMap* iI18nResourcePropertyMap, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nResourcePropertyMap_addRef) (ICEI18nResourcePropertyMap* iI18nResourcePropertyMap);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nResourcePropertyMap_release) (ICEI18nResourcePropertyMap* iI18nResourcePropertyMap);

typedef CEHResult (*iCEI18nResourcePropertyMap_setString) (ICEI18nResourcePropertyMap* iI18nResourcePropertyMap, const char* category, const struct ICEUString* iName, const struct ICEI18nLocale* locale, const struct ICEUString* iStr);

typedef CEHResult (*iCEI18nResourcePropertyMap_getString) (ICEI18nResourcePropertyMap* iI18nResourcePropertyMap, const char* category, const struct ICEUString* iName, const struct ICEI18nLocale* locale, struct ICEUString* *const iStrOut);

typedef CEHResult (*iCEI18nResourcePropertyMap_setStringById) (ICEI18nResourcePropertyMap* iI18nResourcePropertyMap, const char* category, UINT32 id32, const struct ICEI18nLocale* locale, const struct ICEUString* iStr);

typedef CEHResult (*iCEI18nResourcePropertyMap_getStringById) (ICEI18nResourcePropertyMap* iI18nResourcePropertyMap, const char* category, UINT32 id32, const struct ICEI18nLocale* locale, struct ICEUString* *const iStrOut);

/*!
 * V-table of ICEI18nResourcePropertyMap
 */
struct ICEI18nResourcePropertyMap_vtbl
{
	iCEI18nResourcePropertyMap_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nResourcePropertyMap_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nResourcePropertyMap_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nResourcePropertyMap_setString	_setString;
	iCEI18nResourcePropertyMap_getString	_getString;
	iCEI18nResourcePropertyMap_setStringById	_setStringById;
	iCEI18nResourcePropertyMap_getStringById	_getStringById;
};

/*! @}
 * end of ICEI18nResourcePropertyMap
 */

/*! \defgroup ICEI18nCharMapper ICEI18nCharMapper
 * @{
 */

/*!
 * ID of ICEI18nCharMapper
 */
#define CEComIID_ICEI18nCharMapper 0x6246658c

/*!
 * ICEI18nCharMapper
 */
typedef struct ICEI18nCharMapper
{
	const struct ICEI18nCharMapper_vtbl* _vtbl;
} ICEI18nCharMapper;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nCharMapper	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nCharMapper_queryInterface) (ICEI18nCharMapper* iI18nCharMapper, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nCharMapper_addRef) (ICEI18nCharMapper* iI18nCharMapper);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nCharMapper_release) (ICEI18nCharMapper* iI18nCharMapper);

typedef CEHResult (*iCEI18nCharMapper_mapChar) (ICEI18nCharMapper* iI18nCharMapper, UTF32CHAR c32In, UTF32CHAR *const c32Out);

/*!
 * V-table of ICEI18nCharMapper
 */
struct ICEI18nCharMapper_vtbl
{
	iCEI18nCharMapper_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nCharMapper_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nCharMapper_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nCharMapper_mapChar	_mapChar;
};

/*! @}
 * end of ICEI18nCharMapper
 */

/*! \defgroup ICEI18nTextBreakTester ICEI18nTextBreakTester
 * @{
 */

/*!
 * ID of ICEI18nTextBreakTester
 */
#define CEComIID_ICEI18nTextBreakTester 0x3a088f27

/*!
 * ICEI18nTextBreakTester
 */
typedef struct ICEI18nTextBreakTester
{
	const struct ICEI18nTextBreakTester_vtbl* _vtbl;
} ICEI18nTextBreakTester;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nTextBreakTester	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nTextBreakTester_queryInterface) (ICEI18nTextBreakTester* iI18nTextBreakTester, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nTextBreakTester_addRef) (ICEI18nTextBreakTester* iI18nTextBreakTester);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nTextBreakTester_release) (ICEI18nTextBreakTester* iI18nTextBreakTester);

typedef CEHResult (*iCEI18nTextBreakTester_canBreakBetween) (ICEI18nTextBreakTester* iI18nTextBreakTester, UTF32CHAR c1, UTF32CHAR c2, bool *const bCanBreakOut);

/*!
 * V-table of ICEI18nTextBreakTester
 */
struct ICEI18nTextBreakTester_vtbl
{
	iCEI18nTextBreakTester_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nTextBreakTester_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nTextBreakTester_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nTextBreakTester_canBreakBetween	_canBreakBetween;
};

/*! @}
 * end of ICEI18nTextBreakTester
 */

/*! \defgroup ICEI18nCTLEngine ICEI18nCTLEngine
 * @{
 */

/*!
 * ID of ICEI18nCTLEngine
 */
#define CEComIID_ICEI18nCTLEngine 0x173a8faf

/*!
 * ICEI18nCTLEngine
 */
typedef struct ICEI18nCTLEngine
{
	const struct ICEI18nCTLEngine_vtbl* _vtbl;
} ICEI18nCTLEngine;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nCTLEngine	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nCTLEngine_queryInterface) (ICEI18nCTLEngine* iI18nCTLEngine, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nCTLEngine_addRef) (ICEI18nCTLEngine* iI18nCTLEngine);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nCTLEngine_release) (ICEI18nCTLEngine* iI18nCTLEngine);

typedef CEHResult (*iCEI18nCTLEngine_mapChar32) (ICEI18nCTLEngine* iI18nCTLEngine, bool bFilterControlCode, bool bMirror, struct ICEI18nCharMapper* pOptCharMapper, UTF32CHAR c32In, UTF32CHAR *const c32Out);

typedef CEHResult (*iCEI18nCTLEngine_layoutChars) (ICEI18nCTLEngine* iI18nCTLEngine, struct ICEUIPlatformFont* iPhyFont, struct ICEI18nCharMapper* iCharMapper, eCEI18nScriptCode scriptCode, struct ICEI18nLocale* pOptLocale,  bool isRtoL, const UTF16CHAR* pC16s, const UINT32 numOfC16s, CEI18nGlyphInfo *const pOptGlyphInfoRef, CEUTextMeasurement *const pOptWdithOut);

typedef CEHResult (*iCEI18nCTLEngine_initGlyphInfo) (ICEI18nCTLEngine* iI18nCTLEngine, CEI18nGlyphInfo* pGlyphInfoOut, UINT32 eCEI18nGIFlagsIn);

typedef CEHResult (*iCEI18nCTLEngine_deleteGlyphInfo) (ICEI18nCTLEngine* iI18nCTLEngine, CEI18nGlyphInfo* pGlyphInfoOut);

/*!
 * V-table of ICEI18nCTLEngine
 */
struct ICEI18nCTLEngine_vtbl
{
	iCEI18nCTLEngine_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nCTLEngine_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nCTLEngine_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nCTLEngine_mapChar32	_mapChar32;
	iCEI18nCTLEngine_layoutChars	_layoutChars;
	iCEI18nCTLEngine_initGlyphInfo	_initGlyphInfo;
	iCEI18nCTLEngine_deleteGlyphInfo	_deleteGlyphInfo;
};

/*! @}
 * end of ICEI18nCTLEngine
 */

/*! \defgroup ICEI18nBreakIterator ICEI18nBreakIterator
 * @{
 */

/*!
 * ID of ICEI18nBreakIterator
 */
#define CEComIID_ICEI18nBreakIterator 0xde0640e7

/*!
 * ICEI18nBreakIterator
 */
typedef struct ICEI18nBreakIterator
{
	const struct ICEI18nBreakIterator_vtbl* _vtbl;
} ICEI18nBreakIterator;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nBreakIterator	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nBreakIterator_queryInterface) (ICEI18nBreakIterator* iI18nBreakIterator, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nBreakIterator_addRef) (ICEI18nBreakIterator* iI18nBreakIterator);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nBreakIterator_release) (ICEI18nBreakIterator* iI18nBreakIterator);

typedef CEHResult (*iCEI18nBreakIterator_getInfo) (ICEI18nBreakIterator* iI18nBreakIterator, eCEI18nBreakIteratorType *const optTypeOut, struct ICEI18nLocale* *const optLocaleOut);

	//! standard iterator interface

typedef CEHResult (*iCEI18nBreakIterator_setUTF16CHARArray) (ICEI18nBreakIterator* iI18nBreakIterator, const UTF16CHAR* pC16s, UINT32 numOfC16s, bool bAppend);

typedef CEHResult (*iCEI18nBreakIterator_moveNext) (ICEI18nBreakIterator* iI18nBreakIterator, INT32 *const boundaryOut, UTF32CHAR *const optC32Out, eCEI18nGCB *const optGCBOut);

typedef CEHResult (*iCEI18nBreakIterator_current) (ICEI18nBreakIterator* iI18nBreakIterator, INT32 *const boundaryOut, UTF32CHAR *const optC32Out, eCEI18nGCB *const optGCBOut);

typedef CEHResult (*iCEI18nBreakIterator_reset) (ICEI18nBreakIterator* iI18nBreakIterator);

/*!
 * V-table of ICEI18nBreakIterator
 */
struct ICEI18nBreakIterator_vtbl
{
	iCEI18nBreakIterator_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nBreakIterator_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nBreakIterator_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nBreakIterator_getInfo	_getInfo;
	iCEI18nBreakIterator_setUTF16CHARArray	_setUTF16CHARArray;
	iCEI18nBreakIterator_moveNext	_moveNext;
	iCEI18nBreakIterator_current	_current;
	iCEI18nBreakIterator_reset	_reset;
};

/*! @}
 * end of ICEI18nBreakIterator
 */

/*! \defgroup ICEI18nBiDiUtil ICEI18nBiDiUtil
 * @{
 */

/*!
 * ID of ICEI18nBiDiUtil
 */
#define CEComIID_ICEI18nBiDiUtil 0xe1a03934

/*!
 * ICEI18nBiDiUtil
 */
typedef struct ICEI18nBiDiUtil
{
	const struct ICEI18nBiDiUtil_vtbl* _vtbl;
} ICEI18nBiDiUtil;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nBiDiUtil	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nBiDiUtil_queryInterface) (ICEI18nBiDiUtil* iI18nBiDiUtil, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nBiDiUtil_addRef) (ICEI18nBiDiUtil* iI18nBiDiUtil);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nBiDiUtil_release) (ICEI18nBiDiUtil* iI18nBiDiUtil);

typedef CEHResult (*iCEI18nBiDiUtil_reset) (ICEI18nBiDiUtil* iI18nBiDiUtil);


	/*!
	* bResolveParagraphLevel: true: search first strong character in pC16s to determine the paragraph level. If no such characters are found, parafraphLevelIn is used for paragraph level.
	*						  false: simply, use paragraphLevelIn for paragraph level.
	*/

typedef CEHResult (*iCEI18nBiDiUtil_reorder) (ICEI18nBiDiUtil* iI18nBiDiUtil, bool bResolveParagraphLevel, eCEI18nBiDiLevel paragraphLevelIn, const UTF16CHAR* pC16sIn, UINT32 numOfC16sIn, const CEI18nBiDiCCodeStack* optLeadCCodeStack, CEI18nBiDiCCodeStack* *const optTrailCCodeStackOut);

typedef CEHResult (*iCEI18nBiDiUtil_getVisualRunCount) (ICEI18nBiDiUtil* iI18nBiDiUtil, INT32 *const numOfVisualRunsOut);

typedef CEHResult (*iCEI18nBiDiUtil_getVisualRun) (ICEI18nBiDiUtil* iI18nBiDiUtil, INT32 visualRunIndexIn, eCEI18nBiDiDirection *const optDirOut, INT32 *const optStart16Out, INT32 *const optNumOfC16sOut);

/*!
 * V-table of ICEI18nBiDiUtil
 */
struct ICEI18nBiDiUtil_vtbl
{
	iCEI18nBiDiUtil_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nBiDiUtil_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nBiDiUtil_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nBiDiUtil_reset	_reset;
	iCEI18nBiDiUtil_reorder	_reorder;
	iCEI18nBiDiUtil_getVisualRunCount	_getVisualRunCount;
	iCEI18nBiDiUtil_getVisualRun	_getVisualRun;
};

/*! @}
 * end of ICEI18nBiDiUtil
 */

/*! \defgroup ICEI18nBiDiUtilClass ICEI18nBiDiUtilClass
 * @{
 */

/*!
 * ID of ICEI18nBiDiUtilClass
 */
#define CEComIID_ICEI18nBiDiUtilClass 0xd90e1bb6

/*!
 * ICEI18nBiDiUtilClass
 */
typedef struct ICEI18nBiDiUtilClass
{
	const struct ICEI18nBiDiUtilClass_vtbl* _vtbl;
} ICEI18nBiDiUtilClass;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iI18nBiDiUtilClass	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEI18nBiDiUtilClass_queryInterface) (ICEI18nBiDiUtilClass* iI18nBiDiUtilClass, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nBiDiUtilClass_addRef) (ICEI18nBiDiUtilClass* iI18nBiDiUtilClass);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEI18nBiDiUtilClass_release) (ICEI18nBiDiUtilClass* iI18nBiDiUtilClass);

	//! returns ICEI18nBiDiUtil interface pointer. 

typedef CEHResult (*iCEI18nBiDiUtilClass_newBiDiUtil) (ICEI18nBiDiUtilClass* iI18nBiDiUtilClass, struct ICEI18nBiDiUtil* *const bidiUtilOut);

typedef CEHResult (*iCEI18nBiDiUtilClass_newCCodeStack) (ICEI18nBiDiUtilClass* iI18nBiDiUtilClass, bool bRtlParagraphLevel, bool bOverride, CEI18nBiDiCCodeStack* *const pCCodeStackOut);

typedef CEHResult (*iCEI18nBiDiUtilClass_deleteCCodeStack) (ICEI18nBiDiUtilClass* iI18nBiDiUtilClass, CEI18nBiDiCCodeStack* pCCodeStack);

/*!
 * V-table of ICEI18nBiDiUtilClass
 */
struct ICEI18nBiDiUtilClass_vtbl
{
	iCEI18nBiDiUtilClass_queryInterface	_queryInterface;	//!< Query interface.
	iCEI18nBiDiUtilClass_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEI18nBiDiUtilClass_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEI18nBiDiUtilClass_newBiDiUtil	_newBiDiUtil;
	iCEI18nBiDiUtilClass_newCCodeStack	_newCCodeStack;
	iCEI18nBiDiUtilClass_deleteCCodeStack	_deleteCCodeStack;
};

/*! @}
 * end of ICEI18nBiDiUtilClass
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEI18nLocaleRef CEComICEI18nLocaleRef
 * @{
 */

class CEComICEI18nLocaleRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nLocaleRef() : _iI18nLocale(0)
	{
#if defined(_DEBUG) || !defined(NDEBUG)
		dbgToString();
#endif //_DEBUG || !NDEBUG
	}
	CEComICEI18nLocaleRef(ICEI18nLocale* iOther) : _iI18nLocale(0)
	{
		if (iOther)
		{
			_iI18nLocale = iOther;
			_iI18nLocale->_vtbl->_addRef(_iI18nLocale);
		}
	}

	CEComICEI18nLocaleRef(const CEComICEI18nLocaleRef& other) : _iI18nLocale(0)
	{
		if (other._iI18nLocale)
		{
			_iI18nLocale = other._iI18nLocale;
			_iI18nLocale->_vtbl->_addRef(_iI18nLocale);
		}
	}

	~CEComICEI18nLocaleRef()
	{
		if (_iI18nLocale)
		{
			ICEI18nLocale* tmp = _iI18nLocale;
			_iI18nLocale = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nLocale* iI18nLocale;
				void* _ptr;
			} uIntf;
			uIntf.iI18nLocale = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nLocale, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nLocale)
				{
					ICEI18nLocale* tmp = _iI18nLocale;
					_iI18nLocale = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nLocale = uIntf.iI18nLocale;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nLocale* object() const	{ return _iI18nLocale; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nLocale* iOther)
	{
		if (_iI18nLocale)
		{
			ICEI18nLocale* tmp = _iI18nLocale;
			_iI18nLocale = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nLocale = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nLocale* detach()
	{
		ICEI18nLocale* iIntf = _iI18nLocale;
		_iI18nLocale = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nLocale* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nLocale;
		if (_iI18nLocale)
		{
			_iI18nLocale->_vtbl->_addRef(_iI18nLocale);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nLocale*() const	{ return _iI18nLocale; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nLocale& operator*() const	{ return *_iI18nLocale; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nLocale == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nLocale* iOther) const	{ return (_iI18nLocale != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nLocale* iOther) const	{ return (_iI18nLocale == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nLocaleRef& operator=(const CEComICEI18nLocaleRef& other)	{ return operator=(other._iI18nLocale); }

	CEComICEI18nLocaleRef& operator=(const ICEI18nLocale* iOther)
	{
		if (_iI18nLocale != iOther)
		{
			if (_iI18nLocale)
			{
				ICEI18nLocale* tmp = _iI18nLocale;
				_iI18nLocale = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nLocale = const_cast<ICEI18nLocale*>(iOther);
			if (_iI18nLocale)
			{
				_iI18nLocale->_vtbl->_addRef(_iI18nLocale);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nLocale** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nLocaleRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nLocale && "has a com object reference. clear first.");
		return & _iI18nLocale; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nLocaleRef& other)
	{
		bool result = false;
		if (_iI18nLocale)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nLocale);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nLocale ? _iI18nLocale->_vtbl->_queryInterface(_iI18nLocale, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nLocale) { _iI18nLocale->_vtbl->_addRef(_iI18nLocale); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nLocale) { _iI18nLocale->_vtbl->_release(_iI18nLocale); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getISO2LLanguageCode(struct ICEUString* *const optIStrOut, struct ICESymbol* *const optISymbolOut)	{ return _iI18nLocale ? _iI18nLocale->_vtbl->_getISO2LLanguageCode(_iI18nLocale, optIStrOut, optISymbolOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getISO2LCountryCode(struct ICEUString* *const optIStrOut, struct ICESymbol* *const optISymbolOut)	{ return _iI18nLocale ? _iI18nLocale->_vtbl->_getISO2LCountryCode(_iI18nLocale, optIStrOut, optISymbolOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLanguageTag(struct ICEUString* *const optIStrOut, struct ICESymbol* *const optISymbolOut)	{ return _iI18nLocale ? _iI18nLocale->_vtbl->_getLanguageTag(_iI18nLocale, optIStrOut, optISymbolOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getISO3LLanguageCode(struct ICEUString* *const optIStrOut, struct ICESymbol* *const optISymbolOut)	{ return _iI18nLocale ? _iI18nLocale->_vtbl->_getISO3LLanguageCode(_iI18nLocale, optIStrOut, optISymbolOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getISO3LCountryCode(struct ICEUString* *const optIStrOut, struct ICESymbol* *const optISymbolOut)	{ return _iI18nLocale ? _iI18nLocale->_vtbl->_getISO3LCountryCode(_iI18nLocale, optIStrOut, optISymbolOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDisplayLanguage(const struct ICEI18nLocale* displayLocale, struct ICEUString* *const strOut)	{ return _iI18nLocale ? _iI18nLocale->_vtbl->_getDisplayLanguage(_iI18nLocale, displayLocale, strOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDisplayCountry(const struct ICEI18nLocale* displayLocale, struct ICEUString* *const strOut)	{ return _iI18nLocale ? _iI18nLocale->_vtbl->_getDisplayCountry(_iI18nLocale, displayLocale, strOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult equalsCStringLiteral(const char* lang, const char* country, bool *const bLangMatchedOut, bool *const bCountryMatchedOut)	{ return _iI18nLocale ? _iI18nLocale->_vtbl->_equalsCStringLiteral(_iI18nLocale, lang, country, bLangMatchedOut, bCountryMatchedOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult equals(const struct ICEI18nLocale* other, bool *const bSameLangOut, bool *const bSameCountryOut)	{ return _iI18nLocale ? _iI18nLocale->_vtbl->_equals(_iI18nLocale, other, bSameLangOut, bSameCountryOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getParentLocale(struct ICEI18nLocale* *const parentOut)	{ return _iI18nLocale ? _iI18nLocale->_vtbl->_getParentLocale(_iI18nLocale, parentOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult dbgToString(const UTF16CHAR* *const pStr16Out)	{ return _iI18nLocale ? _iI18nLocale->_vtbl->_dbgToString(_iI18nLocale, pStr16Out) : CE_SILK_ERR_UNINITIALIZED; }

#if defined(_DEBUG) || !defined(NDEBUG)
	const UTF16CHAR* dbgToString() { const UTF16CHAR* ret=0; if (_iI18nLocale) { dbgToString(&ret); } return ret; }
#endif // _DEBUG || !NDEBUG


private:
	ICEI18nLocale* _iI18nLocale;
};

/*! @}
 * end of CEComICEI18nLocaleRef
 */

/*! \defgroup CEComICEI18nCharsetEncodingRef CEComICEI18nCharsetEncodingRef
 * @{
 */

class CEComICEI18nCharsetEncodingRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nCharsetEncodingRef() : _iI18nCharsetEncoding(0)
	{
#if defined(_DEBUG) || !defined(NDEBUG)
		dbgToString();
#endif //_DEBUG || !NDEBUG
	}
	CEComICEI18nCharsetEncodingRef(ICEI18nCharsetEncoding* iOther) : _iI18nCharsetEncoding(0)
	{
		if (iOther)
		{
			_iI18nCharsetEncoding = iOther;
			_iI18nCharsetEncoding->_vtbl->_addRef(_iI18nCharsetEncoding);
		}
	}

	CEComICEI18nCharsetEncodingRef(const CEComICEI18nCharsetEncodingRef& other) : _iI18nCharsetEncoding(0)
	{
		if (other._iI18nCharsetEncoding)
		{
			_iI18nCharsetEncoding = other._iI18nCharsetEncoding;
			_iI18nCharsetEncoding->_vtbl->_addRef(_iI18nCharsetEncoding);
		}
	}

	~CEComICEI18nCharsetEncodingRef()
	{
		if (_iI18nCharsetEncoding)
		{
			ICEI18nCharsetEncoding* tmp = _iI18nCharsetEncoding;
			_iI18nCharsetEncoding = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nCharsetEncoding* iI18nCharsetEncoding;
				void* _ptr;
			} uIntf;
			uIntf.iI18nCharsetEncoding = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nCharsetEncoding, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nCharsetEncoding)
				{
					ICEI18nCharsetEncoding* tmp = _iI18nCharsetEncoding;
					_iI18nCharsetEncoding = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nCharsetEncoding = uIntf.iI18nCharsetEncoding;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharsetEncoding* object() const	{ return _iI18nCharsetEncoding; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nCharsetEncoding* iOther)
	{
		if (_iI18nCharsetEncoding)
		{
			ICEI18nCharsetEncoding* tmp = _iI18nCharsetEncoding;
			_iI18nCharsetEncoding = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nCharsetEncoding = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nCharsetEncoding* detach()
	{
		ICEI18nCharsetEncoding* iIntf = _iI18nCharsetEncoding;
		_iI18nCharsetEncoding = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nCharsetEncoding* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nCharsetEncoding;
		if (_iI18nCharsetEncoding)
		{
			_iI18nCharsetEncoding->_vtbl->_addRef(_iI18nCharsetEncoding);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nCharsetEncoding*() const	{ return _iI18nCharsetEncoding; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharsetEncoding& operator*() const	{ return *_iI18nCharsetEncoding; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nCharsetEncoding == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nCharsetEncoding* iOther) const	{ return (_iI18nCharsetEncoding != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nCharsetEncoding* iOther) const	{ return (_iI18nCharsetEncoding == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nCharsetEncodingRef& operator=(const CEComICEI18nCharsetEncodingRef& other)	{ return operator=(other._iI18nCharsetEncoding); }

	CEComICEI18nCharsetEncodingRef& operator=(const ICEI18nCharsetEncoding* iOther)
	{
		if (_iI18nCharsetEncoding != iOther)
		{
			if (_iI18nCharsetEncoding)
			{
				ICEI18nCharsetEncoding* tmp = _iI18nCharsetEncoding;
				_iI18nCharsetEncoding = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nCharsetEncoding = const_cast<ICEI18nCharsetEncoding*>(iOther);
			if (_iI18nCharsetEncoding)
			{
				_iI18nCharsetEncoding->_vtbl->_addRef(_iI18nCharsetEncoding);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharsetEncoding** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nCharsetEncodingRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nCharsetEncoding && "has a com object reference. clear first.");
		return & _iI18nCharsetEncoding; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nCharsetEncodingRef& other)
	{
		bool result = false;
		if (_iI18nCharsetEncoding)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nCharsetEncoding);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nCharsetEncoding ? _iI18nCharsetEncoding->_vtbl->_queryInterface(_iI18nCharsetEncoding, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nCharsetEncoding) { _iI18nCharsetEncoding->_vtbl->_addRef(_iI18nCharsetEncoding); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nCharsetEncoding) { _iI18nCharsetEncoding->_vtbl->_release(_iI18nCharsetEncoding); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getIanaName(struct ICEUString* *const ianaNameOut)	{ return _iI18nCharsetEncoding ? _iI18nCharsetEncoding->_vtbl->_getIanaName(_iI18nCharsetEncoding, ianaNameOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getId(eCEI18nIanaMIBEnum *const ianaMIBEnumOut, UINT32 *const idOut)	{ return _iI18nCharsetEncoding ? _iI18nCharsetEncoding->_vtbl->_getId(_iI18nCharsetEncoding, ianaMIBEnumOut, idOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDisplayName(const struct ICEI18nLocale* displayLocale, struct ICEUString* *const displayNameOut)	{ return _iI18nCharsetEncoding ? _iI18nCharsetEncoding->_vtbl->_getDisplayName(_iI18nCharsetEncoding, displayLocale, displayNameOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getUnicodeEncoder(struct ICEI18nUnicodeEncoder* *const unicodeEncoderOut)	{ return _iI18nCharsetEncoding ? _iI18nCharsetEncoding->_vtbl->_getUnicodeEncoder(_iI18nCharsetEncoding, unicodeEncoderOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getUnicodeDecoder(struct ICEI18nUnicodeDecoder* *const unicodeDecoderOut)	{ return _iI18nCharsetEncoding ? _iI18nCharsetEncoding->_vtbl->_getUnicodeDecoder(_iI18nCharsetEncoding, unicodeDecoderOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLocale(struct ICEI18nLocale* *const localeOut)	{ return _iI18nCharsetEncoding ? _iI18nCharsetEncoding->_vtbl->_getLocale(_iI18nCharsetEncoding, localeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult equals(const struct ICEI18nCharsetEncoding* other, bool *const bEqualsOut)	{ return _iI18nCharsetEncoding ? _iI18nCharsetEncoding->_vtbl->_equals(_iI18nCharsetEncoding, other, bEqualsOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult dbgToString(const UTF16CHAR* *const pStr16Out)	{ return _iI18nCharsetEncoding ? _iI18nCharsetEncoding->_vtbl->_dbgToString(_iI18nCharsetEncoding, pStr16Out) : CE_SILK_ERR_UNINITIALIZED; }

#if defined(_DEBUG) || !defined(NDEBUG)
	const UTF16CHAR* dbgToString() { const UTF16CHAR* ret=0; if (_iI18nCharsetEncoding) { dbgToString(&ret); } return ret; }
#endif // _DEBUG || !NDEBUG


private:
	ICEI18nCharsetEncoding* _iI18nCharsetEncoding;
};

/*! @}
 * end of CEComICEI18nCharsetEncodingRef
 */

/*! \defgroup CEComICEI18nClassFactoryRef CEComICEI18nClassFactoryRef
 * @{
 */

class CEComICEI18nClassFactoryRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nClassFactoryRef() : _iI18nClassFactory(0) {}
	CEComICEI18nClassFactoryRef(ICEI18nClassFactory* iOther) : _iI18nClassFactory(0)
	{
		if (iOther)
		{
			_iI18nClassFactory = iOther;
			_iI18nClassFactory->_vtbl->_addRef(_iI18nClassFactory);
		}
	}

	CEComICEI18nClassFactoryRef(const CEComICEI18nClassFactoryRef& other) : _iI18nClassFactory(0)
	{
		if (other._iI18nClassFactory)
		{
			_iI18nClassFactory = other._iI18nClassFactory;
			_iI18nClassFactory->_vtbl->_addRef(_iI18nClassFactory);
		}
	}

	~CEComICEI18nClassFactoryRef()
	{
		if (_iI18nClassFactory)
		{
			ICEI18nClassFactory* tmp = _iI18nClassFactory;
			_iI18nClassFactory = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nClassFactory* iI18nClassFactory;
				void* _ptr;
			} uIntf;
			uIntf.iI18nClassFactory = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nClassFactory, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nClassFactory)
				{
					ICEI18nClassFactory* tmp = _iI18nClassFactory;
					_iI18nClassFactory = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nClassFactory = uIntf.iI18nClassFactory;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nClassFactory* object() const	{ return _iI18nClassFactory; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nClassFactory* iOther)
	{
		if (_iI18nClassFactory)
		{
			ICEI18nClassFactory* tmp = _iI18nClassFactory;
			_iI18nClassFactory = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nClassFactory = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nClassFactory* detach()
	{
		ICEI18nClassFactory* iIntf = _iI18nClassFactory;
		_iI18nClassFactory = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nClassFactory* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nClassFactory;
		if (_iI18nClassFactory)
		{
			_iI18nClassFactory->_vtbl->_addRef(_iI18nClassFactory);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nClassFactory*() const	{ return _iI18nClassFactory; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nClassFactory& operator*() const	{ return *_iI18nClassFactory; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nClassFactory == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nClassFactory* iOther) const	{ return (_iI18nClassFactory != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nClassFactory* iOther) const	{ return (_iI18nClassFactory == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nClassFactoryRef& operator=(const CEComICEI18nClassFactoryRef& other)	{ return operator=(other._iI18nClassFactory); }

	CEComICEI18nClassFactoryRef& operator=(const ICEI18nClassFactory* iOther)
	{
		if (_iI18nClassFactory != iOther)
		{
			if (_iI18nClassFactory)
			{
				ICEI18nClassFactory* tmp = _iI18nClassFactory;
				_iI18nClassFactory = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nClassFactory = const_cast<ICEI18nClassFactory*>(iOther);
			if (_iI18nClassFactory)
			{
				_iI18nClassFactory->_vtbl->_addRef(_iI18nClassFactory);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nClassFactory** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nClassFactoryRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nClassFactory && "has a com object reference. clear first.");
		return & _iI18nClassFactory; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nClassFactoryRef& other)
	{
		bool result = false;
		if (_iI18nClassFactory)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nClassFactory);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_queryInterface(_iI18nClassFactory, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nClassFactory) { _iI18nClassFactory->_vtbl->_addRef(_iI18nClassFactory); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nClassFactory) { _iI18nClassFactory->_vtbl->_release(_iI18nClassFactory); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCharsetEncodings(bool bIanaRegisteredOnly, struct ICEI18nCharsetEncoding* *const encodingsOut, UINT32 *const numOfEncodingsRef)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getCharsetEncodings(_iI18nClassFactory, bIanaRegisteredOnly, encodingsOut, numOfEncodingsRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCharsetEncodingFromIanaName(const UTF16CHAR* str16, const UINT32 numOfChar16, struct ICEI18nCharsetEncoding* *const charsetEncodingOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getCharsetEncodingFromIanaName(_iI18nClassFactory, str16, numOfChar16, charsetEncodingOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCharsetEncodingFromIanaCStringLiteral(const char* str8, struct ICEI18nCharsetEncoding* *const charsetEncodingOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getCharsetEncodingFromIanaCStringLiteral(_iI18nClassFactory, str8, charsetEncodingOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCharsetDetectorInfos(struct ICEI18nCharsetDetectorInfo* *const charsetDetectorInfosOut, UINT32 *const numOfCharsetDetectorInfosRef)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getCharsetDetectorInfos(_iI18nClassFactory, charsetDetectorInfosOut, numOfCharsetDetectorInfosRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCharsetDetectorInfo(UINT32 charsetDetectorCLSID, struct ICEI18nCharsetDetectorInfo* *const charsetDetectorInfoOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getCharsetDetectorInfo(_iI18nClassFactory, charsetDetectorCLSID, charsetDetectorInfoOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getAvailableLocales(struct ICEI18nLocale* *const availableLocalesOut, UINT32 *const numOfAvailableLocalesRef)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getAvailableLocales(_iI18nClassFactory, availableLocalesOut, numOfAvailableLocalesRef) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLocaleFromCStringLiteral(const char* isoLangCStrLiteral, const char* isoCountryCStrLiteral, struct ICEI18nLocale* *const localeOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getLocaleFromCStringLiteral(_iI18nClassFactory, isoLangCStrLiteral, isoCountryCStrLiteral, localeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLocaleFromUtf16Array(UTF16CHAR* isoLang16, UINT32 numOfIsoLangChars16, UTF16CHAR* isoCountry16, UINT32 numOfisoCountryChars16, struct ICEI18nLocale* *const localeOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getLocaleFromUtf16Array(_iI18nClassFactory, isoLang16, numOfIsoLangChars16, isoCountry16, numOfisoCountryChars16, localeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLocaleFromLanguageTagCStringLiteral(const char* langTagCStringLiteral, struct ICEI18nLocale* *const localeOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getLocaleFromLanguageTagCStringLiteral(_iI18nClassFactory, langTagCStringLiteral, localeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLocaleFromLanguageTagUtf16Array(const UTF16CHAR* langTag16, UINT32 numOfLangTagChars16, struct ICEI18nLocale* *const localeOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getLocaleFromLanguageTagUtf16Array(_iI18nClassFactory, langTag16, numOfLangTagChars16, localeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCharacterClassForLocale(const struct ICEI18nLocale* i18nLocale, struct ICEI18nCharacterClass* *const charClassOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getCharacterClassForLocale(_iI18nClassFactory, i18nLocale, charClassOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initConverter(CEI18nConverterInitArg* arg)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_initConverter(_iI18nClassFactory, arg) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult shutdownConverter()	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_shutdownConverter(_iI18nClassFactory) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initResource(struct ICEI18nClassFactory* parentFactory)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_initResource(_iI18nClassFactory, parentFactory) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult shutdownResource(struct ICEI18nClassFactory* parentFactory)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_shutdownResource(_iI18nClassFactory, parentFactory) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getResourcePropertyMap(struct ICEI18nResourcePropertyMap* *const propMapOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getResourcePropertyMap(_iI18nClassFactory, propMapOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCTLEngine(struct ICEI18nCTLEngine* *const ctlEngineOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getCTLEngine(_iI18nClassFactory, ctlEngineOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getBreakIterator(struct ICEI18nLocale* locale, eCEI18nBreakIteratorType type, struct ICEI18nBreakIterator* *const breakIteratorOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getBreakIterator(_iI18nClassFactory, locale, type, breakIteratorOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getBiDiUtilClass(struct ICEI18nBiDiUtilClass* *const bidiUtilClassOut)	{ return _iI18nClassFactory ? _iI18nClassFactory->_vtbl->_getBiDiUtilClass(_iI18nClassFactory, bidiUtilClassOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEI18nClassFactory* _iI18nClassFactory;
};

/*! @}
 * end of CEComICEI18nClassFactoryRef
 */

/*! \defgroup CEComICEI18nCharacterClassRef CEComICEI18nCharacterClassRef
 * @{
 */

class CEComICEI18nCharacterClassRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nCharacterClassRef() : _iI18nCharacterClass(0) {}
	CEComICEI18nCharacterClassRef(ICEI18nCharacterClass* iOther) : _iI18nCharacterClass(0)
	{
		if (iOther)
		{
			_iI18nCharacterClass = iOther;
			_iI18nCharacterClass->_vtbl->_addRef(_iI18nCharacterClass);
		}
	}

	CEComICEI18nCharacterClassRef(const CEComICEI18nCharacterClassRef& other) : _iI18nCharacterClass(0)
	{
		if (other._iI18nCharacterClass)
		{
			_iI18nCharacterClass = other._iI18nCharacterClass;
			_iI18nCharacterClass->_vtbl->_addRef(_iI18nCharacterClass);
		}
	}

	~CEComICEI18nCharacterClassRef()
	{
		if (_iI18nCharacterClass)
		{
			ICEI18nCharacterClass* tmp = _iI18nCharacterClass;
			_iI18nCharacterClass = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nCharacterClass* iI18nCharacterClass;
				void* _ptr;
			} uIntf;
			uIntf.iI18nCharacterClass = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nCharacterClass, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nCharacterClass)
				{
					ICEI18nCharacterClass* tmp = _iI18nCharacterClass;
					_iI18nCharacterClass = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nCharacterClass = uIntf.iI18nCharacterClass;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharacterClass* object() const	{ return _iI18nCharacterClass; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nCharacterClass* iOther)
	{
		if (_iI18nCharacterClass)
		{
			ICEI18nCharacterClass* tmp = _iI18nCharacterClass;
			_iI18nCharacterClass = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nCharacterClass = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nCharacterClass* detach()
	{
		ICEI18nCharacterClass* iIntf = _iI18nCharacterClass;
		_iI18nCharacterClass = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nCharacterClass* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nCharacterClass;
		if (_iI18nCharacterClass)
		{
			_iI18nCharacterClass->_vtbl->_addRef(_iI18nCharacterClass);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nCharacterClass*() const	{ return _iI18nCharacterClass; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharacterClass& operator*() const	{ return *_iI18nCharacterClass; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nCharacterClass == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nCharacterClass* iOther) const	{ return (_iI18nCharacterClass != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nCharacterClass* iOther) const	{ return (_iI18nCharacterClass == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nCharacterClassRef& operator=(const CEComICEI18nCharacterClassRef& other)	{ return operator=(other._iI18nCharacterClass); }

	CEComICEI18nCharacterClassRef& operator=(const ICEI18nCharacterClass* iOther)
	{
		if (_iI18nCharacterClass != iOther)
		{
			if (_iI18nCharacterClass)
			{
				ICEI18nCharacterClass* tmp = _iI18nCharacterClass;
				_iI18nCharacterClass = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nCharacterClass = const_cast<ICEI18nCharacterClass*>(iOther);
			if (_iI18nCharacterClass)
			{
				_iI18nCharacterClass->_vtbl->_addRef(_iI18nCharacterClass);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharacterClass** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nCharacterClassRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nCharacterClass && "has a com object reference. clear first.");
		return & _iI18nCharacterClass; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nCharacterClassRef& other)
	{
		bool result = false;
		if (_iI18nCharacterClass)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nCharacterClass);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nCharacterClass ? _iI18nCharacterClass->_vtbl->_queryInterface(_iI18nCharacterClass, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nCharacterClass) { _iI18nCharacterClass->_vtbl->_addRef(_iI18nCharacterClass); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nCharacterClass) { _iI18nCharacterClass->_vtbl->_release(_iI18nCharacterClass); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isBreakableBetween(UTF32CHAR c1, UTF32CHAR c2, bool *const bBreakableOut)	{ return _iI18nCharacterClass ? _iI18nCharacterClass->_vtbl->_isBreakableBetween(_iI18nCharacterClass, c1, c2, bBreakableOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isBreakableChar(UTF32CHAR c, bool *const bBreakableOut)	{ return _iI18nCharacterClass ? _iI18nCharacterClass->_vtbl->_isBreakableChar(_iI18nCharacterClass, c, bBreakableOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isBreakableBetweenGCB(UTF32CHAR c1, UTF32CHAR c2, bool *const bBreakableOut)	{ return _iI18nCharacterClass ? _iI18nCharacterClass->_vtbl->_isBreakableBetweenGCB(_iI18nCharacterClass, c1, c2, bBreakableOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isBreakableBetweenWord(UTF32CHAR c1, UTF32CHAR c2, bool *const bBreakableOut)	{ return _iI18nCharacterClass ? _iI18nCharacterClass->_vtbl->_isBreakableBetweenWord(_iI18nCharacterClass, c1, c2, bBreakableOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult toUpper(UTF32CHAR c32, UTF32CHAR *const c32Out)	{ return _iI18nCharacterClass ? _iI18nCharacterClass->_vtbl->_toUpper(_iI18nCharacterClass, c32, c32Out) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult toLower(UTF32CHAR c32, UTF32CHAR *const c32Out)	{ return _iI18nCharacterClass ? _iI18nCharacterClass->_vtbl->_toLower(_iI18nCharacterClass, c32, c32Out) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult toTitle(UTF32CHAR c32, UTF32CHAR *const c32Out)	{ return _iI18nCharacterClass ? _iI18nCharacterClass->_vtbl->_toTitle(_iI18nCharacterClass, c32, c32Out) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getScriptCode(UTF32CHAR c32, eCEI18nScriptCode *const scOut)	{ return _iI18nCharacterClass ? _iI18nCharacterClass->_vtbl->_getScriptCode(_iI18nCharacterClass, c32, scOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getURangeCode(UTF32CHAR c32, eCEI18nURangeCode *const rangeOut)	{ return _iI18nCharacterClass ? _iI18nCharacterClass->_vtbl->_getURangeCode(_iI18nCharacterClass, c32, rangeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getBiDIDirection(UTF32CHAR c32, eCEI18nBiDiDirection *const bidiDirectionOut)	{ return _iI18nCharacterClass ? _iI18nCharacterClass->_vtbl->_getBiDIDirection(_iI18nCharacterClass, c32, bidiDirectionOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEI18nCharacterClass* _iI18nCharacterClass;
};

/*! @}
 * end of CEComICEI18nCharacterClassRef
 */

/*! \defgroup CEComICEI18nCharsetDetectorInfoRef CEComICEI18nCharsetDetectorInfoRef
 * @{
 */

class CEComICEI18nCharsetDetectorInfoRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nCharsetDetectorInfoRef() : _iI18nCharsetDetectorInfo(0) {}
	CEComICEI18nCharsetDetectorInfoRef(ICEI18nCharsetDetectorInfo* iOther) : _iI18nCharsetDetectorInfo(0)
	{
		if (iOther)
		{
			_iI18nCharsetDetectorInfo = iOther;
			_iI18nCharsetDetectorInfo->_vtbl->_addRef(_iI18nCharsetDetectorInfo);
		}
	}

	CEComICEI18nCharsetDetectorInfoRef(const CEComICEI18nCharsetDetectorInfoRef& other) : _iI18nCharsetDetectorInfo(0)
	{
		if (other._iI18nCharsetDetectorInfo)
		{
			_iI18nCharsetDetectorInfo = other._iI18nCharsetDetectorInfo;
			_iI18nCharsetDetectorInfo->_vtbl->_addRef(_iI18nCharsetDetectorInfo);
		}
	}

	~CEComICEI18nCharsetDetectorInfoRef()
	{
		if (_iI18nCharsetDetectorInfo)
		{
			ICEI18nCharsetDetectorInfo* tmp = _iI18nCharsetDetectorInfo;
			_iI18nCharsetDetectorInfo = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nCharsetDetectorInfo* iI18nCharsetDetectorInfo;
				void* _ptr;
			} uIntf;
			uIntf.iI18nCharsetDetectorInfo = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nCharsetDetectorInfo, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nCharsetDetectorInfo)
				{
					ICEI18nCharsetDetectorInfo* tmp = _iI18nCharsetDetectorInfo;
					_iI18nCharsetDetectorInfo = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nCharsetDetectorInfo = uIntf.iI18nCharsetDetectorInfo;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharsetDetectorInfo* object() const	{ return _iI18nCharsetDetectorInfo; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nCharsetDetectorInfo* iOther)
	{
		if (_iI18nCharsetDetectorInfo)
		{
			ICEI18nCharsetDetectorInfo* tmp = _iI18nCharsetDetectorInfo;
			_iI18nCharsetDetectorInfo = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nCharsetDetectorInfo = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nCharsetDetectorInfo* detach()
	{
		ICEI18nCharsetDetectorInfo* iIntf = _iI18nCharsetDetectorInfo;
		_iI18nCharsetDetectorInfo = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nCharsetDetectorInfo* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nCharsetDetectorInfo;
		if (_iI18nCharsetDetectorInfo)
		{
			_iI18nCharsetDetectorInfo->_vtbl->_addRef(_iI18nCharsetDetectorInfo);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nCharsetDetectorInfo*() const	{ return _iI18nCharsetDetectorInfo; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharsetDetectorInfo& operator*() const	{ return *_iI18nCharsetDetectorInfo; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nCharsetDetectorInfo == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nCharsetDetectorInfo* iOther) const	{ return (_iI18nCharsetDetectorInfo != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nCharsetDetectorInfo* iOther) const	{ return (_iI18nCharsetDetectorInfo == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nCharsetDetectorInfoRef& operator=(const CEComICEI18nCharsetDetectorInfoRef& other)	{ return operator=(other._iI18nCharsetDetectorInfo); }

	CEComICEI18nCharsetDetectorInfoRef& operator=(const ICEI18nCharsetDetectorInfo* iOther)
	{
		if (_iI18nCharsetDetectorInfo != iOther)
		{
			if (_iI18nCharsetDetectorInfo)
			{
				ICEI18nCharsetDetectorInfo* tmp = _iI18nCharsetDetectorInfo;
				_iI18nCharsetDetectorInfo = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nCharsetDetectorInfo = const_cast<ICEI18nCharsetDetectorInfo*>(iOther);
			if (_iI18nCharsetDetectorInfo)
			{
				_iI18nCharsetDetectorInfo->_vtbl->_addRef(_iI18nCharsetDetectorInfo);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharsetDetectorInfo** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nCharsetDetectorInfoRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nCharsetDetectorInfo && "has a com object reference. clear first.");
		return & _iI18nCharsetDetectorInfo; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nCharsetDetectorInfoRef& other)
	{
		bool result = false;
		if (_iI18nCharsetDetectorInfo)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nCharsetDetectorInfo);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nCharsetDetectorInfo ? _iI18nCharsetDetectorInfo->_vtbl->_queryInterface(_iI18nCharsetDetectorInfo, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nCharsetDetectorInfo) { _iI18nCharsetDetectorInfo->_vtbl->_addRef(_iI18nCharsetDetectorInfo); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nCharsetDetectorInfo) { _iI18nCharsetDetectorInfo->_vtbl->_release(_iI18nCharsetDetectorInfo); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCLSID(UINT32 *const clsidOut)	{ return _iI18nCharsetDetectorInfo ? _iI18nCharsetDetectorInfo->_vtbl->_getCLSID(_iI18nCharsetDetectorInfo, clsidOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getMinimumInputSizeInBytes(UINT32 *const minInputSizeInBytesOut)	{ return _iI18nCharsetDetectorInfo ? _iI18nCharsetDetectorInfo->_vtbl->_getMinimumInputSizeInBytes(_iI18nCharsetDetectorInfo, minInputSizeInBytesOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDisplayName(const struct ICEI18nLocale* displayLocale, struct ICEUString* *const displayNameOut)	{ return _iI18nCharsetDetectorInfo ? _iI18nCharsetDetectorInfo->_vtbl->_getDisplayName(_iI18nCharsetDetectorInfo, displayLocale, displayNameOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createCharsetDetector(struct ICEI18nCharsetDetector* *const detectorOut)	{ return _iI18nCharsetDetectorInfo ? _iI18nCharsetDetectorInfo->_vtbl->_createCharsetDetector(_iI18nCharsetDetectorInfo, detectorOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEI18nCharsetDetectorInfo* _iI18nCharsetDetectorInfo;
};

/*! @}
 * end of CEComICEI18nCharsetDetectorInfoRef
 */

/*! \defgroup CEComICEI18nCharsetDetectorRef CEComICEI18nCharsetDetectorRef
 * @{
 */

class CEComICEI18nCharsetDetectorRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nCharsetDetectorRef() : _iI18nCharsetDetector(0)
	{
#if defined(_DEBUG) || !defined(NDEBUG)
		dbgToString();
#endif //_DEBUG || !NDEBUG
	}
	CEComICEI18nCharsetDetectorRef(ICEI18nCharsetDetector* iOther) : _iI18nCharsetDetector(0)
	{
		if (iOther)
		{
			_iI18nCharsetDetector = iOther;
			_iI18nCharsetDetector->_vtbl->_addRef(_iI18nCharsetDetector);
		}
	}

	CEComICEI18nCharsetDetectorRef(const CEComICEI18nCharsetDetectorRef& other) : _iI18nCharsetDetector(0)
	{
		if (other._iI18nCharsetDetector)
		{
			_iI18nCharsetDetector = other._iI18nCharsetDetector;
			_iI18nCharsetDetector->_vtbl->_addRef(_iI18nCharsetDetector);
		}
	}

	~CEComICEI18nCharsetDetectorRef()
	{
		if (_iI18nCharsetDetector)
		{
			ICEI18nCharsetDetector* tmp = _iI18nCharsetDetector;
			_iI18nCharsetDetector = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nCharsetDetector* iI18nCharsetDetector;
				void* _ptr;
			} uIntf;
			uIntf.iI18nCharsetDetector = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nCharsetDetector, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nCharsetDetector)
				{
					ICEI18nCharsetDetector* tmp = _iI18nCharsetDetector;
					_iI18nCharsetDetector = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nCharsetDetector = uIntf.iI18nCharsetDetector;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharsetDetector* object() const	{ return _iI18nCharsetDetector; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nCharsetDetector* iOther)
	{
		if (_iI18nCharsetDetector)
		{
			ICEI18nCharsetDetector* tmp = _iI18nCharsetDetector;
			_iI18nCharsetDetector = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nCharsetDetector = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nCharsetDetector* detach()
	{
		ICEI18nCharsetDetector* iIntf = _iI18nCharsetDetector;
		_iI18nCharsetDetector = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nCharsetDetector* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nCharsetDetector;
		if (_iI18nCharsetDetector)
		{
			_iI18nCharsetDetector->_vtbl->_addRef(_iI18nCharsetDetector);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nCharsetDetector*() const	{ return _iI18nCharsetDetector; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharsetDetector& operator*() const	{ return *_iI18nCharsetDetector; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nCharsetDetector == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nCharsetDetector* iOther) const	{ return (_iI18nCharsetDetector != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nCharsetDetector* iOther) const	{ return (_iI18nCharsetDetector == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nCharsetDetectorRef& operator=(const CEComICEI18nCharsetDetectorRef& other)	{ return operator=(other._iI18nCharsetDetector); }

	CEComICEI18nCharsetDetectorRef& operator=(const ICEI18nCharsetDetector* iOther)
	{
		if (_iI18nCharsetDetector != iOther)
		{
			if (_iI18nCharsetDetector)
			{
				ICEI18nCharsetDetector* tmp = _iI18nCharsetDetector;
				_iI18nCharsetDetector = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nCharsetDetector = const_cast<ICEI18nCharsetDetector*>(iOther);
			if (_iI18nCharsetDetector)
			{
				_iI18nCharsetDetector->_vtbl->_addRef(_iI18nCharsetDetector);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharsetDetector** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nCharsetDetectorRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nCharsetDetector && "has a com object reference. clear first.");
		return & _iI18nCharsetDetector; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nCharsetDetectorRef& other)
	{
		bool result = false;
		if (_iI18nCharsetDetector)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nCharsetDetector);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nCharsetDetector ? _iI18nCharsetDetector->_vtbl->_queryInterface(_iI18nCharsetDetector, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nCharsetDetector) { _iI18nCharsetDetector->_vtbl->_addRef(_iI18nCharsetDetector); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nCharsetDetector) { _iI18nCharsetDetector->_vtbl->_release(_iI18nCharsetDetector); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult doIt(const UCHAR8* inbytes, UINT32 inLen, eCEI18nCharsetDetectStatus *const detectStatusOut, struct ICEI18nCharsetEncoding* *const charsetEncodingOut)	{ return _iI18nCharsetDetector ? _iI18nCharsetDetector->_vtbl->_doIt(_iI18nCharsetDetector, inbytes, inLen, detectStatusOut, charsetEncodingOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult finish(eCEI18nCharsetDetectStatus *const detectStatusOut, struct ICEI18nCharsetEncoding* *const charsetEncodingOut)	{ return _iI18nCharsetDetector ? _iI18nCharsetDetector->_vtbl->_finish(_iI18nCharsetDetector, detectStatusOut, charsetEncodingOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult reset()	{ return _iI18nCharsetDetector ? _iI18nCharsetDetector->_vtbl->_reset(_iI18nCharsetDetector) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult dbgToString(const UTF16CHAR* *const pStr16Out)	{ return _iI18nCharsetDetector ? _iI18nCharsetDetector->_vtbl->_dbgToString(_iI18nCharsetDetector, pStr16Out) : CE_SILK_ERR_UNINITIALIZED; }

#if defined(_DEBUG) || !defined(NDEBUG)
	const UTF16CHAR* dbgToString() { const UTF16CHAR* ret=0; if (_iI18nCharsetDetector) { dbgToString(&ret); } return ret; }
#endif // _DEBUG || !NDEBUG


private:
	ICEI18nCharsetDetector* _iI18nCharsetDetector;
};

/*! @}
 * end of CEComICEI18nCharsetDetectorRef
 */

/*! \defgroup CEComICEI18nUnicodeEncoderRef CEComICEI18nUnicodeEncoderRef
 * @{
 */

class CEComICEI18nUnicodeEncoderRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nUnicodeEncoderRef() : _iI18nUnicodeEncoder(0)
	{
#if defined(_DEBUG) || !defined(NDEBUG)
		dbgToString();
#endif //_DEBUG || !NDEBUG
	}
	CEComICEI18nUnicodeEncoderRef(ICEI18nUnicodeEncoder* iOther) : _iI18nUnicodeEncoder(0)
	{
		if (iOther)
		{
			_iI18nUnicodeEncoder = iOther;
			_iI18nUnicodeEncoder->_vtbl->_addRef(_iI18nUnicodeEncoder);
		}
	}

	CEComICEI18nUnicodeEncoderRef(const CEComICEI18nUnicodeEncoderRef& other) : _iI18nUnicodeEncoder(0)
	{
		if (other._iI18nUnicodeEncoder)
		{
			_iI18nUnicodeEncoder = other._iI18nUnicodeEncoder;
			_iI18nUnicodeEncoder->_vtbl->_addRef(_iI18nUnicodeEncoder);
		}
	}

	~CEComICEI18nUnicodeEncoderRef()
	{
		if (_iI18nUnicodeEncoder)
		{
			ICEI18nUnicodeEncoder* tmp = _iI18nUnicodeEncoder;
			_iI18nUnicodeEncoder = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nUnicodeEncoder* iI18nUnicodeEncoder;
				void* _ptr;
			} uIntf;
			uIntf.iI18nUnicodeEncoder = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nUnicodeEncoder, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nUnicodeEncoder)
				{
					ICEI18nUnicodeEncoder* tmp = _iI18nUnicodeEncoder;
					_iI18nUnicodeEncoder = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nUnicodeEncoder = uIntf.iI18nUnicodeEncoder;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nUnicodeEncoder* object() const	{ return _iI18nUnicodeEncoder; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nUnicodeEncoder* iOther)
	{
		if (_iI18nUnicodeEncoder)
		{
			ICEI18nUnicodeEncoder* tmp = _iI18nUnicodeEncoder;
			_iI18nUnicodeEncoder = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nUnicodeEncoder = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nUnicodeEncoder* detach()
	{
		ICEI18nUnicodeEncoder* iIntf = _iI18nUnicodeEncoder;
		_iI18nUnicodeEncoder = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nUnicodeEncoder* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nUnicodeEncoder;
		if (_iI18nUnicodeEncoder)
		{
			_iI18nUnicodeEncoder->_vtbl->_addRef(_iI18nUnicodeEncoder);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nUnicodeEncoder*() const	{ return _iI18nUnicodeEncoder; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nUnicodeEncoder& operator*() const	{ return *_iI18nUnicodeEncoder; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nUnicodeEncoder == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nUnicodeEncoder* iOther) const	{ return (_iI18nUnicodeEncoder != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nUnicodeEncoder* iOther) const	{ return (_iI18nUnicodeEncoder == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nUnicodeEncoderRef& operator=(const CEComICEI18nUnicodeEncoderRef& other)	{ return operator=(other._iI18nUnicodeEncoder); }

	CEComICEI18nUnicodeEncoderRef& operator=(const ICEI18nUnicodeEncoder* iOther)
	{
		if (_iI18nUnicodeEncoder != iOther)
		{
			if (_iI18nUnicodeEncoder)
			{
				ICEI18nUnicodeEncoder* tmp = _iI18nUnicodeEncoder;
				_iI18nUnicodeEncoder = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nUnicodeEncoder = const_cast<ICEI18nUnicodeEncoder*>(iOther);
			if (_iI18nUnicodeEncoder)
			{
				_iI18nUnicodeEncoder->_vtbl->_addRef(_iI18nUnicodeEncoder);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nUnicodeEncoder** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nUnicodeEncoderRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nUnicodeEncoder && "has a com object reference. clear first.");
		return & _iI18nUnicodeEncoder; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nUnicodeEncoderRef& other)
	{
		bool result = false;
		if (_iI18nUnicodeEncoder)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nUnicodeEncoder);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nUnicodeEncoder ? _iI18nUnicodeEncoder->_vtbl->_queryInterface(_iI18nUnicodeEncoder, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nUnicodeEncoder) { _iI18nUnicodeEncoder->_vtbl->_addRef(_iI18nUnicodeEncoder); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nUnicodeEncoder) { _iI18nUnicodeEncoder->_vtbl->_release(_iI18nUnicodeEncoder); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult requiredSize(const UTF16CHAR* utf16Array, UINT32 numOfUtf16, UINT32 *const numOfBytesOut, eCEI18nErr *const i18nErrOut)	{ return _iI18nUnicodeEncoder ? _iI18nUnicodeEncoder->_vtbl->_requiredSize(_iI18nUnicodeEncoder, utf16Array, numOfUtf16, numOfBytesOut, i18nErrOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult convert(const UTF16CHAR* srcUtf16Chars, UINT32 *const numOfSrcUtf16CharsRef, void *const destBytesOut, UINT32 *const numOfDestBytesRef, eCEI18nErr *const i18nErrOut)	{ return _iI18nUnicodeEncoder ? _iI18nUnicodeEncoder->_vtbl->_convert(_iI18nUnicodeEncoder, srcUtf16Chars, numOfSrcUtf16CharsRef, destBytesOut, numOfDestBytesRef, i18nErrOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult convertEx(const UTF16CHAR* srcUtf16Chars, UINT32 *const numOfSrcUtf16CharsRef, void *const destBytesOut, UINT32 *const numOfDestBytesRef, UCHAR8 altCharForUnknownChar8, eCEI18nErr *const i18nErrOut)	{ return _iI18nUnicodeEncoder ? _iI18nUnicodeEncoder->_vtbl->_convertEx(_iI18nUnicodeEncoder, srcUtf16Chars, numOfSrcUtf16CharsRef, destBytesOut, numOfDestBytesRef, altCharForUnknownChar8, i18nErrOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCharsetEncoding(struct ICEI18nCharsetEncoding* *const encodingOut)	{ return _iI18nUnicodeEncoder ? _iI18nUnicodeEncoder->_vtbl->_getCharsetEncoding(_iI18nUnicodeEncoder, encodingOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult dbgToString(const UTF16CHAR* *const pStr16Out)	{ return _iI18nUnicodeEncoder ? _iI18nUnicodeEncoder->_vtbl->_dbgToString(_iI18nUnicodeEncoder, pStr16Out) : CE_SILK_ERR_UNINITIALIZED; }

#if defined(_DEBUG) || !defined(NDEBUG)
	const UTF16CHAR* dbgToString() { const UTF16CHAR* ret=0; if (_iI18nUnicodeEncoder) { dbgToString(&ret); } return ret; }
#endif // _DEBUG || !NDEBUG


private:
	ICEI18nUnicodeEncoder* _iI18nUnicodeEncoder;
};

/*! @}
 * end of CEComICEI18nUnicodeEncoderRef
 */

/*! \defgroup CEComICEI18nUnicodeDecoderRef CEComICEI18nUnicodeDecoderRef
 * @{
 */

class CEComICEI18nUnicodeDecoderRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nUnicodeDecoderRef() : _iI18nUnicodeDecoder(0)
	{
#if defined(_DEBUG) || !defined(NDEBUG)
		dbgToString();
#endif //_DEBUG || !NDEBUG
	}
	CEComICEI18nUnicodeDecoderRef(ICEI18nUnicodeDecoder* iOther) : _iI18nUnicodeDecoder(0)
	{
		if (iOther)
		{
			_iI18nUnicodeDecoder = iOther;
			_iI18nUnicodeDecoder->_vtbl->_addRef(_iI18nUnicodeDecoder);
		}
	}

	CEComICEI18nUnicodeDecoderRef(const CEComICEI18nUnicodeDecoderRef& other) : _iI18nUnicodeDecoder(0)
	{
		if (other._iI18nUnicodeDecoder)
		{
			_iI18nUnicodeDecoder = other._iI18nUnicodeDecoder;
			_iI18nUnicodeDecoder->_vtbl->_addRef(_iI18nUnicodeDecoder);
		}
	}

	~CEComICEI18nUnicodeDecoderRef()
	{
		if (_iI18nUnicodeDecoder)
		{
			ICEI18nUnicodeDecoder* tmp = _iI18nUnicodeDecoder;
			_iI18nUnicodeDecoder = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nUnicodeDecoder* iI18nUnicodeDecoder;
				void* _ptr;
			} uIntf;
			uIntf.iI18nUnicodeDecoder = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nUnicodeDecoder, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nUnicodeDecoder)
				{
					ICEI18nUnicodeDecoder* tmp = _iI18nUnicodeDecoder;
					_iI18nUnicodeDecoder = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nUnicodeDecoder = uIntf.iI18nUnicodeDecoder;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nUnicodeDecoder* object() const	{ return _iI18nUnicodeDecoder; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nUnicodeDecoder* iOther)
	{
		if (_iI18nUnicodeDecoder)
		{
			ICEI18nUnicodeDecoder* tmp = _iI18nUnicodeDecoder;
			_iI18nUnicodeDecoder = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nUnicodeDecoder = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nUnicodeDecoder* detach()
	{
		ICEI18nUnicodeDecoder* iIntf = _iI18nUnicodeDecoder;
		_iI18nUnicodeDecoder = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nUnicodeDecoder* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nUnicodeDecoder;
		if (_iI18nUnicodeDecoder)
		{
			_iI18nUnicodeDecoder->_vtbl->_addRef(_iI18nUnicodeDecoder);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nUnicodeDecoder*() const	{ return _iI18nUnicodeDecoder; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nUnicodeDecoder& operator*() const	{ return *_iI18nUnicodeDecoder; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nUnicodeDecoder == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nUnicodeDecoder* iOther) const	{ return (_iI18nUnicodeDecoder != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nUnicodeDecoder* iOther) const	{ return (_iI18nUnicodeDecoder == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nUnicodeDecoderRef& operator=(const CEComICEI18nUnicodeDecoderRef& other)	{ return operator=(other._iI18nUnicodeDecoder); }

	CEComICEI18nUnicodeDecoderRef& operator=(const ICEI18nUnicodeDecoder* iOther)
	{
		if (_iI18nUnicodeDecoder != iOther)
		{
			if (_iI18nUnicodeDecoder)
			{
				ICEI18nUnicodeDecoder* tmp = _iI18nUnicodeDecoder;
				_iI18nUnicodeDecoder = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nUnicodeDecoder = const_cast<ICEI18nUnicodeDecoder*>(iOther);
			if (_iI18nUnicodeDecoder)
			{
				_iI18nUnicodeDecoder->_vtbl->_addRef(_iI18nUnicodeDecoder);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nUnicodeDecoder** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nUnicodeDecoderRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nUnicodeDecoder && "has a com object reference. clear first.");
		return & _iI18nUnicodeDecoder; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nUnicodeDecoderRef& other)
	{
		bool result = false;
		if (_iI18nUnicodeDecoder)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nUnicodeDecoder);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nUnicodeDecoder ? _iI18nUnicodeDecoder->_vtbl->_queryInterface(_iI18nUnicodeDecoder, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nUnicodeDecoder) { _iI18nUnicodeDecoder->_vtbl->_addRef(_iI18nUnicodeDecoder); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nUnicodeDecoder) { _iI18nUnicodeDecoder->_vtbl->_release(_iI18nUnicodeDecoder); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult requiredSize(const void* bytes, UINT32 numOfBytes, UINT32 *const numOfUtf16Out, eCEI18nErr *const i18nErrOut)	{ return _iI18nUnicodeDecoder ? _iI18nUnicodeDecoder->_vtbl->_requiredSize(_iI18nUnicodeDecoder, bytes, numOfBytes, numOfUtf16Out, i18nErrOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult convert(const void* srcBytes, UINT32 *const numOfSrcBytesRef, UTF16CHAR *const destUtf16CharsOut, UINT32 *const destNumOfUtf16CharsRef, eCEI18nErr *const i18nErrOut)	{ return _iI18nUnicodeDecoder ? _iI18nUnicodeDecoder->_vtbl->_convert(_iI18nUnicodeDecoder, srcBytes, numOfSrcBytesRef, destUtf16CharsOut, destNumOfUtf16CharsRef, i18nErrOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult convertEx(const void* srcBytes, UINT32 *const numOfSrcBytesRef, UTF16CHAR *const destUtf16CharsOut, UINT32 *const destNumOfUtf16CharsRef, UCHAR8 altCharForUnknownChar8, eCEI18nErr *const i18nErrOut)	{ return _iI18nUnicodeDecoder ? _iI18nUnicodeDecoder->_vtbl->_convertEx(_iI18nUnicodeDecoder, srcBytes, numOfSrcBytesRef, destUtf16CharsOut, destNumOfUtf16CharsRef, altCharForUnknownChar8, i18nErrOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getCharsetEncoding(struct ICEI18nCharsetEncoding* *const encodingOut)	{ return _iI18nUnicodeDecoder ? _iI18nUnicodeDecoder->_vtbl->_getCharsetEncoding(_iI18nUnicodeDecoder, encodingOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult dbgToString(const UTF16CHAR* *const pStr16Out)	{ return _iI18nUnicodeDecoder ? _iI18nUnicodeDecoder->_vtbl->_dbgToString(_iI18nUnicodeDecoder, pStr16Out) : CE_SILK_ERR_UNINITIALIZED; }

#if defined(_DEBUG) || !defined(NDEBUG)
	const UTF16CHAR* dbgToString() { const UTF16CHAR* ret=0; if (_iI18nUnicodeDecoder) { dbgToString(&ret); } return ret; }
#endif // _DEBUG || !NDEBUG


private:
	ICEI18nUnicodeDecoder* _iI18nUnicodeDecoder;
};

/*! @}
 * end of CEComICEI18nUnicodeDecoderRef
 */

/*! \defgroup CEComICEI18nResourcePropertyMapRef CEComICEI18nResourcePropertyMapRef
 * @{
 */

class CEComICEI18nResourcePropertyMapRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nResourcePropertyMapRef() : _iI18nResourcePropertyMap(0) {}
	CEComICEI18nResourcePropertyMapRef(ICEI18nResourcePropertyMap* iOther) : _iI18nResourcePropertyMap(0)
	{
		if (iOther)
		{
			_iI18nResourcePropertyMap = iOther;
			_iI18nResourcePropertyMap->_vtbl->_addRef(_iI18nResourcePropertyMap);
		}
	}

	CEComICEI18nResourcePropertyMapRef(const CEComICEI18nResourcePropertyMapRef& other) : _iI18nResourcePropertyMap(0)
	{
		if (other._iI18nResourcePropertyMap)
		{
			_iI18nResourcePropertyMap = other._iI18nResourcePropertyMap;
			_iI18nResourcePropertyMap->_vtbl->_addRef(_iI18nResourcePropertyMap);
		}
	}

	~CEComICEI18nResourcePropertyMapRef()
	{
		if (_iI18nResourcePropertyMap)
		{
			ICEI18nResourcePropertyMap* tmp = _iI18nResourcePropertyMap;
			_iI18nResourcePropertyMap = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nResourcePropertyMap* iI18nResourcePropertyMap;
				void* _ptr;
			} uIntf;
			uIntf.iI18nResourcePropertyMap = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nResourcePropertyMap, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nResourcePropertyMap)
				{
					ICEI18nResourcePropertyMap* tmp = _iI18nResourcePropertyMap;
					_iI18nResourcePropertyMap = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nResourcePropertyMap = uIntf.iI18nResourcePropertyMap;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nResourcePropertyMap* object() const	{ return _iI18nResourcePropertyMap; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nResourcePropertyMap* iOther)
	{
		if (_iI18nResourcePropertyMap)
		{
			ICEI18nResourcePropertyMap* tmp = _iI18nResourcePropertyMap;
			_iI18nResourcePropertyMap = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nResourcePropertyMap = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nResourcePropertyMap* detach()
	{
		ICEI18nResourcePropertyMap* iIntf = _iI18nResourcePropertyMap;
		_iI18nResourcePropertyMap = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nResourcePropertyMap* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nResourcePropertyMap;
		if (_iI18nResourcePropertyMap)
		{
			_iI18nResourcePropertyMap->_vtbl->_addRef(_iI18nResourcePropertyMap);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nResourcePropertyMap*() const	{ return _iI18nResourcePropertyMap; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nResourcePropertyMap& operator*() const	{ return *_iI18nResourcePropertyMap; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nResourcePropertyMap == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nResourcePropertyMap* iOther) const	{ return (_iI18nResourcePropertyMap != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nResourcePropertyMap* iOther) const	{ return (_iI18nResourcePropertyMap == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nResourcePropertyMapRef& operator=(const CEComICEI18nResourcePropertyMapRef& other)	{ return operator=(other._iI18nResourcePropertyMap); }

	CEComICEI18nResourcePropertyMapRef& operator=(const ICEI18nResourcePropertyMap* iOther)
	{
		if (_iI18nResourcePropertyMap != iOther)
		{
			if (_iI18nResourcePropertyMap)
			{
				ICEI18nResourcePropertyMap* tmp = _iI18nResourcePropertyMap;
				_iI18nResourcePropertyMap = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nResourcePropertyMap = const_cast<ICEI18nResourcePropertyMap*>(iOther);
			if (_iI18nResourcePropertyMap)
			{
				_iI18nResourcePropertyMap->_vtbl->_addRef(_iI18nResourcePropertyMap);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nResourcePropertyMap** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nResourcePropertyMapRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nResourcePropertyMap && "has a com object reference. clear first.");
		return & _iI18nResourcePropertyMap; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nResourcePropertyMapRef& other)
	{
		bool result = false;
		if (_iI18nResourcePropertyMap)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nResourcePropertyMap);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nResourcePropertyMap ? _iI18nResourcePropertyMap->_vtbl->_queryInterface(_iI18nResourcePropertyMap, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nResourcePropertyMap) { _iI18nResourcePropertyMap->_vtbl->_addRef(_iI18nResourcePropertyMap); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nResourcePropertyMap) { _iI18nResourcePropertyMap->_vtbl->_release(_iI18nResourcePropertyMap); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setString(const char* category, const struct ICEUString* iName, const struct ICEI18nLocale* locale, const struct ICEUString* iStr)	{ return _iI18nResourcePropertyMap ? _iI18nResourcePropertyMap->_vtbl->_setString(_iI18nResourcePropertyMap, category, iName, locale, iStr) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getString(const char* category, const struct ICEUString* iName, const struct ICEI18nLocale* locale, struct ICEUString* *const iStrOut)	{ return _iI18nResourcePropertyMap ? _iI18nResourcePropertyMap->_vtbl->_getString(_iI18nResourcePropertyMap, category, iName, locale, iStrOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setStringById(const char* category, UINT32 id32, const struct ICEI18nLocale* locale, const struct ICEUString* iStr)	{ return _iI18nResourcePropertyMap ? _iI18nResourcePropertyMap->_vtbl->_setStringById(_iI18nResourcePropertyMap, category, id32, locale, iStr) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStringById(const char* category, UINT32 id32, const struct ICEI18nLocale* locale, struct ICEUString* *const iStrOut)	{ return _iI18nResourcePropertyMap ? _iI18nResourcePropertyMap->_vtbl->_getStringById(_iI18nResourcePropertyMap, category, id32, locale, iStrOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEI18nResourcePropertyMap* _iI18nResourcePropertyMap;
};

/*! @}
 * end of CEComICEI18nResourcePropertyMapRef
 */

/*! \defgroup CEComICEI18nCharMapperRef CEComICEI18nCharMapperRef
 * @{
 */

class CEComICEI18nCharMapperRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nCharMapperRef() : _iI18nCharMapper(0) {}
	CEComICEI18nCharMapperRef(ICEI18nCharMapper* iOther) : _iI18nCharMapper(0)
	{
		if (iOther)
		{
			_iI18nCharMapper = iOther;
			_iI18nCharMapper->_vtbl->_addRef(_iI18nCharMapper);
		}
	}

	CEComICEI18nCharMapperRef(const CEComICEI18nCharMapperRef& other) : _iI18nCharMapper(0)
	{
		if (other._iI18nCharMapper)
		{
			_iI18nCharMapper = other._iI18nCharMapper;
			_iI18nCharMapper->_vtbl->_addRef(_iI18nCharMapper);
		}
	}

	~CEComICEI18nCharMapperRef()
	{
		if (_iI18nCharMapper)
		{
			ICEI18nCharMapper* tmp = _iI18nCharMapper;
			_iI18nCharMapper = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nCharMapper* iI18nCharMapper;
				void* _ptr;
			} uIntf;
			uIntf.iI18nCharMapper = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nCharMapper, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nCharMapper)
				{
					ICEI18nCharMapper* tmp = _iI18nCharMapper;
					_iI18nCharMapper = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nCharMapper = uIntf.iI18nCharMapper;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharMapper* object() const	{ return _iI18nCharMapper; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nCharMapper* iOther)
	{
		if (_iI18nCharMapper)
		{
			ICEI18nCharMapper* tmp = _iI18nCharMapper;
			_iI18nCharMapper = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nCharMapper = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nCharMapper* detach()
	{
		ICEI18nCharMapper* iIntf = _iI18nCharMapper;
		_iI18nCharMapper = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nCharMapper* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nCharMapper;
		if (_iI18nCharMapper)
		{
			_iI18nCharMapper->_vtbl->_addRef(_iI18nCharMapper);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nCharMapper*() const	{ return _iI18nCharMapper; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharMapper& operator*() const	{ return *_iI18nCharMapper; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nCharMapper == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nCharMapper* iOther) const	{ return (_iI18nCharMapper != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nCharMapper* iOther) const	{ return (_iI18nCharMapper == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nCharMapperRef& operator=(const CEComICEI18nCharMapperRef& other)	{ return operator=(other._iI18nCharMapper); }

	CEComICEI18nCharMapperRef& operator=(const ICEI18nCharMapper* iOther)
	{
		if (_iI18nCharMapper != iOther)
		{
			if (_iI18nCharMapper)
			{
				ICEI18nCharMapper* tmp = _iI18nCharMapper;
				_iI18nCharMapper = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nCharMapper = const_cast<ICEI18nCharMapper*>(iOther);
			if (_iI18nCharMapper)
			{
				_iI18nCharMapper->_vtbl->_addRef(_iI18nCharMapper);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nCharMapper** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nCharMapperRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nCharMapper && "has a com object reference. clear first.");
		return & _iI18nCharMapper; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nCharMapperRef& other)
	{
		bool result = false;
		if (_iI18nCharMapper)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nCharMapper);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nCharMapper ? _iI18nCharMapper->_vtbl->_queryInterface(_iI18nCharMapper, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nCharMapper) { _iI18nCharMapper->_vtbl->_addRef(_iI18nCharMapper); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nCharMapper) { _iI18nCharMapper->_vtbl->_release(_iI18nCharMapper); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult mapChar(UTF32CHAR c32In, UTF32CHAR *const c32Out)	{ return _iI18nCharMapper ? _iI18nCharMapper->_vtbl->_mapChar(_iI18nCharMapper, c32In, c32Out) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEI18nCharMapper* _iI18nCharMapper;
};

/*! @}
 * end of CEComICEI18nCharMapperRef
 */

/*! \defgroup CEComICEI18nTextBreakTesterRef CEComICEI18nTextBreakTesterRef
 * @{
 */

class CEComICEI18nTextBreakTesterRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nTextBreakTesterRef() : _iI18nTextBreakTester(0) {}
	CEComICEI18nTextBreakTesterRef(ICEI18nTextBreakTester* iOther) : _iI18nTextBreakTester(0)
	{
		if (iOther)
		{
			_iI18nTextBreakTester = iOther;
			_iI18nTextBreakTester->_vtbl->_addRef(_iI18nTextBreakTester);
		}
	}

	CEComICEI18nTextBreakTesterRef(const CEComICEI18nTextBreakTesterRef& other) : _iI18nTextBreakTester(0)
	{
		if (other._iI18nTextBreakTester)
		{
			_iI18nTextBreakTester = other._iI18nTextBreakTester;
			_iI18nTextBreakTester->_vtbl->_addRef(_iI18nTextBreakTester);
		}
	}

	~CEComICEI18nTextBreakTesterRef()
	{
		if (_iI18nTextBreakTester)
		{
			ICEI18nTextBreakTester* tmp = _iI18nTextBreakTester;
			_iI18nTextBreakTester = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nTextBreakTester* iI18nTextBreakTester;
				void* _ptr;
			} uIntf;
			uIntf.iI18nTextBreakTester = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nTextBreakTester, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nTextBreakTester)
				{
					ICEI18nTextBreakTester* tmp = _iI18nTextBreakTester;
					_iI18nTextBreakTester = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nTextBreakTester = uIntf.iI18nTextBreakTester;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nTextBreakTester* object() const	{ return _iI18nTextBreakTester; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nTextBreakTester* iOther)
	{
		if (_iI18nTextBreakTester)
		{
			ICEI18nTextBreakTester* tmp = _iI18nTextBreakTester;
			_iI18nTextBreakTester = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nTextBreakTester = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nTextBreakTester* detach()
	{
		ICEI18nTextBreakTester* iIntf = _iI18nTextBreakTester;
		_iI18nTextBreakTester = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nTextBreakTester* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nTextBreakTester;
		if (_iI18nTextBreakTester)
		{
			_iI18nTextBreakTester->_vtbl->_addRef(_iI18nTextBreakTester);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nTextBreakTester*() const	{ return _iI18nTextBreakTester; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nTextBreakTester& operator*() const	{ return *_iI18nTextBreakTester; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nTextBreakTester == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nTextBreakTester* iOther) const	{ return (_iI18nTextBreakTester != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nTextBreakTester* iOther) const	{ return (_iI18nTextBreakTester == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nTextBreakTesterRef& operator=(const CEComICEI18nTextBreakTesterRef& other)	{ return operator=(other._iI18nTextBreakTester); }

	CEComICEI18nTextBreakTesterRef& operator=(const ICEI18nTextBreakTester* iOther)
	{
		if (_iI18nTextBreakTester != iOther)
		{
			if (_iI18nTextBreakTester)
			{
				ICEI18nTextBreakTester* tmp = _iI18nTextBreakTester;
				_iI18nTextBreakTester = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nTextBreakTester = const_cast<ICEI18nTextBreakTester*>(iOther);
			if (_iI18nTextBreakTester)
			{
				_iI18nTextBreakTester->_vtbl->_addRef(_iI18nTextBreakTester);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nTextBreakTester** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nTextBreakTesterRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nTextBreakTester && "has a com object reference. clear first.");
		return & _iI18nTextBreakTester; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nTextBreakTesterRef& other)
	{
		bool result = false;
		if (_iI18nTextBreakTester)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nTextBreakTester);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nTextBreakTester ? _iI18nTextBreakTester->_vtbl->_queryInterface(_iI18nTextBreakTester, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nTextBreakTester) { _iI18nTextBreakTester->_vtbl->_addRef(_iI18nTextBreakTester); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nTextBreakTester) { _iI18nTextBreakTester->_vtbl->_release(_iI18nTextBreakTester); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult canBreakBetween(UTF32CHAR c1, UTF32CHAR c2, bool *const bCanBreakOut)	{ return _iI18nTextBreakTester ? _iI18nTextBreakTester->_vtbl->_canBreakBetween(_iI18nTextBreakTester, c1, c2, bCanBreakOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEI18nTextBreakTester* _iI18nTextBreakTester;
};

/*! @}
 * end of CEComICEI18nTextBreakTesterRef
 */

/*! \defgroup CEComICEI18nCTLEngineRef CEComICEI18nCTLEngineRef
 * @{
 */

class CEComICEI18nCTLEngineRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nCTLEngineRef() : _iI18nCTLEngine(0) {}
	CEComICEI18nCTLEngineRef(ICEI18nCTLEngine* iOther) : _iI18nCTLEngine(0)
	{
		if (iOther)
		{
			_iI18nCTLEngine = iOther;
			_iI18nCTLEngine->_vtbl->_addRef(_iI18nCTLEngine);
		}
	}

	CEComICEI18nCTLEngineRef(const CEComICEI18nCTLEngineRef& other) : _iI18nCTLEngine(0)
	{
		if (other._iI18nCTLEngine)
		{
			_iI18nCTLEngine = other._iI18nCTLEngine;
			_iI18nCTLEngine->_vtbl->_addRef(_iI18nCTLEngine);
		}
	}

	~CEComICEI18nCTLEngineRef()
	{
		if (_iI18nCTLEngine)
		{
			ICEI18nCTLEngine* tmp = _iI18nCTLEngine;
			_iI18nCTLEngine = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nCTLEngine* iI18nCTLEngine;
				void* _ptr;
			} uIntf;
			uIntf.iI18nCTLEngine = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nCTLEngine, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nCTLEngine)
				{
					ICEI18nCTLEngine* tmp = _iI18nCTLEngine;
					_iI18nCTLEngine = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nCTLEngine = uIntf.iI18nCTLEngine;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nCTLEngine* object() const	{ return _iI18nCTLEngine; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nCTLEngine* iOther)
	{
		if (_iI18nCTLEngine)
		{
			ICEI18nCTLEngine* tmp = _iI18nCTLEngine;
			_iI18nCTLEngine = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nCTLEngine = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nCTLEngine* detach()
	{
		ICEI18nCTLEngine* iIntf = _iI18nCTLEngine;
		_iI18nCTLEngine = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nCTLEngine* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nCTLEngine;
		if (_iI18nCTLEngine)
		{
			_iI18nCTLEngine->_vtbl->_addRef(_iI18nCTLEngine);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nCTLEngine*() const	{ return _iI18nCTLEngine; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nCTLEngine& operator*() const	{ return *_iI18nCTLEngine; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nCTLEngine == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nCTLEngine* iOther) const	{ return (_iI18nCTLEngine != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nCTLEngine* iOther) const	{ return (_iI18nCTLEngine == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nCTLEngineRef& operator=(const CEComICEI18nCTLEngineRef& other)	{ return operator=(other._iI18nCTLEngine); }

	CEComICEI18nCTLEngineRef& operator=(const ICEI18nCTLEngine* iOther)
	{
		if (_iI18nCTLEngine != iOther)
		{
			if (_iI18nCTLEngine)
			{
				ICEI18nCTLEngine* tmp = _iI18nCTLEngine;
				_iI18nCTLEngine = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nCTLEngine = const_cast<ICEI18nCTLEngine*>(iOther);
			if (_iI18nCTLEngine)
			{
				_iI18nCTLEngine->_vtbl->_addRef(_iI18nCTLEngine);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nCTLEngine** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nCTLEngineRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nCTLEngine && "has a com object reference. clear first.");
		return & _iI18nCTLEngine; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nCTLEngineRef& other)
	{
		bool result = false;
		if (_iI18nCTLEngine)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nCTLEngine);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nCTLEngine ? _iI18nCTLEngine->_vtbl->_queryInterface(_iI18nCTLEngine, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nCTLEngine) { _iI18nCTLEngine->_vtbl->_addRef(_iI18nCTLEngine); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nCTLEngine) { _iI18nCTLEngine->_vtbl->_release(_iI18nCTLEngine); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult mapChar32(bool bFilterControlCode, bool bMirror, struct ICEI18nCharMapper* pOptCharMapper, UTF32CHAR c32In, UTF32CHAR *const c32Out)	{ return _iI18nCTLEngine ? _iI18nCTLEngine->_vtbl->_mapChar32(_iI18nCTLEngine, bFilterControlCode, bMirror, pOptCharMapper, c32In, c32Out) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult layoutChars(struct ICEUIPlatformFont* iPhyFont, struct ICEI18nCharMapper* iCharMapper, eCEI18nScriptCode scriptCode, struct ICEI18nLocale* pOptLocale,  bool isRtoL, const UTF16CHAR* pC16s, const UINT32 numOfC16s, CEI18nGlyphInfo *const pOptGlyphInfoRef, CEUTextMeasurement *const pOptWdithOut)	{ return _iI18nCTLEngine ? _iI18nCTLEngine->_vtbl->_layoutChars(_iI18nCTLEngine, iPhyFont, iCharMapper, scriptCode, pOptLocale, isRtoL, pC16s, numOfC16s, pOptGlyphInfoRef, pOptWdithOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult initGlyphInfo(CEI18nGlyphInfo* pGlyphInfoOut, UINT32 eCEI18nGIFlagsIn)	{ return _iI18nCTLEngine ? _iI18nCTLEngine->_vtbl->_initGlyphInfo(_iI18nCTLEngine, pGlyphInfoOut, eCEI18nGIFlagsIn) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult deleteGlyphInfo(CEI18nGlyphInfo* pGlyphInfoOut)	{ return _iI18nCTLEngine ? _iI18nCTLEngine->_vtbl->_deleteGlyphInfo(_iI18nCTLEngine, pGlyphInfoOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEI18nCTLEngine* _iI18nCTLEngine;
};

/*! @}
 * end of CEComICEI18nCTLEngineRef
 */

/*! \defgroup CEComICEI18nBreakIteratorRef CEComICEI18nBreakIteratorRef
 * @{
 */

class CEComICEI18nBreakIteratorRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nBreakIteratorRef() : _iI18nBreakIterator(0) {}
	CEComICEI18nBreakIteratorRef(ICEI18nBreakIterator* iOther) : _iI18nBreakIterator(0)
	{
		if (iOther)
		{
			_iI18nBreakIterator = iOther;
			_iI18nBreakIterator->_vtbl->_addRef(_iI18nBreakIterator);
		}
	}

	CEComICEI18nBreakIteratorRef(const CEComICEI18nBreakIteratorRef& other) : _iI18nBreakIterator(0)
	{
		if (other._iI18nBreakIterator)
		{
			_iI18nBreakIterator = other._iI18nBreakIterator;
			_iI18nBreakIterator->_vtbl->_addRef(_iI18nBreakIterator);
		}
	}

	~CEComICEI18nBreakIteratorRef()
	{
		if (_iI18nBreakIterator)
		{
			ICEI18nBreakIterator* tmp = _iI18nBreakIterator;
			_iI18nBreakIterator = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nBreakIterator* iI18nBreakIterator;
				void* _ptr;
			} uIntf;
			uIntf.iI18nBreakIterator = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nBreakIterator, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nBreakIterator)
				{
					ICEI18nBreakIterator* tmp = _iI18nBreakIterator;
					_iI18nBreakIterator = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nBreakIterator = uIntf.iI18nBreakIterator;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nBreakIterator* object() const	{ return _iI18nBreakIterator; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nBreakIterator* iOther)
	{
		if (_iI18nBreakIterator)
		{
			ICEI18nBreakIterator* tmp = _iI18nBreakIterator;
			_iI18nBreakIterator = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nBreakIterator = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nBreakIterator* detach()
	{
		ICEI18nBreakIterator* iIntf = _iI18nBreakIterator;
		_iI18nBreakIterator = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nBreakIterator* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nBreakIterator;
		if (_iI18nBreakIterator)
		{
			_iI18nBreakIterator->_vtbl->_addRef(_iI18nBreakIterator);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nBreakIterator*() const	{ return _iI18nBreakIterator; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nBreakIterator& operator*() const	{ return *_iI18nBreakIterator; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nBreakIterator == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nBreakIterator* iOther) const	{ return (_iI18nBreakIterator != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nBreakIterator* iOther) const	{ return (_iI18nBreakIterator == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nBreakIteratorRef& operator=(const CEComICEI18nBreakIteratorRef& other)	{ return operator=(other._iI18nBreakIterator); }

	CEComICEI18nBreakIteratorRef& operator=(const ICEI18nBreakIterator* iOther)
	{
		if (_iI18nBreakIterator != iOther)
		{
			if (_iI18nBreakIterator)
			{
				ICEI18nBreakIterator* tmp = _iI18nBreakIterator;
				_iI18nBreakIterator = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nBreakIterator = const_cast<ICEI18nBreakIterator*>(iOther);
			if (_iI18nBreakIterator)
			{
				_iI18nBreakIterator->_vtbl->_addRef(_iI18nBreakIterator);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nBreakIterator** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nBreakIteratorRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nBreakIterator && "has a com object reference. clear first.");
		return & _iI18nBreakIterator; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nBreakIteratorRef& other)
	{
		bool result = false;
		if (_iI18nBreakIterator)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nBreakIterator);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nBreakIterator ? _iI18nBreakIterator->_vtbl->_queryInterface(_iI18nBreakIterator, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nBreakIterator) { _iI18nBreakIterator->_vtbl->_addRef(_iI18nBreakIterator); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nBreakIterator) { _iI18nBreakIterator->_vtbl->_release(_iI18nBreakIterator); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getInfo(eCEI18nBreakIteratorType *const optTypeOut, struct ICEI18nLocale* *const optLocaleOut)	{ return _iI18nBreakIterator ? _iI18nBreakIterator->_vtbl->_getInfo(_iI18nBreakIterator, optTypeOut, optLocaleOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setUTF16CHARArray(const UTF16CHAR* pC16s, UINT32 numOfC16s, bool bAppend)	{ return _iI18nBreakIterator ? _iI18nBreakIterator->_vtbl->_setUTF16CHARArray(_iI18nBreakIterator, pC16s, numOfC16s, bAppend) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult moveNext(INT32 *const boundaryOut, UTF32CHAR *const optC32Out, eCEI18nGCB *const optGCBOut)	{ return _iI18nBreakIterator ? _iI18nBreakIterator->_vtbl->_moveNext(_iI18nBreakIterator, boundaryOut, optC32Out, optGCBOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult current(INT32 *const boundaryOut, UTF32CHAR *const optC32Out, eCEI18nGCB *const optGCBOut)	{ return _iI18nBreakIterator ? _iI18nBreakIterator->_vtbl->_current(_iI18nBreakIterator, boundaryOut, optC32Out, optGCBOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult reset()	{ return _iI18nBreakIterator ? _iI18nBreakIterator->_vtbl->_reset(_iI18nBreakIterator) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEI18nBreakIterator* _iI18nBreakIterator;
};

/*! @}
 * end of CEComICEI18nBreakIteratorRef
 */

/*! \defgroup CEComICEI18nBiDiUtilRef CEComICEI18nBiDiUtilRef
 * @{
 */

class CEComICEI18nBiDiUtilRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nBiDiUtilRef() : _iI18nBiDiUtil(0) {}
	CEComICEI18nBiDiUtilRef(ICEI18nBiDiUtil* iOther) : _iI18nBiDiUtil(0)
	{
		if (iOther)
		{
			_iI18nBiDiUtil = iOther;
			_iI18nBiDiUtil->_vtbl->_addRef(_iI18nBiDiUtil);
		}
	}

	CEComICEI18nBiDiUtilRef(const CEComICEI18nBiDiUtilRef& other) : _iI18nBiDiUtil(0)
	{
		if (other._iI18nBiDiUtil)
		{
			_iI18nBiDiUtil = other._iI18nBiDiUtil;
			_iI18nBiDiUtil->_vtbl->_addRef(_iI18nBiDiUtil);
		}
	}

	~CEComICEI18nBiDiUtilRef()
	{
		if (_iI18nBiDiUtil)
		{
			ICEI18nBiDiUtil* tmp = _iI18nBiDiUtil;
			_iI18nBiDiUtil = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nBiDiUtil* iI18nBiDiUtil;
				void* _ptr;
			} uIntf;
			uIntf.iI18nBiDiUtil = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nBiDiUtil, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nBiDiUtil)
				{
					ICEI18nBiDiUtil* tmp = _iI18nBiDiUtil;
					_iI18nBiDiUtil = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nBiDiUtil = uIntf.iI18nBiDiUtil;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nBiDiUtil* object() const	{ return _iI18nBiDiUtil; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nBiDiUtil* iOther)
	{
		if (_iI18nBiDiUtil)
		{
			ICEI18nBiDiUtil* tmp = _iI18nBiDiUtil;
			_iI18nBiDiUtil = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nBiDiUtil = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nBiDiUtil* detach()
	{
		ICEI18nBiDiUtil* iIntf = _iI18nBiDiUtil;
		_iI18nBiDiUtil = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nBiDiUtil* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nBiDiUtil;
		if (_iI18nBiDiUtil)
		{
			_iI18nBiDiUtil->_vtbl->_addRef(_iI18nBiDiUtil);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nBiDiUtil*() const	{ return _iI18nBiDiUtil; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nBiDiUtil& operator*() const	{ return *_iI18nBiDiUtil; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nBiDiUtil == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nBiDiUtil* iOther) const	{ return (_iI18nBiDiUtil != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nBiDiUtil* iOther) const	{ return (_iI18nBiDiUtil == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nBiDiUtilRef& operator=(const CEComICEI18nBiDiUtilRef& other)	{ return operator=(other._iI18nBiDiUtil); }

	CEComICEI18nBiDiUtilRef& operator=(const ICEI18nBiDiUtil* iOther)
	{
		if (_iI18nBiDiUtil != iOther)
		{
			if (_iI18nBiDiUtil)
			{
				ICEI18nBiDiUtil* tmp = _iI18nBiDiUtil;
				_iI18nBiDiUtil = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nBiDiUtil = const_cast<ICEI18nBiDiUtil*>(iOther);
			if (_iI18nBiDiUtil)
			{
				_iI18nBiDiUtil->_vtbl->_addRef(_iI18nBiDiUtil);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nBiDiUtil** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nBiDiUtilRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nBiDiUtil && "has a com object reference. clear first.");
		return & _iI18nBiDiUtil; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nBiDiUtilRef& other)
	{
		bool result = false;
		if (_iI18nBiDiUtil)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nBiDiUtil);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nBiDiUtil ? _iI18nBiDiUtil->_vtbl->_queryInterface(_iI18nBiDiUtil, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nBiDiUtil) { _iI18nBiDiUtil->_vtbl->_addRef(_iI18nBiDiUtil); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nBiDiUtil) { _iI18nBiDiUtil->_vtbl->_release(_iI18nBiDiUtil); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult reset()	{ return _iI18nBiDiUtil ? _iI18nBiDiUtil->_vtbl->_reset(_iI18nBiDiUtil) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult reorder(bool bResolveParagraphLevel, eCEI18nBiDiLevel paragraphLevelIn, const UTF16CHAR* pC16sIn, UINT32 numOfC16sIn, const CEI18nBiDiCCodeStack* optLeadCCodeStack, CEI18nBiDiCCodeStack* *const optTrailCCodeStackOut)	{ return _iI18nBiDiUtil ? _iI18nBiDiUtil->_vtbl->_reorder(_iI18nBiDiUtil, bResolveParagraphLevel, paragraphLevelIn, pC16sIn, numOfC16sIn, optLeadCCodeStack, optTrailCCodeStackOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getVisualRunCount(INT32 *const numOfVisualRunsOut)	{ return _iI18nBiDiUtil ? _iI18nBiDiUtil->_vtbl->_getVisualRunCount(_iI18nBiDiUtil, numOfVisualRunsOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getVisualRun(INT32 visualRunIndexIn, eCEI18nBiDiDirection *const optDirOut, INT32 *const optStart16Out, INT32 *const optNumOfC16sOut)	{ return _iI18nBiDiUtil ? _iI18nBiDiUtil->_vtbl->_getVisualRun(_iI18nBiDiUtil, visualRunIndexIn, optDirOut, optStart16Out, optNumOfC16sOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEI18nBiDiUtil* _iI18nBiDiUtil;
};

/*! @}
 * end of CEComICEI18nBiDiUtilRef
 */

/*! \defgroup CEComICEI18nBiDiUtilClassRef CEComICEI18nBiDiUtilClassRef
 * @{
 */

class CEComICEI18nBiDiUtilClassRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEI18nBiDiUtilClassRef() : _iI18nBiDiUtilClass(0) {}
	CEComICEI18nBiDiUtilClassRef(ICEI18nBiDiUtilClass* iOther) : _iI18nBiDiUtilClass(0)
	{
		if (iOther)
		{
			_iI18nBiDiUtilClass = iOther;
			_iI18nBiDiUtilClass->_vtbl->_addRef(_iI18nBiDiUtilClass);
		}
	}

	CEComICEI18nBiDiUtilClassRef(const CEComICEI18nBiDiUtilClassRef& other) : _iI18nBiDiUtilClass(0)
	{
		if (other._iI18nBiDiUtilClass)
		{
			_iI18nBiDiUtilClass = other._iI18nBiDiUtilClass;
			_iI18nBiDiUtilClass->_vtbl->_addRef(_iI18nBiDiUtilClass);
		}
	}

	~CEComICEI18nBiDiUtilClassRef()
	{
		if (_iI18nBiDiUtilClass)
		{
			ICEI18nBiDiUtilClass* tmp = _iI18nBiDiUtilClass;
			_iI18nBiDiUtilClass = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEI18nBiDiUtilClass* iI18nBiDiUtilClass;
				void* _ptr;
			} uIntf;
			uIntf.iI18nBiDiUtilClass = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEI18nBiDiUtilClass, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iI18nBiDiUtilClass)
				{
					ICEI18nBiDiUtilClass* tmp = _iI18nBiDiUtilClass;
					_iI18nBiDiUtilClass = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iI18nBiDiUtilClass = uIntf.iI18nBiDiUtilClass;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEI18nBiDiUtilClass* object() const	{ return _iI18nBiDiUtilClass; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEI18nBiDiUtilClass* iOther)
	{
		if (_iI18nBiDiUtilClass)
		{
			ICEI18nBiDiUtilClass* tmp = _iI18nBiDiUtilClass;
			_iI18nBiDiUtilClass = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iI18nBiDiUtilClass = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEI18nBiDiUtilClass* detach()
	{
		ICEI18nBiDiUtilClass* iIntf = _iI18nBiDiUtilClass;
		_iI18nBiDiUtilClass = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEI18nBiDiUtilClass* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iI18nBiDiUtilClass;
		if (_iI18nBiDiUtilClass)
		{
			_iI18nBiDiUtilClass->_vtbl->_addRef(_iI18nBiDiUtilClass);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEI18nBiDiUtilClass*() const	{ return _iI18nBiDiUtilClass; }
	FORCEINLINE_WITHOUT_DEBUG ICEI18nBiDiUtilClass& operator*() const	{ return *_iI18nBiDiUtilClass; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iI18nBiDiUtilClass == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEI18nBiDiUtilClass* iOther) const	{ return (_iI18nBiDiUtilClass != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEI18nBiDiUtilClass* iOther) const	{ return (_iI18nBiDiUtilClass == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEI18nBiDiUtilClassRef& operator=(const CEComICEI18nBiDiUtilClassRef& other)	{ return operator=(other._iI18nBiDiUtilClass); }

	CEComICEI18nBiDiUtilClassRef& operator=(const ICEI18nBiDiUtilClass* iOther)
	{
		if (_iI18nBiDiUtilClass != iOther)
		{
			if (_iI18nBiDiUtilClass)
			{
				ICEI18nBiDiUtilClass* tmp = _iI18nBiDiUtilClass;
				_iI18nBiDiUtilClass = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iI18nBiDiUtilClass = const_cast<ICEI18nBiDiUtilClass*>(iOther);
			if (_iI18nBiDiUtilClass)
			{
				_iI18nBiDiUtilClass->_vtbl->_addRef(_iI18nBiDiUtilClass);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEI18nBiDiUtilClass** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEI18nBiDiUtilClassRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iI18nBiDiUtilClass && "has a com object reference. clear first.");
		return & _iI18nBiDiUtilClass; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEI18nBiDiUtilClassRef& other)
	{
		bool result = false;
		if (_iI18nBiDiUtilClass)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iI18nBiDiUtilClass);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iI18nBiDiUtilClass ? _iI18nBiDiUtilClass->_vtbl->_queryInterface(_iI18nBiDiUtilClass, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iI18nBiDiUtilClass) { _iI18nBiDiUtilClass->_vtbl->_addRef(_iI18nBiDiUtilClass); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iI18nBiDiUtilClass) { _iI18nBiDiUtilClass->_vtbl->_release(_iI18nBiDiUtilClass); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult newBiDiUtil(struct ICEI18nBiDiUtil* *const bidiUtilOut)	{ return _iI18nBiDiUtilClass ? _iI18nBiDiUtilClass->_vtbl->_newBiDiUtil(_iI18nBiDiUtilClass, bidiUtilOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult newCCodeStack(bool bRtlParagraphLevel, bool bOverride, CEI18nBiDiCCodeStack* *const pCCodeStackOut)	{ return _iI18nBiDiUtilClass ? _iI18nBiDiUtilClass->_vtbl->_newCCodeStack(_iI18nBiDiUtilClass, bRtlParagraphLevel, bOverride, pCCodeStackOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult deleteCCodeStack(CEI18nBiDiCCodeStack* pCCodeStack)	{ return _iI18nBiDiUtilClass ? _iI18nBiDiUtilClass->_vtbl->_deleteCCodeStack(_iI18nBiDiUtilClass, pCCodeStack) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEI18nBiDiUtilClass* _iI18nBiDiUtilClass;
};

/*! @}
 * end of CEComICEI18nBiDiUtilClassRef
 */

#endif // __cplusplus

#endif
