///////////////////////////////////////////////////////////////////////////////
// Copyright 2012 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICEHtmlWindow_h
#define ICEHtmlWindow_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"
#include "ICEHtmlWindowWidget.h"
#include "ICEHtmlWindowWidgetContainer.h"
#include "ICEHtmlElementEventHandler.h"
#include "ICEHtmlDocumentEventHandler.h"
#include "ICEHtmlNetworkEventHandler.h"
#include "ICEHtmlElement.h"
#include "CEComAtomicSupport.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------


//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEHtmlWindow ICEHtmlWindow
 * @{
 */

/*!
 * ID of ICEHtmlWindow
 */
#define CEComIID_ICEHtmlWindow 0x66bb097a


/*!
 * ICEHtmlWindow
 */
typedef struct ICEHtmlWindow
{
	const struct ICEHtmlWindow_vtbl* _vtbl;
} ICEHtmlWindow;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iWindow	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEHtmlWindow_queryInterface) (ICEHtmlWindow* iWindow, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEHtmlWindow_addRef) (ICEHtmlWindow* iWindow);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEHtmlWindow_release) (ICEHtmlWindow* iWindow);

typedef CEHResult (*iCEHtmlWindow_isFrameBorderNecessary) (ICEHtmlWindow* iWindow, bool *const necessary);

typedef CEHResult (*iCEHtmlWindow_getWindowWidget) (ICEHtmlWindow* iWindow, struct ICEHtmlWindowWidget* *const iWindowWidget);

typedef CEHResult (*iCEHtmlWindow_getActiveDocument) (ICEHtmlWindow* iWindow, struct ICEHtmlDocument* *const iDocumentOut);

typedef CEHResult (*iCEHtmlWindow_handleActionEvent) (ICEHtmlWindow* iWindow, CEActionEventParam* action, CEATOMIC* consumingCount, bool *const consumedOut, const bool enableCheckMode);

typedef CEHResult (*iCEHtmlWindow_handleKeyboardEvent) (ICEHtmlWindow* iWindow, CEKeyboardEventParam* eventParam, CEATOMIC* consumingCount, bool *const consumedOut);

typedef CEHResult (*iCEHtmlWindow_handleHtmlEvent) (ICEHtmlWindow* iWindow, CEHtmlEventId eventId, bool *const consumedOut);

typedef CEHResult (*iCEHtmlWindow_setWindowWidget) (ICEHtmlWindow* iWindow, struct ICEHtmlWindowWidget* iWindowWidget);

typedef CEHResult (*iCEHtmlWindow_setWindowWidgetContainer) (ICEHtmlWindow* iWindow, struct ICEHtmlWindowWidgetContainer* iWindowWidgetContainer);

typedef CEHResult (*iCEHtmlWindow_setElementEventHandler) (ICEHtmlWindow* iWindow, struct ICEHtmlElementEventHandler* iElementEventHandler);

typedef CEHResult (*iCEHtmlWindow_setDocumentEventHandler) (ICEHtmlWindow* iWindow, struct ICEHtmlDocumentEventHandler* iDocumentEventHandler);

typedef CEHResult (*iCEHtmlWindow_setNetworkEventHandler) (ICEHtmlWindow* iWindow, struct ICEHtmlNetworkEventHandler* iNetworkEventHandler);

typedef CEHResult (*iCEHtmlWindow_notifyResized) (ICEHtmlWindow* iWindow);

/*!
 * V-table of ICEHtmlWindow
 */
struct ICEHtmlWindow_vtbl
{
	iCEHtmlWindow_queryInterface	_queryInterface;	//!< Query interface.
	iCEHtmlWindow_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEHtmlWindow_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEHtmlWindow_isFrameBorderNecessary	_isFrameBorderNecessary;
	iCEHtmlWindow_getWindowWidget	_getWindowWidget;
	iCEHtmlWindow_getActiveDocument	_getActiveDocument;
	iCEHtmlWindow_handleActionEvent	_handleActionEvent;
	iCEHtmlWindow_handleKeyboardEvent	_handleKeyboardEvent;
	iCEHtmlWindow_handleHtmlEvent	_handleHtmlEvent;
	iCEHtmlWindow_setWindowWidget	_setWindowWidget;
	iCEHtmlWindow_setWindowWidgetContainer	_setWindowWidgetContainer;
	iCEHtmlWindow_setElementEventHandler	_setElementEventHandler;
	iCEHtmlWindow_setDocumentEventHandler	_setDocumentEventHandler;
	iCEHtmlWindow_setNetworkEventHandler	_setNetworkEventHandler;
	iCEHtmlWindow_notifyResized	_notifyResized;
};

/*! @}
 * end of ICEHtmlWindow
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEHtmlWindowRef CEComICEHtmlWindowRef
 * @{
 */

class CEComICEHtmlWindowRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEHtmlWindowRef() : _iWindow(0) {}
	CEComICEHtmlWindowRef(ICEHtmlWindow* iOther) : _iWindow(0)
	{
		if (iOther)
		{
			_iWindow = iOther;
			_iWindow->_vtbl->_addRef(_iWindow);
		}
	}

	CEComICEHtmlWindowRef(const CEComICEHtmlWindowRef& other) : _iWindow(0)
	{
		if (other._iWindow)
		{
			_iWindow = other._iWindow;
			_iWindow->_vtbl->_addRef(_iWindow);
		}
	}

	~CEComICEHtmlWindowRef()
	{
		if (_iWindow)
		{
			ICEHtmlWindow* tmp = _iWindow;
			_iWindow = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEHtmlWindow* iWindow;
				void* _ptr;
			} uIntf;
			uIntf.iWindow = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEHtmlWindow, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iWindow)
				{
					ICEHtmlWindow* tmp = _iWindow;
					_iWindow = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iWindow = uIntf.iWindow;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEHtmlWindow* object() const	{ return _iWindow; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEHtmlWindow* iOther)
	{
		if (_iWindow)
		{
			ICEHtmlWindow* tmp = _iWindow;
			_iWindow = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iWindow = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEHtmlWindow* detach()
	{
		ICEHtmlWindow* iIntf = _iWindow;
		_iWindow = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEHtmlWindow* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iWindow;
		if (_iWindow)
		{
			_iWindow->_vtbl->_addRef(_iWindow);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEHtmlWindow*() const	{ return _iWindow; }
	FORCEINLINE_WITHOUT_DEBUG ICEHtmlWindow& operator*() const	{ return *_iWindow; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iWindow == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEHtmlWindow* iOther) const	{ return (_iWindow != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEHtmlWindow* iOther) const	{ return (_iWindow == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEHtmlWindowRef& operator=(const CEComICEHtmlWindowRef& other)	{ return operator=(other._iWindow); }

	CEComICEHtmlWindowRef& operator=(const ICEHtmlWindow* iOther)
	{
		if (_iWindow != iOther)
		{
			if (_iWindow)
			{
				ICEHtmlWindow* tmp = _iWindow;
				_iWindow = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iWindow = const_cast<ICEHtmlWindow*>(iOther);
			if (_iWindow)
			{
				_iWindow->_vtbl->_addRef(_iWindow);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEHtmlWindow** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEHtmlWindowRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iWindow && "has a com object reference. clear first.");
		return & _iWindow; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEHtmlWindowRef& other)
	{
		bool result = false;
		if (_iWindow)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iWindow);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iWindow ? _iWindow->_vtbl->_queryInterface(_iWindow, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iWindow) { _iWindow->_vtbl->_addRef(_iWindow); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iWindow) { _iWindow->_vtbl->_release(_iWindow); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult isFrameBorderNecessary(bool *const necessary)	{ return _iWindow ? _iWindow->_vtbl->_isFrameBorderNecessary(_iWindow, necessary) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getWindowWidget(struct ICEHtmlWindowWidget* *const iWindowWidget)	{ return _iWindow ? _iWindow->_vtbl->_getWindowWidget(_iWindow, iWindowWidget) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getActiveDocument(struct ICEHtmlDocument* *const iDocumentOut)	{ return _iWindow ? _iWindow->_vtbl->_getActiveDocument(_iWindow, iDocumentOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult handleActionEvent(CEActionEventParam* action, CEATOMIC* consumingCount, bool *const consumedOut, const bool enableCheckMode)	{ return _iWindow ? _iWindow->_vtbl->_handleActionEvent(_iWindow, action, consumingCount, consumedOut, enableCheckMode) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult handleKeyboardEvent(CEKeyboardEventParam* eventParam, CEATOMIC* consumingCount, bool *const consumedOut)	{ return _iWindow ? _iWindow->_vtbl->_handleKeyboardEvent(_iWindow, eventParam, consumingCount, consumedOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult handleHtmlEvent(CEHtmlEventId eventId, bool *const consumedOut)	{ return _iWindow ? _iWindow->_vtbl->_handleHtmlEvent(_iWindow, eventId, consumedOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setWindowWidget(struct ICEHtmlWindowWidget* iWindowWidget)	{ return _iWindow ? _iWindow->_vtbl->_setWindowWidget(_iWindow, iWindowWidget) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setWindowWidgetContainer(struct ICEHtmlWindowWidgetContainer* iWindowWidgetContainer)	{ return _iWindow ? _iWindow->_vtbl->_setWindowWidgetContainer(_iWindow, iWindowWidgetContainer) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setElementEventHandler(struct ICEHtmlElementEventHandler* iElementEventHandler)	{ return _iWindow ? _iWindow->_vtbl->_setElementEventHandler(_iWindow, iElementEventHandler) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setDocumentEventHandler(struct ICEHtmlDocumentEventHandler* iDocumentEventHandler)	{ return _iWindow ? _iWindow->_vtbl->_setDocumentEventHandler(_iWindow, iDocumentEventHandler) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setNetworkEventHandler(struct ICEHtmlNetworkEventHandler* iNetworkEventHandler)	{ return _iWindow ? _iWindow->_vtbl->_setNetworkEventHandler(_iWindow, iNetworkEventHandler) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult notifyResized()	{ return _iWindow ? _iWindow->_vtbl->_notifyResized(_iWindow) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEHtmlWindow* _iWindow;
};

/*! @}
 * end of CEComICEHtmlWindowRef
 */

#endif // __cplusplus

#endif
