///////////////////////////////////////////////////////////////////////////////
// Copyright 2012 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICEDXSurface_h
#define ICEDXSurface_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"
#include "CEUITypes.h"
#include "CEDXTypes.h"
#include "CEPixelFormat.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------

struct ICEDXSurface;
struct ICEDXPalette;

//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

typedef enum
{
	eCEDXResourceType_FIRST = 0,

	eCEDXResourceType_Surface = eCEDXResourceType_FIRST,
	eCEDXResourceType_Texture,
	eCEDXResourceType_VertexBuffer,
	eCEDXResourceType_IndexBuffer,
	eCEDXResourceType_Palette,

	eCEDXResourceType_LAST = eCEDXResourceType_Palette
} eCEDXResourceType;

//! buffer pool type of resources
													//!< when the DXDevice need it to draw. lockable.
typedef enum
{
	eCEDXPoolType_FIRST = 0,

	eCEDXPoolType_Default = eCEDXPoolType_FIRST,	//!< allocated in the video memory. non-lockable.
	eCEDXPoolType_Managed,							//!< allocated in the system memory and copied to the video memory
	eCEDXPoolType_Managed_Opt1,
	eCEDXPoolType_Managed_Opt2,
	eCEDXPoolType_Managed_Opt3,
	eCEDXPoolType_Managed_Opt4,
	eCEDXPoolType_Managed_Opt5,
	eCEDXPoolType_Managed_Opt6,
	eCEDXPoolType_Managed_Opt7,
	eCEDXPoolType_Managed_Opt8,
	eCEDXPoolType_Managed_Opt9,

	eCEDXPoolType_SystemMem,						//!< allocated in the system memory. lockable.

	eCEDXPoolType_LAST = eCEDXPoolType_SystemMem
} eCEDXPoolType;

//! flags of lockRect
typedef enum
{
	eCEDXLockRectFlag_SetNonPurgeable = 1,		//!< set non-purgeable.
	eCEDXLockRectFlag_SetPurgeable = 2,			//!< set purgeable.
	eCEDXLockRectFlag_UseSoftLimit = 4,			//!< use softLimit.
	eCEDXLockRectFlag_EnablePurgeControl = 8	//!< enable purge Control.
} eCEDXLockRectFlag;

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEDXResource2 ICEDXResource2
 * @{
 */

/*!
 * ID of ICEDXResource2
 */
#define CEComIID_ICEDXResource2 0xffb978f1


/*!
 * ICEDXResource
 */
typedef struct ICEDXResource2
{
	const struct ICEDXResource2_vtbl* _vtbl;
} ICEDXResource2;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXResource2	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXResource2_queryInterface) (ICEDXResource2* iDXResource2, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXResource2_addRef) (ICEDXResource2* iDXResource2);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXResource2_release) (ICEDXResource2* iDXResource2);

	//! get the device that creates this resource

typedef CEHResult (*iCEDXResource2_getDevice) (ICEDXResource2* iDXResource2, struct ICEDXDevice* *const iDeviceOut);

	//! get the resource type

typedef CEHResult (*iCEDXResource2_getType) (ICEDXResource2* iDXResource2, eCEDXResourceType *const typeOut);

/*!
 * V-table of ICEDXResource2
 */
struct ICEDXResource2_vtbl
{
	iCEDXResource2_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXResource2_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXResource2_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXResource2_getDevice	_getDevice;
	iCEDXResource2_getType	_getType;
};

/*! @}
 * end of ICEDXResource2
 */

/*! \defgroup ICEDXSurface ICEDXSurface
 * @{
 */

/*!
 * ID of ICEDXSurface
 */
#define CEComIID_ICEDXSurface 0x3d4ad9e0

	//! get the pool type

/*!
 * ICEDXSurface
 */
typedef struct ICEDXSurface
{
	const struct ICEDXSurface_vtbl* _vtbl;
} ICEDXSurface;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXSurface	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXSurface_queryInterface) (ICEDXSurface* iDXSurface, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXSurface_addRef) (ICEDXSurface* iDXSurface);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXSurface_release) (ICEDXSurface* iDXSurface);

	//! get the device that creates this resource

typedef CEHResult (*iCEDXSurface_getDevice) (ICEDXSurface* iDXSurface, struct ICEDXDevice* *const iDeviceOut);

	//! get the resource type

typedef CEHResult (*iCEDXSurface_getType) (ICEDXSurface* iDXSurface, eCEDXResourceType *const typeOut);

	//! get the surface description

typedef CEHResult (*iCEDXSurface_getDesc) (ICEDXSurface* iDXSurface, struct CEDXSurfaceDesc *const descOut);

	//! lock the specified rect on this surface
	//! \return  JGSS_ERR_BADARGS when failed locking.

typedef CEHResult (*iCEDXSurface_lockRect) (ICEDXSurface* iDXSurface, const CERect* rect, UINT32 flags, struct CEDXLockedRect *const lockedRectOut);

	//! unlock the previously locked rect on this surface.

typedef CEHResult (*iCEDXSurface_unlockRect) (ICEDXSurface* iDXSurface);

	//! Get the container interface, such as ICEDXTexture.
	//! If the surface is retrieved by ICEDXDevice::getBackBuffer(), the surface is
	//! considered stand alone. In this case, ICEDXSurface::GetContainer will return
	//! the DX device used to create the surface.
	//! \return  JGSS_ERR_NOTFOUND when failed to find the container.

typedef CEHResult (*iCEDXSurface_getContainer) (ICEDXSurface* iDXSurface, UINT32 iid, void** iContainerOut);

	//! begin painting with DXContext2D  (float version).
	//! \param[in] iUpdateRgn   specifies the region to be repainted. NULL means the entire surface.
	//! \param[out] piCtxOut    context2D to use for painting.

typedef CEHResult (*iCEDXSurface_beginPaintF) (ICEDXSurface* iDXSurface, struct ICEDXImmutableRegion* iUpdateRgn, struct ICEDXContext2D* *const piCtx2DOut);

	//! end painting with DXContext2D  (float version).
	//! \return  JGSS_ERR_BADARGS when the given context is not retrieved via beginPaintF()

typedef CEHResult (*iCEDXSurface_endPaintF) (ICEDXSurface* iDXSurface, struct ICEDXContext2D* iCtx2D);

	//! get region for this surface
	//! \return  JGSS_ERR_BADARGS when the given region is null.

typedef CEHResult (*iCEDXSurface_getRegion) (ICEDXSurface* iDXSurface, struct ICEDXRegion* *const piRegionOut);

/*!
 * V-table of ICEDXSurface
 */
struct ICEDXSurface_vtbl
{
	iCEDXSurface_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXSurface_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXSurface_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXSurface_getDevice	_getDevice;
	iCEDXSurface_getType	_getType;
	iCEDXSurface_getDesc	_getDesc;
	iCEDXSurface_lockRect	_lockRect;
	iCEDXSurface_unlockRect	_unlockRect;
	iCEDXSurface_getContainer	_getContainer;
	iCEDXSurface_beginPaintF	_beginPaintF;
	iCEDXSurface_endPaintF	_endPaintF;
	iCEDXSurface_getRegion	_getRegion;
};

/*! @}
 * end of ICEDXSurface
 */

/*! \defgroup ICEDXPalette ICEDXPalette
 * @{
 */

/*!
 * ID of ICEDXPalette
 */
#define CEComIID_ICEDXPalette 0x7652098f


/*!
 * ICEDXPalette
 */
typedef struct ICEDXPalette
{
	const struct ICEDXPalette_vtbl* _vtbl;
} ICEDXPalette;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXPalette	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXPalette_queryInterface) (ICEDXPalette* iDXPalette, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXPalette_addRef) (ICEDXPalette* iDXPalette);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXPalette_release) (ICEDXPalette* iDXPalette);

	//! get the device that creates this resource

typedef CEHResult (*iCEDXPalette_getDevice) (ICEDXPalette* iDXPalette, struct ICEDXDevice* *const iDeviceOut);

	//! get the resource type

typedef CEHResult (*iCEDXPalette_getType) (ICEDXPalette* iDXPalette, eCEDXResourceType *const typeOut);

	//! set the palette entries

typedef CEHResult (*iCEDXPalette_setPaletteEntries) (ICEDXPalette* iDXPalette, const CERGBAColor* pEntries);

	//! get the palette entries

typedef CEHResult (*iCEDXPalette_getPaletteEntries) (ICEDXPalette* iDXPalette, CERGBAColor* pEntriesOut);

/*!
 * V-table of ICEDXPalette
 */
struct ICEDXPalette_vtbl
{
	iCEDXPalette_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXPalette_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXPalette_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXPalette_getDevice	_getDevice;
	iCEDXPalette_getType	_getType;
	iCEDXPalette_setPaletteEntries	_setPaletteEntries;
	iCEDXPalette_getPaletteEntries	_getPaletteEntries;
};

/*! @}
 * end of ICEDXPalette
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEDXResource2Ref CEComICEDXResource2Ref
 * @{
 */

class CEComICEDXResource2Ref
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXResource2Ref() : _iDXResource2(0) {}
	CEComICEDXResource2Ref(ICEDXResource2* iOther) : _iDXResource2(0)
	{
		if (iOther)
		{
			_iDXResource2 = iOther;
			_iDXResource2->_vtbl->_addRef(_iDXResource2);
		}
	}

	CEComICEDXResource2Ref(const CEComICEDXResource2Ref& other) : _iDXResource2(0)
	{
		if (other._iDXResource2)
		{
			_iDXResource2 = other._iDXResource2;
			_iDXResource2->_vtbl->_addRef(_iDXResource2);
		}
	}

	~CEComICEDXResource2Ref()
	{
		if (_iDXResource2)
		{
			ICEDXResource2* tmp = _iDXResource2;
			_iDXResource2 = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXResource2* iDXResource2;
				void* _ptr;
			} uIntf;
			uIntf.iDXResource2 = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXResource2, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXResource2)
				{
					ICEDXResource2* tmp = _iDXResource2;
					_iDXResource2 = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXResource2 = uIntf.iDXResource2;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXResource2* object() const	{ return _iDXResource2; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXResource2* iOther)
	{
		if (_iDXResource2)
		{
			ICEDXResource2* tmp = _iDXResource2;
			_iDXResource2 = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXResource2 = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXResource2* detach()
	{
		ICEDXResource2* iIntf = _iDXResource2;
		_iDXResource2 = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXResource2* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXResource2;
		if (_iDXResource2)
		{
			_iDXResource2->_vtbl->_addRef(_iDXResource2);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXResource2*() const	{ return _iDXResource2; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXResource2& operator*() const	{ return *_iDXResource2; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXResource2 == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXResource2* iOther) const	{ return (_iDXResource2 != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXResource2* iOther) const	{ return (_iDXResource2 == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXResource2Ref& operator=(const CEComICEDXResource2Ref& other)	{ return operator=(other._iDXResource2); }

	CEComICEDXResource2Ref& operator=(const ICEDXResource2* iOther)
	{
		if (_iDXResource2 != iOther)
		{
			if (_iDXResource2)
			{
				ICEDXResource2* tmp = _iDXResource2;
				_iDXResource2 = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXResource2 = const_cast<ICEDXResource2*>(iOther);
			if (_iDXResource2)
			{
				_iDXResource2->_vtbl->_addRef(_iDXResource2);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXResource2** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXResource2Ref foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXResource2 && "has a com object reference. clear first.");
		return & _iDXResource2; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXResource2Ref& other)
	{
		bool result = false;
		if (_iDXResource2)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXResource2);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXResource2 ? _iDXResource2->_vtbl->_queryInterface(_iDXResource2, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXResource2) { _iDXResource2->_vtbl->_addRef(_iDXResource2); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXResource2) { _iDXResource2->_vtbl->_release(_iDXResource2); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDevice(struct ICEDXDevice* *const iDeviceOut)	{ return _iDXResource2 ? _iDXResource2->_vtbl->_getDevice(_iDXResource2, iDeviceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getType(eCEDXResourceType *const typeOut)	{ return _iDXResource2 ? _iDXResource2->_vtbl->_getType(_iDXResource2, typeOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXResource2* _iDXResource2;
};

/*! @}
 * end of CEComICEDXResource2Ref
 */

/*! \defgroup CEComICEDXSurfaceRef CEComICEDXSurfaceRef
 * @{
 */

class CEComICEDXSurfaceRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXSurfaceRef() : _iDXSurface(0) {}
	CEComICEDXSurfaceRef(ICEDXSurface* iOther) : _iDXSurface(0)
	{
		if (iOther)
		{
			_iDXSurface = iOther;
			_iDXSurface->_vtbl->_addRef(_iDXSurface);
		}
	}

	CEComICEDXSurfaceRef(const CEComICEDXSurfaceRef& other) : _iDXSurface(0)
	{
		if (other._iDXSurface)
		{
			_iDXSurface = other._iDXSurface;
			_iDXSurface->_vtbl->_addRef(_iDXSurface);
		}
	}

	~CEComICEDXSurfaceRef()
	{
		if (_iDXSurface)
		{
			ICEDXSurface* tmp = _iDXSurface;
			_iDXSurface = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXSurface* iDXSurface;
				void* _ptr;
			} uIntf;
			uIntf.iDXSurface = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXSurface, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXSurface)
				{
					ICEDXSurface* tmp = _iDXSurface;
					_iDXSurface = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXSurface = uIntf.iDXSurface;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXSurface* object() const	{ return _iDXSurface; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXSurface* iOther)
	{
		if (_iDXSurface)
		{
			ICEDXSurface* tmp = _iDXSurface;
			_iDXSurface = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXSurface = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXSurface* detach()
	{
		ICEDXSurface* iIntf = _iDXSurface;
		_iDXSurface = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXSurface* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXSurface;
		if (_iDXSurface)
		{
			_iDXSurface->_vtbl->_addRef(_iDXSurface);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXSurface*() const	{ return _iDXSurface; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXSurface& operator*() const	{ return *_iDXSurface; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXSurface == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXSurface* iOther) const	{ return (_iDXSurface != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXSurface* iOther) const	{ return (_iDXSurface == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXSurfaceRef& operator=(const CEComICEDXSurfaceRef& other)	{ return operator=(other._iDXSurface); }

	CEComICEDXSurfaceRef& operator=(const ICEDXSurface* iOther)
	{
		if (_iDXSurface != iOther)
		{
			if (_iDXSurface)
			{
				ICEDXSurface* tmp = _iDXSurface;
				_iDXSurface = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXSurface = const_cast<ICEDXSurface*>(iOther);
			if (_iDXSurface)
			{
				_iDXSurface->_vtbl->_addRef(_iDXSurface);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXSurface** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXSurfaceRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXSurface && "has a com object reference. clear first.");
		return & _iDXSurface; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXSurfaceRef& other)
	{
		bool result = false;
		if (_iDXSurface)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXSurface);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXSurface ? _iDXSurface->_vtbl->_queryInterface(_iDXSurface, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXSurface) { _iDXSurface->_vtbl->_addRef(_iDXSurface); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXSurface) { _iDXSurface->_vtbl->_release(_iDXSurface); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDevice(struct ICEDXDevice* *const iDeviceOut)	{ return _iDXSurface ? _iDXSurface->_vtbl->_getDevice(_iDXSurface, iDeviceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getType(eCEDXResourceType *const typeOut)	{ return _iDXSurface ? _iDXSurface->_vtbl->_getType(_iDXSurface, typeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDesc(struct CEDXSurfaceDesc *const descOut)	{ return _iDXSurface ? _iDXSurface->_vtbl->_getDesc(_iDXSurface, descOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult lockRect(const CERect* rect, UINT32 flags, struct CEDXLockedRect *const lockedRectOut)	{ return _iDXSurface ? _iDXSurface->_vtbl->_lockRect(_iDXSurface, rect, flags, lockedRectOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unlockRect()	{ return _iDXSurface ? _iDXSurface->_vtbl->_unlockRect(_iDXSurface) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getContainer(UINT32 iid, void** iContainerOut)	{ return _iDXSurface ? _iDXSurface->_vtbl->_getContainer(_iDXSurface, iid, iContainerOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult beginPaintF(struct ICEDXImmutableRegion* iUpdateRgn, struct ICEDXContext2D* *const piCtx2DOut)	{ return _iDXSurface ? _iDXSurface->_vtbl->_beginPaintF(_iDXSurface, iUpdateRgn, piCtx2DOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult endPaintF(struct ICEDXContext2D* iCtx2D)	{ return _iDXSurface ? _iDXSurface->_vtbl->_endPaintF(_iDXSurface, iCtx2D) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getRegion(struct ICEDXRegion* *const piRegionOut)	{ return _iDXSurface ? _iDXSurface->_vtbl->_getRegion(_iDXSurface, piRegionOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXSurface* _iDXSurface;
};

/*! @}
 * end of CEComICEDXSurfaceRef
 */

/*! \defgroup CEComICEDXPaletteRef CEComICEDXPaletteRef
 * @{
 */

class CEComICEDXPaletteRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXPaletteRef() : _iDXPalette(0) {}
	CEComICEDXPaletteRef(ICEDXPalette* iOther) : _iDXPalette(0)
	{
		if (iOther)
		{
			_iDXPalette = iOther;
			_iDXPalette->_vtbl->_addRef(_iDXPalette);
		}
	}

	CEComICEDXPaletteRef(const CEComICEDXPaletteRef& other) : _iDXPalette(0)
	{
		if (other._iDXPalette)
		{
			_iDXPalette = other._iDXPalette;
			_iDXPalette->_vtbl->_addRef(_iDXPalette);
		}
	}

	~CEComICEDXPaletteRef()
	{
		if (_iDXPalette)
		{
			ICEDXPalette* tmp = _iDXPalette;
			_iDXPalette = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXPalette* iDXPalette;
				void* _ptr;
			} uIntf;
			uIntf.iDXPalette = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXPalette, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXPalette)
				{
					ICEDXPalette* tmp = _iDXPalette;
					_iDXPalette = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXPalette = uIntf.iDXPalette;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXPalette* object() const	{ return _iDXPalette; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXPalette* iOther)
	{
		if (_iDXPalette)
		{
			ICEDXPalette* tmp = _iDXPalette;
			_iDXPalette = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXPalette = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXPalette* detach()
	{
		ICEDXPalette* iIntf = _iDXPalette;
		_iDXPalette = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXPalette* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXPalette;
		if (_iDXPalette)
		{
			_iDXPalette->_vtbl->_addRef(_iDXPalette);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXPalette*() const	{ return _iDXPalette; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXPalette& operator*() const	{ return *_iDXPalette; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXPalette == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXPalette* iOther) const	{ return (_iDXPalette != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXPalette* iOther) const	{ return (_iDXPalette == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXPaletteRef& operator=(const CEComICEDXPaletteRef& other)	{ return operator=(other._iDXPalette); }

	CEComICEDXPaletteRef& operator=(const ICEDXPalette* iOther)
	{
		if (_iDXPalette != iOther)
		{
			if (_iDXPalette)
			{
				ICEDXPalette* tmp = _iDXPalette;
				_iDXPalette = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXPalette = const_cast<ICEDXPalette*>(iOther);
			if (_iDXPalette)
			{
				_iDXPalette->_vtbl->_addRef(_iDXPalette);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXPalette** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXPaletteRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXPalette && "has a com object reference. clear first.");
		return & _iDXPalette; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXPaletteRef& other)
	{
		bool result = false;
		if (_iDXPalette)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXPalette);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXPalette ? _iDXPalette->_vtbl->_queryInterface(_iDXPalette, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXPalette) { _iDXPalette->_vtbl->_addRef(_iDXPalette); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXPalette) { _iDXPalette->_vtbl->_release(_iDXPalette); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDevice(struct ICEDXDevice* *const iDeviceOut)	{ return _iDXPalette ? _iDXPalette->_vtbl->_getDevice(_iDXPalette, iDeviceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getType(eCEDXResourceType *const typeOut)	{ return _iDXPalette ? _iDXPalette->_vtbl->_getType(_iDXPalette, typeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setPaletteEntries(const CERGBAColor* pEntries)	{ return _iDXPalette ? _iDXPalette->_vtbl->_setPaletteEntries(_iDXPalette, pEntries) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getPaletteEntries(CERGBAColor* pEntriesOut)	{ return _iDXPalette ? _iDXPalette->_vtbl->_getPaletteEntries(_iDXPalette, pEntriesOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXPalette* _iDXPalette;
};

/*! @}
 * end of CEComICEDXPaletteRef
 */

#endif // __cplusplus

#endif
