///////////////////////////////////////////////////////////////////////////////
// Copyright 2012 Sony Corporation
///////////////////////////////////////////////////////////////////////////////

#ifndef ICEDXDevice_h
#define ICEDXDevice_h

//----------------------------------------------------------------
// Includes
//----------------------------------------------------------------

#include "CEApiUnknown.h"
#include "ICEDXSurface.h"
#include "CEUITypes.h"
#include "CEDXTypes.h"
#include "CEPixelFormat.h"

#ifdef __cplusplus
extern "C" {
#endif

//----------------------------------------------------------------
// Prototypes
//----------------------------------------------------------------

struct ICEDXContext2D;
struct ICEDpyFrameWindow;
struct ICEDXDevice;
struct ICEDXVertexBuffer;
struct ICEDXIndexBuffer;
struct ICEDXTexture;
struct ICEDXStateBlock;
struct ICEDXVertexShader;
struct ICEDXPixelShader;
struct ICEDXVertexDeclaration;

//----------------------------------------------------------------
// Enumerators
//----------------------------------------------------------------

//! primitive type
//! enum numbers are compatible with Direct3D. this is important for DX implemented on Direct3D.
typedef enum
{
	eCEDXPrimitiveType_FIRST = 1,

    eCEDXPrimitiveType_PointList = eCEDXPrimitiveType_FIRST,
    eCEDXPrimitiveType_LineList,
    eCEDXPrimitiveType_LineStrip,
	eCEDXPrimitiveType_TriangleList,
	eCEDXPrimitiveType_TriangleStrip,
	eCEDXPrimitiveType_TriangleFan,

	eCEDXPrimitiveType_LAST = eCEDXPrimitiveType_TriangleFan
} eCEDXPrimitiveType;

//! buffer usage type of resource
typedef enum
{
	eCEDXUsage_FIRST = 0,
	eCEDXUsage_Default = eCEDXUsage_FIRST, 
	eCEDXUsage_Dynamic,								//!< Set the dynamic texture.
	eCEDXUsage_RenderTarget,						//!< Set the render target texture.
	eCEDXUsage_Writeonly,							//!< Set the write only mode.
	
	eCEDXUsage_LAST = eCEDXUsage_Writeonly
} eCEDXUsage;

//! Device's render state.
//! those states are set via ICEDXDevice::setRenderState().
//!
//! enum numbers are compatible with Direct3D. this is important for DX implemented on Direct3D.
										//!< Z-buffering state as one member of the eCEDXZBufferType enumerated type. 
                                        //!< Set this state to eCEDXZBufferType_True to enable z-buffering, 
                                        //!< or eCEDXZBufferType_False to disable z-buffering. 
                                        //!< The default value for this render state is eCEDXZBufferType_True.
                                        //!< This can be set to one member of the eCEDXCull enumerated type. 
                                        //!< The default value is eCEDXCull_CCW.
										//!< TRUE to enable alpha-blended transparency. The default value is FALSE. 
                                        //!< The type of alpha blending is determined by the 
										//!< eCEDXRenderStateType_SrcBlend and 
                                        //!< eCEDXRenderStateType_DestBlend render states.
typedef enum
{
	eCEDXRenderStateType_FIRST = 0,

	eCEDXRenderStateType_ZEnable = 7,

	eCEDXRenderStateType_CullMode = 22,	//!< Specifies how back-facing triangles are culled.

	eCEDXRenderStateType_AlphaBlendEnable = 27,

	eCEDXRenderStateType_SrcBlend = 19,	//!< One member of the eCEDXBlend enumerated type. The default value is eCEDXBlend_One. 

	eCEDXRenderStateType_DestBlend = 20,//!< One member of the eCEDXBlend enumerated type. The default value is eCEDXBlend_Zero.

	eCEDXRenderStateType_ScissorTestEnable = 174,	//!< TRUE to enable scissor testing and FALSE to disable it. The default value is FALSE.

	eCEDXRenderStateType_LAST = eCEDXRenderStateType_ScissorTestEnable
} eCEDXRenderStateType;

//! Defines constants that describe Z-buffer formats.
//! Members of this enumerated type are used with the eCEDXRenderStateType_ZEnable render state
//! in ICEDXDevice::setRenderState().
//!
//! enum numbers are compatible with Direct3D. this is important for DX implemented on Direct3D.
typedef enum
{
	eCEDXZBufferType_FIRST = 0,
	eCEDXZBufferType_False = 0,			//!< Disable Z buffering
	eCEDXZBufferType_True = 1,			//!< Enable Z buffering
	eCEDXZBufferType_LAST = eCEDXZBufferType_True
} eCEDXZBufferType;

//! Defines the supported culling modes.
//! The values in this enumerated type are used by the eCEDXRenderStateType_CullMode render state
//! in ICEDXDevice::setRenderState().
//! The culling modes define how back faces are culled when rendering a geometry.
//!
//! enum numbers are compatible with Direct3D. this is important for DX implemented on Direct3D.
typedef enum
{
	eCEDXCull_FIRST = 0,
	eCEDXCull_None = 1,					//!< Do not cull back faces. 
	eCEDXCull_CW = 2,					//!< Cull back faces with clockwise vertices. 
	eCEDXCull_CCW = 3,					//!< Cull back faces with counterclockwise vertices. 
	eCEDXCull_LAST = eCEDXCull_CCW
} eCEDXCull;

//! Supported blend mode. 
//! The values in this enumerated type are used by the eCEDXRenderStateType_SrcBlend and
//! eCEDXRenderStateType_DestBlend render state in ICEDXDevice::setRenderState().
//!
//! enum numbers are compatible with Direct3D. this is important for DX implemented on Direct3D.
typedef enum
{
	eCEDXBlend_FIRST = 0,
	eCEDXBlend_Zero = 1,					//!< Blend factor is (0, 0, 0, 0). 
	eCEDXBlend_One  = 2,					//!< Blend factor is (1, 1, 1, 1).
	eCEDXBlend_SrcAlpha = 5,				//!< Blend factor is (As, As, As, As). 
	eCEDXBlend_InvSrcAlpha = 6,				//!< Blend factor is ( 1 - As, 1 - As, 1 - As, 1 - As). 
	eCEDXBlend_LAST = eCEDXBlend_InvSrcAlpha
} eCEDXBlend;

//! Sampler states define texture sampling operations such as texture addressing and texture filtering. 
//!
//! enum numbers are compatible with Direct3D. this is important for DX implemented on Direct3D.
											//!< The default is eCEDXTextureAddress_Wrap.
											//!< The default is eCEDXTextureAddress_Wrap.
typedef enum
{
	eCEDXSamplerStateType_AddressU = 1,		//!< Texture-address mode for the u coordinate. 
	eCEDXSamplerStateType_AddressV = 2,		//!< Texture-address mode for the v coordinate. 
	eCEDXSamplerStateType_MagFilter = 5,	//!< Magnification filter. The default value is eCEDXTextureFilterType_Point.
	eCEDXSamplerStateType_MinFilter = 6,	//!< Minification filter. The default value is eCEDXTextureFilterType_Point.
	eCEDXSamplerStateType_MipFilter = 7		//!< Mipmap filter to use during minification. The default value is eCEDXTextureFilterType_None.
} eCEDXSamplerStateType;

//! Defines constants that describe the supported texture-addressing modes.
//! enum numbers are compatible with Direct3D. this is important for DX implemented on Direct3D.
											//!< For example, for u values between 0 and 3, 
											//!< the texture is repeated three times.
											//!< are set to the texture color at 0.0 or 1.0, respectively. 
typedef enum
{
	eCEDXTextureAddress_Wrap = 1,			//!< Tile the texture at every integer junction. 
	eCEDXTextureAddress_Clamp = 3			//!< Texture coordinates outside the range [0.0, 1.0]
} eCEDXTextureAddress;

//! Defines texture filtering modes for a texture stage.
//! enum numbers are compatible with Direct3D. this is important for DX implemented on Direct3D.
											//!< minification filter. The texel with coordinates nearest 
											//!< to the desired pixel value is used. 
											//!< magnification or minification filter. 
											//!< A weighted average of a 2 x 2 area of texels surrounding 
											//!< the desired pixel is used. 
typedef enum
{
	eCEDXTextureFilterType_None = 0,		//!< Mipmapping disabled. The rasterizer should use the magnification filter instead. 
	eCEDXTextureFilterType_Point = 1,		//!< Point filtering used as a texture magnification or 
	eCEDXTextureFilterType_Linear = 2		//!< Bilinear interpolation filtering used as a texture 
} eCEDXTextureFilterType;

//! Vertex element semantics
//! enum numbers are compatible with Direct3D. this is important for DX implemented on Direct3D.
typedef enum
{
	eCEDXDeclUsage_Position		= 0,
	eCEDXDeclUsage_TexCoord		= 5,
	eCEDXDeclUsage_Color		= 10
} eCEDXDeclUsage;

//! Defines the vertex declaration method which is a predefined operation performed by 
//! the tessellator (or any procedural geometry routine on the vertex data during tessellation).
//!
//! enum numbers are compatible with Direct3D. this is important for DX implemented on Direct3D.
typedef enum
{
	eCEDXDeclMethod_Default = 0
} eCEDXDeclMethod;

//! Declarations for _type fields
//!
//! enum numbers are compatible with Direct3D. this is important for DX implemented on Direct3D.
typedef enum
{
	eCEDXDeclType_Float2	= 1,		//!< 2D float expanded to (value, value, 0, 1.)
	eCEDXDeclType_Float3	= 2,		//!< 3D float expanded to (value, value, value, 1.)
	eCEDXDeclType_Float4	= 3,		//!< 4D float
	eCEDXDeclType_UByte4N	= 8,		//!< Each of 4 bytes is normalized by dividing to 255.0
	eCEDXDeclType_Unused	= 17,		//!< When the type field in a decl is unused.
} eCEDXDeclType;

//! index buffer's format
//! enum numbers are compatible with Direct3D (D3DFORMAT). this is important for DX implemented on Direct3D.
typedef enum
{
	eCEDXIndexFormat_Index16 = 101,		//!< 16-bit index buffer bit depth.
	eCEDXIndexFormat_Index32 = 102		//!< 32-bit index buffer bit depth.
} eCEDXIndexFormat;

//----------------------------------------------------------------
// Structures
//----------------------------------------------------------------

//! Viewport information
//! 
//! The _x, _y, _width, and _height members describe the position and dimensions 
//! of the viewport on the render-target surface. Usually, applications 
//! render to the entire target surface; when rendering on a 640~480 
//! surface, these members should be 0, 0, 640, and 480, respectively. The 
//! minZ and maxZ are typically set to 0.0 and 1.0 but can be set to other 
//! values to achieve specific effects. For example, you might set them both 
//! to 0.0 to force the system to render objects to the foreground of a 
//! scene, or both to 1.0 to force the objects into the background.
								//!< the render-target surface. Unless you want to render to a subset
								//!< of the surface, this member can be set to 0. 
								//!< the render-target surface. Unless you want to render to a subset
								//!< of the surface, this member can be set to 0. 
								//!< rendering only to a subset of the surface, this member should
								//!< be set to the width dimension of the render-target surface. 
								//!< rendering only to a subset of the surface, this member should
								//!< be set to the height dimension of the render-target surface. 
								//!< into which a scene is to be rendered, the minimum and maximum 
								//!< values of the clip volume. Most applications set this value 
								//!< to 0.0. Clipping is performed after applying the projection matrix. 
								//!< into which a scene is to be rendered, the minimum and maximum 
								//!< values of the clip volume. Most applications set this value 
								//!< to 1.0. Clipping is performed after applying the projection matrix. 
typedef struct CEDXViewport
{
	INT32 _x;					//!< Pixel coordinate of the upper-left corner of the viewport on 

	INT32 _y;					//!< Pixel coordinate of the upper-left corner of the viewport on 
	
	UINT32 _width;				//!< Width dimension of the clip volume, in pixels. Unless you are

	UINT32 _height;				//!< Height dimension of the clip volume, in pixels. Unless you are

	float _minZ;				//!< Together with _maxZ, value describing the range of depth values

	float _maxZ;				//!< Together with _minZ, value describing the range of depth values
} CEDXViewport;

//! Vertex element 
//! structure format is compatible with Direct3D. this is important for DX implemented on Direct3D.
typedef struct CEDXVertexElement
{
	UINT16 _stream;				//!< Stream index
	UINT16 _offset;				//!< Offset in the stream in bytes
	UINT8  _type;				//!< Data type. value of eCEDXDeclType.
	UINT8  _method;				//!< Processing method. value of eCEDXDeclMethod.
	UINT8  _usage;				//!< semantics. value of eCEDXDecUsage.
	UINT8  _usageIndex;			//!< semantic index
} CEDXVertexElement;

//----------------------------------------------------------------
// Interfaces
//----------------------------------------------------------------

/*! \defgroup ICEDX ICEDX
 * @{
 */

/*!
 * ID of ICEDX
 */
#define CEComIID_ICEDX 0x43d7c1da


/*!
 * ICEDX
 */
typedef struct ICEDX
{
	const struct ICEDX_vtbl* _vtbl;
} ICEDX;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDX	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDX_queryInterface) (ICEDX* iDX, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDX_addRef) (ICEDX* iDX);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDX_release) (ICEDX* iDX);

	//! create a device.
	//! \param iWindow		the window used as the front buffer.
	//!						NULL means the front buffer is the full screen (desktop).

typedef CEHResult (*iCEDX_createDevice) (ICEDX* iDX, struct ICEDpyFrameWindow* iWindow, struct ICEDXDevice* *const iGLDeviceOut);

/*!
 * V-table of ICEDX
 */
struct ICEDX_vtbl
{
	iCEDX_queryInterface	_queryInterface;	//!< Query interface.
	iCEDX_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDX_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDX_createDevice	_createDevice;
};

/*! @}
 * end of ICEDX
 */

/*! \defgroup ICEDXDevice ICEDXDevice
 * @{
 */

/*!
 * ID of ICEDXDevice
 */
#define CEComIID_ICEDXDevice 0x396af646


/*!
 * ICEDXDevice
 */
typedef struct ICEDXDevice
{
	const struct ICEDXDevice_vtbl* _vtbl;
} ICEDXDevice;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXDevice	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXDevice_queryInterface) (ICEDXDevice* iDXDevice, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXDevice_addRef) (ICEDXDevice* iDXDevice);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXDevice_release) (ICEDXDevice* iDXDevice);

	//! create a vertex buffer object
	//!
	//! \param[in] length       Size of the vertex buffer, in bytes. 
	//! \param[out] iBufferOut	pointer to the varibale that receives the interface pointer to the created vertex buffer

typedef CEHResult (*iCEDXDevice_createVertexBuffer) (ICEDXDevice* iDXDevice, UINT32 length, struct ICEDXVertexBuffer* *const iBufferOut);

	//! set a vertex buffer as the geometry stream source.
	//! \param[in] stride		the stride should be greater than or equal to the stream size computed
	//!							from the declaration. When calling setStreamSource, the stride is normally 
	//!							required to be equal to the vertex size. However, there are times when you
	//!							may want to draw multiple instances of the same or similar geometry 
	//!							(such as when using instancing to draw). For this case, use a zero stride
	//!							to tell the runtime not to increment the vertex buffer offset
	//!							(ie: use the same vertex data for all instances). 

typedef CEHResult (*iCEDXDevice_setStreamSource) (ICEDXDevice* iDXDevice, UINT32 streamNumber, struct ICEDXVertexBuffer* iVertexBuffer, UINT32 stride);

	//! get a vertex buffer as the geometry stream source
	//! \param[out] pStrideOut	Pointer to a returned stride of the component, in bytes. 

typedef CEHResult (*iCEDXDevice_getStreamSource) (ICEDXDevice* iDXDevice, UINT32 streamNumber, struct ICEDXVertexBuffer* *const iVertexBufferOut, UINT32* pStrideOut);

	//! Sets the stream source frequency divider value. 
	//! This may be used to draw several instances of geometry.
	//! \param[in] param	There are two constants that are designed to use with 
	//!						setStreamSourceFreq: CEDXStreamSource_IndexedData and
	//!						CEDXStreamSource_InstanceData. 
	//!						To see how to use the constants, see the following:
	//!						ms-help://MS.VSCC.v80/MS.VSIPCC.v80/MS.DirectX9.1033.2005.October/DirectX9_c/Efficiently_Drawing_Multiple_Instances_of_Geometry.htm

typedef CEHResult (*iCEDXDevice_setStreamSourceFreq) (ICEDXDevice* iDXDevice, UINT32 streamNumber, UINT32 param);

	//! Gets the stream source frequency divider value.

typedef CEHResult (*iCEDXDevice_getStreamSourceFreq) (ICEDXDevice* iDXDevice, UINT32 streamNumber, UINT32* pParamOut);

	//! Create a vertex shader declaration from the device and the vertex elements.

typedef CEHResult (*iCEDXDevice_createVertexDeclaration) (ICEDXDevice* iDXDevice, const CEDXVertexElement* pVtxElms, struct ICEDXVertexDeclaration* *const piVDOut);

	//! Sets a vertex declaration.

typedef CEHResult (*iCEDXDevice_setVertexDeclaration) (ICEDXDevice* iDXDevice, struct ICEDXVertexDeclaration* iVD);

	//! Gets a vertex declaration.

typedef CEHResult (*iCEDXDevice_getVertexDeclaration) (ICEDXDevice* iDXDevice, struct ICEDXVertexDeclaration* *const piVDOut);

	//! Creates an index buffer.
	//! \param[in] length       Size of the index buffer, in bytes. 
	//! \param[in] format		format of the index buffer. 

typedef CEHResult (*iCEDXDevice_createIndexBuffer) (ICEDXDevice* iDXDevice, UINT32 length, eCEDXIndexFormat format, struct ICEDXIndexBuffer* *const piBufferOut);

	//! Sets index data.

typedef CEHResult (*iCEDXDevice_setIndices) (ICEDXDevice* iDXDevice, struct ICEDXIndexBuffer* pIndexBuf);

	//! Retrieves index data.

typedef CEHResult (*iCEDXDevice_getIndices) (ICEDXDevice* iDXDevice, struct ICEDXIndexBuffer* *const piIndexBufOut);

	//! Create a texture.
	//! Note that textures are used by setting them to pixel shaders.

typedef CEHResult (*iCEDXDevice_createTexture) (ICEDXDevice* iDXDevice, CEPixelFormat format, UINT32 width, UINT32 height, eCEDXPoolType pool, eCEDXUsage usage, struct ICEDXTexture* *const iTextureOut);

	//! Create a texture from external texture
	//! \Note this method creates the wrapper object for the specified texture (usually created by other DXDevices)
	//!       to draw by this DXDevice 

typedef CEHResult (*iCEDXDevice_createTextureFromTexture) (ICEDXDevice* iDXDevice, struct ICEDXTexture* iBaseTexture, struct ICEDXTexture* *const iTextureOut);

	//! get the back buffer surface.
	//! \param[out,retval] ppBBuf  address of a pointer to an ICEDXSurface interface.

typedef CEHResult (*iCEDXDevice_getBackBuffer) (ICEDXDevice* iDXDevice, struct ICEDXSurface* *const ppBBuf);

	//! get the render target.
	//! \param[out,retval] piRT  address of a pointer to an ICEDXSurface interface.

typedef CEHResult (*iCEDXDevice_getRenderTarget) (ICEDXDevice* iDXDevice, struct ICEDXSurface* *const piRT);

	//! fill the rectangle in the surface with specified color
	//! \param[in] iSurface		pointer to an ICEDXSurface to be filled
	//! \param[in] pRect		pointer to an CERect that represent the rectangular area in the surface
	//!                         if NULL, it means the entire surface
	//! \param[in] color        color to fill

typedef CEHResult (*iCEDXDevice_colorFill) (ICEDXDevice* iDXDevice, struct ICEDXSurface* iSurface, const CERect* pRect, CERGBAColor color);

	//! copy the rectangular area of source surface to destination surface
	//! \param[in] iSrcSurface	pointer to an ICEDXSurface that is the source of pixel data
	//! \param[in] pSrcRect		pointer to an CERect that represent the rectangular area to be copied
	//!							if NULL, it means the entire iSrcSurface
	//! \param[in] iDstSurface	pointer to an ICEDXSurface that is the destination of copied pixel data
	//! \param[in] pDstPt		pointer to an CEPointBase that represent the left-top corner of
	//!							the copied pixel data in iDstSurface

typedef CEHResult (*iCEDXDevice_updateSurface) (ICEDXDevice* iDXDevice, struct ICEDXSurface* iSrcSurface, const CERect* pSrcRect, struct ICEDXSurface* iDstSurface, const CEPointBase* pDstPt);

typedef CEHResult (*iCEDXDevice_getLayerCount) (ICEDXDevice* iDXDevice, UINT32 *const pCountOut);

typedef CEHResult (*iCEDXDevice_getLevelCorrespondLayerIdx) (ICEDXDevice* iDXDevice, eCEDpyWindowLevel level, UINT32 *const pIdxOut);

typedef CEHResult (*iCEDXDevice_getLayerCorrespondBackBuffer) (ICEDXDevice* iDXDevice, UINT32 layerIdx, struct ICEDXSurface* *const ppBufOut);

typedef CEHResult (*iCEDXDevice_setPaletteToLayer) (ICEDXDevice* iDXDevice, UINT32 layerIdx, const struct ICEDXPalette* iPalette);

typedef CEHResult (*iCEDXDevice_setTransparencyToLayer) (ICEDXDevice* iDXDevice, UINT32 layerIdx, const float trans);

typedef CEHResult (*iCEDXDevice_setClearIndexToLayer) (ICEDXDevice* iDXDevice, UINT32 layerIdx, UINT8 clearIdx);

typedef CEHResult (*iCEDXDevice_getClearIndexOfLayer) (ICEDXDevice* iDXDevice, UINT32 layerIdx, UINT8 *const pClearIdxOut);

	//! Create the palette.
	//! Note that texture palettes are used by setting them to pixel shaders.

typedef CEHResult (*iCEDXDevice_createPalette) (ICEDXDevice* iDXDevice, struct ICEDXPalette* *const iPaletteOut);

	//! Find a vertex shader, which is pre-loaded in the device.

typedef CEHResult (*iCEDXDevice_findVertexShader) (ICEDXDevice* iDXDevice, CEDXShaderID shaderId, struct ICEDXVertexShader* *const piShaderOut);

	//! Bind a vertex shader.
	//! Note that all parameters specified by ICEDXVertexShader APIs are
	//! loaded into the GPU at this moment. This means that a
	//! parameter value set after binding the shader is not used during the
	//! execution of the shader until the next time the shader is bound.
	//! For more information, read the description of "Parameter Shadowing" in Cg document.

typedef CEHResult (*iCEDXDevice_bindVertexShader) (ICEDXDevice* iDXDevice, struct ICEDXVertexShader* iShader);

	//! Find a pixel shader, which is pre-loaded in the device.

typedef CEHResult (*iCEDXDevice_findPixelShader) (ICEDXDevice* iDXDevice, CEDXShaderID shaderId, struct ICEDXPixelShader* *const piShaderOut);

	//! Bind a pixel shader.
	//! Note that all parameters specified by ICEDXPixelShader APIs are
	//! loaded into the GPU at this moment. This means that a
	//! parameter value set after binding the shader is not used during the
	//! execution of the shader until the next time the shader is bound.
	//! For more information, read the description of "Parameter Shadowing" in Cg document.

typedef CEHResult (*iCEDXDevice_bindPixelShader) (ICEDXDevice* iDXDevice, struct ICEDXPixelShader* iShader);

	//! Creates a new state block that contains the values for all device states, 
	//! vertex-related states, or pixel-related states.
	//! those states are initialized to have the default values as the DX device.
	//! \param[out, retval] piSBOut  Pointer to a state block interface.

typedef CEHResult (*iCEDXDevice_createStateBlock) (ICEDXDevice* iDXDevice, struct ICEDXStateBlock* *const piSBOut);

	//! Purge cached resources such as shader programs.
	//! Application programs have to call this method before terminating the application programs.
	//! Otherwise ICEDXDevice implementation may cause memory leak.

typedef CEHResult (*iCEDXDevice_purgeCachedResources) (ICEDXDevice* iDXDevice);

	//! Sets a single device render-state parameter.

typedef CEHResult (*iCEDXDevice_setRenderState) (ICEDXDevice* iDXDevice, eCEDXRenderStateType state, UINT32 value);

	//! Sets the scissor rectangle.
	//! \param[in] pRect			scissor rectangle

typedef CEHResult (*iCEDXDevice_setScissorRect) (ICEDXDevice* iDXDevice, const CERect* pRect);

	//! Gets the scissor rectangle.
	//! \param[out] pRectOut		fill the scissor rectangle.

typedef CEHResult (*iCEDXDevice_getScissorRect) (ICEDXDevice* iDXDevice, CERect* pRectOut);

	//! Based on indexing, renders the specified geometric primitive into an array of vertices.
	//! \param[in] type				primitive type to render.
	//! \param[in] minIndex			Minimum vertex index for vertices used during this call. 
	//!								This is a zero based index.
	//! \param[in] numVertices		Number of vertices used during this call. The first vertex is 
	//!								located at index : minIndex.
	//! \param[in] startIndex		Index of the first index to use when accesssing the vertex buffer. 
	//!								Beginning at startIndex to index vertices from the vertex buffer.
	//! \param[in] numOfPrims		Number of primitives to render.

typedef CEHResult (*iCEDXDevice_drawIndexedPrimitive) (ICEDXDevice* iDXDevice, eCEDXPrimitiveType type, UINT32 minIndex, UINT32 numVertices, UINT32 startIndex, UINT32 numOfPrims);

	//! Renders a sequence of nonindexed, geometric primitives of the specified type from 
	//! the current set of data input streams.
	//! \param[in] type				primitive type to render.
	//! \param[in] startVertex		Index of the first vertex to load. Beginning at startVertex,
	//!								the correct number of vertices will be read out of the vertex buffer. 
	//! \param[in] numOfPrims		Number of primitives to render.

typedef CEHResult (*iCEDXDevice_drawPrimitive) (ICEDXDevice* iDXDevice, eCEDXPrimitiveType type, UINT32 startVertex, UINT32 numOfPrims);

	//! Sets the viewport parameters for the device.
	//! \param[in] pVP	pointer to CEDXViewport that specifies the viewport parameters to set. 

typedef CEHResult (*iCEDXDevice_setViewport) (ICEDXDevice* iDXDevice, const CEDXViewport* pVP);

typedef CEHResult (*iCEDXDevice_beginScene) (ICEDXDevice* iDXDevice);

typedef CEHResult (*iCEDXDevice_endScene) (ICEDXDevice* iDXDevice);

typedef CEHResult (*iCEDXDevice_present) (ICEDXDevice* iDXDevice);

typedef CEHResult (*iCEDXDevice_clear) (ICEDXDevice* iDXDevice);

typedef CEHResult (*iCEDXDevice_paintRectsOfDirtyRegion) (ICEDXDevice* iDXDevice, struct ICEDXRegion* iRegion);

typedef CEHResult (*iCEDXDevice_getPixelFormat) (ICEDXDevice* iDXDevice, CEPixelFormat* const pPxlFmtOut);

/*!
 * V-table of ICEDXDevice
 */
struct ICEDXDevice_vtbl
{
	iCEDXDevice_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXDevice_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXDevice_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXDevice_createVertexBuffer	_createVertexBuffer;
	iCEDXDevice_setStreamSource	_setStreamSource;
	iCEDXDevice_getStreamSource	_getStreamSource;
	iCEDXDevice_setStreamSourceFreq	_setStreamSourceFreq;
	iCEDXDevice_getStreamSourceFreq	_getStreamSourceFreq;
	iCEDXDevice_createVertexDeclaration	_createVertexDeclaration;
	iCEDXDevice_setVertexDeclaration	_setVertexDeclaration;
	iCEDXDevice_getVertexDeclaration	_getVertexDeclaration;
	iCEDXDevice_createIndexBuffer	_createIndexBuffer;
	iCEDXDevice_setIndices	_setIndices;
	iCEDXDevice_getIndices	_getIndices;
	iCEDXDevice_createTexture	_createTexture;
	iCEDXDevice_createTextureFromTexture	_createTextureFromTexture;
	iCEDXDevice_getBackBuffer	_getBackBuffer;
	iCEDXDevice_getRenderTarget	_getRenderTarget;
	iCEDXDevice_colorFill	_colorFill;
	iCEDXDevice_updateSurface	_updateSurface;
	iCEDXDevice_getLayerCount	_getLayerCount;
	iCEDXDevice_getLevelCorrespondLayerIdx	_getLevelCorrespondLayerIdx;
	iCEDXDevice_getLayerCorrespondBackBuffer	_getLayerCorrespondBackBuffer;
	iCEDXDevice_setPaletteToLayer	_setPaletteToLayer;
	iCEDXDevice_setTransparencyToLayer	_setTransparencyToLayer;
	iCEDXDevice_setClearIndexToLayer	_setClearIndexToLayer;
	iCEDXDevice_getClearIndexOfLayer	_getClearIndexOfLayer;
	iCEDXDevice_createPalette	_createPalette;
	iCEDXDevice_findVertexShader	_findVertexShader;
	iCEDXDevice_bindVertexShader	_bindVertexShader;
	iCEDXDevice_findPixelShader	_findPixelShader;
	iCEDXDevice_bindPixelShader	_bindPixelShader;
	iCEDXDevice_createStateBlock	_createStateBlock;
	iCEDXDevice_purgeCachedResources	_purgeCachedResources;
	iCEDXDevice_setRenderState	_setRenderState;
	iCEDXDevice_setScissorRect	_setScissorRect;
	iCEDXDevice_getScissorRect	_getScissorRect;
	iCEDXDevice_drawIndexedPrimitive	_drawIndexedPrimitive;
	iCEDXDevice_drawPrimitive	_drawPrimitive;
	iCEDXDevice_setViewport	_setViewport;
	iCEDXDevice_beginScene	_beginScene;
	iCEDXDevice_endScene	_endScene;
	iCEDXDevice_present	_present;
	iCEDXDevice_clear	_clear;
	iCEDXDevice_paintRectsOfDirtyRegion	_paintRectsOfDirtyRegion;
	iCEDXDevice_getPixelFormat	_getPixelFormat;
};

/*! @}
 * end of ICEDXDevice
 */

/*! \defgroup ICEDXResource ICEDXResource
 * @{
 */

/*!
 * ID of ICEDXResource
 */
#define CEComIID_ICEDXResource 0x11f48b25


/*!
 * ICEDXResource
 */
typedef struct ICEDXResource
{
	const struct ICEDXResource_vtbl* _vtbl;
} ICEDXResource;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXResource	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXResource_queryInterface) (ICEDXResource* iDXResource, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXResource_addRef) (ICEDXResource* iDXResource);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXResource_release) (ICEDXResource* iDXResource);

	//! get the device that creates this resource

typedef CEHResult (*iCEDXResource_getDevice) (ICEDXResource* iDXResource, struct ICEDXDevice* *const iDeviceOut);

	//! get the resource type

typedef CEHResult (*iCEDXResource_getType) (ICEDXResource* iDXResource, eCEDXResourceType *const typeOut);

/*!
 * V-table of ICEDXResource
 */
struct ICEDXResource_vtbl
{
	iCEDXResource_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXResource_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXResource_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXResource_getDevice	_getDevice;
	iCEDXResource_getType	_getType;
};

/*! @}
 * end of ICEDXResource
 */

/*! \defgroup ICEDXVertexBuffer ICEDXVertexBuffer
 * @{
 */

/*!
 * ID of ICEDXVertexBuffer
 */
#define CEComIID_ICEDXVertexBuffer 0x545145e0

	//! get the pool type

/*!
 * ICEDXVertexBuffer
 */
typedef struct ICEDXVertexBuffer
{
	const struct ICEDXVertexBuffer_vtbl* _vtbl;
} ICEDXVertexBuffer;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXVertexBuffer	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXVertexBuffer_queryInterface) (ICEDXVertexBuffer* iDXVertexBuffer, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXVertexBuffer_addRef) (ICEDXVertexBuffer* iDXVertexBuffer);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXVertexBuffer_release) (ICEDXVertexBuffer* iDXVertexBuffer);

	//! get the device that creates this resource

typedef CEHResult (*iCEDXVertexBuffer_getDevice) (ICEDXVertexBuffer* iDXVertexBuffer, struct ICEDXDevice* *const iDeviceOut);

	//! get the resource type

typedef CEHResult (*iCEDXVertexBuffer_getType) (ICEDXVertexBuffer* iDXVertexBuffer, eCEDXResourceType *const typeOut);

typedef CEHResult (*iCEDXVertexBuffer_lock) (ICEDXVertexBuffer* iDXVertexBuffer, void* *const pDataOut);

typedef CEHResult (*iCEDXVertexBuffer_unlock) (ICEDXVertexBuffer* iDXVertexBuffer);

/*!
 * V-table of ICEDXVertexBuffer
 */
struct ICEDXVertexBuffer_vtbl
{
	iCEDXVertexBuffer_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXVertexBuffer_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXVertexBuffer_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXVertexBuffer_getDevice	_getDevice;
	iCEDXVertexBuffer_getType	_getType;
	iCEDXVertexBuffer_lock	_lock;
	iCEDXVertexBuffer_unlock	_unlock;
};

/*! @}
 * end of ICEDXVertexBuffer
 */

/*! \defgroup ICEDXIndexBuffer ICEDXIndexBuffer
 * @{
 */

/*!
 * ID of ICEDXIndexBuffer
 */
#define CEComIID_ICEDXIndexBuffer 0x2d2869d6


/*!
 * ICEDXIndexBuffer
 */
typedef struct ICEDXIndexBuffer
{
	const struct ICEDXIndexBuffer_vtbl* _vtbl;
} ICEDXIndexBuffer;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXIndexBuffer	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXIndexBuffer_queryInterface) (ICEDXIndexBuffer* iDXIndexBuffer, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXIndexBuffer_addRef) (ICEDXIndexBuffer* iDXIndexBuffer);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXIndexBuffer_release) (ICEDXIndexBuffer* iDXIndexBuffer);

	//! get the device that creates this resource

typedef CEHResult (*iCEDXIndexBuffer_getDevice) (ICEDXIndexBuffer* iDXIndexBuffer, struct ICEDXDevice* *const iDeviceOut);

	//! get the resource type

typedef CEHResult (*iCEDXIndexBuffer_getType) (ICEDXIndexBuffer* iDXIndexBuffer, eCEDXResourceType *const typeOut);

typedef CEHResult (*iCEDXIndexBuffer_lock) (ICEDXIndexBuffer* iDXIndexBuffer, void* *const pDataOut);

typedef CEHResult (*iCEDXIndexBuffer_unlock) (ICEDXIndexBuffer* iDXIndexBuffer);

/*!
 * V-table of ICEDXIndexBuffer
 */
struct ICEDXIndexBuffer_vtbl
{
	iCEDXIndexBuffer_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXIndexBuffer_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXIndexBuffer_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXIndexBuffer_getDevice	_getDevice;
	iCEDXIndexBuffer_getType	_getType;
	iCEDXIndexBuffer_lock	_lock;
	iCEDXIndexBuffer_unlock	_unlock;
};

/*! @}
 * end of ICEDXIndexBuffer
 */

/*! \defgroup ICEDXTexture ICEDXTexture
 * @{
 */

/*!
 * ID of ICEDXTexture
 */
#define CEComIID_ICEDXTexture 0xe8a27e6c


/*!
 * ICEDXTexture
 */
typedef struct ICEDXTexture
{
	const struct ICEDXTexture_vtbl* _vtbl;
} ICEDXTexture;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXTexture	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXTexture_queryInterface) (ICEDXTexture* iDXTexture, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXTexture_addRef) (ICEDXTexture* iDXTexture);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXTexture_release) (ICEDXTexture* iDXTexture);

	//! get the device that creates this resource

typedef CEHResult (*iCEDXTexture_getDevice) (ICEDXTexture* iDXTexture, struct ICEDXDevice* *const iDeviceOut);

	//! get the resource type

typedef CEHResult (*iCEDXTexture_getType) (ICEDXTexture* iDXTexture, eCEDXResourceType *const typeOut);

	//! get the specified MIPMAP level's surface description

typedef CEHResult (*iCEDXTexture_getLevelDesc) (ICEDXTexture* iDXTexture, UINT32 level, struct CEDXSurfaceDesc *const descOut);

	//! get the specified MIPMAP level's surface

typedef CEHResult (*iCEDXTexture_getSurfaceLevel) (ICEDXTexture* iDXTexture, UINT32 level, struct ICEDXSurface* *const iSurfaceOut);

	//! lock the specified rect on this surface

typedef CEHResult (*iCEDXTexture_lockRect) (ICEDXTexture* iDXTexture, UINT32 level, const CERect* rect, UINT32 flags, struct CEDXLockedRect *const lockedRectOut);

	//! unlock the previously locked rect on this surface

typedef CEHResult (*iCEDXTexture_unlockRect) (ICEDXTexture* iDXTexture, UINT32 level);

/*!
 * V-table of ICEDXTexture
 */
struct ICEDXTexture_vtbl
{
	iCEDXTexture_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXTexture_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXTexture_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXTexture_getDevice	_getDevice;
	iCEDXTexture_getType	_getType;
	iCEDXTexture_getLevelDesc	_getLevelDesc;
	iCEDXTexture_getSurfaceLevel	_getSurfaceLevel;
	iCEDXTexture_lockRect	_lockRect;
	iCEDXTexture_unlockRect	_unlockRect;
};

/*! @}
 * end of ICEDXTexture
 */

/*! \defgroup ICEDXStateBlock ICEDXStateBlock
 * @{
 */

/*!
 * ID of ICEDXStateBlock
 */
#define CEComIID_ICEDXStateBlock 0x10d8332a


/*!
 * ICEDXStateBlock
 */
typedef struct ICEDXStateBlock
{
	const struct ICEDXStateBlock_vtbl* _vtbl;
} ICEDXStateBlock;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXStateBlock	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXStateBlock_queryInterface) (ICEDXStateBlock* iDXStateBlock, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXStateBlock_addRef) (ICEDXStateBlock* iDXStateBlock);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXStateBlock_release) (ICEDXStateBlock* iDXStateBlock);

	//! get the device that creates this state block.

typedef CEHResult (*iCEDXStateBlock_getDevice) (ICEDXStateBlock* iDXStateBlock, struct ICEDXDevice* *const piDeviceOut);

	//! Apply the state block to the current device state.

typedef CEHResult (*iCEDXStateBlock_apply) (ICEDXStateBlock* iDXStateBlock);

	//! Capture the current value of states that are included in a stateblock.

typedef CEHResult (*iCEDXStateBlock_capture) (ICEDXStateBlock* iDXStateBlock);

/*!
 * V-table of ICEDXStateBlock
 */
struct ICEDXStateBlock_vtbl
{
	iCEDXStateBlock_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXStateBlock_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXStateBlock_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXStateBlock_getDevice	_getDevice;
	iCEDXStateBlock_apply	_apply;
	iCEDXStateBlock_capture	_capture;
};

/*! @}
 * end of ICEDXStateBlock
 */

/*! \defgroup ICEDXShader ICEDXShader
 * @{
 */

/*!
 * ID of ICEDXShader
 */
#define CEComIID_ICEDXShader 0xfddd90c1


/*!
 * ICEDXShader
 *
 * all APIs are wrapper of Cg runtime APIs. See Cg document for the details.
 * Note that all parameters specified by ICEDXShader APIs are
 * loaded into the GPU when ICEDXDevice::bindVertexShader/bindPixelShader is called. 
 * This means that a parameter value set after binding the shader is not used during the
 * execution of the shader until the next time the shader is bound.
 * For more information, read the description of "Parameter Shadowing" in Cg document.
 */
typedef struct ICEDXShader
{
	const struct ICEDXShader_vtbl* _vtbl;
} ICEDXShader;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXShader	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXShader_queryInterface) (ICEDXShader* iDXShader, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXShader_addRef) (ICEDXShader* iDXShader);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXShader_release) (ICEDXShader* iDXShader);

	//! finds a named parameter.

typedef CEHResult (*iCEDXShader_getNamedParameter) (ICEDXShader* iDXShader, const char* pName, CEDXShaderParameter* pParamOut);

	//! sets the value of scalar and vector parameters.
	//! Be careful about "parameter shadowing".

typedef CEHResult (*iCEDXShader_setParameter1f) (ICEDXShader* iDXShader, CEDXShaderParameter param, float x);

	//! sets the value of scalar and vector parameters.
	//! Be careful about "parameter shadowing".

typedef CEHResult (*iCEDXShader_setParameter4f) (ICEDXShader* iDXShader, CEDXShaderParameter param, float x, float y, float z, float w);

	//! sets the value of matrix parameters.
	//! Be careful about "parameter shadowing".

typedef CEHResult (*iCEDXShader_setMatrixParameter) (ICEDXShader* iDXShader, CEDXShaderParameter param, CEDXMatrix44* pMat);

	//! Sets the texture for a particular sampler parameter.
	//! Be careful about "parameter shadowing".
	//!
	//! \param[in] param	A parameter handle, the parameter whose values are to be set. It must be a sampler.

typedef CEHResult (*iCEDXShader_setTexture) (ICEDXShader* iDXShader, CEDXShaderParameter param, struct ICEDXTexture* pTex);

	//! Sets the state associated with a particular sampler parameter.
	//! Be careful about "parameter shadowing".
	//!
	//! \param[in] param	A parameter handle, the parameter whose values are to be set. It must be a sampler.

typedef CEHResult (*iCEDXShader_setSamplerState) (ICEDXShader* iDXShader, CEDXShaderParameter param, eCEDXSamplerStateType state, UINT32 value);

/*!
 * V-table of ICEDXShader
 */
struct ICEDXShader_vtbl
{
	iCEDXShader_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXShader_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXShader_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXShader_getNamedParameter	_getNamedParameter;
	iCEDXShader_setParameter1f	_setParameter1f;
	iCEDXShader_setParameter4f	_setParameter4f;
	iCEDXShader_setMatrixParameter	_setMatrixParameter;
	iCEDXShader_setTexture	_setTexture;
	iCEDXShader_setSamplerState	_setSamplerState;
};

/*! @}
 * end of ICEDXShader
 */

/*! \defgroup ICEDXVertexShader ICEDXVertexShader
 * @{
 */

/*!
 * ID of ICEDXVertexShader
 */
#define CEComIID_ICEDXVertexShader 0xacb50b59


/*!
 * ICEDXVertexShader
 */
typedef struct ICEDXVertexShader
{
	const struct ICEDXVertexShader_vtbl* _vtbl;
} ICEDXVertexShader;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXVertexShader	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXVertexShader_queryInterface) (ICEDXVertexShader* iDXVertexShader, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXVertexShader_addRef) (ICEDXVertexShader* iDXVertexShader);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXVertexShader_release) (ICEDXVertexShader* iDXVertexShader);

	//! finds a named parameter.

typedef CEHResult (*iCEDXVertexShader_getNamedParameter) (ICEDXVertexShader* iDXVertexShader, const char* pName, CEDXShaderParameter* pParamOut);

	//! sets the value of scalar and vector parameters.
	//! Be careful about "parameter shadowing".

typedef CEHResult (*iCEDXVertexShader_setParameter1f) (ICEDXVertexShader* iDXVertexShader, CEDXShaderParameter param, float x);

	//! sets the value of scalar and vector parameters.
	//! Be careful about "parameter shadowing".

typedef CEHResult (*iCEDXVertexShader_setParameter4f) (ICEDXVertexShader* iDXVertexShader, CEDXShaderParameter param, float x, float y, float z, float w);

	//! sets the value of matrix parameters.
	//! Be careful about "parameter shadowing".

typedef CEHResult (*iCEDXVertexShader_setMatrixParameter) (ICEDXVertexShader* iDXVertexShader, CEDXShaderParameter param, CEDXMatrix44* pMat);

	//! Sets the texture for a particular sampler parameter.
	//! Be careful about "parameter shadowing".
	//!
	//! \param[in] param	A parameter handle, the parameter whose values are to be set. It must be a sampler.

typedef CEHResult (*iCEDXVertexShader_setTexture) (ICEDXVertexShader* iDXVertexShader, CEDXShaderParameter param, struct ICEDXTexture* pTex);

	//! Sets the state associated with a particular sampler parameter.
	//! Be careful about "parameter shadowing".
	//!
	//! \param[in] param	A parameter handle, the parameter whose values are to be set. It must be a sampler.

typedef CEHResult (*iCEDXVertexShader_setSamplerState) (ICEDXVertexShader* iDXVertexShader, CEDXShaderParameter param, eCEDXSamplerStateType state, UINT32 value);

/*!
 * V-table of ICEDXVertexShader
 */
struct ICEDXVertexShader_vtbl
{
	iCEDXVertexShader_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXVertexShader_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXVertexShader_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXVertexShader_getNamedParameter	_getNamedParameter;
	iCEDXVertexShader_setParameter1f	_setParameter1f;
	iCEDXVertexShader_setParameter4f	_setParameter4f;
	iCEDXVertexShader_setMatrixParameter	_setMatrixParameter;
	iCEDXVertexShader_setTexture	_setTexture;
	iCEDXVertexShader_setSamplerState	_setSamplerState;
};

/*! @}
 * end of ICEDXVertexShader
 */

/*! \defgroup ICEDXPixelShader ICEDXPixelShader
 * @{
 */

/*!
 * ID of ICEDXPixelShader
 */
#define CEComIID_ICEDXPixelShader 0xfbff9bb8


/*!
 * ICEDXPixelShader
 */
typedef struct ICEDXPixelShader
{
	const struct ICEDXPixelShader_vtbl* _vtbl;
} ICEDXPixelShader;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXPixelShader	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXPixelShader_queryInterface) (ICEDXPixelShader* iDXPixelShader, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXPixelShader_addRef) (ICEDXPixelShader* iDXPixelShader);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXPixelShader_release) (ICEDXPixelShader* iDXPixelShader);

	//! finds a named parameter.

typedef CEHResult (*iCEDXPixelShader_getNamedParameter) (ICEDXPixelShader* iDXPixelShader, const char* pName, CEDXShaderParameter* pParamOut);

	//! sets the value of scalar and vector parameters.
	//! Be careful about "parameter shadowing".

typedef CEHResult (*iCEDXPixelShader_setParameter1f) (ICEDXPixelShader* iDXPixelShader, CEDXShaderParameter param, float x);

	//! sets the value of scalar and vector parameters.
	//! Be careful about "parameter shadowing".

typedef CEHResult (*iCEDXPixelShader_setParameter4f) (ICEDXPixelShader* iDXPixelShader, CEDXShaderParameter param, float x, float y, float z, float w);

	//! sets the value of matrix parameters.
	//! Be careful about "parameter shadowing".

typedef CEHResult (*iCEDXPixelShader_setMatrixParameter) (ICEDXPixelShader* iDXPixelShader, CEDXShaderParameter param, CEDXMatrix44* pMat);

	//! Sets the texture for a particular sampler parameter.
	//! Be careful about "parameter shadowing".
	//!
	//! \param[in] param	A parameter handle, the parameter whose values are to be set. It must be a sampler.

typedef CEHResult (*iCEDXPixelShader_setTexture) (ICEDXPixelShader* iDXPixelShader, CEDXShaderParameter param, struct ICEDXTexture* pTex);

	//! Sets the state associated with a particular sampler parameter.
	//! Be careful about "parameter shadowing".
	//!
	//! \param[in] param	A parameter handle, the parameter whose values are to be set. It must be a sampler.

typedef CEHResult (*iCEDXPixelShader_setSamplerState) (ICEDXPixelShader* iDXPixelShader, CEDXShaderParameter param, eCEDXSamplerStateType state, UINT32 value);

/*!
 * V-table of ICEDXPixelShader
 */
struct ICEDXPixelShader_vtbl
{
	iCEDXPixelShader_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXPixelShader_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXPixelShader_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXPixelShader_getNamedParameter	_getNamedParameter;
	iCEDXPixelShader_setParameter1f	_setParameter1f;
	iCEDXPixelShader_setParameter4f	_setParameter4f;
	iCEDXPixelShader_setMatrixParameter	_setMatrixParameter;
	iCEDXPixelShader_setTexture	_setTexture;
	iCEDXPixelShader_setSamplerState	_setSamplerState;
};

/*! @}
 * end of ICEDXPixelShader
 */

/*! \defgroup ICEDXVertexDeclaration ICEDXVertexDeclaration
 * @{
 */

/*!
 * ID of ICEDXVertexDeclaration
 */
#define CEComIID_ICEDXVertexDeclaration 0x813f8dd6


/*!
 * ICEDXVertexDeclaration
 */
typedef struct ICEDXVertexDeclaration
{
	const struct ICEDXVertexDeclaration_vtbl* _vtbl;
} ICEDXVertexDeclaration;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXVertexDeclaration	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXVertexDeclaration_queryInterface) (ICEDXVertexDeclaration* iDXVertexDeclaration, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXVertexDeclaration_addRef) (ICEDXVertexDeclaration* iDXVertexDeclaration);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXVertexDeclaration_release) (ICEDXVertexDeclaration* iDXVertexDeclaration);

/*!
 * V-table of ICEDXVertexDeclaration
 */
struct ICEDXVertexDeclaration_vtbl
{
	iCEDXVertexDeclaration_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXVertexDeclaration_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXVertexDeclaration_release	_release;	//!< Decrease the reference count of the specified interface pointer.
};

/*! @}
 * end of ICEDXVertexDeclaration
 */

/*! \defgroup ICEDXConfig ICEDXConfig
 * @{
 */

/*!
 * ID of ICEDXConfig
 */
#define CEComIID_ICEDXConfig 0xc68920b8


/*!
 * ICEDXConfig
 */
typedef struct ICEDXConfig
{
	const struct ICEDXConfig_vtbl* _vtbl;
} ICEDXConfig;


/*! 
 * <b>Summary:</b>
 * Query interface.<br>
 * 
 * \param[in]	iDXConfig	Specifies interface pointer
 * \param[in]	iId	Specifies interface ID
 * \param[out]	iOut	Recieves interface of result
 *
 * \return Error status
 * 
 * <b>Description:</b>
 * - This function gets an address of the interface if it gets the interface.
 * - The list of available interface ID is in the list of CEApiIID. (All interfaces have their own unique IID). 
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */
typedef CEHResult (*iCEDXConfig_queryInterface) (ICEDXConfig* iDXConfig, const UINT32 iId, void* *const iOut);


/*!
 * <b>Summary:</b>
 * Increase the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function increases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes to be referred.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXConfig_addRef) (ICEDXConfig* iDXConfig);


/*!
 * <b>Summary:</b>
 * Decrease the reference count of the specified interface pointer.<br>
 * 
 * <b>Description:</b>
 * - This function decreases the reference count of the specified interface pointer.
 * - It should be called when the instance of the specified interface pointer becomes not to be referred.
 * - The instance of the interface pointer is removed when the reference count becomes zero.
 *
 * <b>Precondition:</b>
 * None
 *
 * <b>Limitation:</b>
 * None
 *
 * <b>Example:</b>
 * None
 *
 * <b>See also:</b>
 * None
 * 
 *
 */

typedef void (*iCEDXConfig_release) (ICEDXConfig* iDXConfig);

typedef CEHResult (*iCEDXConfig_getContext2DType) (ICEDXConfig* iDXConfig, struct ICESymbol* *const pISymbolDX2DOut);

typedef CEHResult (*iCEDXConfig_visualizeDirtyRegion) (ICEDXConfig* iDXConfig, bool* pOut);

typedef CEHResult (*iCEDXConfig_getLevelCorrespondLayerIdx) (ICEDXConfig* iDXConfig, eCEDpyWindowLevel level, UINT32 *const pIdxOut);

typedef CEHResult (*iCEDXConfig_getHWPlaneEmulationConfigs) (ICEDXConfig* iDXConfig, UINT32 *const pPlanesCountOut, CEPixelFormat* *const pixelFormatsOut);

/*!
 * V-table of ICEDXConfig
 */
struct ICEDXConfig_vtbl
{
	iCEDXConfig_queryInterface	_queryInterface;	//!< Query interface.
	iCEDXConfig_addRef	_addRef;	//!< Increase the reference count of the specified interface pointer.
	iCEDXConfig_release	_release;	//!< Decrease the reference count of the specified interface pointer.
	iCEDXConfig_getContext2DType	_getContext2DType;
	iCEDXConfig_visualizeDirtyRegion	_visualizeDirtyRegion;
	iCEDXConfig_getLevelCorrespondLayerIdx	_getLevelCorrespondLayerIdx;
	iCEDXConfig_getHWPlaneEmulationConfigs	_getHWPlaneEmulationConfigs;
};

/*! @}
 * end of ICEDXConfig
 */

#ifdef __cplusplus
} // end of extern "C"
#endif

//----------------------------------------------------------------
// Interface Wrappers
//----------------------------------------------------------------

#ifdef __cplusplus

/*! \defgroup CEComICEDXRef CEComICEDXRef
 * @{
 */

class CEComICEDXRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXRef() : _iDX(0) {}
	CEComICEDXRef(ICEDX* iOther) : _iDX(0)
	{
		if (iOther)
		{
			_iDX = iOther;
			_iDX->_vtbl->_addRef(_iDX);
		}
	}

	CEComICEDXRef(const CEComICEDXRef& other) : _iDX(0)
	{
		if (other._iDX)
		{
			_iDX = other._iDX;
			_iDX->_vtbl->_addRef(_iDX);
		}
	}

	~CEComICEDXRef()
	{
		if (_iDX)
		{
			ICEDX* tmp = _iDX;
			_iDX = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDX* iDX;
				void* _ptr;
			} uIntf;
			uIntf.iDX = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDX, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDX)
				{
					ICEDX* tmp = _iDX;
					_iDX = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDX = uIntf.iDX;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDX* object() const	{ return _iDX; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDX* iOther)
	{
		if (_iDX)
		{
			ICEDX* tmp = _iDX;
			_iDX = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDX = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDX* detach()
	{
		ICEDX* iIntf = _iDX;
		_iDX = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDX* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDX;
		if (_iDX)
		{
			_iDX->_vtbl->_addRef(_iDX);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDX*() const	{ return _iDX; }
	FORCEINLINE_WITHOUT_DEBUG ICEDX& operator*() const	{ return *_iDX; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDX == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDX* iOther) const	{ return (_iDX != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDX* iOther) const	{ return (_iDX == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXRef& operator=(const CEComICEDXRef& other)	{ return operator=(other._iDX); }

	CEComICEDXRef& operator=(const ICEDX* iOther)
	{
		if (_iDX != iOther)
		{
			if (_iDX)
			{
				ICEDX* tmp = _iDX;
				_iDX = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDX = const_cast<ICEDX*>(iOther);
			if (_iDX)
			{
				_iDX->_vtbl->_addRef(_iDX);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDX** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDX && "has a com object reference. clear first.");
		return & _iDX; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXRef& other)
	{
		bool result = false;
		if (_iDX)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDX);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDX ? _iDX->_vtbl->_queryInterface(_iDX, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDX) { _iDX->_vtbl->_addRef(_iDX); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDX) { _iDX->_vtbl->_release(_iDX); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createDevice(struct ICEDpyFrameWindow* iWindow, struct ICEDXDevice* *const iGLDeviceOut)	{ return _iDX ? _iDX->_vtbl->_createDevice(_iDX, iWindow, iGLDeviceOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDX* _iDX;
};

/*! @}
 * end of CEComICEDXRef
 */

/*! \defgroup CEComICEDXDeviceRef CEComICEDXDeviceRef
 * @{
 */

class CEComICEDXDeviceRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXDeviceRef() : _iDXDevice(0) {}
	CEComICEDXDeviceRef(ICEDXDevice* iOther) : _iDXDevice(0)
	{
		if (iOther)
		{
			_iDXDevice = iOther;
			_iDXDevice->_vtbl->_addRef(_iDXDevice);
		}
	}

	CEComICEDXDeviceRef(const CEComICEDXDeviceRef& other) : _iDXDevice(0)
	{
		if (other._iDXDevice)
		{
			_iDXDevice = other._iDXDevice;
			_iDXDevice->_vtbl->_addRef(_iDXDevice);
		}
	}

	~CEComICEDXDeviceRef()
	{
		if (_iDXDevice)
		{
			ICEDXDevice* tmp = _iDXDevice;
			_iDXDevice = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXDevice* iDXDevice;
				void* _ptr;
			} uIntf;
			uIntf.iDXDevice = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXDevice, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXDevice)
				{
					ICEDXDevice* tmp = _iDXDevice;
					_iDXDevice = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXDevice = uIntf.iDXDevice;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXDevice* object() const	{ return _iDXDevice; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXDevice* iOther)
	{
		if (_iDXDevice)
		{
			ICEDXDevice* tmp = _iDXDevice;
			_iDXDevice = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXDevice = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXDevice* detach()
	{
		ICEDXDevice* iIntf = _iDXDevice;
		_iDXDevice = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXDevice* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXDevice;
		if (_iDXDevice)
		{
			_iDXDevice->_vtbl->_addRef(_iDXDevice);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXDevice*() const	{ return _iDXDevice; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXDevice& operator*() const	{ return *_iDXDevice; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXDevice == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXDevice* iOther) const	{ return (_iDXDevice != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXDevice* iOther) const	{ return (_iDXDevice == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXDeviceRef& operator=(const CEComICEDXDeviceRef& other)	{ return operator=(other._iDXDevice); }

	CEComICEDXDeviceRef& operator=(const ICEDXDevice* iOther)
	{
		if (_iDXDevice != iOther)
		{
			if (_iDXDevice)
			{
				ICEDXDevice* tmp = _iDXDevice;
				_iDXDevice = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXDevice = const_cast<ICEDXDevice*>(iOther);
			if (_iDXDevice)
			{
				_iDXDevice->_vtbl->_addRef(_iDXDevice);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXDevice** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXDeviceRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXDevice && "has a com object reference. clear first.");
		return & _iDXDevice; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXDeviceRef& other)
	{
		bool result = false;
		if (_iDXDevice)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXDevice);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_queryInterface(_iDXDevice, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXDevice) { _iDXDevice->_vtbl->_addRef(_iDXDevice); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXDevice) { _iDXDevice->_vtbl->_release(_iDXDevice); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createVertexBuffer(UINT32 length, struct ICEDXVertexBuffer* *const iBufferOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_createVertexBuffer(_iDXDevice, length, iBufferOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setStreamSource(UINT32 streamNumber, struct ICEDXVertexBuffer* iVertexBuffer, UINT32 stride)	{ return _iDXDevice ? _iDXDevice->_vtbl->_setStreamSource(_iDXDevice, streamNumber, iVertexBuffer, stride) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStreamSource(UINT32 streamNumber, struct ICEDXVertexBuffer* *const iVertexBufferOut, UINT32* pStrideOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_getStreamSource(_iDXDevice, streamNumber, iVertexBufferOut, pStrideOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setStreamSourceFreq(UINT32 streamNumber, UINT32 param)	{ return _iDXDevice ? _iDXDevice->_vtbl->_setStreamSourceFreq(_iDXDevice, streamNumber, param) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getStreamSourceFreq(UINT32 streamNumber, UINT32* pParamOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_getStreamSourceFreq(_iDXDevice, streamNumber, pParamOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createVertexDeclaration(const CEDXVertexElement* pVtxElms, struct ICEDXVertexDeclaration* *const piVDOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_createVertexDeclaration(_iDXDevice, pVtxElms, piVDOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setVertexDeclaration(struct ICEDXVertexDeclaration* iVD)	{ return _iDXDevice ? _iDXDevice->_vtbl->_setVertexDeclaration(_iDXDevice, iVD) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getVertexDeclaration(struct ICEDXVertexDeclaration* *const piVDOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_getVertexDeclaration(_iDXDevice, piVDOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createIndexBuffer(UINT32 length, eCEDXIndexFormat format, struct ICEDXIndexBuffer* *const piBufferOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_createIndexBuffer(_iDXDevice, length, format, piBufferOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setIndices(struct ICEDXIndexBuffer* pIndexBuf)	{ return _iDXDevice ? _iDXDevice->_vtbl->_setIndices(_iDXDevice, pIndexBuf) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getIndices(struct ICEDXIndexBuffer* *const piIndexBufOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_getIndices(_iDXDevice, piIndexBufOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createTexture(CEPixelFormat format, UINT32 width, UINT32 height, eCEDXPoolType pool, eCEDXUsage usage, struct ICEDXTexture* *const iTextureOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_createTexture(_iDXDevice, format, width, height, pool, usage, iTextureOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createTextureFromTexture(struct ICEDXTexture* iBaseTexture, struct ICEDXTexture* *const iTextureOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_createTextureFromTexture(_iDXDevice, iBaseTexture, iTextureOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getBackBuffer(struct ICEDXSurface* *const ppBBuf)	{ return _iDXDevice ? _iDXDevice->_vtbl->_getBackBuffer(_iDXDevice, ppBBuf) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getRenderTarget(struct ICEDXSurface* *const piRT)	{ return _iDXDevice ? _iDXDevice->_vtbl->_getRenderTarget(_iDXDevice, piRT) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult colorFill(struct ICEDXSurface* iSurface, const CERect* pRect, CERGBAColor color)	{ return _iDXDevice ? _iDXDevice->_vtbl->_colorFill(_iDXDevice, iSurface, pRect, color) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult updateSurface(struct ICEDXSurface* iSrcSurface, const CERect* pSrcRect, struct ICEDXSurface* iDstSurface, const CEPointBase* pDstPt)	{ return _iDXDevice ? _iDXDevice->_vtbl->_updateSurface(_iDXDevice, iSrcSurface, pSrcRect, iDstSurface, pDstPt) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLayerCount(UINT32 *const pCountOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_getLayerCount(_iDXDevice, pCountOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLevelCorrespondLayerIdx(eCEDpyWindowLevel level, UINT32 *const pIdxOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_getLevelCorrespondLayerIdx(_iDXDevice, level, pIdxOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLayerCorrespondBackBuffer(UINT32 layerIdx, struct ICEDXSurface* *const ppBufOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_getLayerCorrespondBackBuffer(_iDXDevice, layerIdx, ppBufOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setPaletteToLayer(UINT32 layerIdx, const struct ICEDXPalette* iPalette)	{ return _iDXDevice ? _iDXDevice->_vtbl->_setPaletteToLayer(_iDXDevice, layerIdx, iPalette) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setTransparencyToLayer(UINT32 layerIdx, const float trans)	{ return _iDXDevice ? _iDXDevice->_vtbl->_setTransparencyToLayer(_iDXDevice, layerIdx, trans) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setClearIndexToLayer(UINT32 layerIdx, UINT8 clearIdx)	{ return _iDXDevice ? _iDXDevice->_vtbl->_setClearIndexToLayer(_iDXDevice, layerIdx, clearIdx) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getClearIndexOfLayer(UINT32 layerIdx, UINT8 *const pClearIdxOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_getClearIndexOfLayer(_iDXDevice, layerIdx, pClearIdxOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createPalette(struct ICEDXPalette* *const iPaletteOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_createPalette(_iDXDevice, iPaletteOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult findVertexShader(CEDXShaderID shaderId, struct ICEDXVertexShader* *const piShaderOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_findVertexShader(_iDXDevice, shaderId, piShaderOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult bindVertexShader(struct ICEDXVertexShader* iShader)	{ return _iDXDevice ? _iDXDevice->_vtbl->_bindVertexShader(_iDXDevice, iShader) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult findPixelShader(CEDXShaderID shaderId, struct ICEDXPixelShader* *const piShaderOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_findPixelShader(_iDXDevice, shaderId, piShaderOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult bindPixelShader(struct ICEDXPixelShader* iShader)	{ return _iDXDevice ? _iDXDevice->_vtbl->_bindPixelShader(_iDXDevice, iShader) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult createStateBlock(struct ICEDXStateBlock* *const piSBOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_createStateBlock(_iDXDevice, piSBOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult purgeCachedResources()	{ return _iDXDevice ? _iDXDevice->_vtbl->_purgeCachedResources(_iDXDevice) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setRenderState(eCEDXRenderStateType state, UINT32 value)	{ return _iDXDevice ? _iDXDevice->_vtbl->_setRenderState(_iDXDevice, state, value) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setScissorRect(const CERect* pRect)	{ return _iDXDevice ? _iDXDevice->_vtbl->_setScissorRect(_iDXDevice, pRect) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getScissorRect(CERect* pRectOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_getScissorRect(_iDXDevice, pRectOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawIndexedPrimitive(eCEDXPrimitiveType type, UINT32 minIndex, UINT32 numVertices, UINT32 startIndex, UINT32 numOfPrims)	{ return _iDXDevice ? _iDXDevice->_vtbl->_drawIndexedPrimitive(_iDXDevice, type, minIndex, numVertices, startIndex, numOfPrims) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult drawPrimitive(eCEDXPrimitiveType type, UINT32 startVertex, UINT32 numOfPrims)	{ return _iDXDevice ? _iDXDevice->_vtbl->_drawPrimitive(_iDXDevice, type, startVertex, numOfPrims) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setViewport(const CEDXViewport* pVP)	{ return _iDXDevice ? _iDXDevice->_vtbl->_setViewport(_iDXDevice, pVP) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult beginScene()	{ return _iDXDevice ? _iDXDevice->_vtbl->_beginScene(_iDXDevice) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult endScene()	{ return _iDXDevice ? _iDXDevice->_vtbl->_endScene(_iDXDevice) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult present()	{ return _iDXDevice ? _iDXDevice->_vtbl->_present(_iDXDevice) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult clear()	{ return _iDXDevice ? _iDXDevice->_vtbl->_clear(_iDXDevice) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult paintRectsOfDirtyRegion(struct ICEDXRegion* iRegion)	{ return _iDXDevice ? _iDXDevice->_vtbl->_paintRectsOfDirtyRegion(_iDXDevice, iRegion) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getPixelFormat(CEPixelFormat* const pPxlFmtOut)	{ return _iDXDevice ? _iDXDevice->_vtbl->_getPixelFormat(_iDXDevice, pPxlFmtOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXDevice* _iDXDevice;
};

/*! @}
 * end of CEComICEDXDeviceRef
 */

/*! \defgroup CEComICEDXResourceRef CEComICEDXResourceRef
 * @{
 */

class CEComICEDXResourceRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXResourceRef() : _iDXResource(0) {}
	CEComICEDXResourceRef(ICEDXResource* iOther) : _iDXResource(0)
	{
		if (iOther)
		{
			_iDXResource = iOther;
			_iDXResource->_vtbl->_addRef(_iDXResource);
		}
	}

	CEComICEDXResourceRef(const CEComICEDXResourceRef& other) : _iDXResource(0)
	{
		if (other._iDXResource)
		{
			_iDXResource = other._iDXResource;
			_iDXResource->_vtbl->_addRef(_iDXResource);
		}
	}

	~CEComICEDXResourceRef()
	{
		if (_iDXResource)
		{
			ICEDXResource* tmp = _iDXResource;
			_iDXResource = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXResource* iDXResource;
				void* _ptr;
			} uIntf;
			uIntf.iDXResource = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXResource, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXResource)
				{
					ICEDXResource* tmp = _iDXResource;
					_iDXResource = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXResource = uIntf.iDXResource;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXResource* object() const	{ return _iDXResource; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXResource* iOther)
	{
		if (_iDXResource)
		{
			ICEDXResource* tmp = _iDXResource;
			_iDXResource = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXResource = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXResource* detach()
	{
		ICEDXResource* iIntf = _iDXResource;
		_iDXResource = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXResource* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXResource;
		if (_iDXResource)
		{
			_iDXResource->_vtbl->_addRef(_iDXResource);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXResource*() const	{ return _iDXResource; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXResource& operator*() const	{ return *_iDXResource; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXResource == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXResource* iOther) const	{ return (_iDXResource != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXResource* iOther) const	{ return (_iDXResource == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXResourceRef& operator=(const CEComICEDXResourceRef& other)	{ return operator=(other._iDXResource); }

	CEComICEDXResourceRef& operator=(const ICEDXResource* iOther)
	{
		if (_iDXResource != iOther)
		{
			if (_iDXResource)
			{
				ICEDXResource* tmp = _iDXResource;
				_iDXResource = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXResource = const_cast<ICEDXResource*>(iOther);
			if (_iDXResource)
			{
				_iDXResource->_vtbl->_addRef(_iDXResource);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXResource** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXResourceRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXResource && "has a com object reference. clear first.");
		return & _iDXResource; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXResourceRef& other)
	{
		bool result = false;
		if (_iDXResource)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXResource);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXResource ? _iDXResource->_vtbl->_queryInterface(_iDXResource, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXResource) { _iDXResource->_vtbl->_addRef(_iDXResource); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXResource) { _iDXResource->_vtbl->_release(_iDXResource); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDevice(struct ICEDXDevice* *const iDeviceOut)	{ return _iDXResource ? _iDXResource->_vtbl->_getDevice(_iDXResource, iDeviceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getType(eCEDXResourceType *const typeOut)	{ return _iDXResource ? _iDXResource->_vtbl->_getType(_iDXResource, typeOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXResource* _iDXResource;
};

/*! @}
 * end of CEComICEDXResourceRef
 */

/*! \defgroup CEComICEDXVertexBufferRef CEComICEDXVertexBufferRef
 * @{
 */

class CEComICEDXVertexBufferRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXVertexBufferRef() : _iDXVertexBuffer(0) {}
	CEComICEDXVertexBufferRef(ICEDXVertexBuffer* iOther) : _iDXVertexBuffer(0)
	{
		if (iOther)
		{
			_iDXVertexBuffer = iOther;
			_iDXVertexBuffer->_vtbl->_addRef(_iDXVertexBuffer);
		}
	}

	CEComICEDXVertexBufferRef(const CEComICEDXVertexBufferRef& other) : _iDXVertexBuffer(0)
	{
		if (other._iDXVertexBuffer)
		{
			_iDXVertexBuffer = other._iDXVertexBuffer;
			_iDXVertexBuffer->_vtbl->_addRef(_iDXVertexBuffer);
		}
	}

	~CEComICEDXVertexBufferRef()
	{
		if (_iDXVertexBuffer)
		{
			ICEDXVertexBuffer* tmp = _iDXVertexBuffer;
			_iDXVertexBuffer = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXVertexBuffer* iDXVertexBuffer;
				void* _ptr;
			} uIntf;
			uIntf.iDXVertexBuffer = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXVertexBuffer, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXVertexBuffer)
				{
					ICEDXVertexBuffer* tmp = _iDXVertexBuffer;
					_iDXVertexBuffer = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXVertexBuffer = uIntf.iDXVertexBuffer;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXVertexBuffer* object() const	{ return _iDXVertexBuffer; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXVertexBuffer* iOther)
	{
		if (_iDXVertexBuffer)
		{
			ICEDXVertexBuffer* tmp = _iDXVertexBuffer;
			_iDXVertexBuffer = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXVertexBuffer = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXVertexBuffer* detach()
	{
		ICEDXVertexBuffer* iIntf = _iDXVertexBuffer;
		_iDXVertexBuffer = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXVertexBuffer* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXVertexBuffer;
		if (_iDXVertexBuffer)
		{
			_iDXVertexBuffer->_vtbl->_addRef(_iDXVertexBuffer);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXVertexBuffer*() const	{ return _iDXVertexBuffer; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXVertexBuffer& operator*() const	{ return *_iDXVertexBuffer; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXVertexBuffer == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXVertexBuffer* iOther) const	{ return (_iDXVertexBuffer != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXVertexBuffer* iOther) const	{ return (_iDXVertexBuffer == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXVertexBufferRef& operator=(const CEComICEDXVertexBufferRef& other)	{ return operator=(other._iDXVertexBuffer); }

	CEComICEDXVertexBufferRef& operator=(const ICEDXVertexBuffer* iOther)
	{
		if (_iDXVertexBuffer != iOther)
		{
			if (_iDXVertexBuffer)
			{
				ICEDXVertexBuffer* tmp = _iDXVertexBuffer;
				_iDXVertexBuffer = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXVertexBuffer = const_cast<ICEDXVertexBuffer*>(iOther);
			if (_iDXVertexBuffer)
			{
				_iDXVertexBuffer->_vtbl->_addRef(_iDXVertexBuffer);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXVertexBuffer** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXVertexBufferRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXVertexBuffer && "has a com object reference. clear first.");
		return & _iDXVertexBuffer; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXVertexBufferRef& other)
	{
		bool result = false;
		if (_iDXVertexBuffer)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXVertexBuffer);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXVertexBuffer ? _iDXVertexBuffer->_vtbl->_queryInterface(_iDXVertexBuffer, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXVertexBuffer) { _iDXVertexBuffer->_vtbl->_addRef(_iDXVertexBuffer); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXVertexBuffer) { _iDXVertexBuffer->_vtbl->_release(_iDXVertexBuffer); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDevice(struct ICEDXDevice* *const iDeviceOut)	{ return _iDXVertexBuffer ? _iDXVertexBuffer->_vtbl->_getDevice(_iDXVertexBuffer, iDeviceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getType(eCEDXResourceType *const typeOut)	{ return _iDXVertexBuffer ? _iDXVertexBuffer->_vtbl->_getType(_iDXVertexBuffer, typeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult lock(void* *const pDataOut)	{ return _iDXVertexBuffer ? _iDXVertexBuffer->_vtbl->_lock(_iDXVertexBuffer, pDataOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unlock()	{ return _iDXVertexBuffer ? _iDXVertexBuffer->_vtbl->_unlock(_iDXVertexBuffer) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXVertexBuffer* _iDXVertexBuffer;
};

/*! @}
 * end of CEComICEDXVertexBufferRef
 */

/*! \defgroup CEComICEDXIndexBufferRef CEComICEDXIndexBufferRef
 * @{
 */

class CEComICEDXIndexBufferRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXIndexBufferRef() : _iDXIndexBuffer(0) {}
	CEComICEDXIndexBufferRef(ICEDXIndexBuffer* iOther) : _iDXIndexBuffer(0)
	{
		if (iOther)
		{
			_iDXIndexBuffer = iOther;
			_iDXIndexBuffer->_vtbl->_addRef(_iDXIndexBuffer);
		}
	}

	CEComICEDXIndexBufferRef(const CEComICEDXIndexBufferRef& other) : _iDXIndexBuffer(0)
	{
		if (other._iDXIndexBuffer)
		{
			_iDXIndexBuffer = other._iDXIndexBuffer;
			_iDXIndexBuffer->_vtbl->_addRef(_iDXIndexBuffer);
		}
	}

	~CEComICEDXIndexBufferRef()
	{
		if (_iDXIndexBuffer)
		{
			ICEDXIndexBuffer* tmp = _iDXIndexBuffer;
			_iDXIndexBuffer = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXIndexBuffer* iDXIndexBuffer;
				void* _ptr;
			} uIntf;
			uIntf.iDXIndexBuffer = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXIndexBuffer, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXIndexBuffer)
				{
					ICEDXIndexBuffer* tmp = _iDXIndexBuffer;
					_iDXIndexBuffer = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXIndexBuffer = uIntf.iDXIndexBuffer;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXIndexBuffer* object() const	{ return _iDXIndexBuffer; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXIndexBuffer* iOther)
	{
		if (_iDXIndexBuffer)
		{
			ICEDXIndexBuffer* tmp = _iDXIndexBuffer;
			_iDXIndexBuffer = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXIndexBuffer = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXIndexBuffer* detach()
	{
		ICEDXIndexBuffer* iIntf = _iDXIndexBuffer;
		_iDXIndexBuffer = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXIndexBuffer* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXIndexBuffer;
		if (_iDXIndexBuffer)
		{
			_iDXIndexBuffer->_vtbl->_addRef(_iDXIndexBuffer);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXIndexBuffer*() const	{ return _iDXIndexBuffer; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXIndexBuffer& operator*() const	{ return *_iDXIndexBuffer; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXIndexBuffer == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXIndexBuffer* iOther) const	{ return (_iDXIndexBuffer != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXIndexBuffer* iOther) const	{ return (_iDXIndexBuffer == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXIndexBufferRef& operator=(const CEComICEDXIndexBufferRef& other)	{ return operator=(other._iDXIndexBuffer); }

	CEComICEDXIndexBufferRef& operator=(const ICEDXIndexBuffer* iOther)
	{
		if (_iDXIndexBuffer != iOther)
		{
			if (_iDXIndexBuffer)
			{
				ICEDXIndexBuffer* tmp = _iDXIndexBuffer;
				_iDXIndexBuffer = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXIndexBuffer = const_cast<ICEDXIndexBuffer*>(iOther);
			if (_iDXIndexBuffer)
			{
				_iDXIndexBuffer->_vtbl->_addRef(_iDXIndexBuffer);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXIndexBuffer** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXIndexBufferRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXIndexBuffer && "has a com object reference. clear first.");
		return & _iDXIndexBuffer; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXIndexBufferRef& other)
	{
		bool result = false;
		if (_iDXIndexBuffer)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXIndexBuffer);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXIndexBuffer ? _iDXIndexBuffer->_vtbl->_queryInterface(_iDXIndexBuffer, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXIndexBuffer) { _iDXIndexBuffer->_vtbl->_addRef(_iDXIndexBuffer); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXIndexBuffer) { _iDXIndexBuffer->_vtbl->_release(_iDXIndexBuffer); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDevice(struct ICEDXDevice* *const iDeviceOut)	{ return _iDXIndexBuffer ? _iDXIndexBuffer->_vtbl->_getDevice(_iDXIndexBuffer, iDeviceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getType(eCEDXResourceType *const typeOut)	{ return _iDXIndexBuffer ? _iDXIndexBuffer->_vtbl->_getType(_iDXIndexBuffer, typeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult lock(void* *const pDataOut)	{ return _iDXIndexBuffer ? _iDXIndexBuffer->_vtbl->_lock(_iDXIndexBuffer, pDataOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unlock()	{ return _iDXIndexBuffer ? _iDXIndexBuffer->_vtbl->_unlock(_iDXIndexBuffer) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXIndexBuffer* _iDXIndexBuffer;
};

/*! @}
 * end of CEComICEDXIndexBufferRef
 */

/*! \defgroup CEComICEDXTextureRef CEComICEDXTextureRef
 * @{
 */

class CEComICEDXTextureRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXTextureRef() : _iDXTexture(0) {}
	CEComICEDXTextureRef(ICEDXTexture* iOther) : _iDXTexture(0)
	{
		if (iOther)
		{
			_iDXTexture = iOther;
			_iDXTexture->_vtbl->_addRef(_iDXTexture);
		}
	}

	CEComICEDXTextureRef(const CEComICEDXTextureRef& other) : _iDXTexture(0)
	{
		if (other._iDXTexture)
		{
			_iDXTexture = other._iDXTexture;
			_iDXTexture->_vtbl->_addRef(_iDXTexture);
		}
	}

	~CEComICEDXTextureRef()
	{
		if (_iDXTexture)
		{
			ICEDXTexture* tmp = _iDXTexture;
			_iDXTexture = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXTexture* iDXTexture;
				void* _ptr;
			} uIntf;
			uIntf.iDXTexture = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXTexture, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXTexture)
				{
					ICEDXTexture* tmp = _iDXTexture;
					_iDXTexture = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXTexture = uIntf.iDXTexture;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXTexture* object() const	{ return _iDXTexture; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXTexture* iOther)
	{
		if (_iDXTexture)
		{
			ICEDXTexture* tmp = _iDXTexture;
			_iDXTexture = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXTexture = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXTexture* detach()
	{
		ICEDXTexture* iIntf = _iDXTexture;
		_iDXTexture = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXTexture* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXTexture;
		if (_iDXTexture)
		{
			_iDXTexture->_vtbl->_addRef(_iDXTexture);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXTexture*() const	{ return _iDXTexture; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXTexture& operator*() const	{ return *_iDXTexture; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXTexture == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXTexture* iOther) const	{ return (_iDXTexture != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXTexture* iOther) const	{ return (_iDXTexture == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXTextureRef& operator=(const CEComICEDXTextureRef& other)	{ return operator=(other._iDXTexture); }

	CEComICEDXTextureRef& operator=(const ICEDXTexture* iOther)
	{
		if (_iDXTexture != iOther)
		{
			if (_iDXTexture)
			{
				ICEDXTexture* tmp = _iDXTexture;
				_iDXTexture = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXTexture = const_cast<ICEDXTexture*>(iOther);
			if (_iDXTexture)
			{
				_iDXTexture->_vtbl->_addRef(_iDXTexture);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXTexture** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXTextureRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXTexture && "has a com object reference. clear first.");
		return & _iDXTexture; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXTextureRef& other)
	{
		bool result = false;
		if (_iDXTexture)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXTexture);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXTexture ? _iDXTexture->_vtbl->_queryInterface(_iDXTexture, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXTexture) { _iDXTexture->_vtbl->_addRef(_iDXTexture); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXTexture) { _iDXTexture->_vtbl->_release(_iDXTexture); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDevice(struct ICEDXDevice* *const iDeviceOut)	{ return _iDXTexture ? _iDXTexture->_vtbl->_getDevice(_iDXTexture, iDeviceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getType(eCEDXResourceType *const typeOut)	{ return _iDXTexture ? _iDXTexture->_vtbl->_getType(_iDXTexture, typeOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLevelDesc(UINT32 level, struct CEDXSurfaceDesc *const descOut)	{ return _iDXTexture ? _iDXTexture->_vtbl->_getLevelDesc(_iDXTexture, level, descOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getSurfaceLevel(UINT32 level, struct ICEDXSurface* *const iSurfaceOut)	{ return _iDXTexture ? _iDXTexture->_vtbl->_getSurfaceLevel(_iDXTexture, level, iSurfaceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult lockRect(UINT32 level, const CERect* rect, UINT32 flags, struct CEDXLockedRect *const lockedRectOut)	{ return _iDXTexture ? _iDXTexture->_vtbl->_lockRect(_iDXTexture, level, rect, flags, lockedRectOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult unlockRect(UINT32 level)	{ return _iDXTexture ? _iDXTexture->_vtbl->_unlockRect(_iDXTexture, level) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXTexture* _iDXTexture;
};

/*! @}
 * end of CEComICEDXTextureRef
 */

/*! \defgroup CEComICEDXStateBlockRef CEComICEDXStateBlockRef
 * @{
 */

class CEComICEDXStateBlockRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXStateBlockRef() : _iDXStateBlock(0) {}
	CEComICEDXStateBlockRef(ICEDXStateBlock* iOther) : _iDXStateBlock(0)
	{
		if (iOther)
		{
			_iDXStateBlock = iOther;
			_iDXStateBlock->_vtbl->_addRef(_iDXStateBlock);
		}
	}

	CEComICEDXStateBlockRef(const CEComICEDXStateBlockRef& other) : _iDXStateBlock(0)
	{
		if (other._iDXStateBlock)
		{
			_iDXStateBlock = other._iDXStateBlock;
			_iDXStateBlock->_vtbl->_addRef(_iDXStateBlock);
		}
	}

	~CEComICEDXStateBlockRef()
	{
		if (_iDXStateBlock)
		{
			ICEDXStateBlock* tmp = _iDXStateBlock;
			_iDXStateBlock = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXStateBlock* iDXStateBlock;
				void* _ptr;
			} uIntf;
			uIntf.iDXStateBlock = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXStateBlock, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXStateBlock)
				{
					ICEDXStateBlock* tmp = _iDXStateBlock;
					_iDXStateBlock = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXStateBlock = uIntf.iDXStateBlock;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXStateBlock* object() const	{ return _iDXStateBlock; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXStateBlock* iOther)
	{
		if (_iDXStateBlock)
		{
			ICEDXStateBlock* tmp = _iDXStateBlock;
			_iDXStateBlock = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXStateBlock = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXStateBlock* detach()
	{
		ICEDXStateBlock* iIntf = _iDXStateBlock;
		_iDXStateBlock = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXStateBlock* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXStateBlock;
		if (_iDXStateBlock)
		{
			_iDXStateBlock->_vtbl->_addRef(_iDXStateBlock);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXStateBlock*() const	{ return _iDXStateBlock; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXStateBlock& operator*() const	{ return *_iDXStateBlock; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXStateBlock == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXStateBlock* iOther) const	{ return (_iDXStateBlock != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXStateBlock* iOther) const	{ return (_iDXStateBlock == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXStateBlockRef& operator=(const CEComICEDXStateBlockRef& other)	{ return operator=(other._iDXStateBlock); }

	CEComICEDXStateBlockRef& operator=(const ICEDXStateBlock* iOther)
	{
		if (_iDXStateBlock != iOther)
		{
			if (_iDXStateBlock)
			{
				ICEDXStateBlock* tmp = _iDXStateBlock;
				_iDXStateBlock = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXStateBlock = const_cast<ICEDXStateBlock*>(iOther);
			if (_iDXStateBlock)
			{
				_iDXStateBlock->_vtbl->_addRef(_iDXStateBlock);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXStateBlock** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXStateBlockRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXStateBlock && "has a com object reference. clear first.");
		return & _iDXStateBlock; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXStateBlockRef& other)
	{
		bool result = false;
		if (_iDXStateBlock)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXStateBlock);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXStateBlock ? _iDXStateBlock->_vtbl->_queryInterface(_iDXStateBlock, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXStateBlock) { _iDXStateBlock->_vtbl->_addRef(_iDXStateBlock); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXStateBlock) { _iDXStateBlock->_vtbl->_release(_iDXStateBlock); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getDevice(struct ICEDXDevice* *const piDeviceOut)	{ return _iDXStateBlock ? _iDXStateBlock->_vtbl->_getDevice(_iDXStateBlock, piDeviceOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult apply()	{ return _iDXStateBlock ? _iDXStateBlock->_vtbl->_apply(_iDXStateBlock) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult capture()	{ return _iDXStateBlock ? _iDXStateBlock->_vtbl->_capture(_iDXStateBlock) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXStateBlock* _iDXStateBlock;
};

/*! @}
 * end of CEComICEDXStateBlockRef
 */

/*! \defgroup CEComICEDXShaderRef CEComICEDXShaderRef
 * @{
 */

class CEComICEDXShaderRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXShaderRef() : _iDXShader(0) {}
	CEComICEDXShaderRef(ICEDXShader* iOther) : _iDXShader(0)
	{
		if (iOther)
		{
			_iDXShader = iOther;
			_iDXShader->_vtbl->_addRef(_iDXShader);
		}
	}

	CEComICEDXShaderRef(const CEComICEDXShaderRef& other) : _iDXShader(0)
	{
		if (other._iDXShader)
		{
			_iDXShader = other._iDXShader;
			_iDXShader->_vtbl->_addRef(_iDXShader);
		}
	}

	~CEComICEDXShaderRef()
	{
		if (_iDXShader)
		{
			ICEDXShader* tmp = _iDXShader;
			_iDXShader = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXShader* iDXShader;
				void* _ptr;
			} uIntf;
			uIntf.iDXShader = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXShader, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXShader)
				{
					ICEDXShader* tmp = _iDXShader;
					_iDXShader = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXShader = uIntf.iDXShader;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXShader* object() const	{ return _iDXShader; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXShader* iOther)
	{
		if (_iDXShader)
		{
			ICEDXShader* tmp = _iDXShader;
			_iDXShader = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXShader = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXShader* detach()
	{
		ICEDXShader* iIntf = _iDXShader;
		_iDXShader = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXShader* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXShader;
		if (_iDXShader)
		{
			_iDXShader->_vtbl->_addRef(_iDXShader);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXShader*() const	{ return _iDXShader; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXShader& operator*() const	{ return *_iDXShader; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXShader == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXShader* iOther) const	{ return (_iDXShader != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXShader* iOther) const	{ return (_iDXShader == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXShaderRef& operator=(const CEComICEDXShaderRef& other)	{ return operator=(other._iDXShader); }

	CEComICEDXShaderRef& operator=(const ICEDXShader* iOther)
	{
		if (_iDXShader != iOther)
		{
			if (_iDXShader)
			{
				ICEDXShader* tmp = _iDXShader;
				_iDXShader = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXShader = const_cast<ICEDXShader*>(iOther);
			if (_iDXShader)
			{
				_iDXShader->_vtbl->_addRef(_iDXShader);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXShader** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXShaderRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXShader && "has a com object reference. clear first.");
		return & _iDXShader; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXShaderRef& other)
	{
		bool result = false;
		if (_iDXShader)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXShader);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXShader ? _iDXShader->_vtbl->_queryInterface(_iDXShader, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXShader) { _iDXShader->_vtbl->_addRef(_iDXShader); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXShader) { _iDXShader->_vtbl->_release(_iDXShader); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getNamedParameter(const char* pName, CEDXShaderParameter* pParamOut)	{ return _iDXShader ? _iDXShader->_vtbl->_getNamedParameter(_iDXShader, pName, pParamOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setParameter1f(CEDXShaderParameter param, float x)	{ return _iDXShader ? _iDXShader->_vtbl->_setParameter1f(_iDXShader, param, x) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setParameter4f(CEDXShaderParameter param, float x, float y, float z, float w)	{ return _iDXShader ? _iDXShader->_vtbl->_setParameter4f(_iDXShader, param, x, y, z, w) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setMatrixParameter(CEDXShaderParameter param, CEDXMatrix44* pMat)	{ return _iDXShader ? _iDXShader->_vtbl->_setMatrixParameter(_iDXShader, param, pMat) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setTexture(CEDXShaderParameter param, struct ICEDXTexture* pTex)	{ return _iDXShader ? _iDXShader->_vtbl->_setTexture(_iDXShader, param, pTex) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setSamplerState(CEDXShaderParameter param, eCEDXSamplerStateType state, UINT32 value)	{ return _iDXShader ? _iDXShader->_vtbl->_setSamplerState(_iDXShader, param, state, value) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXShader* _iDXShader;
};

/*! @}
 * end of CEComICEDXShaderRef
 */

/*! \defgroup CEComICEDXVertexShaderRef CEComICEDXVertexShaderRef
 * @{
 */

class CEComICEDXVertexShaderRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXVertexShaderRef() : _iDXVertexShader(0) {}
	CEComICEDXVertexShaderRef(ICEDXVertexShader* iOther) : _iDXVertexShader(0)
	{
		if (iOther)
		{
			_iDXVertexShader = iOther;
			_iDXVertexShader->_vtbl->_addRef(_iDXVertexShader);
		}
	}

	CEComICEDXVertexShaderRef(const CEComICEDXVertexShaderRef& other) : _iDXVertexShader(0)
	{
		if (other._iDXVertexShader)
		{
			_iDXVertexShader = other._iDXVertexShader;
			_iDXVertexShader->_vtbl->_addRef(_iDXVertexShader);
		}
	}

	~CEComICEDXVertexShaderRef()
	{
		if (_iDXVertexShader)
		{
			ICEDXVertexShader* tmp = _iDXVertexShader;
			_iDXVertexShader = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXVertexShader* iDXVertexShader;
				void* _ptr;
			} uIntf;
			uIntf.iDXVertexShader = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXVertexShader, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXVertexShader)
				{
					ICEDXVertexShader* tmp = _iDXVertexShader;
					_iDXVertexShader = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXVertexShader = uIntf.iDXVertexShader;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXVertexShader* object() const	{ return _iDXVertexShader; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXVertexShader* iOther)
	{
		if (_iDXVertexShader)
		{
			ICEDXVertexShader* tmp = _iDXVertexShader;
			_iDXVertexShader = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXVertexShader = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXVertexShader* detach()
	{
		ICEDXVertexShader* iIntf = _iDXVertexShader;
		_iDXVertexShader = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXVertexShader* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXVertexShader;
		if (_iDXVertexShader)
		{
			_iDXVertexShader->_vtbl->_addRef(_iDXVertexShader);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXVertexShader*() const	{ return _iDXVertexShader; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXVertexShader& operator*() const	{ return *_iDXVertexShader; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXVertexShader == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXVertexShader* iOther) const	{ return (_iDXVertexShader != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXVertexShader* iOther) const	{ return (_iDXVertexShader == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXVertexShaderRef& operator=(const CEComICEDXVertexShaderRef& other)	{ return operator=(other._iDXVertexShader); }

	CEComICEDXVertexShaderRef& operator=(const ICEDXVertexShader* iOther)
	{
		if (_iDXVertexShader != iOther)
		{
			if (_iDXVertexShader)
			{
				ICEDXVertexShader* tmp = _iDXVertexShader;
				_iDXVertexShader = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXVertexShader = const_cast<ICEDXVertexShader*>(iOther);
			if (_iDXVertexShader)
			{
				_iDXVertexShader->_vtbl->_addRef(_iDXVertexShader);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXVertexShader** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXVertexShaderRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXVertexShader && "has a com object reference. clear first.");
		return & _iDXVertexShader; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXVertexShaderRef& other)
	{
		bool result = false;
		if (_iDXVertexShader)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXVertexShader);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXVertexShader ? _iDXVertexShader->_vtbl->_queryInterface(_iDXVertexShader, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXVertexShader) { _iDXVertexShader->_vtbl->_addRef(_iDXVertexShader); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXVertexShader) { _iDXVertexShader->_vtbl->_release(_iDXVertexShader); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getNamedParameter(const char* pName, CEDXShaderParameter* pParamOut)	{ return _iDXVertexShader ? _iDXVertexShader->_vtbl->_getNamedParameter(_iDXVertexShader, pName, pParamOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setParameter1f(CEDXShaderParameter param, float x)	{ return _iDXVertexShader ? _iDXVertexShader->_vtbl->_setParameter1f(_iDXVertexShader, param, x) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setParameter4f(CEDXShaderParameter param, float x, float y, float z, float w)	{ return _iDXVertexShader ? _iDXVertexShader->_vtbl->_setParameter4f(_iDXVertexShader, param, x, y, z, w) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setMatrixParameter(CEDXShaderParameter param, CEDXMatrix44* pMat)	{ return _iDXVertexShader ? _iDXVertexShader->_vtbl->_setMatrixParameter(_iDXVertexShader, param, pMat) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setTexture(CEDXShaderParameter param, struct ICEDXTexture* pTex)	{ return _iDXVertexShader ? _iDXVertexShader->_vtbl->_setTexture(_iDXVertexShader, param, pTex) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setSamplerState(CEDXShaderParameter param, eCEDXSamplerStateType state, UINT32 value)	{ return _iDXVertexShader ? _iDXVertexShader->_vtbl->_setSamplerState(_iDXVertexShader, param, state, value) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXVertexShader* _iDXVertexShader;
};

/*! @}
 * end of CEComICEDXVertexShaderRef
 */

/*! \defgroup CEComICEDXPixelShaderRef CEComICEDXPixelShaderRef
 * @{
 */

class CEComICEDXPixelShaderRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXPixelShaderRef() : _iDXPixelShader(0) {}
	CEComICEDXPixelShaderRef(ICEDXPixelShader* iOther) : _iDXPixelShader(0)
	{
		if (iOther)
		{
			_iDXPixelShader = iOther;
			_iDXPixelShader->_vtbl->_addRef(_iDXPixelShader);
		}
	}

	CEComICEDXPixelShaderRef(const CEComICEDXPixelShaderRef& other) : _iDXPixelShader(0)
	{
		if (other._iDXPixelShader)
		{
			_iDXPixelShader = other._iDXPixelShader;
			_iDXPixelShader->_vtbl->_addRef(_iDXPixelShader);
		}
	}

	~CEComICEDXPixelShaderRef()
	{
		if (_iDXPixelShader)
		{
			ICEDXPixelShader* tmp = _iDXPixelShader;
			_iDXPixelShader = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXPixelShader* iDXPixelShader;
				void* _ptr;
			} uIntf;
			uIntf.iDXPixelShader = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXPixelShader, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXPixelShader)
				{
					ICEDXPixelShader* tmp = _iDXPixelShader;
					_iDXPixelShader = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXPixelShader = uIntf.iDXPixelShader;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXPixelShader* object() const	{ return _iDXPixelShader; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXPixelShader* iOther)
	{
		if (_iDXPixelShader)
		{
			ICEDXPixelShader* tmp = _iDXPixelShader;
			_iDXPixelShader = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXPixelShader = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXPixelShader* detach()
	{
		ICEDXPixelShader* iIntf = _iDXPixelShader;
		_iDXPixelShader = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXPixelShader* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXPixelShader;
		if (_iDXPixelShader)
		{
			_iDXPixelShader->_vtbl->_addRef(_iDXPixelShader);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXPixelShader*() const	{ return _iDXPixelShader; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXPixelShader& operator*() const	{ return *_iDXPixelShader; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXPixelShader == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXPixelShader* iOther) const	{ return (_iDXPixelShader != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXPixelShader* iOther) const	{ return (_iDXPixelShader == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXPixelShaderRef& operator=(const CEComICEDXPixelShaderRef& other)	{ return operator=(other._iDXPixelShader); }

	CEComICEDXPixelShaderRef& operator=(const ICEDXPixelShader* iOther)
	{
		if (_iDXPixelShader != iOther)
		{
			if (_iDXPixelShader)
			{
				ICEDXPixelShader* tmp = _iDXPixelShader;
				_iDXPixelShader = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXPixelShader = const_cast<ICEDXPixelShader*>(iOther);
			if (_iDXPixelShader)
			{
				_iDXPixelShader->_vtbl->_addRef(_iDXPixelShader);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXPixelShader** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXPixelShaderRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXPixelShader && "has a com object reference. clear first.");
		return & _iDXPixelShader; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXPixelShaderRef& other)
	{
		bool result = false;
		if (_iDXPixelShader)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXPixelShader);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXPixelShader ? _iDXPixelShader->_vtbl->_queryInterface(_iDXPixelShader, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXPixelShader) { _iDXPixelShader->_vtbl->_addRef(_iDXPixelShader); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXPixelShader) { _iDXPixelShader->_vtbl->_release(_iDXPixelShader); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getNamedParameter(const char* pName, CEDXShaderParameter* pParamOut)	{ return _iDXPixelShader ? _iDXPixelShader->_vtbl->_getNamedParameter(_iDXPixelShader, pName, pParamOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setParameter1f(CEDXShaderParameter param, float x)	{ return _iDXPixelShader ? _iDXPixelShader->_vtbl->_setParameter1f(_iDXPixelShader, param, x) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setParameter4f(CEDXShaderParameter param, float x, float y, float z, float w)	{ return _iDXPixelShader ? _iDXPixelShader->_vtbl->_setParameter4f(_iDXPixelShader, param, x, y, z, w) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setMatrixParameter(CEDXShaderParameter param, CEDXMatrix44* pMat)	{ return _iDXPixelShader ? _iDXPixelShader->_vtbl->_setMatrixParameter(_iDXPixelShader, param, pMat) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setTexture(CEDXShaderParameter param, struct ICEDXTexture* pTex)	{ return _iDXPixelShader ? _iDXPixelShader->_vtbl->_setTexture(_iDXPixelShader, param, pTex) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult setSamplerState(CEDXShaderParameter param, eCEDXSamplerStateType state, UINT32 value)	{ return _iDXPixelShader ? _iDXPixelShader->_vtbl->_setSamplerState(_iDXPixelShader, param, state, value) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXPixelShader* _iDXPixelShader;
};

/*! @}
 * end of CEComICEDXPixelShaderRef
 */

/*! \defgroup CEComICEDXVertexDeclarationRef CEComICEDXVertexDeclarationRef
 * @{
 */

class CEComICEDXVertexDeclarationRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXVertexDeclarationRef() : _iDXVertexDeclaration(0) {}
	CEComICEDXVertexDeclarationRef(ICEDXVertexDeclaration* iOther) : _iDXVertexDeclaration(0)
	{
		if (iOther)
		{
			_iDXVertexDeclaration = iOther;
			_iDXVertexDeclaration->_vtbl->_addRef(_iDXVertexDeclaration);
		}
	}

	CEComICEDXVertexDeclarationRef(const CEComICEDXVertexDeclarationRef& other) : _iDXVertexDeclaration(0)
	{
		if (other._iDXVertexDeclaration)
		{
			_iDXVertexDeclaration = other._iDXVertexDeclaration;
			_iDXVertexDeclaration->_vtbl->_addRef(_iDXVertexDeclaration);
		}
	}

	~CEComICEDXVertexDeclarationRef()
	{
		if (_iDXVertexDeclaration)
		{
			ICEDXVertexDeclaration* tmp = _iDXVertexDeclaration;
			_iDXVertexDeclaration = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXVertexDeclaration* iDXVertexDeclaration;
				void* _ptr;
			} uIntf;
			uIntf.iDXVertexDeclaration = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXVertexDeclaration, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXVertexDeclaration)
				{
					ICEDXVertexDeclaration* tmp = _iDXVertexDeclaration;
					_iDXVertexDeclaration = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXVertexDeclaration = uIntf.iDXVertexDeclaration;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXVertexDeclaration* object() const	{ return _iDXVertexDeclaration; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXVertexDeclaration* iOther)
	{
		if (_iDXVertexDeclaration)
		{
			ICEDXVertexDeclaration* tmp = _iDXVertexDeclaration;
			_iDXVertexDeclaration = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXVertexDeclaration = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXVertexDeclaration* detach()
	{
		ICEDXVertexDeclaration* iIntf = _iDXVertexDeclaration;
		_iDXVertexDeclaration = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXVertexDeclaration* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXVertexDeclaration;
		if (_iDXVertexDeclaration)
		{
			_iDXVertexDeclaration->_vtbl->_addRef(_iDXVertexDeclaration);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXVertexDeclaration*() const	{ return _iDXVertexDeclaration; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXVertexDeclaration& operator*() const	{ return *_iDXVertexDeclaration; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXVertexDeclaration == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXVertexDeclaration* iOther) const	{ return (_iDXVertexDeclaration != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXVertexDeclaration* iOther) const	{ return (_iDXVertexDeclaration == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXVertexDeclarationRef& operator=(const CEComICEDXVertexDeclarationRef& other)	{ return operator=(other._iDXVertexDeclaration); }

	CEComICEDXVertexDeclarationRef& operator=(const ICEDXVertexDeclaration* iOther)
	{
		if (_iDXVertexDeclaration != iOther)
		{
			if (_iDXVertexDeclaration)
			{
				ICEDXVertexDeclaration* tmp = _iDXVertexDeclaration;
				_iDXVertexDeclaration = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXVertexDeclaration = const_cast<ICEDXVertexDeclaration*>(iOther);
			if (_iDXVertexDeclaration)
			{
				_iDXVertexDeclaration->_vtbl->_addRef(_iDXVertexDeclaration);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXVertexDeclaration** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXVertexDeclarationRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXVertexDeclaration && "has a com object reference. clear first.");
		return & _iDXVertexDeclaration; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXVertexDeclarationRef& other)
	{
		bool result = false;
		if (_iDXVertexDeclaration)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXVertexDeclaration);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXVertexDeclaration ? _iDXVertexDeclaration->_vtbl->_queryInterface(_iDXVertexDeclaration, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXVertexDeclaration) { _iDXVertexDeclaration->_vtbl->_addRef(_iDXVertexDeclaration); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXVertexDeclaration) { _iDXVertexDeclaration->_vtbl->_release(_iDXVertexDeclaration); } }


private:
	ICEDXVertexDeclaration* _iDXVertexDeclaration;
};

/*! @}
 * end of CEComICEDXVertexDeclarationRef
 */

/*! \defgroup CEComICEDXConfigRef CEComICEDXConfigRef
 * @{
 */

class CEComICEDXConfigRef
{
public:
	//----------------------------------------------------------------
	// constructor / destructor.
	//----------------------------------------------------------------
	CEComICEDXConfigRef() : _iDXConfig(0) {}
	CEComICEDXConfigRef(ICEDXConfig* iOther) : _iDXConfig(0)
	{
		if (iOther)
		{
			_iDXConfig = iOther;
			_iDXConfig->_vtbl->_addRef(_iDXConfig);
		}
	}

	CEComICEDXConfigRef(const CEComICEDXConfigRef& other) : _iDXConfig(0)
	{
		if (other._iDXConfig)
		{
			_iDXConfig = other._iDXConfig;
			_iDXConfig->_vtbl->_addRef(_iDXConfig);
		}
	}

	~CEComICEDXConfigRef()
	{
		if (_iDXConfig)
		{
			ICEDXConfig* tmp = _iDXConfig;
			_iDXConfig = 0;
			tmp->_vtbl->_release(tmp);
		}
	}

	//----------------------------------------------------------------
	// initialize instance which uses queryInterface().
	//   Warning: this method does not increment the reference count.
	//----------------------------------------------------------------
	CEHResult initByQueryInterface(void* iIn)
	{
		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		if (iIn)
		{
			// explicit type-punning to notify aliasing to compiler
			union
			{
				ICEDXConfig* iDXConfig;
				void* _ptr;
			} uIntf;
			uIntf.iDXConfig = 0;
			hr = reinterpret_cast<ICEUnknown*>(iIn)->_vtbl->_queryInterface(reinterpret_cast<ICEUnknown*>(iIn), CEComIID_ICEDXConfig, &uIntf._ptr);
			if (CESucceeded(hr))
			{
				if (_iDXConfig)
				{
					ICEDXConfig* tmp = _iDXConfig;
					_iDXConfig = 0;
					tmp->_vtbl->_release(tmp);
				}
				_iDXConfig = uIntf.iDXConfig;
			}
		}
		else
		{
			hr = CE_SILK_ERR_BADARGS;
		}
		return hr;
	}

	//----------------------------------------------------------------
	// get the interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG ICEDXConfig* object() const	{ return _iDXConfig; }

	//----------------------------------------------------------------
	// attach this smart pointer to an existing interface.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	void attach(ICEDXConfig* iOther)
	{
		if (_iDXConfig)
		{
			ICEDXConfig* tmp = _iDXConfig;
			_iDXConfig = 0;
			tmp->_vtbl->_release(tmp);
		}
		_iDXConfig = iOther;
	}

	//----------------------------------------------------------------
	// detach the interface pointer from this.
	//   Warning: this method does not change the reference count.
	//----------------------------------------------------------------
	ICEDXConfig* detach()
	{
		ICEDXConfig* iIntf = _iDXConfig;
		_iDXConfig = 0;
		return iIntf;
	}

	//----------------------------------------------------------------
	// copy this to an existing interface pointer holder.
	//----------------------------------------------------------------
	CEHResult copyTo(ICEDXConfig* *const iIntfOut)
	{
		if (!iIntfOut)
		{
			return CE_SILK_ERR_BADARGS;
		}
		*iIntfOut = _iDXConfig;
		if (_iDXConfig)
		{
			_iDXConfig->_vtbl->_addRef(_iDXConfig);
		}
		return CE_S_OK;
	}

	//----------------------------------------------------------------
	// operator overwrite.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG operator ICEDXConfig*() const	{ return _iDXConfig; }
	FORCEINLINE_WITHOUT_DEBUG ICEDXConfig& operator*() const	{ return *_iDXConfig; }
	FORCEINLINE_WITHOUT_DEBUG bool operator!() const	{ return (_iDXConfig == 0); }
	FORCEINLINE_WITHOUT_DEBUG bool operator!=(ICEDXConfig* iOther) const	{ return (_iDXConfig != iOther); }
	FORCEINLINE_WITHOUT_DEBUG bool operator==(ICEDXConfig* iOther) const	{ return (_iDXConfig == iOther); }
	FORCEINLINE_WITHOUT_DEBUG CEComICEDXConfigRef& operator=(const CEComICEDXConfigRef& other)	{ return operator=(other._iDXConfig); }

	CEComICEDXConfigRef& operator=(const ICEDXConfig* iOther)
	{
		if (_iDXConfig != iOther)
		{
			if (_iDXConfig)
			{
				ICEDXConfig* tmp = _iDXConfig;
				_iDXConfig = 0;
				tmp->_vtbl->_release(tmp);
			}

			_iDXConfig = const_cast<ICEDXConfig*>(iOther);
			if (_iDXConfig)
			{
				_iDXConfig->_vtbl->_addRef(_iDXConfig);
			}
		}
		return *this;
	}

	FORCEINLINE_WITHOUT_DEBUG ICEDXConfig** operator&() 
	{
		// operator& must be used for [out] pointer.
		// But, If this object has a reference to the com object, 
		// the reference will be leaked. So, clear the reference first:
		// 
		// CEComICEDXConfigRef foo;
		// clazz.createInstance(&foo);  //OK
		// clazz.createInstance(&foo);  //NG (assert).
		// foo = 0;
		// clazz.createInstance(&foo);  //OK
		//CEASSERT(!_iDXConfig && "has a com object reference. clear first.");
		return & _iDXConfig; 
	}
protected:
	void* operator new(size_t) throw()	{ return 0; }
	void operator delete(void*)	{}
	void* operator new[](size_t) throw()	{ return 0; }
#if (__GNUC__ == 2)
public:
#endif
	void operator delete[](void*)	{}

public:
	bool compareICEUnknown(CEComICEDXConfigRef& other)
	{
		bool result = false;
		if (_iDXConfig)
		{
			CEComICEUnknownRef unknown;
			CEHResult hr = unknown.initByQueryInterface(_iDXConfig);
			if(CESucceeded(hr))
			{
				CEComICEUnknownRef otherUnknown;
				hr = otherUnknown.initByQueryInterface(other);
				if (CESucceeded(hr))
				{
					result = (unknown == otherUnknown);
				}
			}
		}
		else
		{
			result = (other == NULL);
		}
		return result;
	}
public:
	//----------------------------------------------------------------
	// interface methods.
	//----------------------------------------------------------------
	FORCEINLINE_WITHOUT_DEBUG CEHResult queryInterface(const UINT32 iId, void* *const iOut)	{ return _iDXConfig ? _iDXConfig->_vtbl->_queryInterface(_iDXConfig, iId, iOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG void addRef()	{ if (_iDXConfig) { _iDXConfig->_vtbl->_addRef(_iDXConfig); } }

	FORCEINLINE_WITHOUT_DEBUG void release()	{ if (_iDXConfig) { _iDXConfig->_vtbl->_release(_iDXConfig); } }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getContext2DType(struct ICESymbol* *const pISymbolDX2DOut)	{ return _iDXConfig ? _iDXConfig->_vtbl->_getContext2DType(_iDXConfig, pISymbolDX2DOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult visualizeDirtyRegion(bool* pOut)	{ return _iDXConfig ? _iDXConfig->_vtbl->_visualizeDirtyRegion(_iDXConfig, pOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getLevelCorrespondLayerIdx(eCEDpyWindowLevel level, UINT32 *const pIdxOut)	{ return _iDXConfig ? _iDXConfig->_vtbl->_getLevelCorrespondLayerIdx(_iDXConfig, level, pIdxOut) : CE_SILK_ERR_UNINITIALIZED; }

	FORCEINLINE_WITHOUT_DEBUG CEHResult getHWPlaneEmulationConfigs(UINT32 *const pPlanesCountOut, CEPixelFormat* *const pixelFormatsOut)	{ return _iDXConfig ? _iDXConfig->_vtbl->_getHWPlaneEmulationConfigs(_iDXConfig, pPlanesCountOut, pixelFormatsOut) : CE_SILK_ERR_UNINITIALIZED; }


private:
	ICEDXConfig* _iDXConfig;
};

/*! @}
 * end of CEComICEDXConfigRef
 */

#endif // __cplusplus

#endif
