///////////////////////////////////////////////////////////////////////////////
// Copyright     2011, 2012 Sony Corporation
// Copyright (C) 2012 Sony Computer Entertainment Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
// 
///////////////////////////////////////////////////////////////////////////////
#include "WebKitSilkPrefix.h"
#include "JSDOMWindow.h"
#include "CEJSSessionStorage.h"
#include "ICEHtmlBrowserManager.h"

#include <runtime/JSLock.h>
#include "UString.h"
#include "PrototypeFunction.h"
#include "runtime/JSValue.h"
#include "runtime/FunctionConstructor.h"
#include "../JSUtilityFunc.h"
#include "JSDOMBinding.h"
#include <runtime/JSGlobalObject.h>
#include <runtime/ObjectPrototype.h>

#include "CEWebKit.h"
#include "ICEHtmlWebKit.h"
#include "CEJSPrivilegeSSObjectDeclaration.h"
#include "CEJSPrivilegeSSObject.h"
#include "CEJSWindowSessionStorage.h"
#include "CEHtmlCredentialGlue.h"

///////////////////////////////////////////////////////////////////////////
// public function
///////////////////////////////////////////////////////////////////////////
void AddWindowSessionStorageJavascriptInterface(WebCore::JSDOMWindow* window)
{
	JSC::JSLock lock(false);
	JSC::ExecState* exec = window->globalExec();
	JSC::JSObject* jsss = new (exec) JSSessionStorage(window, JSSessionStorage::createStructure(JSSessionStorage::createPrototype(exec)));
	if( jsss )
	{
		addStaticProperty(window, JSC::Identifier(exec, "sessionStorage"), jsss);
	}
}

//////////////////////////////////////////////////////////////////////////////
// CEJSSessionStorage
// this is for window.SessionStorage object
//////////////////////////////////////////////////////////////////////////////
JSSessionStorage::JSSessionStorage(WebCore::JSDOMWindow* window, PassRefPtr<JSC::Structure> structure) 
	: JSC::JSObject(structure)
{
	CEASSERT(window);
	_member = new _member_def;
	_member->_window = window;
	_member->_privilegeSSObjectDeclaration = 0;

	_member->_privilegeSSOblectCollection = new CEJSPrivilegeSSObjectCollection(this);
	if( _member->_privilegeSSOblectCollection )
	{
		_member->_privilegeSSOblectCollection->init();
	}

	CEHtmlCredentialGlue* credentialGlue = new CEHtmlCredentialGlue(this);
	if (credentialGlue)
	{
		_member->_credential = CEHtmlCredentialGlue::toICEHtmlCredential(credentialGlue);
	}

#	if defined(ENABLE_TEST_CODE)
	static bool initialized = false;
	if (!initialized)
	{
		initialized = true;
		CEWebKitImpl::getInstance()->setJSPrivilegeSSObjectListener(&jsPrivilegeSSObjectListener);
	}
#	endif //defined(ENABLE_TEST_CODE)
}

JSSessionStorage::~JSSessionStorage()
{
	if( _member )
	{
		if( _member->_privilegeSSObjectDeclaration )
		{
			delete _member->_privilegeSSObjectDeclaration;
			_member->_privilegeSSObjectDeclaration = 0;
		}
		if( _member->_privilegeSSOblectCollection )
		{
			delete _member->_privilegeSSOblectCollection;
			_member->_privilegeSSOblectCollection = 0;
		}
		delete _member;
		_member = 0;
	}
}

const JSC::ClassInfo JSSessionStorage::s_info = {"sessionStorage", 0, 0, 0};

JSC::JSObject* JSSessionStorage::createPrototype(JSC::ExecState* exec)
{
	return new (exec) JSSessionStoragePrototype(JSSessionStoragePrototype::createStructure(exec->lexicalGlobalObject()->objectPrototype()));
}

bool JSSessionStorage::getOwnPropertySlot(
	JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::PropertySlot& slot)
{
	CEComDebugPrintf("JSSessionStorage::getOwnPropertySlot: called. name=%*s\n", propertyName.size(), propertyName.data());

	bool ret = false;

	// search own slot
	JSC::JSValue val = getDirect(propertyName);
	if( val == JSC::JSValue() || true == val.isUndefined() )
	{
		// need handling the name?
		if ( canGetItemsForName(exec, propertyName) )
		{
			// found. set custom getter
			JSC::JSValue ssObjOut;
			CEHResult hr = _member->_privilegeSSOblectCollection->getPrivilegeSSPropertyValue(exec, propertyName, ssObjOut);
			if( CESucceeded(hr) && ssObjOut != JSC::JSValue() )
			{
				CEComDebugPrintf("  JSSessionStorage::getOwnPropertySlot: Found it\n");
				// save it to own object
				putDirect(propertyName, ssObjOut);
				slot.setValue(ssObjOut);
				ret = true;
			}
			else
			{
				// not found
				ret = false;
			}
		}
		else
		{
			// not need handling the name.
			CEComDebugPrintf("  JSSessionStorage::getOwnPropertySlot: not need handling the name.\n");
			ret = false;
		}
	}
	else
	{
		// return own object property value
		slot.setValue(val);
		ret = true;
	}
	return(ret);
}

bool JSSessionStorage::canGetItemsForName(
	JSC::ExecState* exec, const JSC::Identifier& propertyName)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	bool isReadOnly = false;

	if( !_member->_privilegeSSOblectCollection )
	{
		return(false);
	}

	hr = _member->_privilegeSSOblectCollection->hasPrivilegeSSObject(exec, propertyName, isReadOnly);
	if (CESucceeded(hr))
	{
		return(true);
	}

	// not handle it
	return(false);
}

void JSSessionStorage::put(
	JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::JSValue value, JSC::PutPropertySlot& slot)
{
	JSC::JSValue val = value; // do not delete
	if ( canGetItemsForName(exec, propertyName) )
	{
		CEComDebugPrintf("JSSessionStorage::put: set value, name=%*s, value=%*s\n", 
			propertyName.size(), propertyName.data(),  val.toString(exec).size(), val.toString(exec).data());
		return;
	}
	else
	{
		CEComDebugPrintf("JSSessionStorage::put: set value to object. name=%*s, value=%*s\n", 
			propertyName.size(), propertyName.data(),  val.toString(exec).size(), val.toString(exec).data());
		putDirect(propertyName, value);
	}
	return;
}

CEHResult JSSessionStorage::getCredential(CEComICEHtmlCredentialRef& credential)
{
	if (_member) {
		credential = _member->_credential;
		return CE_S_OK;
	} else {
		CEASSERT(0 && "no memory or not initialized?");
		return(CE_SILK_ERR_OPERATION_FAILED);
	}
}


CEHResult JSSessionStorage::getJSPrivilegeSSObjectDeclaration(
	JSC::ExecState* exec, CEJSPrivilegeSSObjectDeclaration** pDeclarationOut)
{
	CEHResult hr = CE_S_OK;
	if (!_member->_privilegeSSObjectDeclaration)
	{
		hr = CEJSPrivilegeSSObjectDeclaration::create(exec, this, _member->_privilegeSSObjectDeclaration);
	}
	if( CESucceeded(hr) && _member->_privilegeSSObjectDeclaration )
	{
		//_member->_privilegeSSObjectDeclaration->AddRef();
		*pDeclarationOut = _member->_privilegeSSObjectDeclaration;
	}
	return hr;
}

///////////////////////////////////////////////////////////////////////////
// implement SessionStorage prototype
///////////////////////////////////////////////////////////////////////////
const JSC::ClassInfo JSSessionStoragePrototype::s_info = { "sessionStoragePrototype", 0, 0, 0 };

bool JSSessionStoragePrototype::getOwnPropertySlot(JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::PropertySlot& slot)
{
	//return WebCore::getStaticFunctionSlot<JSC::JSObject>(exec, 0, this, propertyName, slot);
	// always false.
	return(false);
}

/////////////////////////////////
// CEJSSessionStorageAppCallback implement
/////////////////////////////////
JSC::ExecState* CEJSSessionStorageAppCallback::_exec = 0;

CEHResult CEJSSessionStorageAppCallback::_getListener(CEComICEJSPrivilegeSSObjectListenerRef& listener)
{
	CEWebKitImpl* webKitImpl = CEWebKitImpl::getInstance();
	if( webKitImpl )
	{
		listener = webKitImpl->getJSPrivilegeSSObjectListener();
		CEHResult hr = (listener.object() != 0 ? CE_S_OK : CE_SILK_ERR_OPERATION_FAILED );
		return hr;
	}
	else
	{
		return(CE_SILK_ERR_OPERATION_FAILED);
	}
}

CEHResult CEJSSessionStorageAppCallback::declarePrivilegeSSObj(
	JSC::ExecState* exec, ICEJSPrivilegeSSObjectParent* iParent, ICEJSPrivilegeSSObjectDeclarer* iDeclarer)
{
	CEComICEJSPrivilegeSSObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSSessionStorageAppCallback::_saveExecState saved(exec);
//		hr = listener.handleDeclarePrivilegeSSObject(iDeclarer);
		hr = listener.handleDeclarePrivilegeSSObjectWithParent(iParent, iDeclarer);
	}
	return hr;
}

CEHResult CEJSSessionStorageAppCallback::authorizeOfPrivilegeSSObj(
	JSC::ExecState* exec, INT32 classId, ICEHtmlCredential* iCredential, CEJSAuthStatus* statusOut, INT_PTR* objectIdOut)
{
	CEComICEJSPrivilegeSSObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSSessionStorageAppCallback::_saveExecState saved(exec);
		hr = listener.handleAuthorize(classId, iCredential, statusOut, objectIdOut);
	}
	return hr;
}

CEHResult CEJSSessionStorageAppCallback::finalizeObjectIdOfPrivilegeSSObj(
	INT_PTR objectId)
{
	CEComICEJSPrivilegeSSObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		// now finalize, don't save ExecState.
		hr = listener.handleFinalizeObjectId(objectId);
	}
	return hr;
}

CEHResult CEJSSessionStorageAppCallback::declarePropertyOfPrivilegeSSObj(
	JSC::ExecState* exec, INT_PTR objectId, ICEJSSSPropertyParent* iParent, ICEJSSSPropertyDeclarer* iDeclarer)
{
	CEComICEJSPrivilegeSSObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSSessionStorageAppCallback::_saveExecState saved(exec);
//		hr = listener.handleDeclareProperty(objectId, iDeclarer);
		hr = listener.handleDeclarePropertyWithParent(objectId, iParent, iDeclarer);
	}
	return hr;
}

CEHResult CEJSSessionStorageAppCallback::isDirtyPropertyOfPrivilegeSSObj(
	JSC::ExecState* exec, INT_PTR objectId, INT32 propertyId, CEJSIsDirtyStatus* statusOut)
{
	CEComICEJSPrivilegeSSObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSSessionStorageAppCallback::_saveExecState saved(exec);
		hr = listener.handleIsDirtyProperty(objectId, propertyId, statusOut);
	}
	return hr;
}

CEHResult CEJSSessionStorageAppCallback::getPropertyOfPrivilegeSSObj(
	JSC::ExecState* exec, INT_PTR objectId, INT32 propertyId, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	CEComICEJSPrivilegeSSObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSSessionStorageAppCallback::_saveExecState saved(exec);
		hr = listener.handleGetProperty(objectId, propertyId, iFactory, iValueOut);
	}
	return hr;
}

CEHResult CEJSSessionStorageAppCallback::setPropertyOfPrivilegeSSObj(
	JSC::ExecState* exec, INT_PTR objectId, INT32 propertyId, ICEJSExtVariant* iValue, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	CEComICEJSPrivilegeSSObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSSessionStorageAppCallback::_saveExecState saved(exec);
		hr = listener.handleSetProperty(objectId, propertyId, iValue, iFactory, iValueOut);
	}
	return hr;
}

CEHResult CEJSSessionStorageAppCallback::invokeMethodOfPrivilegeSSObj(
	JSC::ExecState* exec, INT_PTR objectId, INT32 methodId, ICEJSExtVariant** iArgValues, UINT32 argCount, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut)
{
	CEComICEJSPrivilegeSSObjectListenerRef listener;
	CEHResult hr = _getListener(listener);
	if (CESucceeded(hr))
	{
		CEJSSessionStorageAppCallback::_saveExecState saved(exec);
		hr = listener.handleInvokeMethod(objectId, methodId, iArgValues, argCount, iFactory, iValueOut);
	}
	return hr;
}

