///////////////////////////////////////////////////////////////////////////////
// Copyright     2009, 2012 Sony Corporation
// Copyright (C) 2012 Sony Computer Entertainment Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
// 
///////////////////////////////////////////////////////////////////////////////
#ifndef CEJSPrivilegeSSObject_h
#define CEJSPrivilegeSSObject_h

#include "runtime/JSObject.h"
#include "UString.h"
#include "PrototypeFunction.h"
#include "ceuhashtableT.h"
#include "ICEJSSSPropertyDeclarer.h"
#include "ICEJSPrivilegeSSObjectListener.h"
#include "CEComAtomicSupport.h"

///////////////////////////////////////////////////////////////////////////////
// CEJSPrivilegeSSObject
// this is for window.SessionStorage.foo object as application interface
///////////////////////////////////////////////////////////////////////////////
class JSPrivilegeSSObject;
class CEJSPrivilegeSSObject
{
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	CEJSPrivilegeSSObject()
	{
		_jsObject = 0;
		_refCount = 0;
	}
	~CEJSPrivilegeSSObject()
	{
		_jsObject = 0;
		CEASSERT(_refCount == 0);
	}
	void _init(JSPrivilegeSSObject* jsObject)
	{
		CEASSERT(jsObject);
		_jsObject = jsObject;
	}

public:
	void AddRef()
	{
		CEComAtomicIncrement(_refCount);
	}

	void Release()
	{
		CEATOMIC refCount = CEComAtomicDecrement(_refCount);
		if (refCount == 0)
		{
			delete this;
		}
	}
private:
	CEATOMIC _refCount;

	///////////////////////////////////
	//  ICEJSExtVariantFactory methods
	///////////////////////////////////
public:
	CEHResult createFromUndefined(ICEJSExtVariant** pVariantOut);
	CEHResult createFromNull(ICEJSExtVariant** pVariantOut);
	CEHResult createFromBoolean(UINT8 booleanValue, ICEJSExtVariant** pVariantOut);
	CEHResult createFromInt32(INT32 int32Value, ICEJSExtVariant** pVariantOut);
	CEHResult createFromNumber(double doubleValue, ICEJSExtVariant** pVariantOut);
	CEHResult createFromString(const UTF16CHAR* pCharArray16, UINT32 numOfChars16, ICEJSExtVariant** pVariantOut);

protected:
	// ICEJSExtVariantFactory vptr
	const void* _variantFactoryVptr;

	/////////////////////////////////////
	//  ICEJSSSPropertyDeclarer methods
	/////////////////////////////////////
public:
	CEHResult declareProperty(INT32 propertyId, const UTF16CHAR* pPropertyName, UINT32 numOfChars16, UINT32 attr);
	CEHResult declareMethod(INT32 propertyId, const UTF16CHAR* pPropertyName, UINT32 numOfChars16, UINT32 attr, UINT32 argCount);

	CEHResult declareChildProperty(INT32 propertyId, const UTF16CHAR* pPropertyName, UINT32 numOfChars16, UINT32 attr);
	CEHResult declareChildMethod(INT32 propertyId, const UTF16CHAR* pPropertyName, UINT32 numOfChars16, UINT32 attr, UINT32 argCount);
protected:
	// ICEJSSSPropertyDeclarer vptr
	const void* _propertyDeclarerVptr;
	// ICEJSSSPropertyParent vptr
	const void* _propertyParentVptr;

private:
	JSPrivilegeSSObject* _jsObject;
};

///////////////////////////////////////////////////////////////////////////////
// JSPrivilegeSSObject
// this is for window.SessionStorage.foo object
///////////////////////////////////////////////////////////////////////////////
class JSPrivilegeSSObjectPrototype;
class JSSessionStorage;
class CEJSPrivilegeSSObjectGlue;

class JSPrivilegeSSObject : public JSC::JSObject
{
	friend class CEJSPrivilegeSSObject;
public:
	explicit JSPrivilegeSSObject(PassRefPtr<JSC::Structure> structure);
	virtual ~JSPrivilegeSSObject();

	static JSC::JSObject* createPrototype(JSC::ExecState* exec);

	virtual bool getOwnPropertySlot(JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::PropertySlot& slot);
	virtual void put(JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::JSValue value, JSC::PutPropertySlot& slot);

	virtual const JSC::ClassInfo* classInfo() const { return &s_info; }
	static const JSC::ClassInfo s_info;

	static PassRefPtr<JSC::Structure> createStructure(JSC::JSValue prototype)
	{
		return JSC::Structure::create(prototype, JSC::TypeInfo(JSC::ObjectType));
	}

	static CEHResult create(JSC::ExecState* exec, INT_PTR objectId, JSPrivilegeSSObject** rObjectOut);

private:
	CEHResult _init(JSC::ExecState* exec, INT_PTR objectId);
	CEHResult _shutdown();

	/////////////////////////////////
	// operations
	/////////////////////////////////
public:
	INT_PTR getObjectId() const { return _member ? _member->_objectId : 0; }

private:
	// User Defined Properties/Methods Implement
	class PropertyKey
	{
	 public:
		PropertyKey(const JSC::Identifier& iName) : _rName(iName) { }
		~PropertyKey() {} // no inherit

		UINT32 getHash() const { return _rName.ustring().rep()->hash(); }
		const JSC::Identifier& getName() const { return _rName; }

	 private:
		const JSC::Identifier _rName;
	};

	class PropertyEntry
	{
		friend class JSPrivilegeSSObject;
	 public:
		PropertyEntry(const JSC::Identifier& iName, INT32 propertyId, UINT32 attr, bool isMethod, UINT32 defaultArgCount, JSPrivilegeSSObject* pOwner)
			: _key(iName), _propertyId(propertyId), _attr(attr), _isMethod(isMethod), _defaultArgCount(defaultArgCount), _pOwner(pOwner)
		{
		}
		// no inherit
		~PropertyEntry() {}

		// callback to application
		CEHResult callbackSetProperty(JSC::ExecState* exec, JSC::JSValue setValue, JSC::JSValue& valueOut);
		CEHResult callbackGetValue(JSC::ExecState* exec, JSC::JSValue& valueOut);
		CEHResult invoke(JSC::ExecState* exec, const JSC::ArgList& arglist, JSC::JSValue& valueOut);

		UINT32 getDefaultArgCount() const { return _defaultArgCount; }
		bool isReadOnly() const { return _attr & CEJSSSPropertyAttr_ReadOnly; }
		bool isMethod() const { return _isMethod; }

		// hash implementation
		const PropertyKey& getKey() const { return _key; }
		bool isEqual(const PropertyKey& key) const { return _key.getName() == key.getName(); }

	 private:
		PropertyKey _key;
		INT32 _propertyId;
		UINT32 _attr;
		bool _isMethod : 1;
		UINT32 _defaultArgCount;
		JSPrivilegeSSObject* _pOwner;
	};
private:
	CEJSPrivilegeSSObjectGlue* getGlueObject()
	{
		return( _member ? _member->_glueObject : 0 );
	}

	// add/get User Defined Properties/Methods.
private:
	CEHResult addUserPropertyEntry(JSC::Identifier& idt, INT32 propertyId, UINT32 attr);
	CEHResult addUserMethodEntry(JSC::ExecState* exec, JSC::Identifier& idt, INT32 propertyId, UINT32 attr, UINT32 argCount);
	bool canGetItemsForName(const JSC::Identifier&, PropertyEntry*& pEntry);
public:
	CEHResult invokeUserMethod(JSC::ExecState* exec, const JSC::Identifier& idt, const JSC::ArgList& arglist, JSC::JSValue& rValue);
	CEHResult callbackUserGetValue(JSC::ExecState* exec, JSC::JSValue& valueOut);
	void callbackEnable() { _isCallback = true; }
	void setPropertyId(INT32 propertyId) { _propertyId = propertyId; }
private:
	// the heap couldn't allocate big class.
	struct _member_def {
		CEJSPrivilegeSSObjectGlue* _glueObject;
		INT_PTR _objectId;
		CEUHashtableT<PropertyEntry, PropertyKey> _propertyHash;
		bool _isDeclared : 1;
	};
	_member_def* _member;
	bool _isCallback;
	INT32 _propertyId;
};


//
// class JSPrivilegeSSObjectPrototype
//
class JSPrivilegeSSObjectPrototype : public JSC::JSObject {
public:
	//static JSC::JSObject* self(JSC::ExecState*);
	virtual const JSC::ClassInfo* classInfo() const { return &s_info; }
	static const JSC::ClassInfo s_info;
	virtual bool getOwnPropertySlot(JSC::ExecState*, const JSC::Identifier&, JSC::PropertySlot&);
	static PassRefPtr<JSC::Structure> createStructure(JSC::JSValue prototype)
	{
		return JSC::Structure::create(prototype, JSC::TypeInfo(JSC::ObjectType));
	}
	JSPrivilegeSSObjectPrototype(PassRefPtr<JSC::Structure> structure) : JSC::JSObject(structure) { }
};


///////////////////////////////////////////////////////////////////////////////
// CEJSPrivilegeSSObjectCollection
///////////////////////////////////////////////////////////////////////////////
class CEJSPrivilegeSSObjectCollection
{
public:
	CEALLOCATORS;

	CEJSPrivilegeSSObjectCollection(JSSessionStorage* owner);
	~CEJSPrivilegeSSObjectCollection();

	CEHResult init();
	CEHResult shutdown();

	/////////////////////////////////
	// operation
	/////////////////////////////////
public:
	CEHResult hasPrivilegeSSObject(JSC::ExecState* exec, const JSC::Identifier& iPropName, bool& isReadOnly);
	CEHResult getPrivilegeSSObject(JSC::ExecState* exec, const JSC::Identifier& iPropName, JSC::JSValue& valueOut);
	CEHResult getPrivilegeSSPropertyValue(JSC::ExecState* exec, const JSC::Identifier& iPropName, JSC::JSValue& valueOut);

private:
	class PrivilegeSSObjectKey
	{
	 public:
		CEALLOCATORS;

		PrivilegeSSObjectKey(INT32 classId) : _classId(classId) {}
		~PrivilegeSSObjectKey() {} // no inherit

		UINT32 getHash() const { return static_cast<UINT32>(_classId); }
		INT32 getClassId() const { return _classId; }

	 private:
		INT32 _classId;
	};

	class PrivilegeSSObjectEntry
	{
	 public:
		CEALLOCATORS;

		PrivilegeSSObjectEntry(INT32 classId) : _key(classId), _authStatus(CEJSAuthStatus_UnauthorizedTemporarily), _rObject(0) {}
		~PrivilegeSSObjectEntry() {} // no inherit

		CEHResult getObject(JSC::ExecState* exec, JSSessionStorage* owner, JSC::JSValue& valueOut);
		CEHResult getProperty(JSC::ExecState* exec, JSSessionStorage* owner, JSC::JSValue& valueOut);

		// hash implementation
		const PrivilegeSSObjectKey& getKey() const { return _key; }
		bool isEqual(const PrivilegeSSObjectKey& key) const { return _key.getClassId() == key.getClassId(); }

	 private:
		PrivilegeSSObjectKey _key;
		CEJSAuthStatus _authStatus;
		JSPrivilegeSSObject* _rObject;
	};

private:
	JSSessionStorage* _owner;
	CEUHashtableT<PrivilegeSSObjectEntry, PrivilegeSSObjectKey> _privilegeSSObjectHash;
};

#endif
