///////////////////////////////////////////////////////////////////////////////
// Copyright     2009, 2012 Sony Corporation
// Copyright (C) 2012 Sony Computer Entertainment Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
// 
///////////////////////////////////////////////////////////////////////////////
#include "WebKitSilkPrefix.h"
#include "JSDOMWindow.h"
#include "CEJSSessionStorage.h"
#include "CEJSPrivilegeSSObject.h"

#include <runtime/JSLock.h>
#include "UString.h"
#include "PrototypeFunction.h"
#include "runtime/JSValue.h"
#include "runtime/FunctionConstructor.h"
#include "../JSUtilityFunc.h"
#include "JSDOMBinding.h"
#include <runtime/JSGlobalObject.h>
#include <runtime/ObjectPrototype.h>

#include "CEJSPrivilegeSSObject.h"
#include "CEJSExtVariant.h"
#include "CEJSExtVariantGlue.h"
#include "CEJSPrivilegeSSObjectDeclaration.h"
#include "CEJSPrivilegeSSObjectGlue.h"

// callback from JS engine
static JSC::JSValue JSC_HOST_CALL _invokeMethod(
	JSC::ExecState* exec, JSC::JSObject* obj, JSC::JSValue thisValue, const JSC::ArgList& arglist);

//////////////////////////////////////////////////////////////////////////////
// JSPrivilegeSSObject
//////////////////////////////////////////////////////////////////////////////
CEHResult JSPrivilegeSSObject::create(
	JSC::ExecState* exec, INT_PTR objectId, JSPrivilegeSSObject** rObjectOut)
{
	CEHResult hr = CE_S_OK;
	*rObjectOut = new (exec) JSPrivilegeSSObject(
		JSPrivilegeSSObject::createStructure(JSPrivilegeSSObject::createPrototype(exec)));

	if( *rObjectOut )
	{
		hr = (*rObjectOut)->_init(exec, objectId);
		CEASSERT(hr == CE_S_OK && "TODO why error, delete rObjectOut.");
	}
	else
	{
		hr = CE_SILK_ERR_MEMERR;
	}
	return hr;
}

JSPrivilegeSSObject::JSPrivilegeSSObject(
	PassRefPtr<JSC::Structure> structure) 
	: JSC::JSObject(structure), _member(0), _isCallback(false), _propertyId(0)
{
}

JSPrivilegeSSObject::~JSPrivilegeSSObject()
{
	_shutdown();
}

CEHResult JSPrivilegeSSObject::_init(JSC::ExecState* exec, INT_PTR objectId)
{
	CEHResult hr = CE_SILK_ERR_MEMERR;
	_member = new _member_def();
	if( _member )
	{
		_member->_glueObject = new CEJSPrivilegeSSObjectGlue();
		if( _member->_glueObject )
		{
			_member->_glueObject->_init(this);
			_member->_glueObject->AddRef();
		}
		else
		{
			return(CE_SILK_ERR_MEMERR);
		}
		_member->_objectId = objectId;
		_member->_isDeclared = false;
		hr = _member->_propertyHash.init(CEComGetAllocatorRec(), 8);

		if(CESucceeded(hr))
		{
			_member->_isDeclared = true;

			if ( false == _isCallback )
			{
				// declare properties at application
				CEJSSessionStorageAppCallback::declarePropertyOfPrivilegeSSObj(exec, _member->_objectId,
											       CEJSPrivilegeSSObjectGlue::toICEJSSSPropertyParent(_member->_glueObject),
											       CEJSPrivilegeSSObjectGlue::toICEJSSSPropertyDeclarer(_member->_glueObject));
			}
		}
	}
	return hr;
}

CEHResult JSPrivilegeSSObject::_shutdown()
{
	CEHResult hr = CE_S_OK;

	if( _member )
	{
		_member->_glueObject->Release();
		_member->_glueObject = 0;

		CEUHashtableT<PropertyEntry, PropertyKey>::Iterator iter(_member->_propertyHash);
		PropertyEntry* pEntry;
		while ((pEntry = iter.nextElement()))
		{
			delete pEntry;
		}
		_member->_propertyHash.shutdown();
		
		CEJSSessionStorageAppCallback::finalizeObjectIdOfPrivilegeSSObj(_member->_objectId);
	}
	delete _member;
	_member = 0;
	return hr;
}

const JSC::ClassInfo JSPrivilegeSSObject::s_info = {"JSPrivilegeSSObject", 0, 0, 0};

JSC::JSObject* JSPrivilegeSSObject::createPrototype(JSC::ExecState* exec)
{
	return new (exec) JSPrivilegeSSObjectPrototype(JSPrivilegeSSObjectPrototype::createStructure(exec->lexicalGlobalObject()->objectPrototype()));
}

CEHResult JSPrivilegeSSObject::callbackUserGetValue(
	JSC::ExecState* exec, JSC::JSValue& valueOut)
{
	CEHResult hr = CE_S_OK;

	valueOut = JSC::JSValue();
	if( _member->_glueObject == 0 )
	{
		CEASSERT(0 && "why?");
		return(CE_SILK_ERR_OPERATION_FAILED);
	}

	// get property
	ICEJSExtVariant* iJSExtVariant = 0;
	hr = CEJSSessionStorageAppCallback::getPropertyOfPrivilegeSSObj(exec, _member->_objectId, _propertyId,
			CEJSPrivilegeSSObjectGlue::toICEJSExtVariantFactory(_member->_glueObject), &iJSExtVariant);
	if (CESucceeded(hr))
	{
		if (iJSExtVariant)
		{
			CEComICEJSExtVariantRef rJSExtVariant = iJSExtVariant;
			valueOut = CEJSExtVariantGlue::toSubstance(iJSExtVariant)->getPropertyValue();
		}
		else
		{
			hr = CE_SILK_ERR_OPERATION_FAILED;
		}
	}
    return hr;
}

bool JSPrivilegeSSObject::getOwnPropertySlot(
	JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::PropertySlot& slot)
{
	CEComDebugPrintf("JSPrivilegeSSObject::getOwnPropertySlot: called. name=%*s\n", propertyName.size(), propertyName.data());

	CEHResult hr = CE_S_OK;
	bool ret = false;
	PropertyEntry* pEntry = NULL;
	JSC::JSValue localVal = getDirect(propertyName);
	bool bNeedCallback = canGetItemsForName(propertyName, pEntry);
	if ( false == bNeedCallback )
	{
		// return local.
		if( localVal != JSC::JSValue() )
		{
			slot.setValue(localVal);
			ret = true;
		}
	}
	else if( pEntry && pEntry->isMethod() && localVal.isObject() )
	{
		// there is method in local.
		slot.setValue(localVal);
		ret = true;
	}
	else
	{
		// check local and get from callbacks if need.
		if( localVal != JSC::JSValue() )
		{
			// already have. but is it need update?
			if( !(pEntry->_attr & CEJSSSPropertyAttr_Const) )
			{
				CEJSIsDirtyStatus status;
				hr = CEJSSessionStorageAppCallback::isDirtyPropertyOfPrivilegeSSObj(exec, getObjectId(), pEntry->_propertyId, &status);
				if (CESucceeded(hr))
				{
					if( status == CEJSIsDirtyStatus_Dirty )
					{
						// value is dirty. get from callbacks again.
						localVal = JSC::JSValue();
					}
					else
					{
						// not dirty. return localVal
						slot.setValue(localVal);
						ret = true;
					}
				}
			}
		}
		if( localVal == JSC::JSValue() )
		{
			// get from callbacks
			hr = pEntry->callbackGetValue(exec, localVal);
			if(CESucceeded(hr))
			{
				CEComDebugPrintf("  Found it. %hs=[%*s]\n", propertyName.ascii(), localVal.toString(exec).size(), localVal.toString(exec).data());
				// save local
				putDirect(propertyName, localVal);
				slot.setValue(localVal);
				ret = true;
			}
		}
		else
		{
			// return local value
			slot.setValue(localVal);
			ret = true;
		}
	}

	if( ret == true && localVal == JSC::JSValue() )
	{
		CEASSERT(0 && "why?");
		ret = false;
	}
	return(ret);
}

bool JSPrivilegeSSObject::canGetItemsForName(
	const JSC::Identifier& propertyName, PropertyEntry*& pEntry)
{
	if( _member == 0 )
	{
		return(false);
	}

	CEASSERT(_member->_isDeclared);

	PropertyKey key(propertyName);
	pEntry = NULL;
	if (_member->_propertyHash.find(key, pEntry))
	{
		return(true);
	}
	else
	{
		return(false);
	}
}

void JSPrivilegeSSObject::put(
	JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::JSValue value, JSC::PutPropertySlot& slot)
{
	JSC::JSValue val = value; // do not delete
	PropertyEntry* pEntry = NULL;
	if ( false == canGetItemsForName(propertyName, pEntry) )
	{
		CEComDebugPrintf("JSPrivilegeSSObject::put: to object, name=%*s, value=%*s\n", 
			propertyName.size(), propertyName.data(),  val.toString(exec).size(), val.toString(exec).data());

		// put to object
		putDirect(propertyName, val);
	}
	else
	{
		CEComDebugPrintf("JSPrivilegeSSObject::put: set value, name=%*s, value=%*s\n", 
			propertyName.size(), propertyName.data(),  val.toString(exec).size(), val.toString(exec).data());

		CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
		CEASSERT(pEntry);
		if( _member && pEntry )
		{
			if (pEntry->isReadOnly())
			{
				hr = CE_S_OK;
			}
			else
			{
				JSC::JSValue valueOut;
				hr = pEntry->callbackSetProperty(exec, val, valueOut);
				if(CESucceeded(hr))
				{
					// set to local property
					putDirect(propertyName, valueOut);
				}
			}
		}
	}
	return;
}

CEHResult JSPrivilegeSSObject::addUserPropertyEntry(
	JSC::Identifier& idt, INT32 propertyId, UINT32 attr)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if( _member )
	{
		PropertyEntry* pEntry = NULL;
		if( false == canGetItemsForName(idt, pEntry) )
		{
			pEntry = new PropertyEntry(idt, propertyId, attr, false, 0, this);
			if (pEntry)
			{
				hr = _member->_propertyHash.insert(pEntry);
			}
			else
			{
				hr = CE_SILK_ERR_MEMERR;
			}
		}
		else
		{
			hr = CE_SILK_ERR_OPERATION_FAILED;
		}
	}
	return hr;
}

CEHResult JSPrivilegeSSObject::addUserMethodEntry(
	JSC::ExecState* exec, JSC::Identifier& idt, INT32 propertyId, UINT32 attr, UINT32 argCount)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if ( _member )
	{
		PropertyEntry* pEntry = NULL;
		if( false == canGetItemsForName(idt, pEntry) )
		{
			pEntry = new PropertyEntry(idt, propertyId, attr, true, argCount, this);
			if (pEntry)
			{
				hr = _member->_propertyHash.insert(pEntry);
			}
			else
			{
				hr = CE_SILK_ERR_MEMERR;
			}
		}
		else
		{
			hr = CE_SILK_ERR_OPERATION_FAILED;
		}

		//
		// add to JS object
		// this is to call method directory.
		JSC::PrototypeFunction* propFunc = new (exec) JSC::PrototypeFunction(exec, 0, idt, _invokeMethod);
		putDirectFunction(exec, propFunc, JSC::DontEnum | JSC::ReadOnly | JSC::DontDelete);
	}
	return hr;
}

CEHResult JSPrivilegeSSObject::invokeUserMethod(
	JSC::ExecState* exec, const JSC::Identifier& idt, const JSC::ArgList& arglist, JSC::JSValue& rValue)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if( _member )
	{
		PropertyKey key(idt);
		PropertyEntry* pEntry = NULL;
		if (_member->_propertyHash.find(key, pEntry))
		{
			hr = pEntry->invoke(exec, arglist, rValue);
			return hr;
		}
	}
	return hr;
}

///////////////////////////////////////////////////////////////////////////
// implement SessionStorage prototype
///////////////////////////////////////////////////////////////////////////
const JSC::ClassInfo JSPrivilegeSSObjectPrototype::s_info = { "JSPrivilegeSSObjectPrototype", 0, 0, 0 };

bool JSPrivilegeSSObjectPrototype::getOwnPropertySlot(JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::PropertySlot& slot)
{
	//return WebCore::getStaticFunctionSlot<JSC::JSObject>(exec, 0, this, propertyName, slot);
	// always false.
	return(false);
}

CEHResult JSPrivilegeSSObject::PropertyEntry::callbackSetProperty(
	JSC::ExecState* exec, JSC::JSValue setValue, JSC::JSValue& valueOut)
{
	CEJSExtVariant* pVariant;
	valueOut = JSC::JSValue();
	CEHResult hr = CEJSExtVariant::createFromPropertyValue(exec, &setValue, &pVariant);
	if (CESucceeded(hr))
	{
		// callback
		CEComICEJSExtVariantRef rJSExtVariantIn = CEJSExtVariantGlue::toICEJSExtVariant(pVariant);
		ICEJSExtVariant* iJSExtVariantOut;
		hr = CEJSSessionStorageAppCallback::setPropertyOfPrivilegeSSObj(exec, _pOwner->getObjectId(), _propertyId, rJSExtVariantIn.object(),
					CEJSPrivilegeSSObjectGlue::toICEJSExtVariantFactory(_pOwner->getGlueObject()), &iJSExtVariantOut);
		if (CESucceeded(hr))
		{
			if (iJSExtVariantOut)
			{
				// get callback func returned value.
				CEComICEJSExtVariantRef rJSExtVariantOut = iJSExtVariantOut;
				valueOut = CEJSExtVariantGlue::toSubstance(rJSExtVariantOut)->getPropertyValue();
			}
		}
	}
    return hr;
}

CEHResult JSPrivilegeSSObject::PropertyEntry::callbackGetValue(
	JSC::ExecState* exec, JSC::JSValue& valueOut)
{
	CEHResult hr = CE_S_OK;

	valueOut = JSC::JSValue();
	if( _pOwner == 0 || _pOwner->getGlueObject() == 0 )
	{
		CEASSERT(0 && "why?");
		return(CE_SILK_ERR_OPERATION_FAILED);
	}

	// get property
	ICEJSExtVariant* iJSExtVariant = 0;
	hr = CEJSSessionStorageAppCallback::getPropertyOfPrivilegeSSObj(exec, _pOwner->getObjectId(), _propertyId,
			CEJSPrivilegeSSObjectGlue::toICEJSExtVariantFactory(_pOwner->getGlueObject()), &iJSExtVariant);
	if (CESucceeded(hr))
	{
		if (iJSExtVariant)
		{
			CEComICEJSExtVariantRef rJSExtVariant = iJSExtVariant;
			valueOut = CEJSExtVariantGlue::toSubstance(iJSExtVariant)->getPropertyValue();
		}
		else
		{
			hr = CE_SILK_ERR_OPERATION_FAILED;
		}
	}
    return hr;
}

static JSC::JSValue JSC_HOST_CALL _invokeMethod(
	JSC::ExecState* exec, JSC::JSObject* obj, JSC::JSValue thisValue, const JSC::ArgList& arglist)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	JSPrivilegeSSObject* thisObj = static_cast<JSPrivilegeSSObject*>(thisValue.toObject(exec));

	JSC::PrototypeFunction* funcObj = static_cast<JSC::PrototypeFunction*>(obj);
	JSC::Identifier methodName(exec, funcObj->name(&exec->globalData()));

	CEComDebugPrintf("CEJSPrivilegeSSObject.cpp:_invokeMethod: method name=%*s\n", methodName.size(), methodName.data());

	JSC::JSValue retValue;
	hr = thisObj->invokeUserMethod(exec, methodName, arglist, retValue);

	return(retValue);
}

CEHResult JSPrivilegeSSObject::PropertyEntry::invoke(
	JSC::ExecState* exec, const JSC::ArgList& arglist, JSC::JSValue& valueOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	UINT32 argCount = arglist.size();

	ICEJSExtVariant** pArgValues = CENEW_ARRAY(ICEJSExtVariant*, argCount);
	if (pArgValues)
	{
		hr = CE_S_OK;
		for (UINT32 i = 0; i < argCount; ++i)
		{
			pArgValues[i] = 0;
			if (CESucceeded(hr))
			{
				CEJSExtVariant* pVariant;
				JSC::JSValue val = arglist.at(i);
				hr = CEJSExtVariant::createFromPropertyValue(exec, &val, &pVariant);
				if (CESucceeded(hr))
				{
					pVariant->AddRef();
					pArgValues[i] = CEJSExtVariantGlue::toICEJSExtVariant(pVariant);
				}
			}
		}

		if (CESucceeded(hr))
		{
			ICEJSExtVariant* iJSExtVariant = 0;
			hr = CEJSSessionStorageAppCallback::invokeMethodOfPrivilegeSSObj(
					exec, _pOwner->getObjectId(), _propertyId, pArgValues, argCount,
					CEJSPrivilegeSSObjectGlue::toICEJSExtVariantFactory(_pOwner->getGlueObject()), &iJSExtVariant);
			if (CESucceeded(hr))
			{
				if (iJSExtVariant)
				{
					CEComICEJSExtVariantRef rJSExtVariant = iJSExtVariant;
					valueOut = CEJSExtVariantGlue::toSubstance(iJSExtVariant)->getPropertyValue();
				}
				else
				{
					valueOut = JSC::jsUndefined();
				}
			}
		}
		for (UINT32 i = 0; i < argCount; ++i)
		{
			if (pArgValues[i])
			{
				CEJSExtVariantGlue::toSubstance(pArgValues[i])->Release();
			}
		}
		CEDELETE_ARRAY(pArgValues);
	}
	else
	{
		hr = CE_SILK_ERR_MEMERR;
	}
    return hr;
}


///////////////////////////////////////////////////////////////////////////////
// CEJSPrivilegeSSObject
// this is for window.SessionStorage.foo object as application interface
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////
//  ICEJSExtVariantFactory methods
///////////////////////////////////
CEHResult CEJSPrivilegeSSObject::createFromUndefined(ICEJSExtVariant** pVariantOut)
{
	CEJSExtVariant* pVariant;
	CEHResult hr = CEJSExtVariant::createFromUndefined(CEJSSessionStorageAppCallback::getExecState(), &pVariant);
	if (CESucceeded(hr))
	{
		*pVariantOut = CEJSExtVariantGlue::toICEJSExtVariant(pVariant);
	}
	return hr;
}

CEHResult CEJSPrivilegeSSObject::createFromNull(ICEJSExtVariant** pVariantOut)
{
	CEJSExtVariant* pVariant;
	CEHResult hr = CEJSExtVariant::createFromNull(CEJSSessionStorageAppCallback::getExecState(), &pVariant);
	if (CESucceeded(hr))
	{
		*pVariantOut = CEJSExtVariantGlue::toICEJSExtVariant(pVariant);
	}
	return hr;
}

CEHResult CEJSPrivilegeSSObject::createFromBoolean(UINT8 booleanValue, ICEJSExtVariant** pVariantOut)
{
	CEJSExtVariant* pVariant;
	CEHResult hr = CEJSExtVariant::createFromBoolean(CEJSSessionStorageAppCallback::getExecState(), booleanValue, &pVariant);
	if (CESucceeded(hr))
	{
		*pVariantOut = CEJSExtVariantGlue::toICEJSExtVariant(pVariant);
	}
	return hr;
}

CEHResult CEJSPrivilegeSSObject::createFromInt32(INT32 int32Value, ICEJSExtVariant** pVariantOut)
{
	CEJSExtVariant* pVariant;
	CEHResult hr = CEJSExtVariant::createFromInt32(CEJSSessionStorageAppCallback::getExecState(), int32Value, &pVariant);
	if (CESucceeded(hr))
	{
		*pVariantOut = CEJSExtVariantGlue::toICEJSExtVariant(pVariant);
	}
	return hr;
}

CEHResult CEJSPrivilegeSSObject::createFromNumber(double doubleValue, ICEJSExtVariant** pVariantOut)
{
	CEJSExtVariant* pVariant;
	CEHResult hr = CEJSExtVariant::createFromNumber(CEJSSessionStorageAppCallback::getExecState(), doubleValue, &pVariant);
	if (CESucceeded(hr))
	{
		*pVariantOut = CEJSExtVariantGlue::toICEJSExtVariant(pVariant);
	}
	return hr;
}

CEHResult CEJSPrivilegeSSObject::createFromString(const UTF16CHAR* pCharArray16, UINT32 numOfChars16, ICEJSExtVariant** pVariantOut)
{
	CEJSExtVariant* pVariant;
	CEHResult hr = CEJSExtVariant::createFromString(CEJSSessionStorageAppCallback::getExecState(), pCharArray16, numOfChars16, &pVariant);
	if (CESucceeded(hr))
	{
		*pVariantOut = CEJSExtVariantGlue::toICEJSExtVariant(pVariant);
	}
	return hr;
}

/////////////////////////////////////
//  ICEJSSSPropertyDeclarer methods
/////////////////////////////////////
CEHResult CEJSPrivilegeSSObject::declareProperty(
	INT32 propertyId, const UTF16CHAR* pPropertyName, UINT32 numOfChars16, UINT32 attr)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if ( _jsObject && pPropertyName && numOfChars16 > 0)
	{
		JSC::Identifier symbol = JSC::Identifier(CEJSSessionStorageAppCallback::getExecState(), reinterpret_cast<const UChar*>(pPropertyName), numOfChars16);
		hr = _jsObject->addUserPropertyEntry(symbol, propertyId, attr);
	}
	return hr;
}

CEHResult CEJSPrivilegeSSObject::declareMethod(INT32 propertyId, const UTF16CHAR* pPropertyName, UINT32 numOfChars16, UINT32 attr, UINT32 argCount)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if ( _jsObject && pPropertyName && numOfChars16 > 0)
	{
		JSC::Identifier symbol = JSC::Identifier(CEJSSessionStorageAppCallback::getExecState(), reinterpret_cast<const UChar*>(pPropertyName), numOfChars16);
		hr = _jsObject->addUserMethodEntry(CEJSSessionStorageAppCallback::getExecState(), symbol, propertyId, attr, argCount);
	}
	return hr;
}

/////////////////////////////////////
//  ICEJSSSPropertyParent methods
/////////////////////////////////////
CEHResult CEJSPrivilegeSSObject::declareChildProperty(
	INT32 propertyId, const UTF16CHAR* pPropertyName, UINT32 numOfChars16, UINT32 attr)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if ( _jsObject && pPropertyName && numOfChars16 > 0)
	{
		JSC::Identifier symbol = JSC::Identifier(CEJSSessionStorageAppCallback::getExecState(), reinterpret_cast<const UChar*>(pPropertyName), numOfChars16);
		hr = _jsObject->addUserPropertyEntry(symbol, propertyId, attr);
		if (CESucceeded(hr))
		{
			_jsObject->callbackEnable();
			_jsObject->setPropertyId(propertyId);
		}
	}
	return hr;
}

CEHResult CEJSPrivilegeSSObject::declareChildMethod(INT32 propertyId, const UTF16CHAR* pPropertyName, UINT32 numOfChars16, UINT32 attr, UINT32 argCount)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	if ( _jsObject && pPropertyName && numOfChars16 > 0)
	{
		JSC::Identifier symbol = JSC::Identifier(CEJSSessionStorageAppCallback::getExecState(), reinterpret_cast<const UChar*>(pPropertyName), numOfChars16);
		hr = _jsObject->addUserMethodEntry(CEJSSessionStorageAppCallback::getExecState(), symbol, propertyId, attr, argCount);
	}
	return hr;
}

///////////////////////////////////////////////////////////////////////////////
// CEJSPrivilegeSSObjectCollection
///////////////////////////////////////////////////////////////////////////////
CEJSPrivilegeSSObjectCollection::CEJSPrivilegeSSObjectCollection(JSSessionStorage* owner)
{
	_owner = owner;
}

CEJSPrivilegeSSObjectCollection::~CEJSPrivilegeSSObjectCollection()
{
	shutdown();
}

CEHResult CEJSPrivilegeSSObjectCollection::init()
{
	CEHResult hr = _privilegeSSObjectHash.init(CEComGetAllocatorRec(), 8);
	return hr;
}

CEHResult CEJSPrivilegeSSObjectCollection::shutdown()
{
	//CEASSERT(_isShutdownCompleted == false);
	CEHResult hr = CE_S_OK;
	CEUHashtableT<PrivilegeSSObjectEntry, PrivilegeSSObjectKey>::Iterator iter(_privilegeSSObjectHash);
	PrivilegeSSObjectEntry* pEntry;
	while ((pEntry = iter.nextElement()))
	{
		delete pEntry;
	}
	_privilegeSSObjectHash.shutdown();
	return hr;
}

////////////////////////////
// operations 
////////////////////////////
CEHResult CEJSPrivilegeSSObjectCollection::hasPrivilegeSSObject(
	JSC::ExecState* exec, const JSC::Identifier& iPropName, bool& isReadOnly)
{
    isReadOnly = false;
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	CEJSPrivilegeSSObjectDeclaration* declaration = 0;
	if( !_owner )
	{
		return(CE_SILK_ERR_OPERATION_FAILED);
	}

	hr = _owner->getJSPrivilegeSSObjectDeclaration(exec, &declaration);
	if (CESucceeded(hr) && declaration)
	{
		INT32 classId;
		UINT32 attr;
		hr = declaration->isDeclared(iPropName, &classId, &attr);
		if (CESucceeded(hr))
		{
			isReadOnly = attr & CEJSPrivilegeSSObjectAttr_ReadOnly;
		}
	}
    return hr;
}

CEHResult CEJSPrivilegeSSObjectCollection::getPrivilegeSSPropertyValue(
	JSC::ExecState* exec, const JSC::Identifier& iPropName, JSC::JSValue& valueOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	CEJSPrivilegeSSObjectDeclaration* declaration = 0;

	if( _owner == 0 )
	{
		return(CE_SILK_ERR_OPERATION_FAILED);
	}

	hr = _owner->getJSPrivilegeSSObjectDeclaration(exec, &declaration);
	if (CESucceeded(hr))
	{
		INT32 classId;
		UINT32 attr;
		hr = declaration->isDeclared(iPropName, &classId, &attr);
		if (CESucceeded(hr))
		{
			PrivilegeSSObjectKey key(classId);
			PrivilegeSSObjectEntry* pEntry = 0;
			if (!_privilegeSSObjectHash.find(key, pEntry))
			{
				pEntry = new PrivilegeSSObjectEntry(classId);
				if (pEntry)
				{
					hr = _privilegeSSObjectHash.insert(pEntry);
					if (CEFailed(hr))
					{
						delete pEntry;
						pEntry = 0;
					}
				}
				else
				{
					hr = CE_SILK_ERR_MEMERR;
				}
			}
			if (pEntry)
			{
				hr = pEntry->getProperty(exec, _owner, valueOut);
			}
		}
	}
    return hr;
}

CEHResult CEJSPrivilegeSSObjectCollection::getPrivilegeSSObject(
	JSC::ExecState* exec, const JSC::Identifier& iPropName, JSC::JSValue& valueOut)
{
	CEHResult hr = CE_SILK_ERR_OPERATION_FAILED;
	CEJSPrivilegeSSObjectDeclaration* declaration = 0;

	if( _owner == 0 )
	{
		return(CE_SILK_ERR_OPERATION_FAILED);
	}

	hr = _owner->getJSPrivilegeSSObjectDeclaration(exec, &declaration);
	if (CESucceeded(hr))
	{
		INT32 classId;
		UINT32 attr;
		hr = declaration->isDeclared(iPropName, &classId, &attr);
		if (CESucceeded(hr))
		{
			PrivilegeSSObjectKey key(classId);
			PrivilegeSSObjectEntry* pEntry = 0;
			if (!_privilegeSSObjectHash.find(key, pEntry))
			{
				pEntry = new PrivilegeSSObjectEntry(classId);
				if (pEntry)
				{
					hr = _privilegeSSObjectHash.insert(pEntry);
					if (CEFailed(hr))
					{
						delete pEntry;
						pEntry = 0;
					}
				}
				else
				{
					hr = CE_SILK_ERR_MEMERR;
				}
			}
			if (pEntry)
			{
				hr = pEntry->getObject(exec, _owner, valueOut);
			}
		}
	}
    return hr;
}

CEHResult CEJSPrivilegeSSObjectCollection::PrivilegeSSObjectEntry::getObject(
	JSC::ExecState* exec, JSSessionStorage* owner, JSC::JSValue& valueOut)
{
	CEHResult hr = CE_S_OK;
	switch (_authStatus) {
	case CEJSAuthStatus_Authorized:
		valueOut = JSC::JSValue(_rObject);
		break;
	case CEJSAuthStatus_Unauthorized:
		valueOut = JSC::jsUndefined();
		break;
	case CEJSAuthStatus_UnauthorizedTemporarily:
		{
			CEComICEHtmlCredentialRef rCredential;
			hr = owner->getCredential(rCredential);
			if (CESucceeded(hr))
			{
				INT_PTR objectId = 0;
				hr = CEJSSessionStorageAppCallback::authorizeOfPrivilegeSSObj(exec, _key.getClassId(), rCredential.object(), &_authStatus, &objectId);
				if (CESucceeded(hr) && _authStatus == CEJSAuthStatus_Authorized)
				{
					hr = JSPrivilegeSSObject::create(exec,objectId, &_rObject);
					if (CESucceeded(hr))
					{
						valueOut = JSC::JSValue(_rObject);
					}
				}
			}
		}
		break;
	}
	return hr;
}

CEHResult CEJSPrivilegeSSObjectCollection::PrivilegeSSObjectEntry::getProperty(
	JSC::ExecState* exec, JSSessionStorage* owner, JSC::JSValue& valueOut)
{
	CEHResult hr = CE_S_OK;
	switch (_authStatus) {
	case CEJSAuthStatus_Authorized:
		valueOut = JSC::JSValue(_rObject);
		break;
	case CEJSAuthStatus_Unauthorized:
		valueOut = JSC::jsUndefined();
		break;
	case CEJSAuthStatus_UnauthorizedTemporarily:
		{
			CEComICEHtmlCredentialRef rCredential;
			hr = owner->getCredential(rCredential);
			if (CESucceeded(hr))
			{
				INT_PTR objectId = 0;
				hr = CEJSSessionStorageAppCallback::authorizeOfPrivilegeSSObj(exec, _key.getClassId(), rCredential.object(), &_authStatus, &objectId);
				if (CESucceeded(hr) && _authStatus == CEJSAuthStatus_Authorized)
				{
					hr = JSPrivilegeSSObject::create(exec,objectId, &_rObject);
					if (CESucceeded(hr))
					{
						hr = _rObject->callbackUserGetValue(exec, valueOut);
					}
				}
			}
		}
		break;
	}
	return hr;
}
