///////////////////////////////////////////////////////////////////////////////
// Copyright     2009, 2012 Sony Corporation
// Copyright (C) 2012 Sony Computer Entertainment Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
// 
///////////////////////////////////////////////////////////////////////////////
#ifndef JSExternal_h
#define JSExternal_h

#include "runtime/JSObject.h"
#include "ICEHtmlBrowserManager.h"
#include "CEJSPrivilegeExtObject.h"
#include "JSDOMWindow.h"
#include "../CEPrivilegeObjectBase.h"

///////////////////////////////////////////////////////////////////////////////
// CEJSExternal
// this is for window.external object
///////////////////////////////////////////////////////////////////////////////
class JSExternalPrototype;
class CEJSPrivilegeExtObjectDeclaration;
class JSExternal : public JSC::JSObject, public PrivilegeObjectBase
{
public:
	explicit JSExternal(WebCore::JSDOMWindow* window, PassRefPtr<JSC::Structure> structure);
	virtual ~JSExternal();

	static JSC::JSObject* createPrototype(JSC::ExecState* exec);

	virtual bool getOwnPropertySlot(JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::PropertySlot& slot);
	bool canGetItemsForName(JSC::ExecState*, const JSC::Identifier&);
	virtual void put(JSC::ExecState* exec, const JSC::Identifier& propertyName, JSC::JSValue value, JSC::PutPropertySlot& slot);

	virtual const JSC::ClassInfo* classInfo() const { return &s_info; }
	static const JSC::ClassInfo s_info;

	static PassRefPtr<JSC::Structure> createStructure(JSC::JSValue prototype)
	{
		return JSC::Structure::create(prototype, JSC::TypeInfo(JSC::ObjectType));
	}

	//virtual bool deleteProperty(JSC::ExecState*, const JSC::Identifier&);
	//virtual void getPropertyNames(JSC::ExecState*, JSC::PropertyNameArray&);
	//bool customGetPropertyNames(JSC::ExecState*, JSC::PropertyNameArray&);

	//static JSC::JSValue nameGetter(JSC::ExecState*, const JSC::Identifier&, const JSC::PropertySlot&);

	WebCore::JSDOMWindow* getDOMWindow()
	{
		return(_member ? _member->_window : 0);
	}

	CEHResult getCredential(CEComICEHtmlCredentialRef& credential);

	/////////////////////////////////////////////
	// CEJSPrivilegeExtObjectDeclaration support
	/////////////////////////////////////////////
public:
	CEHResult getJSPrivilegeExtObjectDeclaration(JSC::ExecState* exec, CEJSPrivilegeExtObjectDeclaration** pDeclarationOut);
	void useDeclareObjectForParent() { _useDeclareForeParent = true; }
	bool declareObjectForParent() { return _useDeclareForeParent; }
	void clearOldDeclaredObject();
	INT32 getPropertyId();
	void setPropertyId(INT32 propertyId);
private:
	// the heap couldn't allocate big class.
	struct _member_def {
		WebCore::JSDOMWindow* _window;
		CEJSPrivilegeExtObjectDeclaration* _privilegeExtObjectDeclaration;
		CEJSPrivilegeExtObjectCollection* _privilegeExtOblectCollection;
		CEComICEHtmlCredentialRef _credential;
	};
	_member_def* _member;
	bool _useDeclareForeParent;
	INT32 _propertyId;
};

//
// class JSExternalPrototype
//
class JSExternalPrototype : public JSC::JSObject {
public:
	//static JSC::JSObject* self(JSC::ExecState*);
	virtual const JSC::ClassInfo* classInfo() const { return &s_info; }
	static const JSC::ClassInfo s_info;
	virtual bool getOwnPropertySlot(JSC::ExecState*, const JSC::Identifier&, JSC::PropertySlot&);
	static PassRefPtr<JSC::Structure> createStructure(JSC::JSValue prototype)
	{
		return JSC::Structure::create(prototype, JSC::TypeInfo(JSC::ObjectType));
	}
	JSExternalPrototype(PassRefPtr<JSC::Structure> structure) : JSC::JSObject(structure) { }
};


/////////////////////////////////
// CEJSExternalAppCallback
/////////////////////////////////
class CEJSExternalAppCallback {

private:
	CEJSExternalAppCallback();

private:
	// for save ExecState during callback application side.
	class _saveExecState
	{
		public:
		_saveExecState(JSC::ExecState* exec)
		{
			// Do not re enter.
			CEASSERT(CEJSExternalAppCallback::_exec == 0);
			CEJSExternalAppCallback::_exec = exec;
		}
		~_saveExecState()
		{
			CEASSERT(CEJSExternalAppCallback::_exec != 0);
			CEJSExternalAppCallback::_exec = 0;
		}
	};

public:
	static JSC::ExecState* getExecState()
	{
		CEASSERT(_exec && "why did not set exec?");
		return(_exec);
	}

public:
	// Bug15969 tell application some query about external object.
	static CEHResult declarePrivilegeExtObj(JSC::ExecState* exec, ICEJSPrivilegeExtObjectDeclarer* iDeclarer);
	static CEHResult declarePrivilegeExtObjWithParent(JSC::ExecState* exec, ICEJSPrivilegeExtObjectParent* iParent, ICEJSPrivilegeExtObjectDeclarer* iDeclarer);
	static CEHResult authorizeOfPrivilegeExtObj(JSC::ExecState* exec, INT32 classId, ICEHtmlCredential* iCredential, CEJSAuthStatus* statusOut, INT_PTR* objectIdOut);
	static CEHResult finalizeObjectIdOfPrivilegeExtObj(INT_PTR objectId);
	static CEHResult declarePropertyOfPrivilegeExtObj(JSC::ExecState* exec, INT_PTR objectId, ICEJSExtPropertyDeclarer* iDeclarer);
	static CEHResult declarePropertyOfPrivilegeExtObjWithParent(JSC::ExecState* exec, INT_PTR objectId, ICEJSExtPropertyParent* iParent, ICEJSExtPropertyDeclarer* iDeclarer);
	static CEHResult isDirtyPropertyOfPrivilegeExtObj(JSC::ExecState* exec, INT_PTR objectId, INT32 propertyId, CEJSIsDirtyStatus* statusOut);
	static CEHResult getPropertyOfPrivilegeExtObj(JSC::ExecState* exec, INT_PTR objectId, INT32 propertyId, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut);
	static CEHResult setPropertyOfPrivilegeExtObj(JSC::ExecState* exec, INT_PTR objectId, INT32 propertyId, ICEJSExtVariant* iValue, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut);
	static CEHResult invokeMethodOfPrivilegeExtObj(JSC::ExecState* exec, INT_PTR objectId, INT32 methodId, ICEJSExtVariant** iArgValues, UINT32 argCount, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut);

private:
	static CEHResult _getListener(CEComICEJSPrivilegeExtObjectListenerRef& listener);
	static JSC::ExecState* _exec;
};

////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////              test code              ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////
//#define ENABLE_TEST_CODE

#if defined(ENABLE_TEST_CODE)

#include "ICEHtmlCredential.h"
#include "ICEJSPrivilegeExtObjectListener.h"
#include "ICEJSPrivilegeExtObjectDeclarer.h"
#include "ICEJSExtPropertyDeclarer.h"
#include "ICEJSExtVariant.h"
#include "ICEJSExtVariantFactory.h"

////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////   base class of PrivilegeExtObject  ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////

class CEJSExtObjectBase
{
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	CEJSExtObjectBase() {}
	virtual ~CEJSExtObjectBase() {}

	virtual CEHResult declareProperty(ICEJSExtPropertyDeclarer* iDeclarer) = 0;
	virtual CEHResult getProperty(INT32 propertyId, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut) = 0;
	virtual CEHResult setProperty(INT32 propertyId, ICEJSExtVariant* iValue, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut) = 0;
	virtual CEHResult isDirtyProperty(INT32 propertyId, CEJSIsDirtyStatus* statusOut) = 0;
	virtual CEHResult invokeMethod(INT32 methodId, ICEJSExtVariant** iArgValues, UINT32 argCount, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut) = 0;

	typedef CEHResult (*AuthorizerFunc)(ICEHtmlCredential* iCredential, CEJSAuthStatus* statusOut);
	typedef CEHResult (*CreaterFunc)(CEJSExtObjectBase** objOut);

	struct ClassTemplate
	{
		const UTF16CHAR* pObjectName;
		UINT32 numOfChars16;
		UINT32 attribute;
		AuthorizerFunc authorizer;
		CreaterFunc creater;
	};

protected:
	struct PropertyTemplate
	{
		INT32 propertyId;
		const UTF16CHAR* pPropertyName;
		UINT32 numOfChars16;
		UINT32 attr;
	};

	struct MethodTemplate
	{
		INT32 methodId;
		const UTF16CHAR* pMethodName;
		UINT32 numOfChars16;
		UINT32 attr;
		UINT32 argCount;
	};
};

////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////       window.external.SampleA       ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////
//// Implemented JavaScript property :
////     window.external.SampleA.propBoolean
////     window.external.SampleA.propInt32
////     window.external.SampleA.propString
//// Implemented JavaScript method :
////     window.external.SampleA.myAdd(v0,v1)
////     window.external.SampleA.incPropInt32()
////////////////////////////////////////////////////////////////////////////////////////////////////////

class CEJSExtObjectSampleA : public CEJSExtObjectBase
{
public:
	static const ClassTemplate classTemplate;

	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	CEJSExtObjectSampleA() : _propBoolean(0), _propInt32(0), _isDirtyPropInt32(false), _propStringLen(0) {}
	virtual ~CEJSExtObjectSampleA() {}

	virtual CEHResult declareProperty(ICEJSExtPropertyDeclarer* iDeclarer);
	virtual CEHResult getProperty(INT32 propertyId, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut);
	virtual CEHResult setProperty(INT32 propertyId, ICEJSExtVariant* iValue, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut);
	virtual CEHResult isDirtyProperty(INT32 propertyId, CEJSIsDirtyStatus* statusOut);
	virtual CEHResult invokeMethod(INT32 methodId, ICEJSExtVariant** iArgValues, UINT32 argCount, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut);

private:
	static CEHResult _authorizer(ICEHtmlCredential* iCredential, CEJSAuthStatus* statusOut);
	static CEHResult _creater(CEJSExtObjectBase** objOut);
	static bool _isSafeDomain(const UTF16CHAR* pCharArray16Domain, UINT32 numOfChars16Domain);

	enum PropertyId
	{
		eProprtyId_propBoolean,
		eProprtyId_propInt32,
		eProprtyId_propString
	};

	enum MethodId
	{
		eMethodId_myAdd,
		eMethodId_incPropInt32
	};

	static const UTF16CHAR _name_SampleA[];
	static const UTF16CHAR _name_propBoolean[];
	static const UTF16CHAR _name_propInt32[];
	static const UTF16CHAR _name_propString[];
	static const UTF16CHAR _name_myAdd[];
	static const UTF16CHAR _name_incPropInt32[];
	static const PropertyTemplate _propertyTemplateTable[];
	static const MethodTemplate _methodTemplateTable[];

	UINT8 _propBoolean;
	INT32 _propInt32;
	bool _isDirtyPropInt32;
	UINT32 _propStringLen;
	UTF16CHAR _propStringValue[64];
};

////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////       window.external.SampleB       ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////
//// Implemented JavaScript property :
////     window.external.SampleB.propStringConst
////     window.external.SampleB.propNumberReadOnly
//// Implemented JavaScript method :
////     window.external.SampleB.setPropNumberReadOnly(v)
////////////////////////////////////////////////////////////////////////////////////////////////////////

class CEJSExtObjectSampleB : public CEJSExtObjectBase
{
public:
	static const ClassTemplate classTemplate;

	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	CEJSExtObjectSampleB() : _propNumberReadOnly(0.0), _isDirtyPropNumberReadOnly(false) {}
	virtual ~CEJSExtObjectSampleB() {}

	virtual CEHResult declareProperty(ICEJSExtPropertyDeclarer* iDeclarer);
	virtual CEHResult getProperty(INT32 propertyId, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut);
	virtual CEHResult setProperty(INT32 propertyId, ICEJSExtVariant* iValue, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut);
	virtual CEHResult isDirtyProperty(INT32 propertyId, CEJSIsDirtyStatus* statusOut);
	virtual CEHResult invokeMethod(INT32 methodId, ICEJSExtVariant** iArgValues, UINT32 argCount, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut);

private:
	static CEHResult _authorizer(ICEHtmlCredential* iCredential, CEJSAuthStatus* statusOut);
	static CEHResult _creater(CEJSExtObjectBase** objOut);
	static bool _isSafeDomain(const UTF16CHAR* pCharArray16Domain, UINT32 numOfChars16Domain);

	enum PropertyId
	{
		eProprtyId_propStringConst,
		eProprtyId_propNumberReadOnly
	};

	enum MethodId
	{
		eMethodId_setPropNumberReadOnly
	};

	static const UTF16CHAR _name_SampleB[];
	static const UTF16CHAR _name_propStringConst[];
	static const UTF16CHAR _name_propNumberReadOnly[];
	static const UTF16CHAR _name_setPropNumberReadOnly[];
	static const PropertyTemplate _propertyTemplateTable[];
	static const MethodTemplate _methodTemplateTable[];

	double _propNumberReadOnly;
	bool _isDirtyPropNumberReadOnly;
};


////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////// implement class of ICEJSPrivilegeExtObjectListener ////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////

class CEJSPrivilegeExtObjectListenerImpl
{
public:
	////////////////////////////////////////////////////////////////
	// operator new, delete, new[] and delete[].
	////////////////////////////////////////////////////////////////
	CEALLOCATORS;

	CEJSPrivilegeExtObjectListenerImpl() {}
	virtual ~CEJSPrivilegeExtObjectListenerImpl() {}

	CEHResult declarePrivilegeExtObject(ICEJSPrivilegeExtObjectDeclarer* iDeclarer);
	CEHResult authorize(INT32 classId, ICEHtmlCredential* iCredential, CEJSAuthStatus* statusOut, INT_PTR* objectIdOut);
	CEHResult declareProperty(INT_PTR objectId, ICEJSExtPropertyDeclarer* iDeclarer);
	CEHResult getProperty(INT_PTR objectId, INT32 propertyId, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut);
	CEHResult setProperty(INT_PTR objectId, INT32 propertyId, ICEJSExtVariant* iValue, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut);
	CEHResult isDirtyProperty(INT_PTR objectId, INT32 propertyId, CEJSIsDirtyStatus* statusOut);
	CEHResult invokeMethod(INT_PTR objectId, INT32 methodId, ICEJSExtVariant** iArgValues, UINT32 argCount, ICEJSExtVariantFactory* iFactory, ICEJSExtVariant** iValueOut);
	CEHResult finalizeObjectId(INT_PTR objectId);

private:
	CEJSExtObjectBase* _objectIdToObj(INT_PTR objectId) const { return reinterpret_cast<CEJSExtObjectBase*>(objectId); }
	INT_PTR _objToObjectId(CEJSExtObjectBase* obj) const { return reinterpret_cast<INT_PTR>(obj); }

	static const CEJSExtObjectBase::ClassTemplate* _classTemplateTable[];
};

////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////// ICEJSPrivilegeExtObjectListener I/F ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////

extern ICEJSPrivilegeExtObjectListener jsPrivilegeExtObjectListener;

#endif // defined(ENABLE_TEST_CODE)

#endif


